"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
function anyEcsService() {
    const stack = new cdk.Stack();
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDefinition');
    taskDefinition.addContainer('MainContainer', {
        image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
    });
    const vpc = new ec2.Vpc(stack, 'VPC');
    const cluster = new ecs.Cluster(stack, 'Cluster', {
        vpc,
    });
    return new ecs.FargateService(stack, 'FargateService', {
        cluster,
        taskDefinition,
    });
}
module.exports = {
    'ECS deploy Action': {
        'throws an exception if neither inputArtifact nor imageFile were provided'(test) {
            const service = anyEcsService();
            test.throws(() => {
                new cpactions.EcsDeployAction({
                    actionName: 'ECS',
                    service,
                });
            }, /one of 'input' or 'imageFile' is required/);
            test.done();
        },
        'can be created just by specifying the inputArtifact'(test) {
            const service = anyEcsService();
            const artifact = new codepipeline.Artifact('Artifact');
            test.doesNotThrow(() => {
                new cpactions.EcsDeployAction({
                    actionName: 'ECS',
                    service,
                    input: artifact,
                });
            });
            test.done();
        },
        'can be created just by specifying the imageFile'(test) {
            const service = anyEcsService();
            const artifact = new codepipeline.Artifact('Artifact');
            test.doesNotThrow(() => {
                new cpactions.EcsDeployAction({
                    actionName: 'ECS',
                    service,
                    imageFile: artifact.atPath('imageFile.json'),
                });
            });
            test.done();
        },
        'throws an exception if both inputArtifact and imageFile were provided'(test) {
            const service = anyEcsService();
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.EcsDeployAction({
                    actionName: 'ECS',
                    service,
                    input: artifact,
                    imageFile: artifact.atPath('file.json'),
                });
            }, /one of 'input' or 'imageFile' can be provided/);
            test.done();
        },
        "sets the target service as the action's backing resource"(test) {
            const service = anyEcsService();
            const artifact = new codepipeline.Artifact('Artifact');
            const action = new cpactions.EcsDeployAction({
                actionName: 'ECS',
                service,
                input: artifact
            });
            test.equal(action.actionProperties.resource, service);
            test.done();
        },
        'can be created by existing service'(test) {
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'Vpc');
            const service = ecs.FargateService.fromFargateServiceAttributes(stack, 'FargateService', {
                serviceName: 'service-name',
                cluster: ecs.Cluster.fromClusterAttributes(stack, 'Cluster', {
                    vpc,
                    securityGroups: [],
                    clusterName: 'cluster-name',
                }),
            });
            const artifact = new codepipeline.Artifact('Artifact');
            const bucket = new s3.Bucket(stack, 'PipelineBucket', {
                versioned: true,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            const source = new cpactions.S3SourceAction({
                actionName: 'Source',
                output: artifact,
                bucket,
                bucketKey: 'key',
            });
            const action = new cpactions.EcsDeployAction({
                actionName: 'ECS',
                service,
                imageFile: artifact.atPath('imageFile.json'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [source],
                    },
                    {
                        stageName: 'Deploy',
                        actions: [action],
                    }
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Name: 'ECS',
                                ActionTypeId: {
                                    Category: 'Deploy',
                                    Provider: 'ECS'
                                },
                                Configuration: {
                                    ClusterName: 'cluster-name',
                                    ServiceName: 'service-name',
                                    FileName: 'imageFile.json'
                                }
                            }
                        ]
                    }
                ]
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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