# aio-binance-library
# Async library for connecting to the Binance API on Python
[![Python 3.7](https://img.shields.io/badge/python-3.7|3.8|3.9|3.10-blue.svg)](https://www.python.org/downloads/release/python-370/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Aiohttp: 3.8.1](https://img.shields.io/badge/aiohttp-3.8.1-blue.svg)](https://github.com/aio-libs/aiohttp)
[![Loguru: 0.5.3](https://img.shields.io/badge/loguru-0.5.3-blue.svg)](https://github.com/Delgan/loguru)
[![Ujson: 0.5.3](https://img.shields.io/badge/ujson-5.1.0-blue.svg)](https://github.com/ultrajson/ultrajson)
[![Docstrings: Google](https://img.shields.io/badge/Docstrings-Google-black.svg)](https://sphinxcontrib-napoleon.readthedocs.io/en/latest/example_google.html)

This is a lightweight library that works as a connector to [Binance Futures public API](https://binance-docs.github.io/apidocs/futures/en/)



- Supported APIs:
    - USDT-M Futures `/fapi/*``
    - Futures/Delivery Websocket Market Stream
    - Futures/Delivery User Data Stream
- Inclusion of examples
- Response metadata can be displayed

## Installation

```bash
pip install aio-binance-library
```

## Getting started

### REST API

Usage examples:
```python
import asyncio
from aio_binance.futures.usdt import Client 

async def main():
    client = Client()
    res = await client.get_public_time()
    print(res)

    client = Client(key='<api_key>', secret='<api_secret>')

    # Get account information
    res = await client.get_private_account_info()
    print(res)

    # Post a new order
    params = {
        'symbol': 'BTCUSDT',
        'side': 'SELL',
        'type_order': 'LIMIT',
        'time_in_force': 'GTC',
        'quantity': 0.002,
        'price': 59808
    }

    res = await client.create_private_order(**params)
    print(res)

asyncio.run(main())

```
Or you can use session (For multiple requests, this acts faster):
```python
import asyncio
from aio_binance.futures.usdt import ApiSession 

async def main():
  
    async with ApiSession(key='<api_key>', secret='<api_secret>') as session:
      
        res = await session.get_public_time()
        print(res)
    
        # Get account information
        res = await session.get_private_account_info()
        print(res)
    
        # Post a new order
        params = {
            'symbol': 'BTCUSDT',
            'side': 'SELL',
            'type_order': 'LIMIT',
            'time_in_force': 'GTC',
            'quantity': 0.002,
            'price': 59808
        }
    
        res = await session.create_private_order(**params)
        print(res)

asyncio.run(main())

```
Please find `examples` folder to check for more endpoints.

### Notes
The methods you need, adheres to a hierarchy
```
<method>_<availability>_<method_name>

create_private_order()
or
get_public_time()
```
#### Methods:
`create, get, delete, change, update`

#### Availability:
`private` - methods where key_api and secret_api are required

`public` - you can get information without a key


### Testnet

You can choose testnet

```python
from aio_binance.futures.usdt import Client

client= Client(testnet=True)
```

### Optional parameters


Parameters can be passed in different formats as in Binance api documents or PEP8 suggests _lowercase with words separated by underscores_

```python
# Binance api
response = await client.get_private_open_order('BTCUSDT', orderListId=1)

# PEP8
response = await client.get_private_open_order('BTCUSDT', order_list_id=1)
```

### Timeout

`timeout` is available to be assigned with the number of seconds you find most appropriate to wait for a server response.<br/>
Please remember the value as it won't be shown in error message _no bytes have been received on the underlying socket for timeout seconds_.<br/>
By default, `timeout=5`

```python
from aio_binance.futures.usdt import Client

client= Client(timeout=1)
```

### Response Metadata

The Binance API server provides weight usages in the headers of each response.
You can display them by initializing the client with `show_limit_usage=True`:

```python
from aio_binance.futures.usdt import Client

client = Client(show_limit_usage=True)
res = await client.time()
print(res)
```
returns:

```python
{'data': {'serverTime': 1647990837551}, 'limit_usage': 40}
```
You can also display full response metadata to help in debugging:

```python
client = Client(show_header=True)
res = await client.time()
print(res)
```

returns:

```python
{'data': {'serverTime': 1587990847650}, 'header': {'Context-Type': 'application/json;charset=utf-8', ...}}
```

### User agent

```python
client = Client(agent='name_app')
```

You can pass the name of your application.


## Websocket

This is an example of connecting to multiple streams

```python
import asyncio

from aio_binance.futures.usdt import WsClient


async def callback_event(data: dict):
    print(data)


async def main():

    ws = WsClient()
    stream = [
        ws.stream_liquidation_order(),
        ws.stream_book_ticker(),
        ws.stream_ticker('BTCUSDT')
    ]
    res = await asyncio.gather(*stream)
    await ws.subscription_streams(res, callback_event)

asyncio.run(main())
```
More websocket examples are available in the `examples` folder

### Note
Stream methods start with the word `stream` Example: `stream_<name_method>`

Subscribing to multiple streams: `subscription_streams()`

### Heartbeat

Once connected, the websocket server sends a ping frame every 3 minutes and requires a response pong frame back within
a 5 minutes period. This package handles the pong responses automatically.
