"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const origins = require("../lib");
let stack;
let bucket;
let primaryOrigin;
beforeEach(() => {
    stack = new core_1.Stack();
    bucket = new s3.Bucket(stack, 'Bucket');
    primaryOrigin = new origins.S3Origin(bucket);
});
describe('Origin Groups', () => {
    test('correctly render the OriginGroups property of DistributionConfig', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
            fallbackStatusCodes: [500],
        });
        new cloudfront.Distribution(stack, 'Distribution', {
            defaultBehavior: { origin: originGroup },
        });
        const primaryOriginId = 'DistributionOrigin13547B94F';
        const failoverOriginId = 'DistributionOrigin2C85CC43B';
        const originGroupId = 'DistributionOriginGroup1A1A31B49';
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    TargetOriginId: originGroupId,
                },
                Origins: [
                    {
                        Id: primaryOriginId,
                        DomainName: {
                            'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'],
                        },
                        S3OriginConfig: {
                            OriginAccessIdentity: {
                                'Fn::Join': ['', [
                                        'origin-access-identity/cloudfront/',
                                        { Ref: 'DistributionOrigin1S3Origin5F5C0696' },
                                    ]],
                            },
                        },
                    },
                    {
                        Id: failoverOriginId,
                        DomainName: {
                            'Fn::Join': ['', [
                                    'imported-bucket.s3.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' },
                                ]],
                        },
                        S3OriginConfig: {
                            OriginAccessIdentity: {
                                'Fn::Join': ['', [
                                        'origin-access-identity/cloudfront/',
                                        { Ref: 'DistributionOrigin2S3OriginE484D4BF' },
                                    ]],
                            },
                        },
                    },
                ],
                OriginGroups: {
                    Items: [
                        {
                            FailoverCriteria: {
                                StatusCodes: {
                                    Items: [500],
                                    Quantity: 1,
                                },
                            },
                            Id: 'DistributionOriginGroup1A1A31B49',
                            Members: {
                                Items: [
                                    { OriginId: primaryOriginId },
                                    { OriginId: failoverOriginId },
                                ],
                                Quantity: 2,
                            },
                        },
                    ],
                    Quantity: 1,
                },
            },
        });
    });
    test('cannot have an Origin with their own failover configuration as the primary Origin', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
        });
        const groupOfGroups = new origins.OriginGroup({
            primaryOrigin: originGroup,
            fallbackOrigin: failoverOrigin,
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: groupOfGroups },
            });
        }).toThrow(/An OriginGroup cannot use an Origin with its own failover configuration as its primary origin!/);
    });
    test('cannot have an Origin with their own failover configuration as the fallback Origin', () => {
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket')),
        });
        const groupOfGroups = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: originGroup,
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: groupOfGroups },
            });
        }).toThrow(/An Origin cannot use an Origin with its own failover configuration as its fallback origin!/);
    });
    test('cannot have an empty array of fallbackStatusCodes', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
            fallbackStatusCodes: [],
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: originGroup },
            });
        }).toThrow(/fallbackStatusCodes cannot be empty/);
    });
});
//# sourceMappingURL=data:application/json;base64,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