# pylint: disable=missing-docstring

from click.testing import CliRunner

from nengo_bones.scripts import check_bones, generate_bones
from nengo_bones.tests.utils import assert_exit, write_file


def _write_nengo_yml(tmpdir, nengo_yml=None):
    if nengo_yml is None:
        nengo_yml = """
            project_name: Dumdum
            pkg_name: dummy
            repo_name: dummy_org/dummy
            contributors_rst: {}
            """

    write_file(tmpdir=tmpdir, filename=".nengobones.yml", contents=nengo_yml)


def _generate_valid_file(tmpdir):
    """Generate a valid contributors.rst with bones-generate"""
    result = CliRunner().invoke(
        generate_bones.main,
        [
            "--conf-file",
            str(tmpdir.join(".nengobones.yml")),
            "--output-dir",
            str(tmpdir),
            "contributors-rst",
        ],
    )
    assert_exit(result, 0)


def _run_check_bones(tmpdir):
    cmdline_args = [
        "--root-dir",
        str(tmpdir),
        "--conf-file",
        str(tmpdir.join(".nengobones.yml")),
        "--verbose",
    ]
    return CliRunner().invoke(check_bones.main, cmdline_args)


def test_success(tmpdir):
    _write_nengo_yml(tmpdir)
    _generate_valid_file(tmpdir)
    result = _run_check_bones(tmpdir)
    assert_exit(result, 0)
    assert "CONTRIBUTORS.rst:\n  Up to date" in result.output


def test_no_file(tmpdir):
    _write_nengo_yml(tmpdir)
    result = _run_check_bones(tmpdir)
    assert_exit(result, 0)
    assert ".travis.yml:\n  File not found" in result.output


def test_modified_file(tmpdir):
    _write_nengo_yml(tmpdir)
    _generate_valid_file(tmpdir)
    tmpdir.join("CONTRIBUTORS.rst").write("x", mode="a")
    result = _run_check_bones(tmpdir)
    assert_exit(result, 1)
    assert "CONTRIBUTORS.rst:\n  Content does not match" in result.output
    assert "  Full diff" in result.output


def test_file_not_generated_by_bones(tmpdir):
    _write_nengo_yml(tmpdir)
    tmpdir.join("CONTRIBUTORS.rst").write("a contributor")
    result = _run_check_bones(tmpdir)
    assert_exit(result, 0)
    assert (
        "CONTRIBUTORS.rst:\n  This file was not generated with nengo-bones"
        in result.output
    )


def test_file_pretending_to_be_bones_generated(tmpdir):
    _write_nengo_yml(
        tmpdir,
        nengo_yml="""
            project_name: Dumdum
            pkg_name: dummy
            repo_name: dummy_org/dummy
            """,
    )
    tmpdir.join("CONTRIBUTORS.rst").write("Automatically generated by nengo-bones")
    result = _run_check_bones(tmpdir)
    assert_exit(result, 1)
    assert (
        "CONTRIBUTORS.rst:\n"
        "  This file contains 'Automatically generated by nengo-bones'" in result.output
    )
