# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['edgescan',
 'edgescan.api',
 'edgescan.cli',
 'edgescan.cli.command_groups',
 'edgescan.data',
 'edgescan.data.types',
 'edgescan.http']

package_data = \
{'': ['*']}

install_requires = \
['hodgepodge>=3.1.2,<4.0.0']

entry_points = \
{'console_scripts': ['edgescan = edgescan.cli:cli']}

setup_kwargs = {
    'name': 'edgescan',
    'version': '1.3.2',
    'description': 'An API client for EdgeScan',
    'long_description': '# edgescan\n\n[![](https://img.shields.io/pypi/pyversions/edgescan)](https://pypi.org/project/edgescan/) [![](https://img.shields.io/pypi/wheel/edgescan)](https://pypi.org/project/edgescan/#files) [![](https://img.shields.io/pypi/l/edgescan)](https://github.com/whitfieldsdad/edgescan/blob/main/LICENSE.md)\n\n---\n\n`edgescan` is a client for [EdgeScan\'s](https://www.edgescan.com/) [REST API](https://s3-eu-west-1.amazonaws.com/live-cdn-content/docs/api-guide-latest.pdf) that allows you to:\n\n- Query and count assets, hosts, vulnerabilities, and licenses via the command line or programmatically.\n\n## Installation\n\nTo install `edgescan` using `pip`:\n\n```shell\n$ pip install edgescan\n```\n\nTo install `edgescan` from source (requires [`poetry`](https://github.com/python-poetry/poetry)):\n\n```shell\n$ git clone git@github.com:whitfieldsdad/edgescan.git\n$ cd edgescan\n$ make install\n```\n\nTo install `edgescan` from source using `setup.py` (i.e. if you\'re not using `poetry`):\n\n```shell\n$ git clone git@github.com:whitfieldsdad/edgescan.git\n$ cd edgescan\n$ python3 setup.py install\n```\n\n## Environment variables\n\n|Name              |Default value      |Required|\n|------------------|-------------------|--------|\n|`EDGESCAN_API_KEY`|                   |true    |\n|`EDGESCAN_HOST`   |`live.edgescan.com`|false   |\n\n## Testing\n\nYou can run the integration tests for this package as follows:\n\n```shell\n$ make test\n```\n\n> Note: the integration tests will only be run if the `EDGESCAN_API_KEY` environment variable has been set.\n\n## Tutorials\n\n### Command-line interface\n\n- [Assets](#assets)\n    - [List assets](#list-assets)\n- [Hosts](#hosts)\n    - [List hosts](#list-hosts)\n- [Vulnerabilities](#vulnerabilities)\n    - [List vulnerabilities](#list-vulnerabilities)\n- [Licenses](#licenses)\n    - [List licenses](#list-licenses)\n\n#### Setup\n\nAfter installing `edgescan` you can access the command-line interface as follows:\n\nIf you\'re using `poetry`:\n\n```shell\n$ poetry run edgescan\nUsage: edgescan [OPTIONS] COMMAND [ARGS]...\n\nOptions:\n  --host TEXT     ${EDGESCAN_HOST} ✖\n  --api-key TEXT  ${EDGESCAN_API_KEY} ✔\n  --help          Show this message and exit.\n\nCommands:\n  assets           Query or count assets.\n  hosts            Query or count hosts.\n  licenses         Query or count licenses.\n  vulnerabilities  Query or count vulnerabilities.\n```\n\nIf you\'re not using `poetry`:\n\n```shell\n$ python3 -m edgescan.cli\n```\n\n#### Assets\n\nThe following options are available when working with assets:\n\n```shell\n$ poetry run edgescan assets --help\nUsage: edgescan assets [OPTIONS] COMMAND [ARGS]...\n\n  Query or count assets.\n\nOptions:\n  --help  Show this message and exit.\n\nCommands:\n  count-assets\n  get-asset\n  get-asset-names\n  get-asset-tags\n  get-assets\n  get-location-specifiers-by-asset-name\n```\n\n##### List assets\n\nThe following options are available when listing assets:\n\n```shell\n$ poetry run edgescan assets get-assets --help\nUsage: edgescan assets get-assets [OPTIONS]\n\nOptions:\n  --asset-ids TEXT\n  --asset-names TEXT\n  --asset-tags TEXT\n  --host-ids TEXT\n  --hostnames TEXT\n  --ip-addresses TEXT\n  --os-types TEXT\n  --os-versions TEXT\n  --alive / --dead\n  --vulnerability-ids TEXT\n  --vulnerability-names TEXT\n  --cve-ids TEXT\n  --min-asset-create-time TEXT\n  --max-asset-create-time TEXT\n  --min-asset-update-time TEXT\n  --max-asset-update-time TEXT\n  --min-next-assessment-time TEXT\n  --max-next-assessment-time TEXT\n  --min-last-assessment-time TEXT\n  --max-last-assessment-time TEXT\n  --min-last-host-scan-time TEXT\n  --max-last-host-scan-time TEXT\n  --min-host-last-seen-time TEXT\n  --max-host-last-seen-time TEXT\n  --min-vulnerability-create-time TEXT\n  --max-vulnerability-create-time TEXT\n  --min-vulnerability-update-time TEXT\n  --max-vulnerability-update-time TEXT\n  --min-vulnerability-open-time TEXT\n  --max-vulnerability-open-time TEXT\n  --min-vulnerability-close-time TEXT\n  --max-vulnerability-close-time TEXT\n  --limit INTEGER\n  --help                          Show this message and\n                                  exit.\n```\n\n#### Hosts\n\nThe following options are available when working with hosts:\n\n```shell\n$ poetry run edgescan hosts --help\nUsage: edgescan hosts [OPTIONS] COMMAND [ARGS]...\n\n  Query or count hosts.\n\nOptions:\n  --help  Show this message and exit.\n\nCommands:\n  count-hosts\n  count-hosts-by-asset-name\n  count-hosts-by-asset-name-and-status\n  count-hosts-by-asset-name-and-status-and-os-type\n  count-hosts-by-asset-name-and-status-and-os-type-and-os-version\n  count-hosts-by-last-seen-time\n  count-hosts-by-os-type\n  count-hosts-by-os-version\n  count-hosts-by-status\n  count-hosts-by-status-and-os-type\n  count-hosts-by-status-and-os-type-and-os-version\n  get-host\n  get-hostnames\n  get-hostnames-by-asset-name\n  get-hosts\n  get-ip-addresses\n  get-ip-addresses-by-asset-name\n  get-os-types\n  get-os-versions\n```\n\n##### List hosts\n\nThe following options are available when listing hosts:\n\n```shell\n$ poetry run edgescan hosts get-hosts --help\nUsage: edgescan hosts get-hosts [OPTIONS]\n\nOptions:\n  --asset-ids TEXT\n  --asset-names TEXT\n  --asset-tags TEXT\n  --host-ids TEXT\n  --hostnames TEXT\n  --ip-addresses TEXT\n  --os-types TEXT\n  --os-versions TEXT\n  --alive / --dead\n  --vulnerability-ids TEXT\n  --vulnerability-names TEXT\n  --cve-ids TEXT\n  --min-asset-create-time TEXT\n  --max-asset-create-time TEXT\n  --min-asset-update-time TEXT\n  --max-asset-update-time TEXT\n  --min-next-assessment-time TEXT\n  --max-next-assessment-time TEXT\n  --min-last-assessment-time TEXT\n  --max-last-assessment-time TEXT\n  --min-last-host-scan-time TEXT\n  --max-last-host-scan-time TEXT\n  --min-host-last-seen-time TEXT\n  --max-host-last-seen-time TEXT\n  --min-vulnerability-create-time TEXT\n  --max-vulnerability-create-time TEXT\n  --min-vulnerability-update-time TEXT\n  --max-vulnerability-update-time TEXT\n  --min-vulnerability-open-time TEXT\n  --max-vulnerability-open-time TEXT\n  --min-vulnerability-close-time TEXT\n  --max-vulnerability-close-time TEXT\n  --limit INTEGER\n  --help                          Show this message and\n                                  exit.\n```\n\n#### Vulnerabilities\n\nThe following options are available when working with vulnerabilities:\n\n```shell\n$ poetry run edgescan vulnerabilities --help\nUsage: edgescan vulnerabilities [OPTIONS] COMMAND [ARGS]...\n\n  Query or count vulnerabilities.\n\nOptions:\n  --help  Show this message and exit.\n\nCommands:\n  count-vulnerabilities\n  count-vulnerabilities-by-asset-group-name\n  count-vulnerabilities-by-close-time\n  count-vulnerabilities-by-cve-id\n  count-vulnerabilities-by-location\n  count-vulnerabilities-by-open-time\n  count-vulnerabilities-by-os-type\n  count-vulnerabilities-by-os-version\n  get-vulnerabilities\n  get-vulnerability\n```\n\n##### List vulnerabilities\n\nThe following options are available when listing vulnerabilities:\n\n```shell\n$ poetry run edgescan vulnerabilities get-vulnerabilities --help\nUsage: edgescan vulnerabilities get-vulnerabilities\n           [OPTIONS]\n\nOptions:\n  --asset-ids TEXT\n  --asset-names TEXT\n  --asset-tags TEXT\n  --host-ids TEXT\n  --hostnames TEXT\n  --ip-addresses TEXT\n  --os-types TEXT\n  --os-versions TEXT\n  --alive / --dead\n  --vulnerability-ids TEXT\n  --vulnerability-names TEXT\n  --affects-pci-compliance / --does-not-affect-pci-compliance\n  --include-application-layer-vulnerabilities / --exclude-application-layer-vulnerabilities\n  --include-network-layer-vulnerabilities / --exclude-network-layer-vulnerabilities\n  --cve-ids TEXT\n  --min-asset-create-time TEXT\n  --max-asset-create-time TEXT\n  --min-asset-update-time TEXT\n  --max-asset-update-time TEXT\n  --min-next-assessment-time TEXT\n  --max-next-assessment-time TEXT\n  --min-last-assessment-time TEXT\n  --max-last-assessment-time TEXT\n  --min-last-host-scan-time TEXT\n  --max-last-host-scan-time TEXT\n  --min-host-last-seen-time TEXT\n  --max-host-last-seen-time TEXT\n  --min-vulnerability-create-time TEXT\n  --max-vulnerability-create-time TEXT\n  --min-vulnerability-update-time TEXT\n  --max-vulnerability-update-time TEXT\n  --min-vulnerability-open-time TEXT\n  --max-vulnerability-open-time TEXT\n  --min-vulnerability-close-time TEXT\n  --max-vulnerability-close-time TEXT\n  --limit INTEGER\n  --help                          Show this message and\n                                  exit.\n```\n\n#### Licenses\n\nThe following options are available when working with licenses:\n\n```shell\n$ poetry run edgescan licenses --help\nUsage: edgescan licenses [OPTIONS] COMMAND [ARGS]...\n\n  Query or count licenses.\n\nOptions:\n  --help\n\nCommands:\n  count-licenses\n  get-license\n  get-licenses\n```\n\n##### List licenses\n\nThe following options are available when listing licenses:\n\n```shell\n$ poetry run edgescan licenses get-licenses --help\nUsage: edgescan licenses get-licenses [OPTIONS]\n\nOptions:\n  --ids TEXT\n  --names TEXT\n  --expired / --not-expired\n  --limit INTEGER\n  --help\n```\n\n### Development\n\n- [Count assets by tag](#count-assets-by-tag)\n- [Count hosts by asset tag](#count-hosts-by-asset-tag)\n- [Count vulnerabilities by asset tag](#count-vulnerabilities-by-asset-tag)\n- [Count hosts by OS type](#count-hosts-by-os-type)\n- [Count hosts by OS version](#count-hosts-by-os-version)\n- [Count hosts by asset group name](#count-hosts-by-asset-group-name)\n- [Count vulnerabilities by asset group name](#count-vulnerabilities-by-asset-group-name)\n- [Count vulnerabilities by location (i.e. by IP address or hostname)](#count-vulnerabilities-by-location-ie-by-ip-address-or-hostname)\n\n#### Count assets by tag\n\nLet\'s count the number of asset groups with a tag of "DMZ":\n\n```python\nfrom edgescan.api.client import EdgeScan\n\napi = EdgeScan()\ntotal = api.count_assets(tags=[\'DMZ\'])\nprint(total)\n```\n\n```shell\n1\n```\n\n#### Count hosts by asset tag\n\nLet\'s count the number of hosts within any asset group with a tag of "DMZ":\n\n```python\nfrom edgescan.api.client import EdgeScan\n\napi = EdgeScan()\ntotal = api.count_hosts(asset_tags=[\'DMZ\'])\nprint(total)\n```\n\n```shell\n306\n```\n\n#### Count vulnerabilities by asset tag\n\nLet\'s count the number of vulnerabilities present on any hosts within any asset group with an asset tag of "DMZ":\n\n```python\nfrom edgescan.api.client import EdgeScan\n\napi = EdgeScan()\ntotal = api.count_vulnerabilities(asset_tags=[\'DMZ\'])\nprint(total)\n```\n\n```shell\n1450\n```\n\n#### Count hosts by OS type\n\nHere\'s an example of how to calculate the OS type distribution of all hosts:\n\n```python\nfrom edgescan.api.client import EdgeScan\n\nimport json\nimport collections\n\napi = EdgeScan()\n\ntally = collections.defaultdict(int)\nfor host in api.get_hosts():\n    if host.os_type:\n        tally[host.os_type] += 1\n\ntxt = json.dumps(tally, indent=4)\nprint(txt)\n```\n\n```shell\n{\n    "bsd": 168,\n    "darwin": 7,\n    "linux": 175,\n    "other": 300,\n    "solaris": 3,\n    "windows": 50\n}\n```\n\n#### Count hosts by OS version\n\nHere\'s an example of how to calculate the OS version distribution of all Windows hosts:\n\n```python\nfrom edgescan.api.client import EdgeScan\n\nimport json\nimport collections\n\napi = EdgeScan()\n\ntally = collections.defaultdict(int)\nfor host in api.get_hosts(os_types=["windows"]):\n    if host.os_version:\n        tally[host.os_version] += 1\n\ntxt = json.dumps(tally, indent=4)\nprint(txt)\n```\n\n```shell\n{\n    "Microsoft Windows 2008": 9,\n    "Microsoft Windows 2012": 15,\n    "Microsoft Windows 2016": 5,\n    "Microsoft Windows 7": 11,\n    "Microsoft Windows Phone": 3,\n    "Microsoft Windows Vista": 7\n}\n```\n\n#### Count hosts by asset group name\n\nHere\'s an example of how to calculate how many hosts are associated with each asset group:\n\n```python\nfrom edgescan.api.client import EdgeScan\n\nimport json\n\napi = EdgeScan()\n\ntally = {}\nfor asset in api.get_assets():\n    tally[asset.name] = asset.host_count\n\ntxt = json.dumps(tally, indent=4)\nprint(txt)\n```\n\n```shell\n{\n    "External IP Monitoring 66.249.64.0 – 66.249.95.255": 62,\n    "External IP Monitoring 72.14.192.0 – 72.14.255.255": 57,\n    "104.154.0.0/15": 34,\n    "64.233.160.0/19": 23,\n    "66.102.0.0/20": 13,\n    "208.117.224.0/19": 56\n}\n```\n\n#### Count vulnerabilities by asset group name\n\nHere\'s an example of how to calculate how many vulnerabilities are associated with hosts within each asset group:\n\n```python\nfrom edgescan.api.client import EdgeScan\n\nimport collections\nimport json\n\napi = EdgeScan()\n\n#: Count vulnerabilities by `asset.id`.\nvulnerabilities_by_asset_id = collections.defaultdict(int)\nfor vulnerability in api.get_vulnerabilities():\n    vulnerabilities_by_asset_id[vulnerability.asset_id] += 1\n\n#: List the number of vulnerabilities by `asset.name`.\ntally = {}\nfor asset in api.get_assets():\n    if asset.id in vulnerabilities_by_asset_id:\n        tally[asset.name] = vulnerabilities_by_asset_id[asset.id]\n\ntxt = json.dumps(tally, indent=4)\nprint(txt)\n```\n\n```shell\n{\n    "104.154.0.0/15": 1553,\n    "64.233.160.0/19": 759,\n    "66.102.0.0/20": 94,\n    "208.117.224.0/19": 432\n}\n```\n\n#### Count vulnerabilities by location (i.e. by IP address or hostname)\n\nAs an example, let\'s list the number of vulnerabilities associated with all hosts by IP address or hostname:\n\n```python\nfrom edgescan.api.client import EdgeScan\n\nimport json\nimport collections\n\napi = EdgeScan()\n\ntally = collections.defaultdict(int)\nfor vulnerability in api.get_vulnerabilities():\n    tally[vulnerability.location] += 1\n\ntxt = json.dumps(tally, indent=4)\nprint(txt)\n```\n\n```shell\n{\n    "142.251.32.69": 75,\n    "172.217.1.14": 56,\n    "142.251.33.163": 47,\n    "142.251.41.78": 41,\n    "172.217.165.3": 33,\n}\n```\n',
    'author': 'Tyler Fisher',
    'author_email': 'tylerfisher@tylerfisher.ca',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/whitfieldsdad/edgescan',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
