# Vulture - Find dead code

![CI:Test](https://github.com/jendrikseipp/vulture/workflows/CI/badge.svg)
[![image](https://coveralls.io/repos/github/jendrikseipp/vulture/badge.svg?branch=master)](https://coveralls.io/github/jendrikseipp/vulture?branch=master)

Vulture finds unused code in Python programs. This is useful for
cleaning up and finding errors in large code bases. If you run Vulture
on both your library and test suite you can find untested code.

Due to Python's dynamic nature, static code analyzers like Vulture are
likely to miss some dead code. Also, code that is only called implicitly
may be reported as unused. Nonetheless, Vulture can be a very helpful
tool for higher code quality.

## Features

* fast: uses static code analysis
* tested: tests itself and has complete test coverage
* complements pyflakes and has the same output syntax
* sorts unused classes and functions by size with `--sort-by-size`
* supports Python 2.7 and Python \>= 3.5

## Installation

    $ pip install vulture  # from PyPI
    $ pip install .        # from cloned repo

## Usage

    $ vulture myscript.py  # or
    $ python3 -m vulture myscript.py
    $ vulture myscript.py mypackage/
    $ vulture myscript.py --min-confidence 100  # Only report 100% dead code.

The provided arguments may be Python files or directories. For each
directory Vulture analyzes all contained
<span class="title-ref">\*.py</span> files.

Vulture assigns each chunk of dead code a confidence value. A confidence
value of 100% means that the code will never be executed. Values below
100% are only estimates for how likely it is that the code is unused.

After you have found and deleted dead code, run Vulture again, because
it may discover more dead code.

**Handling false positives**

You can add used code that is reported as unused to a Python module and
add it to the list of scanned paths. To obtain such a whitelist
automatically, pass `--make-whitelist` to Vulture:

    $ vulture mydir --make-whitelist > whitelist.py
    $ vulture mydir whitelist.py

Note that the resulting `whitelist.py` file will contain valid Python
syntax, but for Python to be able to *run* it, you will usually have to
make some modifications.

We collect whitelists for common Python modules and packages in
`vulture/whitelists/` (pull requests are welcome). If you want to ignore
a whole file or directory, use the `--exclude` parameter (e.g.,
`--exclude *settings.py,docs/`).

<!-- Hide noqa docs until we decide whether we want to support it.
Another way of ignoring errors is to annotate the line causing the false
positive with `# noqa: <ERROR_CODE>` in a trailing comment (e.g., `#
noqa: V103`). The `ERROR_CODE` specifies what kind of dead code to
ignore (see the table below for the list of error codes). In case no
error code is specified, Vulture ignores all results for the line.
(Note that the line number for decorated objects is the line number of
the first decorator.)
-->

For compatibility with [flake8](https://flake8.pycqa.org/), Vulture
supports the [F401 and
F841](https://flake8.pycqa.org/en/latest/user/error-codes.html) error
codes for ignoring unused imports (`# noqa: F401`) and unused local
variables (`# noqa: F841`). However, we recommend using whitelists instead
of `noqa` comments, since `noqa` comments add visual noise to the code and
make it harder to read.

**Ignoring names**

You can use `--ignore-names foo*,ba[rz]` to let Vulture ignore all names
starting with `foo` and the names `bar` and `baz`. Additionally, the
`--ignore-decorators` option can be used to ignore functions decorated
with the given decorator. This is helpful for example in Flask projects,
where you can use `--ignore-decorators "@app.route"` to ignore all
functions with the `@app.route` decorator.

We recommend using whitelists instead of `--ignore-names` or
`--ignore-decorators` whenever possible, since whitelists are
automatically checked for syntactic correctness when passed to Vulture
and often you can even pass them to your Python interpreter and let it
check that all whitelisted code actually still exists in your project.

**Marking unused variables**

There are situations where you can't just remove unused variables, e.g.,
in tuple assignments or function signatures. Vulture will ignore these
variables if they start with an underscore (e.g., `_x, y = get_pos()`).

**Minimum confidence**

You can use the `--min-confidence` flag to set the minimum confidence
for code to be reported as unused. Use `--min-confidence 100` to only
report code that is guaranteed to be unused within the analyzed files.

## How does it work?

Vulture uses the `ast` module to build abstract syntax trees for all
given files. While traversing all syntax trees it records the names of
defined and used objects. Afterwards, it reports the objects which have
been defined, but not used. This analysis ignores scopes and only takes
object names into account.

Vulture also detects unreachable code by looking for code after
`return`, `break`, `continue` and `raise` statements, and by searching
for unsatisfiable `if`- and `while`-conditions.

## Sort by size

When using the `--sort-by-size` option, Vulture sorts unused code by its
number of lines. This helps developers prioritize where to look for dead
code first.

## Examples

Consider the following Python script (`dead_code.py`):

``` python
import os

class Greeter:
    def greet(self):
        print("Hi")

def hello_world():
    message = "Hello, world!"
    greeter = Greeter()
    greet_func = getattr(greeter, "greet")
    greet_func()

if __name__ == "__main__":
    hello_world()
```

Calling :

    vulture dead_code.py

results in the following output:

    dead_code.py:1: V104 unused import 'os' (90% confidence)
    dead_code.py:4: V103 unused function 'greet' (60% confidence)
    dead_code.py:8: V107 unused variable 'message' (60% confidence)

Vulture correctly reports "os" and "message" as unused, but it fails to
detect that "greet" is actually used. The recommended method to deal
with false positives like this is to create a whitelist Python file.

**Preparing whitelists**

In a whitelist we simulate the usage of variables, attributes, etc. For
the program above, a whitelist could look as follows:

``` python
# whitelist_dead_code.py
from dead_code import Greeter
Greeter.greet
```

Alternatively, you can pass `--make-whitelist` to Vulture and obtain an
automatically generated whitelist.

Passing both the original program and the whitelist to Vulture

    vulture dead_code.py whitelist_dead_code.py

makes Vulture ignore the `greet` method:

    dead_code.py:1: V104 unused import 'os' (90% confidence)
    dead_code.py:8: V107 unused variable 'message' (60% confidence)

<!-- Hide noqa docs until we decide whether we want to support it.
**Using "# noqa"**

```python
import os  # noqa

class Greeter:  # noqa: V102
    def greet(self):  # noqa: V103
        print("Hi")
```

## Error codes

For compatibility with [flake8](https://flake8.pycqa.org/), Vulture
supports the [F401 and
F841](https://flake8.pycqa.org/en/latest/user/error-codes.html) error
codes.

| Error codes |    Description    |
| ----------- | ----------------- |
| V101        | Unused attribute  |
| V102        | Unused class      |
| V103        | Unused function   |
| V104, F401  | Unused import     |
| V105        | Unused property   |
| V106        | Unused method     |
| V107, F841  | Unused variable   |
| V201        | Unreachable code  |

-->

## Exit codes

| Exit code |                          Description                          |
| --------- | ------------------------------------------------------------- |
|     0     | No dead code found                                            |
|     1     | Dead code found                                               |
|     1     | Invalid input (file missing, syntax error, wrong encoding)    |
|     2     | Invalid command line arguments                                |

## Similar programs

  - [pyflakes](https://pypi.org/project/pyflakes/) finds unused imports
    and unused local variables (in addition to many other programmatic
    errors).
  - [coverage](https://pypi.org/project/coverage/) finds unused code
    more reliably than Vulture, but requires all branches of the code to
    actually be run.
  - [uncalled](https://pypi.org/project/uncalled/) finds dead code by
    using the abstract syntax tree (like Vulture), regular expressions,
    or both.
  - [dead](https://pypi.org/project/dead/) finds dead code by using the
    abstract syntax tree (like Vulture).

## Participate

Please visit <https://github.com/jendrikseipp/vulture> to report any
issues or to make pull requests.

  - Contributing guide:
    [CONTRIBUTING.md](https://github.com/jendrikseipp/vulture/blob/master/CONTRIBUTING.md)
  - Release notes:
    [CHANGELOG.md](https://github.com/jendrikseipp/vulture/blob/master/CHANGELOG.md)
  - Roadmap:
    [TODO.md](https://github.com/jendrikseipp/vulture/blob/master/TODO.md)
