"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Model = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
class ModelBase extends cdk.Resource {
    /**
     * An accessor for the Connections object that will fail if this Model does not have a VPC
     * configured.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Cannot manage network access without configuring a VPC');
        }
        return this._connections;
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
}
/**
 * Defines a SageMaker Model.
 */
class Model extends ModelBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.modelName,
        });
        this.containers = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ModelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Model);
            }
            throw error;
        }
        this._connections = this.configureNetworking(props);
        this.subnets = (props.vpc) ? props.vpc.selectSubnets(props.vpcSubnets) : undefined;
        // set the sagemaker role or create new one
        this.role = props.role || this.createSageMakerRole();
        this.grantPrincipal = this.role;
        (props.containers || []).map(c => this.addContainer(c));
        const model = new aws_sagemaker_1.CfnModel(this, 'Model', {
            executionRoleArn: this.role.roleArn,
            modelName: this.physicalName,
            primaryContainer: cdk.Lazy.any({ produce: () => this.renderPrimaryContainer() }),
            vpcConfig: cdk.Lazy.any({ produce: () => this.renderVpcConfig() }),
            containers: cdk.Lazy.any({ produce: () => this.renderContainers() }),
        });
        this.modelName = this.getResourceNameAttribute(model.attrModelName);
        this.modelArn = this.getResourceArnAttribute(model.ref, {
            service: 'sagemaker',
            resource: 'model',
            resourceName: this.physicalName,
        });
        /*
         * SageMaker model creation will fail if the model's execution role does not have read access to
         * its model data in S3. Since the CDK uses a separate AWS::IAM::Policy CloudFormation resource
         * to attach inline policies to IAM roles, the following line ensures that the role and its
         * AWS::IAM::Policy resource are deployed prior to model creation.
         */
        model.node.addDependency(this.role);
    }
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelArn the ARN of the model.
     */
    static fromModelArn(scope, id, modelArn) {
        return Model.fromModelAttributes(scope, id, { modelArn });
    }
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelName the name of the model.
     */
    static fromModelName(scope, id, modelName) {
        const modelArn = cdk.Stack.of(scope).formatArn({
            service: 'sagemaker',
            resource: 'model',
            resourceName: modelName,
        });
        return Model.fromModelAttributes(scope, id, { modelArn });
    }
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param attrs the attributes of the model to import.
     */
    static fromModelAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ModelAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromModelAttributes);
            }
            throw error;
        }
        const modelArn = attrs.modelArn;
        const modelName = cdk.Stack.of(scope).splitArn(modelArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        const role = attrs.role;
        class Import extends ModelBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: attrs.modelArn,
                });
                this.modelArn = modelArn;
                this.modelName = modelName;
                this.role = role;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroups) {
                    this._connections = new ec2.Connections({
                        securityGroups: attrs.securityGroups,
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add containers to the model.
     *
     * @param container The container definition to add.
     */
    addContainer(container) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_ContainerDefinition(container);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addContainer);
            }
            throw error;
        }
        this.containers.push(this.renderContainer(container));
    }
    validateContainers() {
        // validate number of containers
        if (this.containers.length < 1) {
            throw new Error('Must configure at least 1 container for model');
        }
        else if (this.containers.length > 15) {
            throw new Error('Cannot have more than 15 containers in inference pipeline');
        }
    }
    renderPrimaryContainer() {
        return (this.containers.length === 1) ? this.containers[0] : undefined;
    }
    renderContainers() {
        this.validateContainers();
        return (this.containers.length === 1) ? undefined : this.containers;
    }
    renderContainer(container) {
        return {
            image: container.image.bind(this, this).imageName,
            containerHostname: container.containerHostname,
            environment: container.environment,
            modelDataUrl: container.modelData ? container.modelData.bind(this, this).uri : undefined,
        };
    }
    configureNetworking(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroups\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroups');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        return new ec2.Connections({ securityGroups });
    }
    renderVpcConfig() {
        if (!this._connections) {
            return undefined;
        }
        return {
            subnets: this.subnets.subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    createSageMakerRole() {
        const sagemakerRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        });
        // Grant SageMaker FullAccess
        sagemakerRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'));
        return sagemakerRole;
    }
}
exports.Model = Model;
_a = JSII_RTTI_SYMBOL_1;
Model[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.Model", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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