"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Endpoint = exports.InvocationHttpResponseCode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const appscaling = require("aws-cdk-lib/aws-applicationautoscaling");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const endpoint_config_1 = require("./endpoint-config");
const util_1 = require("./private/util");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
const scalable_instance_count_1 = require("./scalable-instance-count");
/*
 * Amazon SageMaker automatic scaling doesn't support automatic scaling for burstable instances such
 * as T2, because they already allow for increased capacity under increased workloads.
 * https://docs.aws.amazon.com/sagemaker/latest/dg/endpoint-auto-scaling-add-console.html
 */
const BURSTABLE_INSTANCE_TYPE_PREFIXES = Object.entries(ec2.InstanceClass)
    .filter(([name, _]) => name.startsWith('T'))
    .map(([_, prefix]) => `ml.${prefix}.`);
class EndpointInstanceProductionVariant {
    constructor(endpoint, variant) {
        this.initialInstanceCount = variant.initialInstanceCount;
        this.instanceType = variant.instanceType;
        this.variantName = variant.variantName;
        this.endpoint = endpoint;
    }
    metric(namespace, metricName, props) {
        return new cloudwatch.Metric({
            namespace,
            metricName,
            dimensions: {
                EndpointName: this.endpoint.endpointName,
                VariantName: this.variantName,
            },
            statistic: 'Sum',
            ...props,
        });
    }
    metricInvocations(props) {
        return this.metric('AWS/SageMaker', 'Invocations', props);
    }
    metricInvocationsPerInstance(props) {
        return this.metric('AWS/SageMaker', 'InvocationsPerInstance', props);
    }
    metricModelLatency(props) {
        return this.metric('AWS/SageMaker', 'ModelLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    metricOverheadLatency(props) {
        return this.metric('AWS/SageMaker', 'OverheadLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    metricInvocationResponseCode(responseCode, props) {
        return this.metric('AWS/SageMaker', responseCode, props);
    }
    metricDiskUtilization(props) {
        return this.metric('/aws/sagemaker/Endpoints', 'DiskUtilization', {
            statistic: 'Average',
            ...props,
        });
    }
    metricCpuUtilization(props) {
        return this.metric('/aws/sagemaker/Endpoints', 'CPUUtilization', {
            statistic: 'Average',
            ...props,
        });
    }
    metricMemoryUtilization(props) {
        return this.metric('/aws/sagemaker/Endpoints', 'MemoryUtilization', {
            statistic: 'Average',
            ...props,
        });
    }
    metricGpuUtilization(props) {
        return this.metric('/aws/sagemaker/Endpoints', 'GPUUtilization', {
            statistic: 'Average',
            ...props,
        });
    }
    metricGpuMemoryUtilization(props) {
        return this.metric('/aws/sagemaker/Endpoints', 'GPUMemoryUtilization', {
            statistic: 'Average',
            ...props,
        });
    }
    autoScaleInstanceCount(scalingProps) {
        const errors = [];
        if (scalingProps.minCapacity && scalingProps.minCapacity > this.initialInstanceCount) {
            errors.push(`minCapacity cannot be greater than initial instance count: ${this.initialInstanceCount}`);
        }
        if (scalingProps.maxCapacity && scalingProps.maxCapacity < this.initialInstanceCount) {
            errors.push(`maxCapacity cannot be less than initial instance count: ${this.initialInstanceCount}`);
        }
        if (BURSTABLE_INSTANCE_TYPE_PREFIXES.some(prefix => this.instanceType.toString().startsWith(prefix))) {
            errors.push(`AutoScaling not supported for burstable instance types like ${this.instanceType}`);
        }
        if (this.scalableInstanceCount) {
            errors.push('AutoScaling of task count already enabled for this service');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid Application Auto Scaling configuration: ${errors.join(os_1.EOL)}`);
        }
        return this.scalableInstanceCount = new scalable_instance_count_1.ScalableInstanceCount(this.endpoint, 'InstanceCount', {
            serviceNamespace: appscaling.ServiceNamespace.SAGEMAKER,
            resourceId: `endpoint/${this.endpoint.endpointName}/variant/${this.variantName}`,
            dimension: 'sagemaker:variant:DesiredInstanceCount',
            role: this.makeScalingRole(),
            minCapacity: scalingProps.minCapacity || this.initialInstanceCount,
            maxCapacity: scalingProps.maxCapacity || this.initialInstanceCount,
        });
    }
    /**
     * Return the service linked role which will automatically be created by Application Auto Scaling
     * for scaling purposes.
     *
     * @see https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this.endpoint, 'ScalingRole', cdk.Stack.of(this.endpoint).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/sagemaker.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_SageMakerEndpoint',
        }));
    }
}
class EndpointBase extends cdk.Resource {
    /**
     * Permits an IAM principal to invoke this endpoint
     * @param grantee The principal to grant access to
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
/**
 * HTTP response codes for Endpoint invocations
 */
var InvocationHttpResponseCode;
(function (InvocationHttpResponseCode) {
    /**
     * 4xx response codes from Endpoint invocations
     */
    InvocationHttpResponseCode["INVOCATION_4XX_ERRORS"] = "Invocation4XXErrors";
    /**
     * 5xx response codes from Endpoint invocations
     */
    InvocationHttpResponseCode["INVOCATION_5XX_ERRORS"] = "Invocation5XXErrors";
})(InvocationHttpResponseCode = exports.InvocationHttpResponseCode || (exports.InvocationHttpResponseCode = {}));
/**
 * Defines a SageMaker endpoint.
 */
class Endpoint extends EndpointBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.endpointName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_EndpointProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Endpoint);
            }
            throw error;
        }
        this.validateEnvironmentCompatibility(props.endpointConfig);
        this.endpointConfig = props.endpointConfig;
        // create the endpoint resource
        const endpoint = new aws_sagemaker_1.CfnEndpoint(this, 'Endpoint', {
            endpointConfigName: props.endpointConfig.endpointConfigName,
            endpointName: this.physicalName,
        });
        this.endpointName = this.getResourceNameAttribute(endpoint.attrEndpointName);
        this.endpointArn = this.getResourceArnAttribute(endpoint.ref, {
            service: 'sagemaker',
            resource: 'endpoint',
            resourceName: this.physicalName,
        });
    }
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointArn the ARN of the endpoint.
     */
    static fromEndpointArn(scope, id, endpointArn) {
        return Endpoint.fromEndpointAttributes(scope, id, { endpointArn });
    }
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointName the name of the endpoint.
     */
    static fromEndpointName(scope, id, endpointName) {
        const endpointArn = cdk.Stack.of(scope).formatArn({
            service: 'sagemaker',
            resource: 'endpoint',
            resourceName: endpointName,
        });
        return Endpoint.fromEndpointAttributes(scope, id, { endpointArn });
    }
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param attrs the attributes of the endpoint to import.
     */
    static fromEndpointAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_EndpointAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEndpointAttributes);
            }
            throw error;
        }
        const endpointArn = attrs.endpointArn;
        const endpointName = cdk.Stack.of(scope).splitArn(endpointArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends EndpointBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: endpointArn,
                });
                this.endpointArn = endpointArn;
                this.endpointName = endpointName;
            }
        }
        return new Import(scope, id);
    }
    validateEnvironmentCompatibility(endpointConfig) {
        if (!util_1.sameEnv(endpointConfig.env.account, this.env.account)) {
            throw new Error(`Cannot use endpoint configuration in account ${endpointConfig.env.account} for endpoint in account ${this.env.account}`);
        }
        else if (!util_1.sameEnv(endpointConfig.env.region, this.env.region)) {
            throw new Error(`Cannot use endpoint configuration in region ${endpointConfig.env.region} for endpoint in region ${this.env.region}`);
        }
    }
    /**
     * Get instance production variants associated with endpoint.
     */
    get instanceProductionVariants() {
        if (this.endpointConfig instanceof endpoint_config_1.EndpointConfig) {
            return this.endpointConfig._instanceProductionVariants.map(v => new EndpointInstanceProductionVariant(this, v));
        }
        throw new Error('Production variant lookup is not supported for an imported IEndpointConfig');
    }
    /**
     * Find instance production variant based on variant name
     * @param name Variant name from production variant
     */
    findInstanceProductionVariant(name) {
        if (this.endpointConfig instanceof endpoint_config_1.EndpointConfig) {
            const variant = this.endpointConfig._findInstanceProductionVariant(name);
            return new EndpointInstanceProductionVariant(this, variant);
        }
        throw new Error('Production variant lookup is not supported for an imported IEndpointConfig');
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.Endpoint", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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