"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToLambda = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-cdk/core");
/**
 * @summary The KinesisStreamsToLambda class.
 */
class KinesisStreamsToLambda extends core_2.Construct {
    /**
     * @summary Constructs a new instance of the KinesisStreamsToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Add the Lambda event source mapping
        const eventSourceProps = (props.eventSourceProps) ?
            core_1.overrideProps(defaults.DefaultKinesisEventSourceProps(this.kinesisStream.streamArn), props.eventSourceProps) :
            defaults.DefaultKinesisEventSourceProps(this.kinesisStream.streamArn);
        this.lambdaFunction.addEventSourceMapping('LambdaKinesisEventSourceMapping', eventSourceProps);
        // Add permissions for the Lambda function to access Kinesis
        const policy = new iam.Policy(this, 'LambdaFunctionPolicy');
        this.kinesisStreamRole = this.lambdaFunction.role;
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [this.kinesisStream.streamArn],
            actions: [
                'kinesis:GetRecords',
                'kinesis:GetShardIterator',
                'kinesis:DescribeStream'
            ]
        }));
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'kinesis:ListStreams',
            ]
        }));
        policy.attachToRole(this.kinesisStreamRole);
        this.kinesisStream.grantRead(this.lambdaFunction.grantPrincipal);
        // Add appropriate cfn_nag metadata
        const cfnCustomPolicy = policy.node.defaultChild;
        cfnCustomPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "The kinesis:ListStreams action requires a wildcard resource."
                    }
                ]
            }
        };
    }
}
exports.KinesisStreamsToLambda = KinesisStreamsToLambda;
//# sourceMappingURL=data:application/json;base64,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