# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import dataclasses
import json  # type: ignore
import re
from typing import Any, Callable, Dict, List, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import (
    gapic_v1,
    operations_v1,
    path_template,
    rest_helpers,
    rest_streaming,
)
from google.api_core import exceptions as core_exceptions
from google.api_core import retry as retries
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.auth.transport.requests import AuthorizedSession  # type: ignore
from google.protobuf import json_format
import grpc  # type: ignore
from requests import __version__ as requests_version

try:
    OptionalRetry = Union[retries.Retry, gapic_v1.method._MethodDefault]
except AttributeError:  # pragma: NO COVER
    OptionalRetry = Union[retries.Retry, object]  # type: ignore


from google.longrunning import operations_pb2  # type: ignore

from google.cloud.appengine_admin_v1.types import appengine, domain_mapping

from .base import DEFAULT_CLIENT_INFO as BASE_DEFAULT_CLIENT_INFO
from .base import DomainMappingsTransport

DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo(
    gapic_version=BASE_DEFAULT_CLIENT_INFO.gapic_version,
    grpc_version=None,
    rest_version=requests_version,
)


class DomainMappingsRestInterceptor:
    """Interceptor for DomainMappings.

    Interceptors are used to manipulate requests, request metadata, and responses
    in arbitrary ways.
    Example use cases include:
    * Logging
    * Verifying requests according to service or custom semantics
    * Stripping extraneous information from responses

    These use cases and more can be enabled by injecting an
    instance of a custom subclass when constructing the DomainMappingsRestTransport.

    .. code-block:: python
        class MyCustomDomainMappingsInterceptor(DomainMappingsRestInterceptor):
            def pre_create_domain_mapping(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_create_domain_mapping(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_delete_domain_mapping(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_delete_domain_mapping(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_get_domain_mapping(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_get_domain_mapping(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_list_domain_mappings(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_list_domain_mappings(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_update_domain_mapping(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_update_domain_mapping(self, response):
                logging.log(f"Received response: {response}")
                return response

        transport = DomainMappingsRestTransport(interceptor=MyCustomDomainMappingsInterceptor())
        client = DomainMappingsClient(transport=transport)


    """

    def pre_create_domain_mapping(
        self,
        request: appengine.CreateDomainMappingRequest,
        metadata: Sequence[Tuple[str, str]],
    ) -> Tuple[appengine.CreateDomainMappingRequest, Sequence[Tuple[str, str]]]:
        """Pre-rpc interceptor for create_domain_mapping

        Override in a subclass to manipulate the request or metadata
        before they are sent to the DomainMappings server.
        """
        return request, metadata

    def post_create_domain_mapping(
        self, response: operations_pb2.Operation
    ) -> operations_pb2.Operation:
        """Post-rpc interceptor for create_domain_mapping

        Override in a subclass to manipulate the response
        after it is returned by the DomainMappings server but before
        it is returned to user code.
        """
        return response

    def pre_delete_domain_mapping(
        self,
        request: appengine.DeleteDomainMappingRequest,
        metadata: Sequence[Tuple[str, str]],
    ) -> Tuple[appengine.DeleteDomainMappingRequest, Sequence[Tuple[str, str]]]:
        """Pre-rpc interceptor for delete_domain_mapping

        Override in a subclass to manipulate the request or metadata
        before they are sent to the DomainMappings server.
        """
        return request, metadata

    def post_delete_domain_mapping(
        self, response: operations_pb2.Operation
    ) -> operations_pb2.Operation:
        """Post-rpc interceptor for delete_domain_mapping

        Override in a subclass to manipulate the response
        after it is returned by the DomainMappings server but before
        it is returned to user code.
        """
        return response

    def pre_get_domain_mapping(
        self,
        request: appengine.GetDomainMappingRequest,
        metadata: Sequence[Tuple[str, str]],
    ) -> Tuple[appengine.GetDomainMappingRequest, Sequence[Tuple[str, str]]]:
        """Pre-rpc interceptor for get_domain_mapping

        Override in a subclass to manipulate the request or metadata
        before they are sent to the DomainMappings server.
        """
        return request, metadata

    def post_get_domain_mapping(
        self, response: domain_mapping.DomainMapping
    ) -> domain_mapping.DomainMapping:
        """Post-rpc interceptor for get_domain_mapping

        Override in a subclass to manipulate the response
        after it is returned by the DomainMappings server but before
        it is returned to user code.
        """
        return response

    def pre_list_domain_mappings(
        self,
        request: appengine.ListDomainMappingsRequest,
        metadata: Sequence[Tuple[str, str]],
    ) -> Tuple[appengine.ListDomainMappingsRequest, Sequence[Tuple[str, str]]]:
        """Pre-rpc interceptor for list_domain_mappings

        Override in a subclass to manipulate the request or metadata
        before they are sent to the DomainMappings server.
        """
        return request, metadata

    def post_list_domain_mappings(
        self, response: appengine.ListDomainMappingsResponse
    ) -> appengine.ListDomainMappingsResponse:
        """Post-rpc interceptor for list_domain_mappings

        Override in a subclass to manipulate the response
        after it is returned by the DomainMappings server but before
        it is returned to user code.
        """
        return response

    def pre_update_domain_mapping(
        self,
        request: appengine.UpdateDomainMappingRequest,
        metadata: Sequence[Tuple[str, str]],
    ) -> Tuple[appengine.UpdateDomainMappingRequest, Sequence[Tuple[str, str]]]:
        """Pre-rpc interceptor for update_domain_mapping

        Override in a subclass to manipulate the request or metadata
        before they are sent to the DomainMappings server.
        """
        return request, metadata

    def post_update_domain_mapping(
        self, response: operations_pb2.Operation
    ) -> operations_pb2.Operation:
        """Post-rpc interceptor for update_domain_mapping

        Override in a subclass to manipulate the response
        after it is returned by the DomainMappings server but before
        it is returned to user code.
        """
        return response


@dataclasses.dataclass
class DomainMappingsRestStub:
    _session: AuthorizedSession
    _host: str
    _interceptor: DomainMappingsRestInterceptor


class DomainMappingsRestTransport(DomainMappingsTransport):
    """REST backend transport for DomainMappings.

    Manages domains serving an application.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends JSON representations of protocol buffers over HTTP/1.1

    """

    def __init__(
        self,
        *,
        host: str = "appengine.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        url_scheme: str = "https",
        interceptor: Optional[DomainMappingsRestInterceptor] = None,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.

            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Callable[[], Tuple[bytes, bytes]]): Client
                certificate to configure mutual TLS HTTP channel. It is ignored
                if ``channel`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you are developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.
            url_scheme: the protocol scheme for the API endpoint.  Normally
                "https", but for testing or local servers,
                "http" can be specified.
        """
        # Run the base constructor
        # TODO(yon-mg): resolve other ctor params i.e. scopes, quota, etc.
        # TODO: When custom host (api_endpoint) is set, `scopes` must *also* be set on the
        # credentials object
        maybe_url_match = re.match("^(?P<scheme>http(?:s)?://)?(?P<host>.*)$", host)
        if maybe_url_match is None:
            raise ValueError(
                f"Unexpected hostname structure: {host}"
            )  # pragma: NO COVER

        url_match_items = maybe_url_match.groupdict()

        host = f"{url_scheme}://{host}" if not url_match_items["scheme"] else host

        super().__init__(
            host=host,
            credentials=credentials,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )
        self._session = AuthorizedSession(
            self._credentials, default_host=self.DEFAULT_HOST
        )
        self._operations_client: Optional[operations_v1.AbstractOperationsClient] = None
        if client_cert_source_for_mtls:
            self._session.configure_mtls_channel(client_cert_source_for_mtls)
        self._interceptor = interceptor or DomainMappingsRestInterceptor()
        self._prep_wrapped_messages(client_info)

    @property
    def operations_client(self) -> operations_v1.AbstractOperationsClient:
        """Create the client designed to process long-running operations.

        This property caches on the instance; repeated calls return the same
        client.
        """
        # Only create a new client if we do not already have one.
        if self._operations_client is None:
            http_options: Dict[str, List[Dict[str, str]]] = {
                "google.longrunning.Operations.GetOperation": [
                    {
                        "method": "get",
                        "uri": "/v1/{name=apps/*/operations/*}",
                    },
                ],
                "google.longrunning.Operations.ListOperations": [
                    {
                        "method": "get",
                        "uri": "/v1/{name=apps/*}/operations",
                    },
                ],
            }

            rest_transport = operations_v1.OperationsRestTransport(
                host=self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                scopes=self._scopes,
                http_options=http_options,
                path_prefix="v1",
            )

            self._operations_client = operations_v1.AbstractOperationsClient(
                transport=rest_transport
            )

        # Return the client from cache.
        return self._operations_client

    class _CreateDomainMapping(DomainMappingsRestStub):
        def __hash__(self):
            return hash("CreateDomainMapping")

        def __call__(
            self,
            request: appengine.CreateDomainMappingRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, str]] = (),
        ) -> operations_pb2.Operation:
            r"""Call the create domain mapping method over HTTP.

            Args:
                request (~.appengine.CreateDomainMappingRequest):
                    The request object. Request message for
                ``DomainMappings.CreateDomainMapping``.
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, str]]): Strings which should be
                    sent along with the request as metadata.

            Returns:
                ~.operations_pb2.Operation:
                    This resource represents a
                long-running operation that is the
                result of a network API call.

            """

            http_options: List[Dict[str, str]] = [
                {
                    "method": "post",
                    "uri": "/v1/{parent=apps/*}/domainMappings",
                    "body": "domain_mapping",
                },
            ]
            request, metadata = self._interceptor.pre_create_domain_mapping(
                request, metadata
            )
            pb_request = appengine.CreateDomainMappingRequest.pb(request)
            transcoded_request = path_template.transcode(http_options, pb_request)

            # Jsonify the request body

            body = json_format.MessageToJson(
                transcoded_request["body"],
                including_default_value_fields=False,
                use_integers_for_enums=True,
            )
            uri = transcoded_request["uri"]
            method = transcoded_request["method"]

            # Jsonify the query params
            query_params = json.loads(
                json_format.MessageToJson(
                    transcoded_request["query_params"],
                    including_default_value_fields=False,
                    use_integers_for_enums=True,
                )
            )

            query_params["$alt"] = "json;enum-encoding=int"

            # Send the request
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(self._session, method)(
                "{host}{uri}".format(host=self._host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = operations_pb2.Operation()
            json_format.Parse(response.content, resp, ignore_unknown_fields=True)
            resp = self._interceptor.post_create_domain_mapping(resp)
            return resp

    class _DeleteDomainMapping(DomainMappingsRestStub):
        def __hash__(self):
            return hash("DeleteDomainMapping")

        def __call__(
            self,
            request: appengine.DeleteDomainMappingRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, str]] = (),
        ) -> operations_pb2.Operation:
            r"""Call the delete domain mapping method over HTTP.

            Args:
                request (~.appengine.DeleteDomainMappingRequest):
                    The request object. Request message for
                ``DomainMappings.DeleteDomainMapping``.
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, str]]): Strings which should be
                    sent along with the request as metadata.

            Returns:
                ~.operations_pb2.Operation:
                    This resource represents a
                long-running operation that is the
                result of a network API call.

            """

            http_options: List[Dict[str, str]] = [
                {
                    "method": "delete",
                    "uri": "/v1/{name=apps/*/domainMappings/*}",
                },
            ]
            request, metadata = self._interceptor.pre_delete_domain_mapping(
                request, metadata
            )
            pb_request = appengine.DeleteDomainMappingRequest.pb(request)
            transcoded_request = path_template.transcode(http_options, pb_request)

            uri = transcoded_request["uri"]
            method = transcoded_request["method"]

            # Jsonify the query params
            query_params = json.loads(
                json_format.MessageToJson(
                    transcoded_request["query_params"],
                    including_default_value_fields=False,
                    use_integers_for_enums=True,
                )
            )

            query_params["$alt"] = "json;enum-encoding=int"

            # Send the request
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(self._session, method)(
                "{host}{uri}".format(host=self._host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = operations_pb2.Operation()
            json_format.Parse(response.content, resp, ignore_unknown_fields=True)
            resp = self._interceptor.post_delete_domain_mapping(resp)
            return resp

    class _GetDomainMapping(DomainMappingsRestStub):
        def __hash__(self):
            return hash("GetDomainMapping")

        def __call__(
            self,
            request: appengine.GetDomainMappingRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, str]] = (),
        ) -> domain_mapping.DomainMapping:
            r"""Call the get domain mapping method over HTTP.

            Args:
                request (~.appengine.GetDomainMappingRequest):
                    The request object. Request message for ``DomainMappings.GetDomainMapping``.
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, str]]): Strings which should be
                    sent along with the request as metadata.

            Returns:
                ~.domain_mapping.DomainMapping:
                    A domain serving an App Engine
                application.

            """

            http_options: List[Dict[str, str]] = [
                {
                    "method": "get",
                    "uri": "/v1/{name=apps/*/domainMappings/*}",
                },
            ]
            request, metadata = self._interceptor.pre_get_domain_mapping(
                request, metadata
            )
            pb_request = appengine.GetDomainMappingRequest.pb(request)
            transcoded_request = path_template.transcode(http_options, pb_request)

            uri = transcoded_request["uri"]
            method = transcoded_request["method"]

            # Jsonify the query params
            query_params = json.loads(
                json_format.MessageToJson(
                    transcoded_request["query_params"],
                    including_default_value_fields=False,
                    use_integers_for_enums=True,
                )
            )

            query_params["$alt"] = "json;enum-encoding=int"

            # Send the request
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(self._session, method)(
                "{host}{uri}".format(host=self._host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = domain_mapping.DomainMapping()
            pb_resp = domain_mapping.DomainMapping.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)
            resp = self._interceptor.post_get_domain_mapping(resp)
            return resp

    class _ListDomainMappings(DomainMappingsRestStub):
        def __hash__(self):
            return hash("ListDomainMappings")

        def __call__(
            self,
            request: appengine.ListDomainMappingsRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, str]] = (),
        ) -> appengine.ListDomainMappingsResponse:
            r"""Call the list domain mappings method over HTTP.

            Args:
                request (~.appengine.ListDomainMappingsRequest):
                    The request object. Request message for
                ``DomainMappings.ListDomainMappings``.
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, str]]): Strings which should be
                    sent along with the request as metadata.

            Returns:
                ~.appengine.ListDomainMappingsResponse:
                    Response message for
                ``DomainMappings.ListDomainMappings``.

            """

            http_options: List[Dict[str, str]] = [
                {
                    "method": "get",
                    "uri": "/v1/{parent=apps/*}/domainMappings",
                },
            ]
            request, metadata = self._interceptor.pre_list_domain_mappings(
                request, metadata
            )
            pb_request = appengine.ListDomainMappingsRequest.pb(request)
            transcoded_request = path_template.transcode(http_options, pb_request)

            uri = transcoded_request["uri"]
            method = transcoded_request["method"]

            # Jsonify the query params
            query_params = json.loads(
                json_format.MessageToJson(
                    transcoded_request["query_params"],
                    including_default_value_fields=False,
                    use_integers_for_enums=True,
                )
            )

            query_params["$alt"] = "json;enum-encoding=int"

            # Send the request
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(self._session, method)(
                "{host}{uri}".format(host=self._host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = appengine.ListDomainMappingsResponse()
            pb_resp = appengine.ListDomainMappingsResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)
            resp = self._interceptor.post_list_domain_mappings(resp)
            return resp

    class _UpdateDomainMapping(DomainMappingsRestStub):
        def __hash__(self):
            return hash("UpdateDomainMapping")

        def __call__(
            self,
            request: appengine.UpdateDomainMappingRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, str]] = (),
        ) -> operations_pb2.Operation:
            r"""Call the update domain mapping method over HTTP.

            Args:
                request (~.appengine.UpdateDomainMappingRequest):
                    The request object. Request message for
                ``DomainMappings.UpdateDomainMapping``.
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, str]]): Strings which should be
                    sent along with the request as metadata.

            Returns:
                ~.operations_pb2.Operation:
                    This resource represents a
                long-running operation that is the
                result of a network API call.

            """

            http_options: List[Dict[str, str]] = [
                {
                    "method": "patch",
                    "uri": "/v1/{name=apps/*/domainMappings/*}",
                    "body": "domain_mapping",
                },
            ]
            request, metadata = self._interceptor.pre_update_domain_mapping(
                request, metadata
            )
            pb_request = appengine.UpdateDomainMappingRequest.pb(request)
            transcoded_request = path_template.transcode(http_options, pb_request)

            # Jsonify the request body

            body = json_format.MessageToJson(
                transcoded_request["body"],
                including_default_value_fields=False,
                use_integers_for_enums=True,
            )
            uri = transcoded_request["uri"]
            method = transcoded_request["method"]

            # Jsonify the query params
            query_params = json.loads(
                json_format.MessageToJson(
                    transcoded_request["query_params"],
                    including_default_value_fields=False,
                    use_integers_for_enums=True,
                )
            )

            query_params["$alt"] = "json;enum-encoding=int"

            # Send the request
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(self._session, method)(
                "{host}{uri}".format(host=self._host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = operations_pb2.Operation()
            json_format.Parse(response.content, resp, ignore_unknown_fields=True)
            resp = self._interceptor.post_update_domain_mapping(resp)
            return resp

    @property
    def create_domain_mapping(
        self,
    ) -> Callable[[appengine.CreateDomainMappingRequest], operations_pb2.Operation]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._CreateDomainMapping(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def delete_domain_mapping(
        self,
    ) -> Callable[[appengine.DeleteDomainMappingRequest], operations_pb2.Operation]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._DeleteDomainMapping(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def get_domain_mapping(
        self,
    ) -> Callable[[appengine.GetDomainMappingRequest], domain_mapping.DomainMapping]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._GetDomainMapping(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def list_domain_mappings(
        self,
    ) -> Callable[
        [appengine.ListDomainMappingsRequest], appengine.ListDomainMappingsResponse
    ]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._ListDomainMappings(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def update_domain_mapping(
        self,
    ) -> Callable[[appengine.UpdateDomainMappingRequest], operations_pb2.Operation]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._UpdateDomainMapping(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def kind(self) -> str:
        return "rest"

    def close(self):
        self._session.close()


__all__ = ("DomainMappingsRestTransport",)
