"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) The type of VPC traffic to log.
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    FlowLogTrafficType["ALL"] = "ALL";
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * (experimental) The available destination types for Flow Logs.
 *
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * (experimental) The type of resource to create the flow log for.
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * (experimental) The subnet to attach the Flow Log to.
     *
     * @experimental
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * (experimental) The VPC to attach the Flow Log to.
     *
     * @experimental
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * (experimental) The Network Interface to attach the Flow Log to.
     *
     * @experimental
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
_a = JSII_RTTI_SYMBOL_1;
FlowLogResourceType[_a] = { fqn: "@aws-cdk/aws-ec2.FlowLogResourceType", version: "1.90.0" };
/**
 * (experimental) The destination type for the flow log.
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * (experimental) Use CloudWatch logs as the destination.
     *
     * @experimental
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * (experimental) Use S3 as the destination.
     *
     * @experimental
     */
    static toS3(bucket, keyPrefix) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
            keyPrefix,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
_b = JSII_RTTI_SYMBOL_1;
FlowLogDestination[_b] = { fqn: "@aws-cdk/aws-ec2.FlowLogDestination", version: "1.90.0" };
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
            keyPrefix: this.props.keyPrefix,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * (experimental) A VPC flow log.
 *
 * @experimental
 * @resource AWS::EC2::FlowLog
 */
class FlowLog extends FlowLogBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        this.keyPrefix = destinationConfig.keyPrefix;
        let logDestination = undefined;
        if (this.bucket) {
            logDestination = this.keyPrefix ? this.bucket.arnForObjects(this.keyPrefix) : this.bucket.bucketArn;
        }
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * (experimental) Import a Flow Log by it's Id.
     *
     * @experimental
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
_c = JSII_RTTI_SYMBOL_1;
FlowLog[_c] = { fqn: "@aws-cdk/aws-ec2.FlowLog", version: "1.90.0" };
//# sourceMappingURL=data:application/json;base64,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