import re
from enum import Enum
from typing import List, cast, Type, Optional

from collections_extended import RangeMap
from pydantic import Field, BaseModel
from pymultirole_plugins.v1.processor import ProcessorParameters, ProcessorBase
from pymultirole_plugins.v1.schema import Document, Sentence, Annotation


class AugmentationStrategy(str, Enum):
    restore_originals = "restore_originals"
    pattern = "pattern"


class GenerativeAugmenterParameters(ProcessorParameters):
    strategy: AugmentationStrategy = Field(
        AugmentationStrategy.pattern,
        description="""The augmentation strategy to apply""",
    )
    to_lowercase: bool = Field(
        False,
        description="""Put the text of the variants in lower case.""",
    )
    modify_doc_id: bool = Field(
        True,
        description="""If true add a prefix to existing document identifier.""",
    )
    variant_altText: str = Field(
        "variants",
        description="""The alternative text where are stored the variants generated by another component.""",
    )
    variant_separator_regex: Optional[str] = Field(
        None, description="Regex used to split variants", extra="advanced"
    )
    annotation_regex: Optional[str] = Field(
        "\\[(?P<label>[^:\\]]+):(?P<text>[^]]+)\\]", description="Regex used to extract entities with labels",
        extra="advanced"
    )


class GenerativeAugmenterProcessor(ProcessorBase):
    __doc__ = """Replace text of the input document by the variants and replace the existing annotations in text."""

    def process(
            self, documents: List[Document], parameters: ProcessorParameters
    ) -> List[Document]:
        # supported_languages = comma_separated_to_list(SUPPORTED_LANGUAGES)
        params: GenerativeAugmenterParameters = cast(
            GenerativeAugmenterParameters, parameters
        )
        try:
            for document in documents:
                altTexts = document.altTexts or []
                variant_altText = next(alt for alt in altTexts if
                                       alt.name == params.variant_altText)
                sentences = []
                annotations = []
                dtext = ""
                if variant_altText:
                    variant_text = variant_altText.text
                    for stext, sannots in segment_augmented_text(document, variant_text, params):
                        sentences.append(Sentence(start=len(dtext), end=len(dtext) + len(stext)))
                        for a in sannots:
                            a.start += len(dtext)
                            a.end += len(dtext)
                            annotations.append(a)
                        dtext += stext
                    document.sentences = sentences
                    if params.modify_doc_id:
                        document.identifier = f"{document.identifier}-augmentedby-{variant_altText.name}"
                    document.metadata = document.metadata or {}
                    document.metadata['augmentedby'] = variant_altText.name
                    document.altTexts = None
                    document.text = dtext.lower() if params.to_lowercase else dtext
                    document.annotations = annotations

        except BaseException as err:
            raise err
        return documents

    @classmethod
    def get_model(cls) -> Type[BaseModel]:
        return GenerativeAugmenterParameters


def left_longest_match(a: Annotation):
    return a.end - a.start, -a.start


def natural_order(a: Annotation):
    return -a.start, a.end - a.start


def segment_augmented_text(document, variant_text, params):
    def extract_annotations(stext):
        annotations = []
        if params.strategy == AugmentationStrategy.restore_originals:
            text = stext.lower() if params.to_lowercase else stext
            seen_offsets = RangeMap()
            sorted_annotations = sorted(document.annotations, key=left_longest_match, reverse=True)
            for a in sorted_annotations:
                if (
                        seen_offsets.get(a.start) is None
                        and seen_offsets.get(a.end - 1) is None
                ):
                    seen_offsets[a.start: a.end] = a
                    atext = a.text or text[a.start:a.end]
                    index = 0
                    while index < len(document.text):
                        index = document.text.find(atext, index)
                        if index == -1:
                            break
                        annotations.append(
                            Annotation(start=index, end=index + len(atext), labelName=a.labelName, label=a.label,
                                       text=atext))
                        index = index + len(atext)
            annotations = sorted(annotations, key=natural_order, reverse=True)
            return stext, annotations
        elif params.strategy == AugmentationStrategy.pattern:
            rtext = ""
            sstart = 0
            matches = re.finditer(params.annotation_regex, stext)
            for matchNum, match in enumerate(matches, start=1):
                rtext += stext[sstart:match.start()]
                atext = match.group('text')
                annotations.append(
                    Annotation(start=len(rtext), end=len(rtext) + len(atext), **match.groupdict()))
                rtext += atext
                sstart = match.end()
            if sstart < len(stext):
                rtext += stext[sstart:]
            return rtext, annotations

    sstart = -1
    if params.variant_separator_regex:
        matches = re.finditer(params.variant_separator_regex, variant_text, re.MULTILINE)
        for matchNum, match in enumerate(matches, start=1):
            send = match.start()
            if sstart >= 0:
                stext = variant_text[sstart:send]
                yield extract_annotations(stext)
            sstart = match.end()
    if sstart < len(variant_text):
        sstart = 0 if sstart == -1 else sstart
        stext = variant_text[sstart:]
        yield extract_annotations(stext)
