# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/04_bbox_annotator.ipynb (unless otherwise specified).

__all__ = ['BBoxState']

# Internal Cell

import warnings
import functools
from pathlib import Path
from copy import deepcopy
from typing import Optional, Dict, List

from IPython.core.display import display
from ipywidgets import (AppLayout, Button, HBox, VBox, Layout)

from .base import BaseState, AppWidgetState
from .bbox_canvas import BBoxCanvas, BBoxCanvasState
from .navi_widget import Navi
from .storage import JsonCaptureStorage

# Cell

class BBoxState(BaseState):
    coords: Optional[List[Dict[str, float]]]
    image: Optional[str]

# Internal Cell

class BBoxList(VBox):
    def __init__(
        self,
        bbox_state: BBoxCanvasState,
        label_clicked: callable = None
    ):
        super().__init__(
            layout=Layout(
                min_width='400px'
            )
        )

        self._state = bbox_state
        self._label_clicked = label_clicked

        if self._state.bbox_coords:
            self._render_btn_list(self._state.bbox_coords)

        bbox_state.subscribe(self._render_btn_list, 'bbox_coords')

    def _render_btn_list(self, bbox_coords: list):
        elements = []

        for index, coord in enumerate(bbox_coords):
            btn_annotation = Button(description=f'Annotation {index+1}')
            btn_del_annotation = Button(
                layout=Layout(width='auto'),
                icon="fa-trash",
                button_style="danger",
                value=index
            )

            if self._label_clicked:
                btn_annotation.on_click(
                     functools.partial(self._label_clicked, index=index)
                )

            btn_del_annotation.on_click(
                functools.partial(self._del_element, index=index)
            )

            elements.append(
                HBox([
                    btn_annotation,
                    btn_del_annotation
                ])
            )

        self.children = elements

    def _del_element(self, event: dict, index: int = None):
        if index is not None:
            elements = list(self.children)
            coords = self._state.bbox_coords.copy()
            del coords[index]
            del elements[index]
            self.children = elements
            self._state.bbox_coords = coords

# Internal Cell

class BBoxAnnotatorGUI(AppLayout):
    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_state: BBoxState,
        save_callable: callable = None
    ):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._save_callable = save_callable

        self._navi = Navi()

        self._save_btn = Button(description="Save",
                                layout=Layout(width='auto'))

        self._undo_btn = Button(description="Undo",
                                icon="fa-undo",
                                layout=Layout(width='auto'))

        self._redo_btn = Button(description="Redo",
                                icon="fa-rotate-right",
                                layout=Layout(width='auto'))

        self._controls_box = HBox(
            [self._navi, self._save_btn, self._undo_btn, self._redo_btn],
            layout=Layout(
                display='flex',
                flex_flow='row wrap',
                align_items='center'
            )
        )

        self._image_box = BBoxCanvas(*self._app_state.size)

        self._element_list = BBoxList(
            self._image_box.state,
            label_clicked=self._highlight_bbox
        )

        # set the values already instantiated on state
        if self._app_state.max_im_number:
            self._set_max_im_number(self._app_state.max_im_number)

        if self._bbox_state.image:
            self._set_image_path(self._bbox_state.image)

        if self._bbox_state.coords:
            self._set_coords(self._bbox_state.coords)

        # set the GUI interactions as callables
        self._navi.navi_callable = self._idx_changed
        self._save_btn.on_click(self._save_btn_clicked)
        self._undo_btn.on_click(self._undo_clicked)
        self._redo_btn.on_click(self._redo_clicked)

        bbox_state.subscribe(self._set_image_path, 'image')
        bbox_state.subscribe(self._set_coords, 'coords')
        app_state.subscribe(self._set_max_im_number, 'max_im_number')

        super().__init__(header=None,
                 left_sidebar=None,
                 center=self._image_box,
                 right_sidebar=self._element_list,
                 footer=self._controls_box,
                 pane_widths=(2, 8, 0),
                 pane_heights=(1, 4, 1))

    def _highlight_bbox(self, event: dict, index: int):
        self._image_box.highlight = index

    def _redo_clicked(self, event: dict):
        self._image_box.redo_bbox()

    def _undo_clicked(self, event: dict):
        self._image_box.undo_bbox()

    def _set_image_path(self, image: str):
        self._image_box._state.image_path = image

    def _set_coords(self, coords: list):
        if coords:
            tmp_coords = deepcopy(self._image_box._state.bbox_coords)
            tmp_coords.append(coords)
            self._image_box._state.bbox_coords = coords

    def _set_max_im_number(self, max_im_number: int):
        # sync the navi GUI with AppWidgetState
        self._navi.max_im_num = max_im_number

    def _idx_changed(self, index: int):
        # store the last bbox drawn before index update
        self._bbox_state.coords = self._image_box._state.bbox_coords
        self._app_state.index = index

    def _save_btn_clicked(self, *args):
        if self._save_callable:
            self._save_callable(self._image_box._state.bbox_coords)
        else:
            warnings.warn("Save button click didn't triggered any event.")

    def on_client_ready(self, callback):
        self._image_box.observe_client_ready(callback)

# Internal Cell

class BBoxAnnotatorController:
    def __init__(self, app_state: AppWidgetState, bbox_state: BBoxState, storage: JsonCaptureStorage):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._storage = storage
        self._last_index = 0

        app_state.subscribe(self._idx_changed, 'index')

        self._update_im(self._last_index)
        self._app_state.max_im_number = len(self._storage)
        self._update_coords(self._last_index)

    def save_current_annotations(self, coords: dict):
        self._bbox_state.set_quietly('coords', coords)
        self._save_annotations(self._app_state.index)

    def _update_im(self, index: int):
        self._bbox_state.image = self._storage.images[index]

    def _update_coords(self, index: int): # from annotations
        image_path = str(self._storage.images[index])
        self._bbox_state.coords = self._storage.get(image_path) or {}

    def _save_annotations(self, index: int, *args, **kwargs): # to disk
        image_path = str(self._storage.images[index])
        self._storage[image_path] = self._bbox_state.coords
        self._storage.save()

    def _idx_changed(self, index: int):
        """
        On index change save an old state and update
        current image path and bbox coordinates for
        visualisation
        """
        self._save_annotations(self._last_index)
        self._update_im(index)
        self._update_coords(index)
        self._last_index = index

# Internal Cell

class BBoxAnnotator:
    """
    Represents bounding box annotator.

    Gives an ability to itarate through image dataset,
    draw 2D bounding box annotations for object detection and localization,
    export final annotations in json format

    """

    def __init__(
        self,
        project_path: Path,
        input_item,
        output_item,
        annotation_file_path: Path,
        *args, **kwargs
    ):
        self.app_state = AppWidgetState(
            uuid=id(self),
            **{
                'size': (input_item.width, input_item.height),
            }
        )

        self.bbox_state = BBoxState(
            uuid=id(self)
        )

        self.storage = JsonCaptureStorage(
            im_dir=project_path / input_item.dir,
            annotation_file_path=annotation_file_path
        )

        self.controller = BBoxAnnotatorController(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            storage=self.storage
        )

        self.view = BBoxAnnotatorGUI(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            save_callable=self.controller.save_current_annotations
        )

    def __repr__(self):
        display(self.view)
        return ""

    def to_dict(self, only_annotated=True):
        return self.storage.to_dict(only_annotated)