# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_base.ipynb (unless otherwise specified).

__all__ = ['Input', 'Output', 'InputImage', 'OutputImageLabel', 'OutputImageBbox']

# Internal Cell

import json
import random
import uuid
from pubsub import pub
from pathlib import Path
from typing import NamedTuple, Optional, Tuple
from pydantic import BaseModel, BaseSettings

# Internal Cell

def validate_project_path(project_path):
    project_path = Path(project_path)
    assert project_path.exists(), "WARNING: Project path should point to " \
                                  "existing directory"
    assert project_path.is_dir(), "WARNING: Project path should point to " \
                                  "existing directory"
    return project_path

# Internal Cell

class Settings(NamedTuple):
    project_path: Path
    project_file: Optional[Path] = None
    image_dir: str = 'images'
    label_dir: Optional[str] = None
    result_dir: Optional[str] = None

    im_width: int = 50
    im_height: int = 50
    label_width: int = 50
    label_height: int = 50

    n_cols: int = 3
    n_rows: Optional[int] = None

# Cell

# todo: use pydantic
class Input():
    """
    Abstract class to represent input
    """
    def __repr__(self):
            return f"Annotator Input type: {self.__class__.__name__}"


class Output():
    """
    Abstract class to represent input
    """
    def __repr__(self):
        return f"Annotator Output type: {self.__class__.__name__}"

# Cell
class InputImage(Input):
    """
    image_dir: string
        Directory of the image

    image_width: int
        Width of the image

    image_height: int
        Height of the image
    """

    def __init__(self, image_dir='pics', image_width=100, image_height=100):
        self.width = image_width
        self.height = image_height
        self.dir = image_dir

# Cell

class OutputImageLabel(Output):
    """
     Will handle the image output.
     If no `label_dir` is specified, it should generate randomized one.

    """

    def __init__(self, label_dir=None, label_width=50, label_height=50):
        self.width = label_width
        self.height = label_height

        if label_dir is None:
            self.dir = 'class_autogenerated_' + ''.join(random.sample(str(uuid.uuid4()), 5))
        else:
            self.dir = label_dir

# Cell
class OutputImageBbox(Output):
    pass

# Internal Cell
class OutputGridBox(Output):
    pass

# Internal Cell


def generate_subset_anno_json(project_path:Path, project_file,
                              number_of_labels,
                              out_filename='subset_anno.json'):
    """
    generates random subset from full dataset based on <number_of_labels>
    """
    if number_of_labels == -1:
        return project_file

    with project_file.open() as f:
        data = json.load(f)

    all_labels = data.values()
    unique_labels = set(label for item_labels in all_labels for label in item_labels)

    #  get <number_of_labels> random labels and generate annotation file with them:
    assert (number_of_labels <= len(unique_labels))
    subset_labels = random.sample([[a] for a in unique_labels], k=number_of_labels)
    subset_annotations = {k: v for k, v in data.items() if v in subset_labels}

    subset_file = Path(project_path) / out_filename
    with subset_file.open('w', encoding='utf-8') as fi:
        json.dump(subset_annotations, fi, ensure_ascii=False, indent=4)

    return subset_file

# Internal Cell

class StateSettings(BaseSettings):
    class Config:
        validate_assignment = True

class BaseState(StateSettings, BaseModel):
    def __init__(self, uuid: str = None, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.set_quietly('_uuid', uuid)

    def set_quietly(self, key: str, value: any):
        """
        Assigns a value to a state's attribute.

        This function can be used to avoid that
        the state dispatches a PyPubSub event.
        It's very usefull to avoid event recursion,
        ex: a component is listening for an event A
        but it also changes the state that dispatch
        the event A. Using set_quietly to set the
        value at the component will avoid the recursion.
        """
        object.__setattr__(self, key, value)

    @property
    def root_topic(self) -> str:
        if hasattr(self, '_uuid') and self._uuid:
            return f'{self._uuid}.{type(self).__name__}'

        return type(self).__name__

    def subscribe(self, fun: callable, attribute: str):
        pub.subscribe(fun, f'{self.root_topic}.{attribute}')

    def __setattr__(self, key: str, value: any):
        self.set_quietly(key, value)

        if key != '__class__':
            pub.sendMessage(f'{self.root_topic}.{key}', **{key:value})

# Internal Cell

class AppWidgetState(BaseState):
    size: Tuple[int, int] = (640, 400)
    max_im_number: int = 1
    index: int = 0