# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00a_annotator.ipynb (unless otherwise specified).

__all__ = ['Annotator']

# Internal Cell
import json
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Tuple

from skimage import io
from tqdm import tqdm

from .base import (Settings, generate_subset_anno_json, Input, Output,
                   InputImage, OutputImageLabel, OutputImageBbox, OutputGridBox)
from .bbox_annotator import BBoxAnnotator
from .capture_annotator import CaptureAnnotator
from .datasets.generators import draw_bbox
from .im2im_annotator import Im2ImAnnotator
from .storage import (construct_annotation_path, group_files_by_class)

# Internal Cell
class AnnotatorFactory(ABC):
    io: Tuple[Input, Output] = None

    @abstractmethod
    def get_annotator(self):
        pass

    def __new__(cls, input_item, output_item):
        subclass_map = {subclass.io: subclass for subclass in cls.__subclasses__()}
        try:
            subclass = subclass_map[(type(input_item), type(output_item))]
            instance = super(AnnotatorFactory, subclass).__new__(subclass)
            return instance
        except KeyError as vE:
            print(f"Pair {(input_item, output_item)} is not supported!")

#
# Define all supported annotators with correct Input/Output pairs for internal use below:
#

class Bboxer(AnnotatorFactory):
    io = (InputImage, OutputImageBbox)

    def get_annotator(self):
        return BBoxAnnotator


class Im2Imer(AnnotatorFactory):
    io = (InputImage, OutputImageLabel)

    def get_annotator(self):
        return Im2ImAnnotator


class Capturer(AnnotatorFactory):
    io = (InputImage, OutputGridBox)

    def get_annotator(self):
        return CaptureAnnotator

# Cell
class Annotator:
    def __init__(self, settings:Settings, input_item, output_item):
        self.settings = settings
        self.input_item = input_item
        self.output_item = output_item

    def explore(self, k=-1):
        '''
        Lets visualize existing annotated dataset
        As we don't have images for each class we provide label_dir=None for ipyannotator,
        thus class labels will be generated automatically based on annotation.json file.

        To explore a part of dataset set `k` - number of classes to display; By default explore `all` (k == -1)
        '''
        subset = generate_subset_anno_json(project_path=self.settings.project_path,
                                           project_file=self.settings.project_file,
                                           number_of_labels=k)

        anno_ = construct_annotation_path(project_path=self.settings.project_path, file_name=subset, results_dir=None)

        annotator = AnnotatorFactory(self.input_item, self.output_item).get_annotator()

        return annotator(project_path=self.settings.project_path,
                                      input_item=self.input_item,
                                      output_item=self.output_item,
                                      annotation_file_path=anno_,
                                      n_cols=self.settings.n_cols,
                                      question="Classification <explore>",
                                      has_border=True)


    def create(self):
        # to create: file_name=None and define empty results_dir
        anno_ = construct_annotation_path(project_path=self.settings.project_path,
                                          file_name=None,
                                          results_dir=self.settings.result_dir)

        annotator = AnnotatorFactory(self.input_item, self.output_item).get_annotator()

        return annotator(project_path=self.settings.project_path,
                                      input_item=self.input_item,
                                      output_item=self.output_item,
                                      annotation_file_path=anno_,
                                      n_cols=self.settings.n_cols,
                                      question="Classification <create>",
                                      has_border=True)


    def improve(self):
        # open labels from create step
        create_step_annotations = Path(self.settings.project_path) / self.settings.result_dir / 'annotations.json'
        with open(create_step_annotations) as infile:
            loaded_image_annotations = json.load(infile)


        ### @TODO?
        if type(self.output_item) == OutputImageLabel:

            #Construct multiple Capturers for each class
            #
            out = []
            for class_name, class_anno in tqdm(group_files_by_class(loaded_image_annotations).items()):
                anno_ = construct_annotation_path(project_path=self.settings.project_path,
                                                  results_dir=f'{self.settings.result_dir}/missed/{class_name[:-4]}')

                out.append(CaptureAnnotator(self.settings.project_path,
                                            input_item=self.input_item,
                                            output_item=OutputGridBox(),
                                            annotation_file_path=anno_,
                                            n_cols=2, n_rows=5,
                                            question=f'Check incorrect annotation for [{class_name[:-4]}] class',
                                            filter_files=class_anno))

        elif type(self.output_item) == OutputImageBbox:
            out = None
            # back to artificial bbox format ->
            di = {k: [v[0]['x'], v[0]['y'], v[0]['width'], v[0]['height']] if v else [] for k, v in loaded_image_annotations.items()}

            captured_path = Path(self.settings.project_path) / "captured"

            # Save annotated images on disk
            #
            for im, bbx in tqdm(di.items()):
                # use captured_path instead image_dir, keeping the folder structure
                old_im_path = Path(im)

                index = old_im_path.parts.index(self.input_item.dir)+1
                new_im_path = captured_path.joinpath(*old_im_path.parts[index:])
                new_im_path.parent.mkdir(parents=True, exist_ok=True)

                _image = io.imread(im)
                if bbx:
                    rect = [bbx[1], bbx[0]]
                    rect_dimensions = [bbx[3], bbx[2]]

                    _image = draw_bbox(rect=rect, rect_dimensions=rect_dimensions, im=_image, black=True)

                io.imsave(str(new_im_path), _image, check_contrast=False)

            # Construct Capturer
            #
            in_p = InputImage(image_dir="captured", image_width=150, image_height=150)
            out_p = OutputGridBox()
            anno_ = construct_annotation_path(self.settings.project_path,
                                              results_dir=f'{self.settings.result_dir}/missed')
            out = CaptureAnnotator(self.settings.project_path, input_item=in_p, output_item=out_p,
                      annotation_file_path=anno_,
                      n_cols=3, question="Check images with incorrect or empty bbox annotation")

        else:
            raise Exception(f"Improve is not supported for {self.output_item}")

        return out