"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const tmp = require("tmp");
const fs = require("fs");
class Seeder extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.setup || !Array.isArray(props.setup))
            throw new Error('setup value must be an array of JSON objects');
        this.props = props;
        const destinationBucket = new aws_s3_1.Bucket(this, 'acds-bucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            this.writeTempFile(dir, 'setup.json', props.setup);
            if (props.teardown) {
                this.writeTempFile(dir, 'teardown.json', props.teardown);
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false,
            });
        });
        const fn = new aws_lambda_1.Function(this, 'handler', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            timeout: core_1.Duration.seconds(60),
            code: aws_lambda_1.Code.fromInline(`
console.log('function loaded');

const AWS = require('aws-sdk');
const s3 = new AWS.S3();

const writeTypeFromAction = (action) => {
  if (action === "Put")
    return "Item";
  if (action === "Delete")
    return "Key";
}

const run = async (filename, action) => {
  console.log('reading from s3');
  const data = await s3.getObject({
    Bucket: "${destinationBucket.bucketName}", 
    Key: filename
  }).promise();
  console.log('finished reading from s3');
  
  console.log('transforming seed data');
  const seed = JSON.parse(data.Body.toString());
  console.log('finished transforming seed data');
  
  const documentClient = new AWS.DynamoDB.DocumentClient({
    convertEmptyValues: true
  });
  console.log('sending data to dynamodb');
  do {
    const requests = [];
    const batch = seed.splice(0, 25);
    for (let i = 0; i < batch.length; i++) {
      requests.push({
        [action + "Request"]: {
          [writeTypeFromAction(action)]: batch[i]
        }
      });
    }
    await documentClient.batchWrite({
      RequestItems: {
        '${props.table.tableName}': [...requests]
      }
    }).promise();
  }
  while (seed.length > 0);
  console.log('finished sending data to dynamodb');
}

exports.handler = async (event) => {
  if (event.mode === "delete")
    await run("teardown.json", "delete");
  if (event.mode === "create" || event.mode === "update")
    await run("setup.json", "put");
}`),
        });
        destinationBucket.grantRead(fn);
        props.table.grantWriteData(fn);
        const onEvent = new custom_resources_1.AwsCustomResource(this, 'on-event', {
            onCreate: {
                ...this.callLambdaOptions(),
                parameters: {
                    FunctionName: fn.functionArn,
                    InvokeArgs: JSON.stringify({
                        mode: 'create',
                    }),
                },
            },
            onDelete: props.teardown
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'delete',
                        }),
                    },
                }
                : undefined,
            onUpdate: props.refreshOnUpdate
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'update',
                        }),
                    },
                }
                : undefined,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        fn.grantInvoke(onEvent);
    }
    callLambdaOptions() {
        return {
            service: 'Lambda',
            action: 'invokeAsync',
            apiVersion: '2015-03-31',
            physicalResourceId: {
                id: `${this.props.table.tableArn}-seeder`,
            },
        };
    }
    writeTempFile(dir, filename, data) {
        const buffer = Buffer.from(JSON.stringify(data));
        const filepath = dir + '/' + filename;
        fs.writeFileSync(filepath, buffer);
    }
}
exports.Seeder = Seeder;
//# sourceMappingURL=data:application/json;base64,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