# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['deep_ast']

package_data = \
{'': ['*']}

install_requires = \
['click-log>=0.4.0,<0.5.0', 'click>=8.1.2,<9.0.0']

entry_points = \
{'console_scripts': ['deep-ast = deep_ast.app:cli']}

setup_kwargs = {
    'name': 'deep-ast',
    'version': '0.0.1',
    'description': 'Walks the entire AST tree, including calls to other functions.',
    'long_description': '<!-- PROJECT SHIELDS -->\n<!--\n*** I\'m using markdown "reference style" links for readability.\n*** Reference links are enclosed in brackets [ ] instead of parentheses ( ).\n*** See the bottom of this document for the declaration of the reference variables\n*** for contributors-url, forks-url, etc. This is an optional, concise syntax you may use.\n*** https://www.markdownguide.org/basic-syntax/#reference-style-links\n-->\n[![Python][python-shield]][pypi-url]\n[![Latest][version-shield]][pypi-url]\n[![Tests][test-shield]][test-url]\n[![Coverage][codecov-shield]][codecov-url]\n[![License][license-shield]][license-url]\n<!-- [![Contributors][contributors-shield]][contributors-url]\n[![Forks][forks-shield]][forks-url]\n[![Stargazers][stars-shield]][stars-url]\n[![Issues][issues-shield]][issues-url] -->\n\n<!-- PROJECT LOGO -->\n<br />\n<p align="center">\n  <!-- <a href="https://github.com/DontShaveTheYak/deep-ast">\n    <img src="images/logo.png" alt="Logo" width="80" height="80">\n  </a> -->\n\n  <h3 align="center">Deep-AST</h3>\n\n  <p align="center">\n    Parse the AST for any <a href=https://docs.python.org/3/library/ast.html#ast.Call>Call</a> nodes encountered.\n    <!-- <br />\n    <a href="https://github.com/DontShaveTheYak/deep-ast"><strong>Explore the docs »</strong></a>\n    <br /> -->\n    <br />\n    <!-- <a href="https://github.com/DontShaveTheYak/deep-ast">View Demo</a>\n    · -->\n    <a href="https://github.com/DontShaveTheYak/deep-ast/issues">Report Bug</a>\n    ·\n    <a href="https://github.com/DontShaveTheYak/deep-ast/issues">Request Feature</a>\n    ·\n    <!-- <a href="https://la-tech.co/post/hypermodern-cloudformation/getting-started/">Guide</a> -->\n  </p>\n</p>\n\n## About The Project\n\n<!-- [![Product Name Screen Shot][product-screenshot]](https://example.com) -->\n\n`deep-ast` is a Python library that attempts to parse all AST that would be encountered when invoking a callable in python. It does this by extending the [ast.NodeVistor](https://docs.python.org/3/library/ast.html#ast.NodeVisitor) and [ast.NodeTransformer](https://docs.python.org/3/library/ast.html#ast.NodeTransformer) classes, so that when an [ast.Call](https://docs.python.org/3/library/ast.html#ast.Call) node is encountered, it\'s source code is parsed into an [ast.Node](https://docs.python.org/3/library/ast.html#ast.Call) and then passes the node to the [visit()](https://docs.python.org/3/library/ast.html#ast.NodeVisitor.visit) method.\n\nWe say attempts because currently there are some limitations. It cant parse any arbitrary code like `exec(\'rorrEeulaV esiar\'[::-1])` which raises a ValueError, functions that aren\'t written in Python and all python internals like `print()`.\n\nIf you have a way around these limitations then a PR would be greatly appreciated.\n\n## Getting Started\n\n### Prerequisites\n\n`deep-ast` requires python >= 3.7\n\n### Installation\n\n`deep-ast` is available as an easy to install pip package.\n```sh\npip install deep-ast\n```\n\n## Usage\n\ndeep-ast offers drop in replacements for the [ast.NodeVistor](https://docs.python.org/3/library/ast.html#ast.NodeVisitor) and [ast.NodeTransformer](https://docs.python.org/3/library/ast.html#ast.NodeTransformer) classes.\n\n```sh\nfrom deep_ast import DeepVisitor, DeepTransformer\n```\n\nTo start the deep processing you the `deep_visit()` method. Each function takes an optional `module` arugment. This is used if the function/method is not defined in the same file that you calling `deep_visit()` method in.\n\n### Example to get all exceptions\n\nThis example shows how you might deeply parse the AST to\nget all exceptions that might be thrown.\n\n```python3\n# Custom NodeVisitor to visit Raise nodes and record them\nclass ParseExceptions(DeepVisitor):\n    def __init__(self) -> None:\n        self.raw_exceptions = []\n        self.found_exceptions = []\n        super().__init__()\n\n    def _add_exception(self, name: str):\n        self.raw_exceptions.append(name)\n\n        if name not in self.found_exceptions:\n            self.found_exceptions.append(name)\n\n    def visit_Raise(self, node):\n\n        exception_obj = node.exc\n\n        if isinstance(exception_obj, (ast.Call, ast.Name):\n            name = (\n                exception_obj.id\n                if isinstance(exception_obj, ast.Name)\n                else exception_obj.func.id\n            )\n\n            self._add_exception(name)\n            return self.generic_visit(node)\n\n        self._add_exception("EmptyRaise")\n        return self.generic_visit(node)\n\n# Test functions to visit\ndef foo():\n    bar()\n    raise TypeError()\n\n\ndef bar():\n    raise ValueError()\n\nparser = ParseExceptions()\n\nparser.deep_visit(foo)\n\nprint(parser.found_exceptions) # prints [\'ValueError\', \'TypeError\']\n```\n\n## Roadmap\n\n- Parsing of deeply nested attribute calls like `foo().bar().bazz()`\n\n- Tracking of variable assignment:\n  ```python3\n  print_func = print\n  print_func()\n  ```\n\nSee the [open issues](https://github.com/DontShaveTheYak/deep-ast/issues) for a list of proposed features (and known issues).\n\n## Contributing\n\nContributions are what make the open-source community such an amazing place to learn, inspire, and create. Any contributions you make are **greatly appreciated**.\n\nThis project uses poetry to manage dependencies and pre-commit to run formatting, linting and tests. You will need to have both installed to your system as well as python 3.9.\n\n1. Fork the Project\n2. Setup the environment.  \n   This project uses vscode devcontainer to provide a completly configured development environment. If you are using vscode and have the remote container extension installed, you should be asked to use the devcontainer when you open this project inside of vscode.\n\n   If you are not using devcontainers then you will need to have python installed. Install the `poetry`, `nox`, `nox_poetry` and `pre-commit` packages. Then run `poetry install` and `pre-commit install` commands. \n\n   Most of the steps can be found in the [Dockerfile](.devcontainer/Dockerfile).\n2. Create your Feature Branch (`git checkout -b feature/AmazingFeature`)\n3. Commit your Changes (`git commit -m \'Add some AmazingFeature\'`)\n4. Push to the Branch (`git push origin feature/AmazingFeature`)\n5. Open a Pull Request\n\n<!-- ## License\n\nDistributed under the Apache-2.0 License. See [LICENSE.txt](./LICENSE.txt) for more information. -->\n\n## Contact\n\nLevi - [@shady_cuz](https://twitter.com/shady_cuz)\n\n<!-- ACKNOWLEDGEMENTS -->\n## Acknowledgements\n* Stackoverflow for getting me started [down this road](https://stackoverflow.com/questions/32560116/how-to-list-all-exceptions-a-function-could-raise-in-python-3).\n* @sethmlarson for asking me which [exceptions](https://github.com/urllib3/urllib3/issues/2648) can be raised.\n\n<!-- MARKDOWN LINKS & IMAGES -->\n<!-- https://www.markdownguide.org/basic-syntax/#reference-style-links -->\n[python-shield]: https://img.shields.io/pypi/pyversions/deep-ast?style=for-the-badge\n[version-shield]: https://img.shields.io/pypi/v/deep-ast?label=latest&style=for-the-badge\n[pypi-url]: https://pypi.org/project/deep-ast/\n[test-shield]: https://img.shields.io/github/workflow/status/DontShaveTheYak/deep-ast/Tests?label=Tests&style=for-the-badge\n[test-url]: https://github.com/DontShaveTheYak/deep-ast/actions?query=workflow%3ATests+branch%3Amaster\n[codecov-shield]: https://img.shields.io/codecov/c/gh/DontShaveTheYak/deep-ast/master?color=green&style=for-the-badge&token=bfF18q99Fl\n[codecov-url]: https://codecov.io/gh/DontShaveTheYak/deep-ast\n[contributors-shield]: https://img.shields.io/github/contributors/DontShaveTheYak/deep-ast.svg?style=for-the-badge\n[contributors-url]: https://github.com/DontShaveTheYak/deep-ast/graphs/contributors\n[forks-shield]: https://img.shields.io/github/forks/DontShaveTheYak/deep-ast.svg?style=for-the-badge\n[forks-url]: https://github.com/DontShaveTheYak/deep-ast/network/members\n[stars-shield]: https://img.shields.io/github/stars/DontShaveTheYak/deep-ast.svg?style=for-the-badge\n[stars-url]: https://github.com/DontShaveTheYak/deep-ast/stargazers\n[issues-shield]: https://img.shields.io/github/issues/DontShaveTheYak/deep-ast.svg?style=for-the-badge\n[issues-url]: https://github.com/DontShaveTheYak/deep-ast/issues\n[license-shield]: https://img.shields.io/github/license/DontShaveTheYak/deep-ast.svg?style=for-the-badge\n[license-url]: https://github.com/DontShaveTheYak/deep-ast/blob/master/LICENSE.txt\n[product-screenshot]: images/screenshot.png\n',
    'author': 'Levi Blaney',
    'author_email': 'shadycuz@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/DontShaveTheYak/deep-ast',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
