# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['bincombo']
setup_kwargs = {
    'name': 'bincombo',
    'version': '0.1.0',
    'description': 'Make any type binary-combinable with a single line of code.',
    'long_description': 'bincombo\n========\n\nMake any type binary-combinable with a single line of code.\n\nThe most obvious use case arguably is the creation of binary expressions over custom\ntypes, but ``bincombo`` is not limited to combination of boolean values.\nBy providing appropriate map/reduce operations, any type of data of the discrete\nmembers of a combination can be aggregated to form a combined result.\n\nThe implementation solely relies on inheritance, no meta programming is involved.\nYou are free to use your own metaclasses, should you wish to do so.\n\nSuppose you have a custom type ``Check``, which accepts or rejects values depending\non the result of the invocation of some callable.\nYou can make instances of it combinable using the binary operators `&` (and) and `|`\n(or) and also support the unary `~` (invert) by using the ``combinable()`` decorator::\n\n    import bincombo\n\n    @bincombo.combinable(methods=("check",))\n    class Check:\n        __slots__ = ("checker",)\n\n        def __init__(self, checker):\n            self.checker = checker\n\n        def check(self, value):\n            return self.checker(value)\n\nNow, ``Check`` objects can be binary-combined::\n\n    c1 = Check(lambda v: isinstance(v, int) and v >= 42)\n    c2 = Check(lambda v: isinstance(v, str))\n    c3 = Check(lambda v: "hello" in v)\n    c = c1 | c2 & ~c3\n    c.check(41)  # False\n    c.check(42)  # True\n    c.check("hello, world!")  # False\n    c.check("hey, world!")  # True\n\nThe ``combinable()`` decorator creates a number of types needed to represent discrete\nchecks and combinations thereof.\nThese are stored in a ``Config`` object, which is aavailable as class attribute\n``BIN_CONFIG`` of ``Check``.\nIt can be worth storing these types as module attributes alongside your ``Check``\nclass to have them at hand for explicit use or type checking::\n\n    BaseCheck = Check.BIN_CONFIG.base_type\n    CheckCombo = Check.BIN_CONFIG.combo_type\n    AllChecks = Check.BIN_CONFIG.and_type\n    AnyCheck = Check.BIN_CONFIG.or_type\n\nAll types in this module have ``__slots__`` defined for smaller memory footprints\nand improved lookup times, as have the types created by ``combinable()``.\nIt is recommended to also equip your own type with ``__slots__`` if possible to\nbenefit from entirely ``__dict__``-less objects.\n\nTo gain a better understanding of how all the types relate or to further customize\nthem, here is how you would make ``Check`` binary-combinable without using the\n``combinable()`` helper::\n\n    # This is a base class from which both the discrete Check type and the type\n    # representing a combination of Check objects will inherit.\n    # It can be used, for instance, to test whether some object is Check-like\n    # using isinstance(obj, BaseCheck).\n    class BaseCheck:\n        __slots__ = ()\n\n    # Check should support all three operations.\n    feature_mixins = (\n        bincombo.AndSupportMixin, bincombo.OrSupportMixin, bincombo.InvertSupportMixin\n    )\n\n    # This is the type that would normally be returned by combinable() decorator.\n    # Here it replaces the original Check, but you could also give it a different name.\n    class Check(*feature_mixins, Check, BaseCheck):\n        __slots__ = ()\n\n    # Objects of this type represent (possibly negated) AND/OR combinations of\n    # Check objects.\n    class CheckCombo(*feature_mixins, bincombo.Combo, BaseCheck):\n        __slots__ = ()\n\n        # Create a proxy check() method that queries an individual combo member.\n        # bincombo will call this method for all members, combining the returned\n        # boolean values using and/or, depending on the combination type, and possibly\n        # invert the final result.\n        @bincombo.combine_members\n        def check(self, member, value):\n            return member.check(value)\n\n    # This is an AND combination.\n    class AllChecks(bincombo.AndComboMixin, CheckCombo):\n        __slots__ = ()\n\n    # This is an OR combination.\n    class AnyCheck(bincombo.OrComboMixin, CheckCombo):\n        __slots__ = ()\n\n    # Finally, bincombo has to be taught all the types just created.\n    # By attaching the Config object to BaseCheck as a class attribute, both Check\n    # and CheckCombo instances will have it available due to inheritance.\n    BaseCheck.BIN_CONFIG = bincombo.Config(\n        BaseCheck, Check, CheckCombo, AllChecks, AnyCheck\n    )\n\nFurther customization of the combining abilities is possible, the documentations of\n``combinable()``, ``combine_members()`` and ``Config`` have more information.\n',
    'author': 'Robert Schindler',
    'author_email': 'dev@bob1.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bob1de/python-bincombo',
    'py_modules': modules,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
