"""
PyOghma_ML Module - Advanced Machine Learning Framework for Organic Photovoltaics

This module serves as the comprehensive interface for the PyOghma_ML package, 
providing a complete machine learning pipeline for organic photovoltaic (OPV) 
device analysis and prediction. The framework integrates data processing, 
neural network training, prediction capabilities, and automated report generation.

Submodules:
    Input: Handles experimental data processing from various laboratory sources.
        Supports multiple characterization types (JV, TPV, CE, CELIV) and provides
        standardized data parsing for different laboratory formats.

    Networks: Contains neural network architectures and model management systems.
        Implements Point, Ensemble, and Difference network types with support for
        residual connections and multi-input architectures.

    Training: Manages the complete training pipeline for machine learning models.
        Features adaptive learning rates, model persistence, and support for both
        new and continued training from existing models.

    Predicting: Provides prediction capabilities using trained models.
        Handles model loading, feature preprocessing, and batch prediction for
        experimental data analysis.

    Output: Generates comprehensive reports and visualizations.
        Creates LaTeX-based reports with experimental analysis, ML predictions,
        statistical comparisons, and performance metrics.

    Latex: Facilitates programmatic LaTeX document creation and compilation.
        Supports figures, tables, sections, and automatic PDF generation with
        custom styling and formatting.

    Figures: Manages matplotlib-based figure generation and customization.
        Provides utilities for scientific plotting, subplot management, and
        publication-ready visualizations.

    Labels: Handles label management and metadata processing for the OghmaNano framework.
        Interfaces with C-based token libraries for efficient label processing
        and provides unit conversion and formatting capabilities.

Key Features:
    - Multi-laboratory data compatibility
    - Automated hyperparameter tuning
    - Ensemble learning capabilities  
    - Comprehensive error analysis
    - Publication-ready report generation
    - Integration with OghmaNano simulation software

Example Usage:
    >>> from PyOghma_ML import Input, Networks, Output
    >>> 
    >>> # Load experimental data
    >>> data = Input.experiment('device_data/', 'JV', 'Brabec')
    >>> 
    >>> # Initialize and train networks
    >>> networks = Networks.initialise('networks/', 'Point')
    >>> networks.train_networks()
    >>> 
    >>> # Generate comprehensive report
    >>> output = Output(networks, data)
    >>> output.build_report()

Version: 2.0
Author: PyOghma_ML Development Team
License: See package documentation
"""

from .Input import Input

from .Networks import Networks
from .Networks import Model_Settings

from .Training import Training
from .Predicting import Predicting

from .Output import Output
from .Latex import Document
from .Figures import Figures
from .Labels import Label