"""
This module provides functionality for managing and processing labels in the OghmaNano framework.

It includes classes for handling metadata associated with labels, such as units, descriptions, and visibility.
The module also provides tools for interfacing with a C-based token library, enabling efficient retrieval
and conversion of label information. Additionally, it supports the parsing and formatting of labels
for use in machine learning workflows and LaTeX-based reports.
"""
import numpy as np
import ctypes
import os
import platform

lib = {}
lib_r = {}
lib_json = {}

if platform.system() == 'Linux':
    try:
        dll_path = os.path.join('/', 'usr', 'lib', 'liboghma_py.so')
        dll_lib = ctypes.CDLL(dll_path)
    except:
        print('Labels DLL not found.')
else:
    print('Oghma Labels are not currently supported on this operating system.')


class my_data:
    """
    A class to represent metadata for a label.

    Attributes:
        units (str): The units associated with the label.
        info (str): Additional information about the label.
        defaults (None): Default values (currently unused).
        widget (str): The widget type associated with the label.
        units_widget (str): The widget type for displaying units.
        hidden (bool): Whether the label is hidden.
        hide_on_token_eq (None): Conditions for hiding the label (currently unused).
        show_on_token_eq (None): Conditions for showing the label (currently unused).
        pack (list): Additional metadata for the label.
        token (str): The token associated with the label.
    """
    def __init__(self, b: str, info: str, widget: str) -> None:
        """
        Initialize a my_data instance.

        This method sets up metadata for a label with the provided information
        and initializes default values for various attributes used in the
        OghmaNano framework.

        Args:
            b (str): The units associated with the label (e.g., 'V', 'A', 'm/s').
            info (str): Additional descriptive information about the label.
            widget (str): The Qt widget type associated with the label for GUI display.

        Note:
            Default values are set for widget configuration, visibility settings,
            and conditional display rules. The pack list can store additional
            metadata as needed.
        """
        self.units = b
        self.info = info
        self.defaults = None
        self.widget = widget
        self.units_widget = "QLabel"
        self.hidden = False
        self.hide_on_token_eq = None
        self.show_on_token_eq = None
        self.pack = []
        self.token = ""


class c_list(ctypes.Structure):
    """
    A ctypes structure to represent a list in C.

    This structure provides a Python interface to C-style lists used in the
    OghmaNano token library. It manages memory allocation and list properties
    for efficient data exchange between Python and C code.

    Attributes:
        names (ctypes.c_void_p): Pointer to the array of names/strings in the list.
        len (ctypes.c_int): Current number of elements in the list.
        len_max (ctypes.c_int): Maximum capacity of the list before reallocation.

    Note:
        This structure is primarily used for interfacing with the C-based
        token library and should not be modified directly unless you understand
        the underlying memory management.
    """
    _fields_ = [('names', ctypes.c_void_p),
                ('len', ctypes.c_int),
                ('len_max', ctypes.c_int)]


class hash_list(ctypes.Structure):
    """
    A ctypes structure to represent a hash list in C.

    Attributes:
        names (ctypes.c_void_p): Pointer to the names in the hash list.
        data (ctypes.c_void_p): Pointer to the data in the hash list.
        index0 (ctypes.c_void_p): Pointer to the first index.
        index1 (ctypes.c_void_p): Pointer to the second index.
        len (ctypes.c_int): The length of the hash list.
        len_max (ctypes.c_int): The maximum length of the hash list.
        item_size (ctypes.c_int): The size of each item in the hash list.
        free_function (ctypes.c_void_p): Pointer to the free function.
        thread_safe (ctypes.c_int): Whether the hash list is thread-safe.
        all_data_valid (ctypes.c_int): Whether all data in the hash list is valid.
    """
    _fields_ = [('names', ctypes.c_void_p),
                ('data', ctypes.c_void_p),
                ('index0', ctypes.c_void_p),
                ('index1', ctypes.c_void_p),
                ('len', ctypes.c_int),
                ('len_max', ctypes.c_int),
                ('item_size', ctypes.c_int),
                ('free_function', ctypes.c_void_p),
                ('thread_safe', ctypes.c_int),
                ('all_data_valid', ctypes.c_int)]


fast_lib = hash_list()


class token_lib_item(ctypes.Structure):
    """
    A ctypes structure to represent a token library item in C.

    Attributes:
        token (ctypes.c_char * 100): The token string.
        english (ctypes.c_char * 100): The English description of the token.
        widget (ctypes.c_char * 100): The widget type associated with the token.
        units (ctypes.c_char * 100): The units associated with the token.
        log (ctypes.c_int): Whether the token is logged.
        hidden (ctypes.c_int): Whether the token is hidden.
        hide_on_token (c_list): Conditions for hiding the token.
        hide_on_value (c_list): Values for hiding the token.
        show_on_token (c_list): Conditions for showing the token.
        show_on_value (c_list): Values for showing the token.
        default_token (c_list): Default tokens.
        default_value (c_list): Default values.
        pack (c_list): Additional metadata for the token.
    """
    _fields_ = [('token', ctypes.c_char * 100),
                ('english', ctypes.c_char * 100),
                ('widget', ctypes.c_char * 100),
                ('units', ctypes.c_char * 100),
                ('log', ctypes.c_int),
                ('hidden', ctypes.c_int),
                ('hide_on_token', c_list),
                ('hide_on_value', c_list),
                ('show_on_token', c_list),
                ('show_on_value', c_list),
                ('default_token', c_list),
                ('default_value', c_list),
                ('pack', c_list)]


class tokens:
    """
    A class to manage token libraries and retrieve token metadata.

    Methods:
        build_token_lib(): Build the token library.
        find(token): Find a token in the library.
        ctoken_to_python_token(token): Convert a C token to a Python token.
        bytes2str(string): Convert bytes to a string.
        isbytes(string): Check if a string is in bytes format.
        str2bytes(string): Convert a string to bytes.
    """
    def __init__(self) -> None:
        """
        Initialize a tokens instance and build the token library if necessary.
        """
        global lib
        global dll_lib

        self.lib = lib
        self.dll_lib = dll_lib

        self.dll_lib.token_lib_find.restype = ctypes.c_void_p
        if len(lib) == 0:
            self.build_token_lib()

    def build_token_lib(self) -> None:
        """
        Build the token library by initializing and populating it.
        """
        global fast_list
        self.dll_lib.token_lib_init(ctypes.byref(fast_lib))
        self.dll_lib.token_lib_build(ctypes.byref(fast_lib))

    def find(self, token: str) -> list["my_data", bool]:
        """
        Find a token in the library.

        Args:
            token (str): The token to search for.

        Returns:
            my_data or bool: The token metadata as a my_data object, or False if not found.
        """
        search_token = token.strip()

        ret = self.dll_lib.token_lib_find(ctypes.byref(fast_lib), ctypes.c_char_p(self.str2bytes(search_token)))

        if ret != None:
            tok = token_lib_item.from_address(ret)
            token = self.ctoken_to_python_token(tok)
            return token

        else:
            return False

    def ctoken_to_python_token(self, token: "token_lib_item") -> "my_data":
        """
        Convert a C token to a Python token.

        Args:
            token (token_lib_item): The C token to convert.

        Returns:
            my_data: The converted Python token.
        """
        a = my_data(self.bytes2str(token.units), self.bytes2str(token.english), self.bytes2str(token.widget))
        a.token = self.bytes2str(token.token)
        a.hidden = bool(token.hidden)

        return token

    @staticmethod
    def bytes2str(string: bytes) -> str:
        """
        Convert bytes to a string.

        Args:
            string (bytes): The bytes to convert.

        Returns:
            str: The converted string.
        """
        ret = string
        try:
            ret = string.decode()
        except:
            pass
        return ret

    @staticmethod
    def isbytes(string: list[str, bytes]) -> bool:
        """
        Check if a string is in bytes format.

        Args:
            string (str or bytes): The string to check.

        Returns:
            bool: True if the string is in bytes format, False otherwise.
        """
        ret = string
        try:
            ret = string.decode()
            return True
        except:
            pass
        return False

    def str2bytes(self, string: str) -> bytes:
        """
        Convert a string to bytes.

        Args:
            string (str): The string to convert.

        Returns:
            bytes: The converted bytes.
        """
        if self.isbytes(string) == False:
            return string.encode()
        return string


class Label:
    """
    A class to represent and process Oghma labels.

    Attributes:
        oghma_label (str): The original Oghma label.
        token (tokens): The token library instance.
        english (str): The English description of the label.
        units (str): The units associated with the label.
        widget (str): The widget type associated with the label.
        hidden (bool): Whether the label is hidden.
        pack (list): Additional metadata for the label.

    Methods:
        __init__(oghma_label): Initialize a Label instance.
    """
    def __init__(self, oghma_label: str) -> None:
        """
        Initialize a Label instance and process the Oghma label.

        Args:
            oghma_label (str): The original Oghma label.
        """
        self.token = tokens()
        self.oghma_label = oghma_label

        if '.' in oghma_label:
            self.oghma_label = self.oghma_label.split('.')[-1]

        self.token = self.token.find(self.oghma_label)
        if self.token == False:
            self.english = self.oghma_label
            self.units = "Not Found"
            self.token = "Not Found"
        else:
            self.english = self.token.english.decode()
            self.units = self.token.units.decode()
            self.widget = self.token.widget.decode()
            self.hidden = self.token.hidden
            self.pack = self.token.pack
            self.token = self.token.token.decode()

        latex_labels = ['^', '_', '{', '}']
        r = []
        for l in latex_labels:
            if l in self.english:
                r.append(np.argwhere(l == np.asarray(list(self.english))).ravel()[0])
        r = np.asarray(r)

        if len(r) == 1:
            self.english = self.english.replace('_', ' ').replace('{', '').replace('}', '').replace('^', '')
        elif len(r) >= 2:
            self.english = self.english[0:r[0]] + '$' + self.english[r[0]:r[-1] + 1] + '$'


if __name__ == "__main__":
    A = Label("light_1.0.voc")
    print(A.oghma_label)
    print(A.english)

    A = Label("light_1.0.k_voc")
    print(A.oghma_label)
    print(A.english)

    A = Label("light_1.0.tau_voc")
    print(A.oghma_label)
    print(A.english)

    A = Label("light_1.0.tau_pmax")
    print(A.oghma_label)
    print(A.english)

    A = Label("light_1.0.mu_geom_pmax")
    print(A.oghma_label)
    print(A.english)