# PyOghma_ML Documentation System

This directory contains a comprehensive, automated documentation system for PyOghma_ML with consistent visual styling and automatic updates.

## 📁 Directory Structure

```
docs/
├── index.html                 # Main landing page
├── PyOghma_ML/               # API documentation modules
│   ├── Input.html            # Input module documentation
│   ├── Networks.html         # Networks module documentation
│   ├── Training.html         # Training module documentation
│   └── ...                   # Other module documentation
├── docs_config.json          # Documentation configuration
└── README.md                 # This file

Scripts:
├── ensure_style_consistency.py  # Applies consistent styling
├── update_documentation.py      # Main documentation updater
└── update_docs.sh              # Bash wrapper script
```

## 🎨 Design System Features

### **Consistent Visual Identity**
- **Professional color palette** with CSS variables
- **Typography scale** with Inter font family
- **Consistent spacing** using rem-based system
- **Modern shadows and borders** for depth
- **Responsive design** for all screen sizes

### **Enhanced Code Readability**
- **Dark theme code blocks** with syntax highlighting
- **Professional monospace fonts** (Fira Code, SF Mono)
- **Color-coded Python syntax** for better readability
- **Copy buttons** on code blocks
- **High contrast** function/class labels

### **Improved Navigation**
- **Gradient sidebar** with excellent contrast
- **Breadcrumb navigation** for better orientation
- **Hover effects** and smooth transitions
- **Mobile-responsive** navigation menus

## 🚀 Automated Documentation Updates

### **Usage**

#### Manual Update
```bash
# Update documentation (only if changes detected)
python3 update_documentation.py

# Force update regardless of changes
python3 update_documentation.py --force

# Using the bash wrapper
./update_docs.sh
```

#### Automated Updates
```bash
# Add to crontab for automatic updates every hour
0 * * * * /path/to/PyOghma_ML/update_docs.sh >/tmp/pyoghma_docs.log 2>&1

# Or check for changes every 15 minutes (recommended)
*/15 * * * * /path/to/PyOghma_ML/update_docs.sh >/tmp/pyoghma_docs.log 2>&1
```

### **Features**

#### **Smart Change Detection**
- **MD5 hash comparison** of source files
- **Incremental updates** only when needed
- **Automatic backup** before updates
- **Version history** maintenance

#### **Comprehensive Generation**
- **pdoc3 integration** for API documentation
- **Automatic styling** application
- **Landing page updates** with current module info
- **Error handling** and recovery

#### **Backup and Recovery**
- **Automatic backups** before each update
- **Timestamp-based** backup naming
- **Cleanup of old backups** (keeps 5 most recent)
- **Rollback capability** if needed

## 🔧 Configuration

### **docs_config.json**
```json
{
  "project_name": "PyOghma_ML",
  "source_directory": "src",
  "docs_directory": "docs",
  "auto_update": {
    "enabled": true,
    "watch_extensions": [".py"],
    "exclude_patterns": ["__pycache__", "*.pyc", ".git"]
  },
  "pdoc_options": {
    "format": "html",
    "force": true,
    "include_source": true
  },
  "styling": {
    "apply_custom_css": true,
    "theme": "professional"
  }
}
```

## 📋 Style Consistency

### **CSS Variables System**
```css
:root {
    /* Colors */
    --primary-color: #2563eb;
    --secondary-color: #0f766e;
    --accent-color: #ef4444;
    
    /* Typography */
    --font-family-sans: 'Inter', sans-serif;
    --font-family-mono: 'Fira Code', monospace;
    
    /* Spacing */
    --space-4: 1rem;
    --space-8: 2rem;
    
    /* Shadows */
    --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.1);
}
```

### **Component Styles**
- **Function/class labels**: Blue gradient backgrounds with white text
- **Code blocks**: Dark theme with syntax highlighting
- **Sidebar navigation**: Gradient background with transparent code elements
- **Cards and sections**: Consistent padding, shadows, and hover effects

## 🛠️ Maintenance

### **Regular Tasks**
1. **Run style consistency check** after major changes
2. **Update documentation** when code changes
3. **Clean up old backups** periodically
4. **Review configuration** settings as needed

### **Troubleshooting**

#### **Common Issues**
- **pdoc3 not found**: Install with `pip3 install pdoc3`
- **Permission errors**: Ensure scripts are executable (`chmod +x update_docs.sh`)
- **Style inconsistencies**: Run `python3 ensure_style_consistency.py`

#### **Manual Fixes**
```bash
# Fix permissions
chmod +x update_docs.sh

# Reinstall pdoc3
pip3 install --upgrade pdoc3

# Force regenerate all documentation
python3 update_documentation.py --force

# Apply styling only
python3 ensure_style_consistency.py
```

## 📊 Monitoring

### **Log Files**
- Check `/tmp/pyoghma_docs.log` for cron job output
- Use `--verbose` flag for detailed logging
- Monitor backup directory growth

### **Health Checks**
```bash
# Check if documentation is up to date
python3 update_documentation.py

# Apply style consistency
python3 ensure_style_consistency.py

# Manual update with force
python3 update_documentation.py --force
```

## 🎯 Future Enhancements

### **Planned Features**
- [ ] **Watch mode** for real-time updates
- [ ] **Link validation** and broken link detection
- [ ] **Search functionality** integration
- [ ] **PDF export** capabilities
- [ ] **Multi-language** support
- [ ] **Performance metrics** and analytics

### **Integration Opportunities**
- [ ] **GitHub Actions** for CI/CD
- [ ] **Docker containers** for consistent environments
- [ ] **Webhook triggers** for automatic updates
- [ ] **Quality gates** for documentation standards

---

**Generated by PyOghma_ML Documentation System**  
*Professional documentation for scientific Python packages*
- **`PyOghma_ML/index.html`** - Main API documentation entry point
- **`PyOghma_ML/*.html`** - Individual module documentation files

### Module Documentation Files
- **`Input.html`** - Data processing and laboratory input handling
- **`Networks.html`** - Neural network architectures and management
- **`Training.html`** - Model training pipeline and utilities
- **`Predicting.html`** - Prediction capabilities and model inference
- **`Output.html`** - Report generation and visualization
- **`Figures.html`** - Scientific plotting and visualization utilities
- **`Latex.html`** - LaTeX document generation
- **`Labels.html`** - Label management and metadata processing
- **`Tuning.html`** - Hyperparameter optimization utilities

## Viewing the Documentation

### Local Viewing
Open `index.html` in any modern web browser to start exploring the documentation.

### Key Features
- **Modern UI**: Clean, responsive design optimized for scientific documentation
- **Easy Navigation**: Breadcrumb navigation between sections and modules
- **Comprehensive Coverage**: Complete API documentation with examples and usage patterns
- **Search Integration**: Built-in code highlighting and collapsible sections
- **Mobile Friendly**: Responsive design that works on all devices

## Documentation Generation

The API documentation was generated using `pdoc3` and enhanced with:
- Custom styling and responsive design
- Interactive navigation elements
- Code syntax highlighting
- Collapsible documentation sections
- Enhanced accessibility features

## Contributing to Documentation

When adding new modules or updating existing ones:
1. Update docstrings in the source code
2. Regenerate API documentation using pdoc3
3. Apply consistent navigation and styling patterns
4. Test documentation on multiple devices and browsers

## Browser Compatibility

The documentation is optimized for:
- Chrome 90+
- Firefox 88+
- Safari 14+
- Edge 90+

Legacy browser support is available with graceful degradation of advanced styling features.
