"""
SubPredict Script for PyOghma ML Individual Predictions

This script performs individual predictions using trained PyOghma ML models on
experimental data. It serves as a subprocess called by the main Predict.py script
to handle specific prediction tasks for individual experimental files.

Usage:
    python SubPredict.py <sim_dir> <abs_dir> <exp_dir> <res_dir> <net_typ> <exp_typ> <Source_Lab> [pop]

Arguments:
    sim_dir: Path to the simulation directory containing trained networks
    abs_dir: Path to the directory containing absolute/conversion data  
    exp_dir: Path to the experimental data file to process
    res_dir: Path to the directory where results will be saved
    net_typ: Type of network architecture ('Difference', 'Residual', 'Point')
    exp_typ: Type of experiment ('JV', 'TPV', etc.)
    Source_Lab: Source laboratory identifier ('Deibel', 'Brabec', etc.)
    pop: Optional population identifier for output organization

The script will:
    1. Load and standardize experimental data
    2. Initialize the appropriate neural network model
    3. Generate predictions based on network type
    4. Build and save a comprehensive output report

Author: Cai Williams
Email: cai.williams@physik.tu-chemnitz.de
"""

from argparse import ArgumentParser
import os
import PyOghma_ML_Private as OML
from PyOghma_ML_Private.Training import Model_Settings, lr


def main():
    """Main prediction workflow."""
    # Parse command-line arguments for prediction configuration
    parser = ArgumentParser(
        description="Individual prediction script for PyOghma ML models",
        epilog="""
Examples:
  python SubPredict.py /path/to/sim /path/to/abs /path/to/exp.dat /path/to/results Residual JV Deibel
  python SubPredict.py /path/to/sim /path/to/abs /path/to/exp.dat /path/to/results Point TPV Brabec 2
        """
    )
    
    parser.add_argument("sim_dir", type=str, 
                       help="Path to the simulation directory containing the trained networks.")
    parser.add_argument("abs_dir", type=str,
                       help="Path to the directory containing absolute/conversion data.")
    parser.add_argument("exp_dir", type=str,
                       help="Path to the experimental data file to process.")
    parser.add_argument("res_dir", type=str,
                       help="Path to the directory where results will be saved.")
    parser.add_argument("net_typ", type=str,
                       help="Type of network architecture ('Difference', 'Residual', 'Point').")
    parser.add_argument("exp_typ", type=str,
                       help="Type of experiment ('JV', 'TPV', 'impedance', etc.).")
    parser.add_argument("Source_Lab", type=str,
                       help="Source laboratory identifier ('Deibel', 'Brabec', etc.).")
    parser.add_argument("pop", type=str, nargs='?', default='.',
                       help="Population identifier for output organization (default: '.').")
    
    args = parser.parse_args()
    
    print(f"Starting prediction for: {os.path.basename(args.exp_dir)}")
    print(f"Network type: {args.net_typ}")
    print(f"Experiment type: {args.exp_typ}")
    print(f"Source laboratory: {args.Source_Lab}")

    try:
        # Prepare experimental input
        print("Loading and standardizing experimental data...")
        Exp = OML.Input.experiment(
            device_dir=args.exp_dir,
            characterisation_type=args.exp_typ,
            source_laboratory=args.Source_Lab,
        )
        Exp.standardise_inputs()  # Standardize the experimental inputs
        print(f"✓ Loaded experimental data: {len(Exp.x)} data points")

        # Initialize the network and make predictions based on network type
        print(f"Initializing {args.net_typ} network...")
        N = OML.Networks.initialise(args.sim_dir, network_type=args.net_typ)
        
        print("Generating predictions...")
        if args.net_typ in ['Difference', 'Residual']:
            # These network types require absolute/conversion data
            N.predict(args.abs_dir, Exp)
        elif args.net_typ == 'Point':
            # Point networks predict directly from experimental data
            N.predict(Exp)
        else:
            # Default behavior for other network types
            N.predict(args.abs_dir, Exp)
        
        print("✓ Predictions generated successfully")

        # Generate and save the output report
        print("Building output report...")
        O = OML.Output(N, Exp, abs_dir=args.abs_dir)
        O.build_report()
        
        # Determine output file path based on population argument
        if args.pop != '.':
            output_dir = os.path.join(args.res_dir, f"{args.net_typ}_{args.pop}")
        else:
            output_dir = os.path.join(args.res_dir, args.net_typ)
        
        # Create output directory if it doesn't exist
        os.makedirs(output_dir, exist_ok=True)
        
        output_name = os.path.join(output_dir, os.path.basename(args.exp_dir).split('.')[0])
        O.save_report(output_name)
        
        print(f"✓ Results saved to: {output_name}.csv")
        print("Prediction completed successfully!")
        
    except Exception as e:
        print(f"✗ Error during prediction: {e}")
        raise


if __name__ == "__main__":
    main()