"""
Loading File Test Script for PyOghma ML

This script demonstrates how to load and visualize experimental data using the
PyOghma ML Input module. It serves as both a test script and an example of
how to properly load experimental data from different laboratories.

The script loads the same experimental file with different optical density (OD)
filters and visualizes the results to verify data loading functionality.

Usage:
    python LoadingFileTest.py

Features:
    - Loads experimental data from Brabec laboratory format
    - Applies different optical density filters (0.0, 1.0, 2.0, 3.0)
    - Standardizes inputs for consistent processing
    - Creates scatter plots with logarithmic x-axis for visualization

Author: Cai Williams
Email: cai.williams@physik.tu-chemnitz.de
"""

import PyOghma_ML as OML
import matplotlib.pyplot as plt
import os


def main():
    """Main testing workflow for loading experimental files."""
    # Define the experimental data file path
    exp_dir = os.path.join(
        '/', 'media', 'cai', 'Big', 'Experimental_Data', 'Brabec', 
        '2022-01-28-PTQ10-Y12-9281-457nm-timer-5degree', 'pin_2_11,4suns', 
        'TPV', '1.001.dat'
    )
    
    # Configuration
    characterisation_type = 'TPV'  # Transient photovoltage measurement
    source_laboratory = 'Brabec'   # Data source laboratory
    pixel = 1  # Pixel identifier (currently unused)
    
    print(f"Loading experimental data from: {exp_dir}")
    print(f"Characterisation type: {characterisation_type}")
    print(f"Source laboratory: {source_laboratory}")
    
    # Test different optical density (OD) values
    od_values = [0.0, 1.0, 2.0, 3.0]
    colors = ['blue', 'orange', 'green', 'red']
    
    plt.figure(figsize=(10, 6))
    
    for i, (od, color) in enumerate(zip(od_values, colors)):
        print(f"\nProcessing OD = {od}")
        
        try:
            # Create experiment object with specific OD value
            Exp = OML.Input.experiment(
                device_dir=exp_dir, 
                characterisation_type=characterisation_type, 
                source_laboratory=source_laboratory, 
                OD=od
            )
            
            # Standardize the inputs for consistent processing
            Exp.standardise_inputs()
            
            # Verify data loading
            print(f"✓ Loaded {len(Exp.x)} data points")
            print(f"  X range: {min(Exp.x):.2e} to {max(Exp.x):.2e}")
            print(f"  Y range: {min(Exp.y):.2e} to {max(Exp.y):.2e}")
            
            # Create scatter plot
            plt.scatter(Exp.x, Exp.y, s=1, color=color, 
                       label=f'OD = {od}', alpha=0.7)
            
        except Exception as e:
            print(f"✗ Error loading data with OD = {od}: {e}")
            continue
    
    # Configure plot
    plt.xscale('log')
    plt.xlabel('Time (s)')
    plt.ylabel('Voltage (V)')
    plt.title('TPV Data Loading Test - Different Optical Densities')
    plt.legend()
    plt.grid(True, alpha=0.3)
    plt.tight_layout()
    
    # Show the plot
    print("\nDisplaying visualization...")
    plt.show()
    
    print("\n✓ File loading test completed successfully!")


if __name__ == "__main__":
    main()