import numpy as np


class InputEdges(object):

    _emax = None
    _emin = None

    def __init__(self, edges: np.array):

        # check that the edges are monotonic

        diff = np.diff(edges)

        if not np.all(diff > 0.):

            raise RuntimeError(
                "The input edges are not monotonically increasing")

        if len(edges) % 2 == 0:

            raise RuntimeError(
                "The edges must have an odd length with the end points closed")

        self._edges = edges

    @property
    def edges(self):

        return self._edges

    @classmethod
    def from_log_bins(cls, n_bins: int):

        if n_bins % 2 == 0:

            n_bins += 1

        edges = np.geomspace(cls._emin, cls._emax, n_bins)

        return cls(edges)

    @classmethod
    def from_custom_array(cls, edges: np.array):
        """
        create edges from a custom set of energies.

        The end points must be those of the original edges.

        This will be corrected if not done... but might cause errors

        """

        if edges.min() > cls._emin:

            edges = np.insert(edges, 0, cls._min)

        elif edges.min() < cls._emin:

            edges = np.append(edges[edges <= cls._emin], 0, cls._emin)

        if edges.max() < cls._emax:

            edges = np.append(edges, cls._emax)

        elif edges.max() > cls._emax:

            edges = np.append(edges[edges <= cls._emax], cls._emax)

        return cls(edges)


tte_edges = {}
tte_edges["nai"] = np.array(
    [
        5.00000,
        5.34000,
        5.70312,
        6.09094,
        6.50513,
        6.94748,
        7.41991,
        7.92447,
        8.46333,
        9.03884,
        9.65349,
        10.3099,
        11.0110,
        11.7598,
        12.5594,
        13.4135,
        14.3256,
        15.2997,
        16.3401,
        17.4513,
        18.6380,
        19.9054,
        21.2589,
        22.7045,
        24.2485,
        25.8974,
        27.6584,
        29.5392,
        31.5479,
        33.6931,
        35.9843,
        38.4312,
        41.0446,
        43.8356,
        46.8164,
        50.0000,
        53.4000,
        57.0312,
        60.9094,
        65.0513,
        69.4748,
        74.1991,
        79.2446,
        84.6333,
        90.3884,
        96.5349,
        103.099,
        110.110,
        117.598,
        125.594,
        134.135,
        143.256,
        152.997,
        163.401,
        174.513,
        186.380,
        199.054,
        212.589,
        227.045,
        242.485,
        258.974,
        276.584,
        295.392,
        315.479,
        336.931,
        359.843,
        384.312,
        410.446,
        438.356,
        468.164,
        500.000,
        534.000,
        570.312,
        609.094,
        650.512,
        694.748,
        741.991,
        792.446,
        846.333,
        903.884,
        965.349,
        1030.99,
        1101.10,
        1175.98,
        1255.94,
        1341.35,
        1432.56,
        1529.97,
        1634.01,
        1745.13,
        1863.80,
        1990.54,
        2125.89,
        2270.45,
        2424.85,
        2589.74,
        2765.84,
        2953.92,
        3154.79,
        3369.31,
        3598.43,
        3843.12,
        4104.46,
        4383.56,
        4681.65,
        5000.00,
        5340.00,
        5703.12,
        6090.94,
        6505.12,
        6947.48,
        7419.91,
        7924.46,
        8463.33,
        9038.84,
        9653.49,
        10309.9,
        11011.0,
        11759.8,
        12559.4,
        13413.5,
        14325.6,
        15299.7,
        16340.1,
        17451.3,
        18637.9,
        19905.3,
        21258.9,
        22704.5,
        24248.5,
        25897.3,
        27658.4,
        29539.2,
        31547.8,
        33693.1,
        35984.3,
        38431.2,
        41044.6,
        43835.6,
        46816.4,
        50000.0,
    ],
    dtype=np.float32,
)

tte_edges["bgo"] = np.array(
    [
        100.000,
        105.579,
        111.470,
        117.689,
        124.255,
        131.188,
        138.507,
        146.235,
        154.394,
        163.008,
        172.103,
        181.705,
        191.843,
        202.546,
        213.847,
        225.778,
        238.375,
        251.675,
        265.716,
        280.541,
        296.194,
        312.719,
        330.167,
        348.588,
        368.036,
        388.570,
        410.250,
        433.139,
        457.305,
        482.820,
        509.757,
        538.198,
        568.226,
        599.929,
        633.401,
        668.740,
        706.052,
        745.444,
        787.035,
        830.946,
        877.307,
        926.255,
        977.933,
        1032.49,
        1090.10,
        1150.92,
        1215.13,
        1282.93,
        1354.51,
        1430.08,
        1509.87,
        1594.11,
        1683.05,
        1776.95,
        1876.09,
        1980.77,
        2091.28,
        2207.96,
        2331.15,
        2461.21,
        2598.53,
        2743.51,
        2896.58,
        3058.18,
        3228.81,
        3408.95,
        3599.15,
        3799.96,
        4011.97,
        4235.81,
        4472.14,
        4721.65,
        4985.09,
        5263.22,
        5556.87,
        5866.90,
        6194.24,
        6539.83,
        6904.71,
        7289.95,
        7696.67,
        8126.09,
        8579.47,
        9058.15,
        9563.53,
        10097.1,
        10660.5,
        11255.2,
        11883.2,
        12546.2,
        13246.2,
        13985.2,
        14765.5,
        15589.3,
        16459.1,
        17377.4,
        18346.9,
        19370.6,
        20451.3,
        21592.4,
        22797.1,
        24069.0,
        25411.8,
        26829.7,
        28326.6,
        29907.0,
        31575.6,
        33337.3,
        35197.3,
        37161.0,
        39234.4,
        41423.4,
        43734.5,
        46174.6,
        48750.8,
        51470.7,
        54342.5,
        57374.4,
        60575.5,
        63955.2,
        67523.4,
        71290.7,
        75268.2,
        79467.7,
        83901.5,
        88582.6,
        93524.9,
        98742.9,
        104252.0,
        110069.0,
        116210.0,
        122693.0,
        129539.0,
        136766.0,
        144397.0,
        152453.0,
        160959.0,
        169939.0,
        179421.0,
        189431.0,
        200000.0,
    ],
    dtype=np.float32,
)


class NaiTTEEdges(InputEdges):
    _emin = 5.0
    _emax = 50000.

    def __init__(self, edges):

        super(NaiTTEEdges, self).__init__(edges)


class BgoTTEEdges(InputEdges):
    _emin = 100.
    _emax = 200000.

    def __init__(self, edges):

        super(BgoTTEEdges, self).__init__(edges)


trigdat_edges = {}

trigdat_edges["nai"] = np.array(
    [
        5.00000,
        5.34000,
        5.70312,
        6.09094,
        6.50513,
        6.94748,
        7.41991,
        7.92447,
        8.46333,
        9.03884,
        9.65349,
        10.3099,
        11.0110,
        11.7598,
        12.5594,
        13.4135,
        14.3256,
        15.2997,
        16.3401,
        17.4513,
        18.6380,
        19.9054,
        21.2589,
        22.7045,
        24.2485,
        25.8974,
        27.6584,
        29.5392,
        31.5479,
        33.6931,
        35.9843,
        38.4312,
        41.0446,
        43.8356,
        46.8164,
        50.0000,
        53.4000,
        57.0312,
        60.9094,
        65.0513,
        69.4748,
        74.1991,
        79.2446,
        84.6333,
        90.3884,
        96.5349,
        103.099,
        110.110,
        117.598,
        125.594,
        134.135,
        143.256,
        152.997,
        163.401,
        174.513,
        186.380,
        199.054,
        212.589,
        227.045,
        242.485,
        258.974,
        276.584,
        295.392,
        315.479,
        336.931,
        359.843,
        384.312,
        410.446,
        438.356,
        468.164,
        500.000,
        534.000,
        570.312,
        609.094,
        650.512,
        694.748,
        741.991,
        792.446,
        846.333,
        903.884,
        965.349,
        1030.99,
        1101.10,
        1175.98,
        1255.94,
        1341.35,
        1432.56,
        1529.97,
        1634.01,
        1745.13,
        1863.80,
        1990.54,
        2125.89,
        2270.45,
        2424.85,
        2589.74,
        2765.84,
        2953.92,
        3154.79,
        3369.31,
        3598.43,
        3843.12,
        4104.46,
        4383.56,
        4681.65,
        5000.00,
        5340.00,
        5703.12,
        6090.94,
        6505.12,
        6947.48,
        7419.91,
        7924.46,
        8463.33,
        9038.84,
        9653.49,
        10309.9,
        11011.0,
        11759.8,
        12559.4,
        13413.5,
        14325.6,
        15299.7,
        16340.1,
        17451.3,
        18637.9,
        19905.3,
        21258.9,
        22704.5,
        24248.5,
        25897.3,
        27658.4,
        29539.2,
        31547.8,
        33693.1,
        35984.3,
        38431.2,
        41044.6,
        43835.6,
        46816.4,
        50000.0,
    ],
    dtype=np.float32,
)


trigdat_edges["bgo"] = np.array(
    [
        100.000,
        105.579,
        111.470,
        117.689,
        124.255,
        131.188,
        138.507,
        146.235,
        154.394,
        163.008,
        172.103,
        181.705,
        191.843,
        202.546,
        213.847,
        225.778,
        238.375,
        251.675,
        265.716,
        280.541,
        296.194,
        312.719,
        330.167,
        348.588,
        368.036,
        388.570,
        410.250,
        433.139,
        457.305,
        482.820,
        509.757,
        538.198,
        568.226,
        599.929,
        633.401,
        668.740,
        706.052,
        745.444,
        787.035,
        830.946,
        877.307,
        926.255,
        977.933,
        1032.49,
        1090.10,
        1150.92,
        1215.13,
        1282.93,
        1354.51,
        1430.08,
        1509.87,
        1594.11,
        1683.05,
        1776.95,
        1876.09,
        1980.77,
        2091.28,
        2207.96,
        2331.15,
        2461.21,
        2598.53,
        2743.51,
        2896.58,
        3058.18,
        3228.81,
        3408.95,
        3599.15,
        3799.96,
        4011.97,
        4235.81,
        4472.14,
        4721.65,
        4985.09,
        5263.22,
        5556.87,
        5866.90,
        6194.24,
        6539.83,
        6904.71,
        7289.95,
        7696.67,
        8126.09,
        8579.47,
        9058.15,
        9563.53,
        10097.1,
        10660.5,
        11255.2,
        11883.2,
        12546.2,
        13246.2,
        13985.2,
        14765.5,
        15589.3,
        16459.1,
        17377.4,
        18346.9,
        19370.6,
        20451.3,
        21592.4,
        22797.1,
        24069.0,
        25411.8,
        26829.7,
        28326.6,
        29907.0,
        31575.6,
        33337.3,
        35197.3,
        37161.0,
        39234.4,
        41423.4,
        43734.5,
        46174.6,
        48750.8,
        51470.7,
        54342.5,
        57374.4,
        60575.5,
        63955.2,
        67523.4,
        71290.7,
        75268.2,
        79467.7,
        83901.5,
        88582.6,
        93524.9,
        98742.9,
        104252.0,
        110069.0,
        116210.0,
        122693.0,
        129539.0,
        136766.0,
        144397.0,
        152453.0,
        160959.0,
        169939.0,
        179421.0,
        189431.0,
        200000.0,
    ],
    dtype=np.float32,
)


trigdat_out_edge = {}
trigdat_out_edge["nai"] = np.array(
    [3.4, 10.0, 22.0, 44.0, 95.0, 300.0, 500.0, 800.0, 2000.0], dtype=np.float32,
)


trigdat_out_edge["bgo"] = np.array(
    [150.0, 400.0, 850.0, 1500.0, 3000.0, 5500.0, 10000.0, 20000.0, 50000.0, ],
    dtype=np.float32,
)
