"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("@alicloud/ros-cdk-cxapi");
const prepare_app_1 = require("./prepare-app");
const stage_1 = require("../stage");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node
            ._actualNode;
        const allAspectsHere = [...(inheritedAspects ?? []), ...node._aspects];
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, "post", (construct) => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, "post", (construct) => construct.onSynthesize({
        outdir: builder.outdir,
        assembly: builder,
    }));
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, "pre", (construct) => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors
            .map((e) => `[${e.source.node.path}] ${e.message}`)
            .join("\n  ");
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === "pre") {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === "post") {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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