"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() { }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return "<unresolved-lazy>";
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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