"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.AppNetworkAccessType = exports.AuthMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const path = require("path");
const sage_maker_image_js_1 = require("./sage-maker-image.js");
const user_profile_js_1 = require("./user-profile.js");
var AuthMode;
(function (AuthMode) {
    AuthMode["SSO"] = "SSO";
    AuthMode["IAM"] = "IAM";
})(AuthMode || (exports.AuthMode = AuthMode = {}));
// export enum DomainEncryption {
//   KMS = "KMS",
// }
var AppNetworkAccessType;
(function (AppNetworkAccessType) {
    AppNetworkAccessType["VPC_ONLY"] = "VpcOnly";
    AppNetworkAccessType["PUBLIC_INTERNET_ONLY"] = "PublicInternetOnly";
})(AppNetworkAccessType || (exports.AppNetworkAccessType = AppNetworkAccessType = {}));
class Domain extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const removalPolicy = props.removalPolicy ?? core_1.RemovalPolicy.RETAIN;
        this.users = new constructs_1.Construct(this, "Users");
        const defaultImage = props.defaultImage ?? sage_maker_image_js_1.SageMakerImage.CPU_V1;
        const domainExecutionRole = new aws_iam_1.Role(this, "ExecutionRole", {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal("sagemaker.amazonaws.com"), new aws_iam_1.ServicePrincipal("glue.amazonaws.com")),
        });
        this.grantPrincipal = domainExecutionRole;
        domainExecutionRole.applyRemovalPolicy(removalPolicy);
        // sagemaker needs permission to call GetRole and PassRole on the Role it assumed
        // e.g. arn:aws:iam::123456789012:role/role-name/SageMaker will call GetRole on arn:aws:iam::123456789012:role/role-name
        // When you run `spark` in a Jupyter notebook, it will:
        // 1. GetRole
        // 2. CreateSession
        // 3. PassRole to the Session
        domainExecutionRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["iam:GetRole", "iam:PassRole"],
            resources: [domainExecutionRole.roleArn],
        }));
        // If EMR, the driver node must be in the same private subnet as the SageMaker notebook
        //
        // Principal Tags (for SSO)
        // AWS Global Config file on the local instance of the notebook
        // Lifecycle script
        // -> Git
        // note: image/lifecycle script should include ssh
        // error: cannot run ssh: No such file or directory
        // or: GitHub CLI ...
        // %glue_version 4.0
        // Athena -> External Catalog (or sync to Glue Iceberg)
        //  https://docs.aws.amazon.com/athena/latest/ug/connect-to-data-source-hive.html
        // https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html
        const sageMakerSg = props.sageMakerSg ??
            new aws_ec2_1.SecurityGroup(this, "SageMakerSecurityGroup", {
                vpc: props.vpc,
            });
        this.sageMakerSg = sageMakerSg;
        this.connections = sageMakerSg.connections;
        this.resource = new aws_sagemaker_1.CfnDomain(this, "Resource", {
            authMode: props.authMode ?? AuthMode.SSO,
            domainName: props.domainName,
            vpcId: props.vpc.vpcId,
            subnetIds: (props.subnetSelection
                ? props.vpc.selectSubnets(props.subnetSelection).subnets
                : props.vpc.privateSubnets).map((subnet) => subnet.subnetId),
            defaultUserSettings: {
                executionRole: domainExecutionRole.roleArn,
                studioWebPortal: props.defaultUserSettings?.studioWebPortal ?? true
                    ? "ENABLED"
                    : "DISABLED",
                securityGroups: [sageMakerSg.securityGroupId],
            },
            appNetworkAccessType: props.appNetworkAccessType ?? AppNetworkAccessType.VPC_ONLY,
            defaultSpaceSettings: {
                executionRole: domainExecutionRole.roleArn,
                kernelGatewayAppSettings: {
                    defaultResourceSpec: {
                        instanceType: "system",
                        sageMakerImageArn: defaultImage.getArnForStack(core_1.Stack.of(this)),
                        // TODO:
                        // lifecycleConfigArn: ??
                    },
                },
                // jupyterServerAppSettings: {
                //   defaultResourceSpec: {
                //     what is the image
                //   },
                // },
            },
        });
        this.resource.applyRemovalPolicy(removalPolicy);
        this.domainId = this.resource.ref;
        this.domainArn = this.resource.attrDomainArn;
        this.domainUrl = this.resource.attrUrl;
        this.homeEfsFileSystemId = this.resource.attrHomeEfsFileSystemId;
        this.homeEfsFileSystemArn = core_1.Arn.format({
            service: "elasticfilesystem",
            resource: "file-system",
            resourceName: this.homeEfsFileSystemId,
        }, core_1.Stack.of(this));
        this.singleSignOnManagedApplicationInstanceId =
            this.resource.attrSingleSignOnManagedApplicationInstanceId;
        this.singleSignOnApplicationArn =
            this.resource.attrSingleSignOnApplicationArn;
        this.enableCleanup(removalPolicy);
        // TODO: CustomResource to spin down Spaces when destroyed
        // TODO: should this be configurable?
        this.grantStudioAccess(domainExecutionRole);
        // TODO: should this be configurable?
        this.grantGlueInteractiveSession(domainExecutionRole);
    }
    grantGlueInteractiveSession(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "glue:CreateSession",
                "glue:DeleteSession",
                "glue:GetSession",
                "glue:StopSession",
                "glue:CancelStatement",
                "glue:GetStatement",
                "glue:ListStatements",
                "glue:RunStatement",
            ],
            resources: [
                core_1.Arn.format({
                    service: "glue",
                    resource: "session",
                    resourceName: "*",
                }, core_1.Stack.of(this)),
            ],
        }));
    }
    /**
     * Creates a CustomResource that will clean up the domain prior to it being destroyed:
     * 1. Delete any running Apps (i.e. instances of a Space)
     * 2. Delete the Domain's spaces.
     * 2. Delete the Domain's EFS file system (first, by deleting any mounted access points, then the FS).
     */
    enableCleanup(removalPolicy) {
        if (this.cleanup) {
            return;
        }
        // const dirname = path.dirname(fileURLToPath(import.meta.url));
        const dirname = __dirname;
        const code = path.join(dirname, "delete-domain");
        const cleanupFunction = new aws_lambda_1.Function(this, "CleanupFunction", {
            code: aws_lambda_1.Code.fromAsset(code),
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            architecture: aws_lambda_1.Architecture.ARM_64,
            handler: "index.handler",
            timeout: core_1.Duration.minutes(1),
            environment: {
                FileSystemId: this.homeEfsFileSystemId,
                DomainId: this.domainId,
                RemovalPolicy: removalPolicy,
                NODE_OPTIONS: "--experimental-modules=true",
            },
        });
        cleanupFunction.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "elasticfilesystem:DeleteFileSystem",
                "elasticfilesystem:DescribeMountTargets",
            ],
            resources: [this.homeEfsFileSystemArn],
        }));
        cleanupFunction.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["elasticfilesystem:DeleteMountTarget"],
            resources: [
                // See: https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticfilesystem.html#amazonelasticfilesystem-actions-as-permissions
                core_1.Arn.format({
                    service: "elasticfilesystem",
                    resource: "file-system",
                    // TODO: can we constrain this more?
                    resourceName: this.homeEfsFileSystemId,
                }, core_1.Stack.of(this)),
            ],
        }));
        this.grantDeleteApp(cleanupFunction);
        this.grantDeleteSpace(cleanupFunction);
        this.grantDescribeApp(cleanupFunction);
        this.grantDescribeSpace(cleanupFunction);
        this.grantListApps(cleanupFunction);
        this.grantListSpaces(cleanupFunction);
        const cleanupProvider = new custom_resources_1.Provider(this, "CleanupProvider", {
            onEventHandler: cleanupFunction,
        });
        this.cleanup = new core_1.CustomResource(this, "Cleanup", {
            resourceType: "Custom::PackYakCleanupDomain",
            serviceToken: cleanupProvider.serviceToken,
            properties: {
                FileSystemId: this.homeEfsFileSystemId,
                DomainId: this.domainId,
                RemovalPolicy: removalPolicy,
            },
            removalPolicy,
        });
    }
    grantStudioAccess(grantee) {
        this.grantCreateSpace(grantee);
        this.grantCreateApp(grantee);
        this.grantDeleteSpace(grantee);
        this.grantDeleteApp(grantee);
        this.grantUpdateSpace(grantee);
        this.grantCreatePresignedDomainUrl(grantee);
        this.grantDescribeApp(grantee);
        this.grantDescribeDomain(grantee);
        this.grantDescribeSpace(grantee);
        this.grantDescribeUserProfile(grantee);
        this.grantListApps(grantee);
        this.grantListSessions(grantee);
        this.grantListTags(grantee);
        this.grantListSpaces(grantee);
        this.grantEMRClusterAccess(grantee);
    }
    /**
     * Grants access to list and describe clusters in the JupyterNotebook.
     */
    grantEMRClusterAccess(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "elasticmapreduce:ListClusters",
                "elasticmapreduce:ListInstances",
                "elasticmapreduce:ListInstanceFleets",
                "elasticmapreduce:ListInstanceGroups",
                "elasticmapreduce:DescribeCluster",
                // TODO: this should be cluster specific
                "elasticmapreduce:GetOnClusterAppUIPresignedURL",
            ],
            resources: ["*"],
        }));
    }
    // sagemaker:Search
    grantSageMakerSearch(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:Search"],
            resource: "user-profile",
        });
    }
    grantListApps(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListApps"],
            resource: "app",
        });
    }
    grantListSessions(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["glue:ListSessions"],
            // TODO: tag-based auth
            resources: ["*"],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantListTags(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListTags"],
            resource: "user-profile",
        });
        // grantee.grantPrincipal.addToPrincipalPolicy(
        //   new PolicyStatement({
        //     actions: ["sagemaker:ListTags"],
        //     resources: [this.domainArn],
        //     effect: Effect.ALLOW,
        //   }),
        // );
    }
    grantSearchServiceCatalogProducts(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["servicecatalog:SearchProducts"],
            // sagemaker scans the whole account
            resources: ["*"],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantListSpaces(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListSpaces"],
            resource: "space",
        });
    }
    grantCreateApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreateApp"],
            resource: "app",
        });
    }
    grantCreatePresignedDomainUrl(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreatePresignedDomainUrl"],
            resource: "user-profile",
        });
    }
    grantCreateSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreateSpace"],
            resource: "space",
        });
    }
    grantDeleteApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DeleteApp"],
            resource: "app",
        });
    }
    grantDeleteSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DeleteSpace"],
            resource: "space",
        });
    }
    grantDescribeApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeApp"],
            resource: "app",
        });
    }
    grantDescribeDomain(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["sagemaker:DescribeDomain"],
            resources: [this.domainArn],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantDescribeSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeSpace"],
            resource: "space",
        });
    }
    grantDescribeUserProfile(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeUserProfile"],
            resource: "user-profile",
        });
    }
    grantUpdateSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:UpdateSpace"],
            resource: "space",
        });
    }
    grant(grantee, props) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: props.actions,
            resources: [
                core_1.Arn.format({
                    service: "sagemaker",
                    resource: props.resource,
                    resourceName: `${this.domainId}/*`,
                }, core_1.Stack.of(this)),
            ],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    addUserProfile(username, props) {
        return new user_profile_js_1.UserProfile(this.users, username, {
            domain: this,
            userProfileName: username,
            executionRole: props?.executionRole,
        });
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@packyak/aws-cdk.Domain", version: "0.3.6" };
//# sourceMappingURL=data:application/json;base64,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