"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseNessieCatalog = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const nessie_version_store_js_1 = require("./nessie-version-store.js");
const nessie_config_js_1 = require("./nessie-config.js");
const core_1 = require("aws-cdk-lib/core");
const spark_sql_extension_js_1 = require("../emr/spark-sql-extension.js");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
class BaseNessieCatalog extends constructs_1.Construct {
    /**
     * Endpoint for the Nessie API v1.
     *
     * @deprecated use {@link apiV2Url} instead
     */
    get apiV1Url() {
        return `${this.endpoint}/api/v1`;
    }
    /**
     * Endpoint for the Nessie API v2.
     *
     * Note: Nessie CLI is not compatible with V1. For CLI use {@link apiV2Url}
     */
    get apiV2Url() {
        return `${this.endpoint}/api/v2`;
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.catalogName = props?.catalogName ?? "spark_catalog";
        this.warehouseBucket =
            props?.warehouseBucket ?? new aws_s3_1.Bucket(this, "Warehouse");
        this.warehousePrefix = props?.warehousePrefix;
        this.defaultMainBranch = props?.defaultMainBranch ?? "main";
        // @see https://github.com/projectnessie/nessie/blob/09762d2b80ca448782c2f4326e3e41f1447ae6e0/versioned/storage/dynamodb/src/main/java/org/projectnessie/versioned/storage/dynamodb/DynamoDBConstants.java#L37
        this.versionStore =
            props.versionStore ??
                new nessie_version_store_js_1.DynamoDBNessieVersionStore(this, "VersionStore", {
                    versionStoreName: `${props.catalogName}-nessie`,
                    ...(props?.versionStore ?? {}),
                });
        this.config = {
            "nessie.version.store.type": nessie_config_js_1.NessieVersionStoreType.DYNAMODB,
            "nessie.version.store.persist.dynamodb.table-prefix": this.versionStore.tablePrefix,
            "nessie.server.default-branch": this.defaultMainBranch,
            "quarkus.dynamodb.async-client.type": "aws-crt",
            "quarkus.dynamodb.sync-client.type": "aws-crt",
            "quarkus.oidc.tenant-enabled": false,
            "quarkus.dynamodb.aws.region": core_1.Stack.of(this).region,
        };
    }
    configAsEnvVars() {
        // @ts-ignore
        return (0, nessie_config_js_1.nessieConfigToEnvironment)(this.config);
    }
    bind(cluster, catalogName) {
        // TODO: should we limit this to the warehouse prefix
        this.warehouseBucket.grantReadWrite(cluster, "*");
        const sparkVersion = cluster.release.sparkVersion;
        const scalaVersion = cluster.release.scalaVersion;
        const nessieExt = spark_sql_extension_js_1.SparkSqlExtension.Nessie.maven(sparkVersion, scalaVersion);
        const icebergExt = spark_sql_extension_js_1.SparkSqlExtension.Iceberg.maven(sparkVersion, scalaVersion);
        // see: https://project-nessie.zulipchat.com/#narrow/stream/371187-general/topic/.E2.9C.94.20Merge.20author/near/421208974
        const catalogNamespace = `spark.sql.catalog.${catalogName}`;
        cluster.addConfig({
            classification: "spark-defaults",
            configurationProperties: {
                // set up Nessie catalog
                "spark.jars.packages": `${icebergExt},${nessieExt}`,
                "spark.sql.extensions": `${spark_sql_extension_js_1.SparkSqlExtension.Iceberg.className},${spark_sql_extension_js_1.SparkSqlExtension.Nessie.className}`,
                // TODO: is s3a:// right?
                [`${catalogNamespace}.warehouse`]: `s3://${this.warehouseBucket.bucketName}${this.warehousePrefix
                    ? `/${this.warehousePrefix.replace(/^[\/]*/g, "")}`
                    : ""}`,
                // TODO: not sure if Spark uses V1 or V2
                // see thread: https://project-nessie.zulipchat.com/#narrow/stream/371187-general/topic/.E2.9C.94.20Merge.20author/near/421198168
                // V1
                // "spark.sql.catalog.nessie.uri": this.apiV1Url,
                // V2
                // // After Iceberg 1.5.0 release, just configuring v2 URI is enough (version is inferred from URI).
                [`${catalogNamespace}.uri`]: this.apiV2Url,
                [`${catalogNamespace}.ref`]: this.defaultMainBranch,
                [`${catalogNamespace}.client-api-version`]: "2",
                [`${catalogNamespace}.authentication.type`]: "AWS",
                [`${catalogNamespace}.catalog-impl`]: "org.apache.iceberg.nessie.NessieCatalog",
                [catalogNamespace]: "org.apache.iceberg.spark.SparkCatalog",
                [`${catalogNamespace}.io-impl`]: "org.apache.iceberg.aws.s3.S3FileIO",
                // "spark.sql.catalog.nessie.cache-enabled": false
            },
        });
    }
}
exports.BaseNessieCatalog = BaseNessieCatalog;
_a = JSII_RTTI_SYMBOL_1;
BaseNessieCatalog[_a] = { fqn: "@packyak/aws-cdk.BaseNessieCatalog", version: "0.3.6" };
//# sourceMappingURL=data:application/json;base64,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