"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ScalingUnit = exports.ScaleDownBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_emr_1 = require("aws-cdk-lib/aws-emr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const core_1 = require("aws-cdk-lib/core");
const path = require("path");
const application_js_1 = require("./application.js");
const configuration_js_1 = require("./configuration.js");
const jdbc_js_1 = require("./jdbc.js");
const market_js_1 = require("./market.js");
const release_label_js_1 = require("./release-label.js");
const spark_config_js_1 = require("./spark-config.js");
var ScaleDownBehavior;
(function (ScaleDownBehavior) {
    ScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
    ScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
})(ScaleDownBehavior || (exports.ScaleDownBehavior = ScaleDownBehavior = {}));
var ScalingUnit;
(function (ScalingUnit) {
    ScalingUnit["INSTANCES"] = "Instances";
    ScalingUnit["INSTANCE_FLEET_UNITS"] = "InstanceFleetUnits";
    ScalingUnit["VCPU"] = "VCPU";
})(ScalingUnit || (exports.ScalingUnit = ScalingUnit = {}));
class Cluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.extraJavaOptions = { ...(props.extraJavaOptions ?? {}) };
        this.steps = [...(props.steps ?? [])];
        this.configurations = [...(props.configurations ?? [])];
        this.bootstrapActions = [...(props.bootstrapActions ?? [])];
        this.release = props.releaseLabel ?? release_label_js_1.ReleaseLabel.EMR_7_0_0;
        const m5xlarge = aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.M5, aws_ec2_1.InstanceSize.XLARGE);
        const masterInstanceType = props.masterInstanceGroup?.instanceType ?? m5xlarge;
        const coreInstanceType = props.coreInstanceGroup?.instanceType ?? m5xlarge;
        // const taskInstanceType = props.taskInstanceGroup?.instanceType ?? m5xlarge;
        // for least privileges, see:
        // https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role-for-ec2.html#emr-ec2-role-least-privilege
        this.jobFlowRole = new aws_iam_1.Role(this, "JobFlowRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com"),
        });
        // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role-for-ec2.html
        this.instanceProfile = new aws_iam_1.InstanceProfile(this, "InstanceProfile", {
            role: this.jobFlowRole,
        });
        this.grantPrincipal = this.jobFlowRole;
        this.serviceRole = new aws_iam_1.Role(this, "ServiceRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("elasticmapreduce.amazonaws.com"),
            managedPolicies: [
                // TODO: fine-grained policies
                // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-iam-policies.html
                // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role.html
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonEMRServicePolicy_v2"),
                // TODO: remove
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AdministratorAccess"),
            ],
        });
        this.serviceRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["ec2:CreateSecurityGroup"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                props.vpc.vpcArn,
                core_1.Arn.format({
                    service: "ec2",
                    resource: "security-group",
                    resourceName: `*`,
                }, core_1.Stack.of(this)),
            ],
        }));
        this.primarySg = new aws_ec2_1.SecurityGroup(this, "PrimarySG", {
            vpc: props.vpc,
            description: "The security group for the primary instance (private subnets). See: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private",
            allowAllOutbound: true,
        });
        this.connections = this.primarySg.connections;
        this.coreSg = new aws_ec2_1.SecurityGroup(this, "CoreSG", {
            vpc: props.vpc,
            description: "Security group for core and task instances (private subnets). See: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private",
            allowAllOutbound: true,
        });
        this.serviceAccessSg = new aws_ec2_1.SecurityGroup(this, "ServiceAccessSG", {
            vpc: props.vpc,
            allowAllOutbound: false,
        });
        this.configureSecurityGroups();
        // this constructs a globally unique identifier for the cluster for use in ResourceTag IAM policies
        // should work when clusters are deployed via CDK or Service Catalog
        this.clusterID = `${core_1.Aws.ACCOUNT_ID}/${core_1.Aws.REGION}/${core_1.Aws.STACK_NAME}/${props.clusterName}`;
        const cluster = new aws_emr_1.CfnCluster(this, "Resource", {
            name: props.clusterName,
            // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-policy-fullaccess-v2.html
            tags: [
                {
                    key: "for-use-with-amazon-emr-managed-policies",
                    value: "true",
                },
                {
                    key: "ClusterID",
                    value: this.clusterID,
                },
            ],
            jobFlowRole: this.instanceProfile.instanceProfileArn,
            serviceRole: this.serviceRole.roleArn,
            releaseLabel: props.releaseLabel?.label ?? release_label_js_1.ReleaseLabel.LATEST.label,
            applications: [
                { name: application_js_1.Application.AMAZON_CLOUDWATCH_AGENT },
                { name: application_js_1.Application.LIVY },
                { name: application_js_1.Application.SPARK },
            ],
            steps: core_1.Lazy.any({
                produce: () => this.steps,
            }),
            stepConcurrencyLevel: props.stepConcurrencyLevel,
            bootstrapActions: core_1.Lazy.any({
                produce: () => this.bootstrapActions,
            }),
            instances: {
                // TODO: is 1 subnet OK?
                // TODO: required for instance fleets
                ec2SubnetId: props.vpc.privateSubnets[0].subnetId,
                emrManagedMasterSecurityGroup: this.primarySg.securityGroupId,
                emrManagedSlaveSecurityGroup: this.coreSg.securityGroupId,
                serviceAccessSecurityGroup: this.serviceAccessSg.securityGroupId,
                // TODO: add advanced options
                // masterInstanceFleet: {},
                masterInstanceGroup: {
                    instanceCount: 1,
                    instanceType: masterInstanceType.toString(),
                    market: props.masterInstanceGroup?.market ?? market_js_1.Market.ON_DEMAND,
                },
                // TODO: add advanced options
                // coreInstanceFleet: {},
                coreInstanceGroup: {
                    instanceCount: props.coreInstanceGroup?.instanceCount ?? 1,
                    instanceType: coreInstanceType.toString(),
                    market: props.coreInstanceGroup?.market ?? market_js_1.Market.SPOT,
                },
                // TODO: support tasks
                // taskInstanceFleets: {},
                // taskInstanceGroups: {},
            },
            autoTerminationPolicy: props.idleTimeout
                ? {
                    idleTimeout: props.idleTimeout.toSeconds(),
                }
                : undefined,
            configurations: core_1.Lazy.any({
                produce: () => (0, configuration_js_1.combineConfigurations)({
                    classification: "spark-defaults",
                    configurationProperties: {
                        // configure spark to use the virtual environment
                        "spark.pyspark.python": "python3",
                        "spark.pyspark.virtualenv.enabled": "true",
                        "spark.pyspark.virtualenv.type": "native",
                        "spark.pyspark.virtualenv.bin.path": "/usr/bin/virtualenv",
                        "spark.driver.extraJavaOptions": (0, spark_config_js_1.toCLIArgs)(this.extraJavaOptions),
                    },
                }, ...this.configurations),
            }),
            scaleDownBehavior: props.scaleDownBehavior ??
                ScaleDownBehavior.TERMINATE_AT_TASK_COMPLETION,
            managedScalingPolicy: props.managedScalingPolicy,
            // TODO: configure specific Role
            // autoScalingRole: "EMR_AutoScaling_DefaultRole",
        });
        Object.entries(props.catalogs).forEach(([catalogName, catalog]) => catalog.bind(this, catalogName));
        this.resource = cluster;
        cluster.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.DESTROY);
        if (props.installSSMAgent) {
            this.installSSMAgent();
        }
        if (props.installGitHubCLI) {
            this.installGitHubCLI();
        }
    }
    /**
     * Configure the EMR cluster start the Thrift Server and serve JDBC requests on the specified port.
     *
     * @param options to set when running the JDBC server
     * @returns a reference to the JDBC server
     * @example
     * const sparkSQL = cluster.jdbc({
     *  port: 10000,
     * });
     * sparkSQL.allowFrom(sageMakerDomain);
     */
    jdbc(options) {
        return new jdbc_js_1.Jdbc(this, options);
    }
    /**
     * Add an EMR Step to the cluster.
     *
     * This step will run when the cluster is started.
     *
     * @param step the step to add
     */
    addStep(step) {
        this.steps.push(step);
    }
    /**
     * Add EMR Configurations to the cluster.
     *
     * E.g. spark or hive configurations.
     *
     * @param configurations additional configurations to add
     */
    addConfig(...configurations) {
        this.configurations.push(...configurations);
    }
    /**
     * Add a Bootstrap Action to the cluster.
     *
     * Bootstrap actions are scripts that run on the cluster before Hadoop starts.
     *
     * @param action the bootstrap action to add
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-bootstrap.html
     */
    addBootstrapAction(action) {
        this.bootstrapActions.push(action);
    }
    /**
     * Installs the SSM Agent on Primary, Core, and Task nodes.
     *
     * Authorizes the EC2 instances to communicate with the SSM service.
     *
     * @see https://aws.amazon.com/blogs/big-data/securing-access-to-emr-clusters-using-aws-systems-manager/
     */
    installSSMAgent() {
        if (this.isSSMAgentInstalled) {
            return;
        }
        this.isSSMAgentInstalled = true;
        // const __dirname = path.dirname(new URL(import.meta.url).pathname);
        const singletonId = "packyak::emr::install-ssm-agent";
        const stack = core_1.Stack.of(this);
        const bootstrapScript = stack.node.tryFindChild(singletonId) ??
            new aws_s3_assets_1.Asset(stack, singletonId, {
                path: path.join(__dirname, "..", "..", "scripts", "install-ssm-agent.sh"),
            });
        bootstrapScript.grantRead(this.jobFlowRole);
        this.addBootstrapAction({
            name: "Install SSM Agent",
            scriptBootstrapAction: {
                path: bootstrapScript.s3ObjectUrl,
            },
        });
        // this allows the SSM agent to communicate with the SSM service
        this.jobFlowRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore"));
    }
    installGitHubCLI() {
        if (this.isGitHubCLIInstalled) {
            return;
        }
        this.isGitHubCLIInstalled = false;
        this.addBootstrapAction({
            name: "Install GitHub CLI",
            scriptBootstrapAction: {
                path: this.getScript("install-github-cli.sh").s3ObjectUrl,
            },
        });
    }
    getScript(name) {
        const singletonId = `packyak::emr::${name}`;
        const stack = core_1.Stack.of(this);
        const bootstrapScript = stack.node.tryFindChild(singletonId) ??
            new aws_s3_assets_1.Asset(stack, singletonId, {
                path: path.join(__dirname, "..", "..", "scripts", name),
            });
        return bootstrapScript;
    }
    /**
     * Grant an permission to start an SSM Session on the EMR cluster.
     *
     * @param grantee the principal to grant the permission to
     *
     * // TODO: figure out how to use SSM Session Documents to:
     * //       1. customize where state is store and encrypt it
     * //       2. customize other session properties
     * //       3. constrain access with IAM Condition: ssm:SessionDocumentAccessCheck
     * @see https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-specify-session-document.html
     */
    grantStartSSMSession(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "ssm:DescribeInstanceProperties",
                "ssm:DescribeSessions",
                "ec2:describeInstances",
                "ssm:GetConnectionStatus",
            ],
            // TODO: not sure if this can be constrained
            resources: ["*"],
        }));
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["ssm:StartSession"],
            resources: [
                core_1.Arn.format({
                    service: "ec2",
                    resource: "instance",
                    resourceName: "*",
                }, core_1.Stack.of(this)),
            ],
            conditions: {
                StringEquals: {
                    // restrict access to only this cluster, as identified by AccountID, Region, StackName and ClusterName
                    "ssm:resourceTag/ClusterID": this.clusterID,
                },
            },
        }));
    }
    /**
     * Allows connections to the Livy server on port 8998 from the specified {@link other} security group.
     */
    allowLivyFrom(other) {
        this.connections.allowFrom(other, aws_ec2_1.Port.tcp(8998));
    }
    /**
     * Configure the rules for the Primary, Core, and Service Access security groups.
     */
    configureSecurityGroups() {
        this.configureMasterSecurityGroup();
        this.configureCoreSecurityGroup();
        this.configureServiceAccessSecurityGroup();
    }
    /**
     * Configure security group for Primary instance (master)
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private
     */
    configureMasterSecurityGroup() {
        this.primarySg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with each other over ICMP or any TCP or UDP port.");
        this.primarySg.connections.allowFrom(this.coreSg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with the core and task nodes over ICMP or any TCP or UDP port.");
        this.primarySg.connections.allowFrom(this.serviceAccessSg, aws_ec2_1.Port.tcp(8443), "This rule allows the cluster manager to communicate with the primary node.");
    }
    /**
     * Configure security group for Core & Task nodes
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private
     */
    configureCoreSecurityGroup() {
        this.coreSg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with the core and task nodes over ICMP or any TCP or UDP port.");
        this.coreSg.connections.allowFrom(this.coreSg, aws_ec2_1.Port.allTraffic(), "Allows core and task node(s) to communicate with each other over ICMP or any TCP or UDP port.");
        this.coreSg.connections.allowFrom(this.serviceAccessSg, aws_ec2_1.Port.tcp(8443), "This rule allows the cluster manager to communicate with core and task nodes.");
    }
    /**
     * Configure security group for Service Access.
     *
     * It allows inbound traffic on 8443 from the primary security group.
     * It allows outbound traffic on 8443 to the primary and core security groups.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-sa-private
     */
    configureServiceAccessSecurityGroup() {
        this.serviceAccessSg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.tcp(9443));
        this.serviceAccessSg.connections.allowTo(this.primarySg, aws_ec2_1.Port.tcp(8443));
        this.serviceAccessSg.connections.allowTo(this.coreSg, aws_ec2_1.Port.tcp(8443));
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@packyak/aws-cdk.Cluster", version: "0.3.6" };
//# sourceMappingURL=data:application/json;base64,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