import { Construct } from "constructs";
import { DynamoDBNessieVersionStore } from "./nessie-version-store.js";
import { RemovalPolicy } from "aws-cdk-lib/core";
import { ICatalog } from "../emr/catalog.js";
import { Cluster } from "../emr/cluster.js";
import { IBucket } from "aws-cdk-lib/aws-s3";
import { ILogGroup } from "aws-cdk-lib/aws-logs";
export interface INessieCatalog extends ICatalog {
    /**
     * The Nessie service endpoint.
     */
    readonly endpoint: string;
    /**
     * Endpoint for the Nessie API v1.
     *
     * This endpoint provides access to the version 1 of the Nessie API. It is recommended to use the v2 endpoint for the latest features and improvements.
     *
     * @deprecated This version of the API is deprecated and will be removed in future releases. Use {@link apiV2Url} instead.
     */
    readonly apiV1Url: string;
    /**
     * Endpoint for the Nessie API v2.
     *
     * This endpoint provides access to the version 2 of the Nessie API. It is the recommended endpoint to use for all interactions with the Nessie service.
     *
     * Note: The Nessie CLI is compatible only with this version of the API. For CLI interactions, ensure to use this endpoint.
     */
    readonly apiV2Url: string;
    /**
     * The default main branch of the Nessie repository.
     *
     * This property specifies the main branch that will be used by default for all operations within the Nessie service.
     *
     * @default main
     */
    readonly defaultMainBranch: string;
}
export interface BaseNessieRepoProps {
    /**
     * The name of this catalog in the Spark Context.
     *
     * @default spark_catalog - i.e. the default catalog
     */
    readonly catalogName?: string;
    /**
     * @default - one is created for you
     */
    readonly warehouseBucket?: IBucket;
    /**
     * The prefix to use for the warehouse path.
     *
     * @default - no prefix (e.g. use the root: `s3://bucket/`)
     */
    readonly warehousePrefix?: string;
    /**
     * The default main branch of a Nessie repository.
     *
     * @default main
     */
    readonly defaultMainBranch?: string;
    /**
     * Properties for configuring the {@link DynamoDBNessieVersionStore}
     */
    readonly versionStore?: DynamoDBNessieVersionStore;
    /**
     * The log group to use for the Nessie service.
     *
     * @default - a new log group is created for you
     */
    readonly logGroup?: ILogGroup;
    /**
     * The removal policy to apply to the Nessie service.
     *
     * @default RemovalPolicy.DESTROY - dynamodb tables will be destroyed.
     */
    readonly removalPolicy?: RemovalPolicy;
}
export declare abstract class BaseNessieCatalog extends Construct implements INessieCatalog {
    /**
     *
     */
    readonly catalogName: string;
    /**
     * The {@link NessieConfig} for this service.
     *
     * This will translate to environment variables set at runtime.
     *
     * @see https://projectnessie.org/try/configuration/#configuration
     */
    protected readonly config: Record<string, any>;
    /**
     * The DynamoDB Table storing all
     *
     * @see https://projectnessie.org/develop/kernel/#high-level-abstract
     */
    readonly versionStore: DynamoDBNessieVersionStore;
    /**
     * The default main branch of a Nessie repository created in this service.
     */
    readonly defaultMainBranch: string;
    /**
     * The URL to this Nessie service.
     */
    abstract readonly endpoint: string;
    /**
     * The S3 bucket used as the warehouse for Nessie.
     */
    readonly warehouseBucket: IBucket;
    /**
     * The prefix to use for the warehouse path.
     */
    readonly warehousePrefix: string | undefined;
    /**
     * Endpoint for the Nessie API v1.
     *
     * @deprecated use {@link apiV2Url} instead
     */
    get apiV1Url(): string;
    /**
     * Endpoint for the Nessie API v2.
     *
     * Note: Nessie CLI is not compatible with V1. For CLI use {@link apiV2Url}
     */
    get apiV2Url(): string;
    constructor(scope: Construct, id: string, props: BaseNessieRepoProps);
    protected configAsEnvVars(): {
        [key: string]: string;
    };
    bind(cluster: Cluster, catalogName: string): void;
}
