import { Connections, IConnectable, IVpc, InstanceType, SecurityGroup } from "aws-cdk-lib/aws-ec2";
import { CfnCluster } from "aws-cdk-lib/aws-emr";
import { IGrantable, IPrincipal, InstanceProfile, Role } from "aws-cdk-lib/aws-iam";
import { Duration, RemovalPolicy, Resource } from "aws-cdk-lib/core";
import { Construct } from "constructs";
import { BootstrapAction } from "./bootstrap-action.js";
import { ICatalog } from "./catalog.js";
import { Configuration } from "./configuration.js";
import { Jdbc, JdbcProps } from "./jdbc.js";
import { Market } from "./market.js";
import { ReleaseLabel } from "./release-label.js";
import { Step } from "./step.js";
export interface InstanceGroup {
    /**
     * @default 1
     */
    readonly instanceCount?: number;
    /**
     * @default m5.xlarge
     */
    readonly instanceType?: InstanceType;
    /**
     * @default SPOT
     */
    readonly market?: Market;
}
export declare enum ScaleDownBehavior {
    TERMINATE_AT_INSTANCE_HOUR = "TERMINATE_AT_INSTANCE_HOUR",
    TERMINATE_AT_TASK_COMPLETION = "TERMINATE_AT_TASK_COMPLETION"
}
export declare enum ScalingUnit {
    INSTANCES = "Instances",
    INSTANCE_FLEET_UNITS = "InstanceFleetUnits",
    VCPU = "VCPU"
}
export interface ManagedScalingPolicy {
    readonly computeLimits: ComputeLimits;
}
export interface ComputeLimits {
    readonly unitType: ScalingUnit;
    readonly minimumCapacityUnits: number;
    readonly maximumCapacityUnits: number;
}
export interface ClusterProps {
    /**
     * Name of the EMR Cluster.
     */
    readonly clusterName: string;
    /**
     * The VPC to deploy the EMR cluster into.
     */
    readonly vpc: IVpc;
    /**
     * @default - 1 m5.xlarge from SPOT market
     */
    readonly masterInstanceGroup?: InstanceGroup;
    /**
     * @default - 1 m5.xlarge from SPOT market
     */
    readonly coreInstanceGroup?: InstanceGroup;
    /**
     * TODO: support tasks
     *
     * @default - 1 m5.xlarge from SPOT market
     */
    /**
     * @default None
     */
    readonly idleTimeout?: Duration;
    /**
     * @default - {@link ReleaseLabel.LATEST}
     */
    readonly releaseLabel?: ReleaseLabel;
    /**
     * The catalogs to use for the EMR cluster.
     */
    readonly catalogs: Record<string, ICatalog>;
    /**
     * @default - {@link ScaleDownBehavior.TERMINATE_AT_TASK_COMPLETION}
     */
    readonly scaleDownBehavior?: ScaleDownBehavior;
    /**
     * @default - No managed scaling policy
     */
    readonly managedScalingPolicy?: ManagedScalingPolicy;
    /**
     * Override EMR Configurations.
     *
     * @default - the {@link catalog}'s configurations + .venv for the user code.
     */
    readonly configurations?: Configuration[];
    /**
     * @default {@link RemovalPolicy.DESTROY}
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * @default - No bootstrap actions
     */
    readonly bootstrapActions?: BootstrapAction[];
    /**
     * The EMR Steps to submit to the cluster.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-submit-step.html
     */
    readonly steps?: Step[];
    /**
     * The concurrency level of the cluster.
     *
     * @default 1
     */
    readonly stepConcurrencyLevel?: number;
    /**
     * Extra java options to include in the Spark context by default.
     */
    readonly extraJavaOptions?: Record<string, string>;
    /**
     * Installs and configures the SSM agent to run on all Primary, Core and Task nodes.
     *
     * @default - `true` if {@link enableSSMTunnelOverSSH} is also `true`, otherwise `false`
     */
    readonly installSSMAgent?: boolean;
    /**
     * Install the GitHub CLI on the EMR cluster.
     *
     * @default false
     */
    readonly installGitHubCLI?: boolean;
}
export declare class Cluster extends Resource implements IGrantable, IConnectable {
    readonly release: ReleaseLabel;
    readonly primarySg: SecurityGroup;
    readonly coreSg: SecurityGroup;
    readonly serviceAccessSg: SecurityGroup;
    readonly connections: Connections;
    readonly grantPrincipal: IPrincipal;
    readonly extraJavaOptions: Readonly<Record<string, string>>;
    readonly jobFlowRole: Role;
    readonly instanceProfile: InstanceProfile;
    readonly serviceRole: Role;
    private readonly steps;
    private readonly bootstrapActions;
    private readonly configurations;
    private readonly clusterID;
    protected readonly resource: CfnCluster;
    constructor(scope: Construct, id: string, props: ClusterProps);
    /**
     * Configure the EMR cluster start the Thrift Server and serve JDBC requests on the specified port.
     *
     * @param options to set when running the JDBC server
     * @returns a reference to the JDBC server
     * @example
     * const sparkSQL = cluster.jdbc({
     *  port: 10000,
     * });
     * sparkSQL.allowFrom(sageMakerDomain);
     */
    jdbc(options: JdbcProps): Jdbc;
    /**
     * Add an EMR Step to the cluster.
     *
     * This step will run when the cluster is started.
     *
     * @param step the step to add
     */
    addStep(step: Step): void;
    /**
     * Add EMR Configurations to the cluster.
     *
     * E.g. spark or hive configurations.
     *
     * @param configurations additional configurations to add
     */
    addConfig(...configurations: Configuration[]): void;
    /**
     * Add a Bootstrap Action to the cluster.
     *
     * Bootstrap actions are scripts that run on the cluster before Hadoop starts.
     *
     * @param action the bootstrap action to add
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-bootstrap.html
     */
    addBootstrapAction(action: BootstrapAction): void;
    /**
     * private flag to make {@link installSSMAgent} idempotent
     */
    private isSSMAgentInstalled;
    /**
     * Installs the SSM Agent on Primary, Core, and Task nodes.
     *
     * Authorizes the EC2 instances to communicate with the SSM service.
     *
     * @see https://aws.amazon.com/blogs/big-data/securing-access-to-emr-clusters-using-aws-systems-manager/
     */
    installSSMAgent(): void;
    private isGitHubCLIInstalled;
    installGitHubCLI(): void;
    private getScript;
    /**
     * Grant an permission to start an SSM Session on the EMR cluster.
     *
     * @param grantee the principal to grant the permission to
     *
     * // TODO: figure out how to use SSM Session Documents to:
     * //       1. customize where state is store and encrypt it
     * //       2. customize other session properties
     * //       3. constrain access with IAM Condition: ssm:SessionDocumentAccessCheck
     * @see https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-specify-session-document.html
     */
    grantStartSSMSession(grantee: IGrantable): void;
    /**
     * Allows connections to the Livy server on port 8998 from the specified {@link other} security group.
     */
    allowLivyFrom(other: IConnectable): void;
    /**
     * Configure the rules for the Primary, Core, and Service Access security groups.
     */
    private configureSecurityGroups;
    /**
     * Configure security group for Primary instance (master)
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private
     */
    private configureMasterSecurityGroup;
    /**
     * Configure security group for Core & Task nodes
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private
     */
    private configureCoreSecurityGroup;
    /**
     * Configure security group for Service Access.
     *
     * It allows inbound traffic on 8443 from the primary security group.
     * It allows outbound traffic on 8443 to the primary and core security groups.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-sa-private
     */
    private configureServiceAccessSecurityGroup;
}
