# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pluserable',
 'pluserable.data',
 'pluserable.data.sqlalchemy',
 'pluserable.web',
 'pluserable.web.pyramid']

package_data = \
{'': ['*'], 'pluserable': ['templates/*', 'templates/admin/*']}

install_requires = \
['bag>=5.0.0',
 'deform',
 'kerno>=0.7.0',
 'passlib',
 'pyramid>=2.0',
 'pyramid_mailer',
 'pyramid_mako',
 'redis',
 'sqlalchemy',
 'transaction',
 'zope.interface']

setup_kwargs = {
    'name': 'pluserable',
    'version': '0.11.0',
    'description': 'Generic user registration for the Pyramid web framework',
    'long_description': '============================\nIntroduction to *pluserable*\n============================\n\n*pluserable* provides generic user registration for the Pyramid\nweb framework, if your web app uses SQLAlchemy.\n\nIt is a pluggable web application that provides user registration, login,\nlogout and change password functionality. *pluserable* follows a policy of\nminimal interference, so your app can mostly keep its existing models.\n\n*pluserable* is highly configurable, you can make it do what you want.\nFeatures (all of them optional) include:\n\n- User sign up (registration).\n- Email address confirmation step (user activation).\n- Log in and log out.\n- Forgot password (sends an email with a new activation).\n- Reset password.\n- Brute force prevention by storing in a redis server the IP address of\n  any user who fails authentication.  Then the user must wait before\n  trying to authenticate again, and the time doubles with each attempt.\n- You can replace forms, templates, views, models, UI strings\n  and email message content.\n\nIt is gradually being refactored to support other web frameworks, too.\n\nThe documentation is at http://docs.nando.audio/pluserable/latest/\n\n\nMinimal integration\n===================\n\n- Create a virtualenv and activate it. Install pyramid and create\n  your Pyramid project.\n\n- Ensure you have some SQLAlchemy declarative initialization.\n  This is usually created by the Pyramid scaffold.\n\n- Edit your *setup.py* to add "pluserable" to the dependencies in the\n  *install_requires* list.\n\n- Run ``python setup.py develop`` on your project to install all\n  dependencies into your virtualenv.\n\n- Create models inheriting from pluserable\'s abstract models.\n  Find an example in the file `pluserable/tests/models.py\n  <https://github.com/nandoflorestan/pluserable/blob/master/pluserable/tests/models.py>`_.\n\n- In your Pyramid configuration file, create a section called\n  "kerno utilities" like this::\n\n    [kerno utilities]\n        # Let pluserable know which model classes to use:\n        activation class = some.app.models:Activation\n        group class = some.app.models:Group\n        user class = some.app.models:User\n\n        # Give pluserable a SQLAlchemy session factory:\n        session factory = some.app.models:get_sqlalchemy_session\n\n- Above you are also pointing to a session factory. Just write a\n  function that returns a SQLAlchemy session instance, ready for use.\n  Alternatively, it can be a scoped session.\n\n- Also add to your Pyramid configuration file a "pluserable" section\n  like this::\n\n    [pluserable]\n        # Whether to log a user in directly after registration. Default: false.\n        # autologin = false\n\n        # Email domains we do not accept for registration. One per line.\n        email_domains_blacklist =\n\n        # Prevents brute force. Default: empty string, meaning feature disabled.\n        # Syntax: redis://username:password@localhost:6379/0\n        # redis_url =\n\n        # Number of seconds a user must wait before trying login again.\n        # Default value: 15, doubled on each attempt. Depends on a redis_url.\n        # seconds_after_login_fail =\n\n        # Route or URL after a user confirms their email. Default: "index"\n        # activate_redirect = index\n\n        # Route or URL after a user fills the forgot password form. Default: "index"\n        # forgot_password_redirect = index\n\n        # Route or URL after a user logs in. Default: "index"\n        # login_redirect = index\n\n        # Route or URL after a user logs out. Default: "index"\n        # logout_redirect = index\n\n        # Route or URL after a user signs up for an account. Default: "index"\n        # register_redirect = index\n\n        # Route or URL after a user resets their password. Default: "index"\n        # reset_password_redirect = index\n\n        # Whether to enable retail rendering of deform forms. Default: false.\n        # deform_retail = false\n\n- pluserable includes a (very standard and vanilla) SecurityPolicy.\n  If you wish to use it, do::\n\n    config.include("pluserable.web.pyramid.security")\n\n- You may write a function that returns a configuration for Pyramid routes and\n  views (which is something you probably want to manipulate in code\n  because it won\'t change between dev, staging and production environments),\n  and then inform pluserable about it like this::\n\n    registry.settings["pluserable_configurator"] = "my.package:some_function"\n\n- Your ``pluserable_configurator`` function would look more or less like this::\n\n    from pluserable.settings import get_default_pluserable_settings\n\n    def my_pluserable(config):\n        """This function is called by pluserable during app startup."""\n        adict = get_default_pluserable_settings()\n        # Manipulate adict to customize pluserable for your application, then\n        return adict\n\n- Include **pluserable** into your Pyramid application,\n  just after Pyramid\'s Configurator is instantiated::\n\n    from kerno.start import Eko\n\n    def includeme(config):\n        """Stuff called during Pyramid initialization."""\n        eko = Eko.from_ini("server.ini")\n        eko.include("pluserable")\n        config.include("pluserable")\n\nThis does almost nothing: it only makes a new config method available.\nYou have to use it next::\n\n    config.setup_pluserable(  # Directive that starts pluserable up\n        global_settings["__file__"],  # Path to your INI configuration file\n    )\n\nThe above causes **pluserable** to read your INI file -- especially\nthe ``[Kerno utilities]`` and ``[pluserable]`` sections.\n\nThe backend for database access is in a separate class, this way you can\nsubstitute the implementation. This is called the "repository" pattern.\nOne of the main benefits is, it makes writing tests much easier.\nIt is recommended that you use the repository pattern in your app, too.\nThe `pluserable repository\n<https://github.com/nandoflorestan/pluserable/blob/master/pluserable/data/repository.py>`_.\nis instantiated once per request.\nThe instance is available in the ``request.repo`` variable.\n\n- If you haven\'t done so yet, configure an HTTP session factory according to\n  the Sessions chapter of the Pyramid documentation.\n\n- Create your database and tables. Maybe even an initial user.\n\n- Be sure to pass an ``authentication_policy`` argument in the\n  ``config = Configurator(...)`` call. Refer to Pyramid docs for details.\n\n- By now the login form should appear at /login, but /register shouldn\'t.\n\n- Include the package pyramid_mailer for the validation e-mail and\n  "forgot password" e-mail::\n\n    config.include("pyramid_mailer")\n\n- The /register form should appear, though ugly. Now you have a choice\n  regarding user activation by email:\n\n  - You may just disable user activation by setting, in your .ini file::\n\n      [pluserable]\n          # (other settings, then...)\n          require_activation = false\n\n  - Otherwise, configure pyramid_mailer `according to its documentation\n    <http://docs.pylonsproject.org/projects/pyramid_mailer/en/latest/>`_\n    and test the registration page.\n\n- If you are using pyramid_tm or the ZopeTransactionManager, your minimal\n  integration is done. (The pages are ugly, but working. Keep reading...)\n\n\nNeed to session.commit()?\n=========================\n\n*pluserable* does not require pyramid_tm or the ZopeTransactionManager with your\nsession but if you do not use them you do have to take one extra step.\nWe don\'t commit transactions for you because that just wouldn\'t be nice!\n\nAll you have to do is subscribe to the extension events and\ncommit the session yourself. This also gives you the chance to\ndo some extra processing::\n\n    from pluserable.events import (\n        EventRegistration, EventActivation, EventLogin,\n        EventPasswordReset, EventProfileUpdated,\n    )\n\n    def handle_event(event):\n        request = event.request\n        session = request.registry.getUtility(IDBSession)\n        session.commit()\n\n    kerno.events.subscribe(handle_event, EventRegistration)\n    kerno.events.subscribe(handle_event, EventActivation)\n    kerno.events.subscribe(handle_event, EventLogin)\n    kerno.events.subscribe(handle_event, EventPasswordReset)\n    kerno.events.subscribe(handle_event, EventProfileUpdated)\n\nThe ``kerno`` variable comes from your initialization of the kerno library,\nwhich is useful to define the domain model of your application.\n(The ``kerno`` variable represents a global object for the domain model --\nit does not know anything about the web framework.)\nAt runtime pluserable finds the kerno instance at ``request.kerno``.\nIn the future pluserable will support web frameworks other than Pyramid.\n\n\nWhether or not to have a "username" field\n=========================================\n\nIt is important that you analyze the characteristics of your web application\nand decide whether you need a ``username`` field for users to log in with.\npluserable provides 2 modes of operation:\n\n1) email + username\n-------------------\n\nThe user chooses a username when registering and later she can log in by\nproviding either the username or the email address. Therefore, usernames\nmay NOT contain the @ character.\n\n**This mode is the default.** It is expressed by the Pyramid configuration\nsetting ``pluserable.handle = username``.\n\n2) email only\n-------------\n\nThere is no ``username`` field and users only provide their email address.\nYou enable this mode by:\n\n* Making your User model subclass ``NoUsernameMixin`` instead\n  of ``UsernameMixin``;\n* Adding this configuration setting: ``pluserable.handle = email``,\n  which will make pluserable default to schemas that contain email\n  fields instead of username fields.\n\nThis choice should be made at the beginning of a project.  If later you\nchange it and want to keep your data you must deal with the existing\n(or missing) "username" column yourself.\n\n\nChanging the forms\n==================\n\nIf you would like to modify any of the forms, you just need\nto register the new deform class to be used.\n\nThe interfaces you have available to override from pluserable.interfaces are:\n\n- IPluserableLoginForm\n- IPluserableRegisterForm\n- IPluserableForgotPasswordForm\n- IPluserableResetPasswordForm\n- IPluserableProfileForm\n\nThis is how you would do it (*MyForm* being a custom deform Form class)::\n\n    config.registry.registerUtility(MyForm, IPluserableLoginForm)\n\n\nChanging the templates\n======================\n\nIf you would like to substitute the templates you can use pyramid\'s\n`override_asset <http://pyramid.readthedocs.org/en/latest/narr/assets.html#overriding-assets-section>`_::\n\n    config.override_asset(\n        to_override="pluserable:templates/template.mako",\n        override_with="your_package:templates/anothertemplate.mako",\n    )\n\nThe templates you have available to override are:\n\n- login.mako\n- register.mako\n- forgot_password.mako\n- reset_password.mako\n- profile.mako\n\nIf you would like to override the templates with Jinja2, or any other\ntemplating language, just override the view configuration::\n\n    config.add_view("pluserable.views.AuthController", attr="login",\n        route_name="login", renderer="yourapp:templates/login.jinja2")\n    config.add_view("pluserable.views.ForgotPasswordController",\n        attr="forgot_password", route_name="forgot_password",\n        renderer="yourapp:templates/forgot_password.jinja2")\n    config.add_view("pluserable.views.ForgotPasswordController",\n        attr="reset_password", route_name="reset_password",\n        renderer="yourapp:templates/reset_password.jinja2")\n    config.add_view("pluserable.views.RegisterController", attr="register",\n        route_name="register", renderer="yourapp:templates/register.jinja2")\n    config.add_view("pluserable.views.ProfileController", attr="profile",\n        route_name="profile", renderer="yourapp:templates/profile.jinja2")\n\n\nChanging UI strings\n===================\n\nTake a look at `this class\n<https://github.com/nandoflorestan/pluserable/blob/master/pluserable/strings.py>`_.\nThis is where we store all the UI strings in *pluserable*.\nIf you\'d like to change one or two messages, simply create a subclass\nand configure it::\n\n    [kerno utilities]\n        # (...bla bla bla...)\n\n        # Determining the UI strings is as easy as pointing to a class:\n        string class = pluserable.strings:UIStringsBase\n\nHere is an example implementation of a strings class::\n\n    class AuthStrings(UIStringsBase):\n        """Our alterations to the pluserable UI text."""\n\n        login_done = None   # Do not flash a message after the user logs in\n        logout_done = None  # Do not flash a message after the user logs out\n\n\nChanging the email messages\n===========================\n\n*pluserable* includes functions that send very simple, plain text only,\nemail messages using pyramid_mailer.  Messages are sent synchronously.\n\nYou can replace those with your own functions in order to send emails\nasynchronously (e. g. using celery), or to determine the content of the\nemail messages.  Plug your function in through kerno utilities -- for example\nin configuration::\n\n    [kerno utilities]\n    pluserable.send_activation_email = myapp.actions:send_activation_email\n    pluserable.send_reset_password_email = myapp.actions:send_reset_password_email\n\n...or imperatively in startup code::\n\n    eko.utilities.register(\n        "pluserable.send_activation_email",\n        "myapp.actions:send_activation_email"\n    )\n    eko.utilities.register(\n        "pluserable.send_reset_password_email",\n        "myapp.actions:send_reset_password_email"\n    )\n\n\nBrute force prevention\n======================\n\nBrute force prevention is enabled by configuring ``redis_url``\nas mentioned above.  This will store in a redis server the IP address of\nany user who fails authentication.  Then the user must wait before\ntrying to authenticate again, and the time doubles with each attempt.\n\nIf you wish to tweak the behavior of brute force prevention, or use a different\nstorage, you can create a subclass, and then configure it as a kerno utility::\n\n    [kerno utilities]\n        # Below is the default class, but you can change it to your own.\n        brute force class = pluserable.no_bruteforce:BruteForceAidRedis\n\n\nChanging the primary key column name\n====================================\n\nIf you wish to override the primary key attribute name, you can do so\nby creating a new mixin class::\n\n    class NullPkMixin(Base):\n        abstract = True\n        _idAttribute = "pk"\n\n        @declared_attr\n        def pk(self):\n            return Base.pk\n\n        @declared_attr\n        def id(self):\n            return None\n\n    class User(NullPkMixin, UserMixin):\n        pass\n\n\npluserable development\n======================\n\nSee https://github.com/nandoflorestan/pluserable\n\nIf you would like to help make any changes to *pluserable*, you can run its\nunit tests with py.test::\n\n    py.test\n\nTo check test coverage::\n\n    py.test --cov-report term-missing --cov pluserable\n\nThe tests can also be run in parallel::\n\n    py.test -n4\n\nWe are going to use this build server:\nhttp://travis-ci.org/#!/nandoflorestan/pluserable\n\n\nOrigin of the project\n=====================\n\n*pluserable* started as a fork of *horus* by John Anderson:\nhttps://github.com/eventray/horus\n\n*horus* is no longer maintained since 2015.  *pluserable* is maintained and\nsees 1 or 2 releases per year.\n',
    'author': 'Nando Florestan',
    'author_email': 'nandoflorestan@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/nandoflorestan/pluserable',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9',
}


setup(**setup_kwargs)
