"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.extraLambdaEnv = {};
        this.props = props ?? {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props.vpc,
            vpcSubnets: this.props.vpcSubnets,
            allowPublicSubnet: this.props.allowPublicSubnet,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        this.orchestrator = this.stateMachine();
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine() {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: this.deleteRunner(),
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${provider.label}`), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        return new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const providers = this.providers.map(provider => {
            return {
                type: provider.constructor.name,
                label: provider.label,
                vpcArn: provider.vpc && provider.vpc.vpcArn,
                securityGroup: provider.securityGroup && provider.securityGroup.securityGroupId,
                roleArn: provider.grantPrincipal.grantPrincipal.roleArn,
            };
        });
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                PROVIDERS: JSON.stringify(providers),
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${cdk.Stack.of(this).region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.3.1" };
//# sourceMappingURL=data:application/json;base64,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