import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_iam as iam, aws_lambda as lambda, aws_stepfunctions as stepfunctions } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IRunnerProvider, RunnerRuntimeParameters, RunnerProviderProps, IImageBuilder } from './common';
export interface LambdaRunnerProps extends RunnerProviderProps {
    /**
     * Provider running an image to run inside CodeBuild with GitHub runner pre-configured.
     *
     * The default command (`CMD`) should be `["runner.handler"]` which points to an included `runner.js` with a function named `handler`. The function should start the GitHub runner.
     *
     * @see https://github.com/CloudSnorkel/cdk-github-runners/tree/main/src/providers/docker-images/lambda
     * @default image builder with LambdaRunner.LINUX_X64_DOCKERFILE_PATH as Dockerfile
     */
    readonly imageBuilder?: IImageBuilder;
    /**
     * GitHub Actions label used for this provider.
     *
     * @default 'lambda'
     */
    readonly label?: string;
    /**
     * The amount of memory, in MB, that is allocated to your Lambda function.
     * Lambda uses this value to proportionally allocate the amount of CPU
     * power. For more information, see Resource Model in the AWS Lambda
     * Developer Guide.
     *
     * @default 2048
     */
    readonly memorySize?: number;
    /**
    * The size of the function’s /tmp directory in MiB.
    *
    * @default 10 GiB
    */
    readonly ephemeralStorageSize?: cdk.Size;
    /**
     * The function execution time (in seconds) after which Lambda terminates
     * the function. Because the execution time affects cost, set this value
     * based on the function's expected execution time.
     *
     * @default Duration.minutes(15)
     */
    readonly timeout?: cdk.Duration;
    /**
    * VPC to launch the runners in.
    *
    * @default no VPC
    */
    readonly vpc?: ec2.IVpc;
    /**
    * Security Group to assign to this instance.
    *
    * @default public lambda with no security group
    */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
    * Where to place the network interfaces within the VPC.
    *
    * @default no subnet
    */
    readonly subnetSelection?: ec2.SubnetSelection;
}
/**
 * GitHub Actions runner provider using Lambda to execute the actions.
 *
 * Creates a Docker-based function that gets executed for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
export declare class LambdaRunner extends Construct implements IRunnerProvider {
    /**
     * Path to Dockerfile for Linux x64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     */
    static readonly LINUX_X64_DOCKERFILE_PATH: string;
    /**
     * Path to Dockerfile for Linux ARM64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     */
    static readonly LINUX_ARM64_DOCKERFILE_PATH: string;
    /**
     * The function hosting the GitHub runner.
     */
    readonly function: lambda.Function;
    /**
     * Label associated with this provider.
     */
    readonly label: string;
    /**
     * VPC used for hosting the function.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security group attached to the function.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Grant principal used to add permissions to the runner role.
     */
    readonly grantPrincipal: iam.IPrincipal;
    constructor(scope: Construct, id: string, props: LambdaRunnerProps);
    /**
     * The network connections associated with this resource.
     */
    get connections(): ec2.Connections;
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters: RunnerRuntimeParameters): stepfunctions.IChainable;
    private addImageUpdater;
}
