"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('Cloudfront Distribution Included', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, IP Filter', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Condition: {
                        IpAddress: {
                            'aws:SourceIp': [
                                '1.1.1.1',
                            ],
                        },
                    },
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Create From Hosted Zone', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
//# sourceMappingURL=data:application/json;base64,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