#!/usr/bin/env python3
###########################################################################
#                                                                         #
#  Copyright (C) 2017-2021 Rafal Kobel <rafalkobel@rafyco.pl>             #
#                                                                         #
#  This program is free software: you can redistribute it and/or modify   #
#  it under the terms of the GNU General Public License as published by   #
#  the Free Software Foundation, either version 3 of the License, or      #
#  (at your option) any later version.                                    #
#                                                                         #
#  This program is distributed in the hope that it will be useful,        #
#  but WITHOUT ANY WARRANTY; without even the implied warranty of         #
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           #
#  GNU General Public License for more details.                           #
#                                                                         #
#  You should have received a copy of the GNU General Public License      #
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.  #
#                                                                         #
###########################################################################
"""
Command line program to generation Podcast in files.

Program to generate podcast in files. It require mp3 files and json files.
That can be generated by ytrss program.

Example usage
=============

To invoke program type in your console::

    python -m ytrss.rssgenerate

for more option call program with flag C{--help}
"""

import logging
import io
import os
from typing import Iterator

from ytrss.configuration.configuration import Configuration
from ytrss.podcast.podcast import Podcast
from ytrss.core.downloaded_movie import DownloadedMovie
from ytrss.core.downloaded_movie import MovieFileError


def list_elements_in_dir(dirname: str, settings: Configuration) -> Iterator[DownloadedMovie]:
    """
    Return list of movie in directory.

    @param dirname: name of directory
    @type dirname: str
    @param settings: Settings handle
    @type settings: L{YTSettings<ytrss.core.settings.YTSettings>}
    @return: list of movie in directory
    @rtype: list
    """
    for filename in os.listdir(os.path.join(settings.output, dirname)):
        if filename.endswith(".json"):
            try:
                yield DownloadedMovie(settings, dirname, filename[0:len(filename) - 5])
            except MovieFileError:
                pass


def rss_generate(settings: Configuration) -> None:
    """
    Generate podcast file.

    @param settings: Settings handle
    @type settings: L{YTSettings<ytrss.core.settings.YTSettings>}
    """
    for dir_name in os.listdir(settings.output):
        if os.path.isdir(os.path.join(settings.output, dir_name)):
            print("Generate Podcast: {}".format(dir_name))
            podcast = Podcast(dir_name,
                              settings.get_podcast_information(dir_name))
            for movie in list_elements_in_dir(dir_name, settings):
                logging.debug("item: %s", movie.name)
                try:
                    podcast.add_item(movie=movie,
                                     dir_name=dir_name)
                    print("add item: %s" % movie.name)
                except ValueError:
                    print("Cannot add item")
            podcast_file = os.path.join(settings.output, dir_name, "podcast.xml")
            logging.debug("try to save: %s", podcast_file)
            file_handler = io.open(podcast_file, "w", encoding="utf-8")
            file_handler.write(podcast.generate())
            file_handler.close()
