"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        // const route_53_zone = aws_route53.HostedZone.fromLookup(this, 'MyZone', {
        //   domainName: props.endpoints.hostedZoneName,
        // });
        // const admin_cert = new aws_certificatemanager.Certificate(this, 'adminCert', {
        //   domainName: props.endpoints.adminDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const cluster_cert = new aws_certificatemanager.Certificate(this, 'clusterCert', {
        //   domainName: props.endpoints.clusterDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const telemetry_cert = new aws_certificatemanager.Certificate(this, 'telemetryCert', {
        //   domainName: props.endpoints.telemetryDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const manager_cert = new aws_certificatemanager.Certificate(this, 'managerCert', {
        //   domainName: props.endpoints.managerDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.clusterDns,
                            props.endpoints.managerDns,
                            props.endpoints.telemetryDns,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                // ingressController: {
                //   enabled: true,
                //   installCRDs: false,
                //   image: {
                //     repository: 'kong/kubernetes-ingress-controller',
                //     tag: '1.3.1-alpine',
                //   },
                // },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${props.clusterIssuerName}-secret/ca.crt`,
                    admin_ssl_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    admin_ssl_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': cluster_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': admin_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    // http: {
                    //   enabled: false,
                    // },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': manager_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    `${props.clusterIssuerName}-secret`,
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi9zcmMvcmVzb3VyY2VzL2Vrcy9oZWxtLWNoYXJ0cy9rb25nL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUF5RTtBQUN6RSwyQ0FBdUM7QUF1QnZDLE1BQWEsbUJBQW9CLFNBQVEsc0JBQVM7SUFDaEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQjtRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQztRQUNsQyxNQUFNLDBCQUEwQixHQUFHLDRCQUE0QixDQUFDO1FBRWhFLDRFQUE0RTtRQUM1RSxnREFBZ0Q7UUFDaEQsTUFBTTtRQUNOLGlGQUFpRjtRQUNqRiwwQ0FBMEM7UUFDMUMscUZBQXFGO1FBQ3JGLE1BQU07UUFDTixxRkFBcUY7UUFDckYsNENBQTRDO1FBQzVDLHFGQUFxRjtRQUNyRixNQUFNO1FBQ04seUZBQXlGO1FBQ3pGLDhDQUE4QztRQUM5QyxxRkFBcUY7UUFDckYsTUFBTTtRQUNOLHFGQUFxRjtRQUNyRiw0Q0FBNEM7UUFDNUMscUZBQXFGO1FBQ3JGLE1BQU07UUFFTixNQUFNLGNBQWMsR0FBRyxJQUFJLHFCQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtZQUMzRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLElBQUk7WUFDZixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLElBQUksRUFBRSxXQUFXO29CQUNqQixRQUFRLEVBQUUsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRTtpQkFDcEM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxxQkFBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzNFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixJQUFJLEVBQUUseUJBQXlCO1lBQy9CLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztTQUMzQixDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDekMsTUFBTSxXQUFXLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBRTNDLHVCQUF1QixDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQU8sQ0FBQyxlQUFlLENBQUM7WUFDdkUsU0FBUyxFQUFFO2dCQUNULDBCQUEwQixVQUFVLElBQUksV0FBVyxXQUFXLEtBQUssQ0FBQyxtQkFBbUIsU0FBUztnQkFDaEcsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxhQUFhLEVBQUU7YUFDckM7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsK0JBQStCO2dCQUMvQiwrQkFBK0I7YUFDaEM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHVCQUF1QixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFM0QsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLHFCQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN4RSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLElBQUk7WUFDZixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsVUFBVSxFQUFFLG9CQUFvQjtvQkFDaEMsSUFBSSxFQUFFLGFBQWE7b0JBQ25CLFFBQVEsRUFBRTt3QkFDUixJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7d0JBQ3RCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztxQkFDM0I7b0JBQ0QsSUFBSSxFQUFFO3dCQUNKLFVBQVUsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLGNBQWM7d0JBQzFDLFFBQVEsRUFBRTs0QkFDUixLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVE7NEJBQ3hCLEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVTs0QkFDMUIsS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVOzRCQUMxQixLQUFLLENBQUMsU0FBUyxDQUFDLFlBQVk7eUJBQzdCO3dCQUNELFFBQVEsRUFBRSxXQUFXO3dCQUNyQixTQUFTLEVBQUU7NEJBQ1QsS0FBSyxFQUFFLHdCQUF3Qjs0QkFDL0IsSUFBSSxFQUFFLHFCQUFxQjs0QkFDM0IsSUFBSSxFQUFFLEtBQUssQ0FBQyxpQkFBaUI7eUJBQzlCO3dCQUNELFdBQVcsRUFBRSxVQUFVO3dCQUN2QixVQUFVLEVBQUUsR0FBRyxLQUFLLENBQUMsaUJBQWlCLFNBQVM7d0JBQy9DLE1BQU0sRUFBRTs0QkFDTixhQUFhOzRCQUNiLGFBQWE7eUJBQ2Q7d0JBQ0QsVUFBVSxFQUFFOzRCQUNWLFNBQVMsRUFBRSxLQUFLOzRCQUNoQixJQUFJLEVBQUUsSUFBSTt5QkFDWDtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUVwRCx3RUFBd0U7UUFDeEUsZ0hBQWdIO1FBRWhILE1BQU0saUJBQWlCLEdBQUcsSUFBSSxxQkFBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDN0UsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFNBQVMsRUFBRSxJQUFJO1lBQ2YsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFVBQVUsRUFBRSwrQkFBK0I7b0JBQzNDLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLFFBQVEsRUFBRTt3QkFDUixJQUFJLEVBQUUsMEJBQTBCO3dCQUNoQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7cUJBQzNCO29CQUNELElBQUksRUFBRTt3QkFDSixRQUFRLEVBQUUsS0FBSzt3QkFDZixVQUFVLEVBQUU7NEJBQ1YsT0FBTyxFQUFFLGtCQUFrQixLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxhQUFhLHdOQUF3Tjt5QkFDblI7d0JBQ0QsYUFBYSxFQUFFOzRCQUNiO2dDQUNFLFVBQVUsRUFBRSxXQUFXO2dDQUN2QixJQUFJLEVBQUUsUUFBUTtnQ0FDZCxJQUFJLEVBQUU7b0NBQ0o7d0NBQ0UsVUFBVSxFQUFFLFlBQVk7d0NBQ3hCLEdBQUcsRUFBRSxZQUFZO3FDQUNsQjtvQ0FDRDt3Q0FDRSxVQUFVLEVBQUUsWUFBWTt3Q0FDeEIsR0FBRyxFQUFFLFlBQVk7cUNBQ2xCO29DQUNEO3dDQUNFLFVBQVUsRUFBRSxRQUFRO3dDQUNwQixHQUFHLEVBQUUsUUFBUTtxQ0FDZDtvQ0FDRDt3Q0FDRSxVQUFVLEVBQUUsUUFBUTt3Q0FDcEIsR0FBRyxFQUFFLFFBQVE7cUNBQ2Q7aUNBTUY7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsVUFBVSxFQUFFLCtCQUErQjtvQkFDM0MsSUFBSSxFQUFFLHFCQUFxQjtvQkFDM0IsUUFBUSxFQUFFO3dCQUNSLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CO3dCQUMvQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7cUJBQzNCO29CQUNELElBQUksRUFBRTt3QkFDSixRQUFRLEVBQUUsS0FBSzt3QkFDZixhQUFhLEVBQUU7NEJBQ2I7Z0NBQ0UsVUFBVSxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7Z0NBQ3JDLElBQUksRUFBRSxRQUFRO2dDQUNkLElBQUksRUFBRTtvQ0FDSjt3Q0FDRSxVQUFVLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjt3Q0FDckMsR0FBRyxFQUFFLFNBQVM7cUNBQ2Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsVUFBVSxFQUFFOzRCQUNWLE9BQU8sRUFBRSxrQkFBa0IsS0FBSyxDQUFDLG1CQUFtQixxQ0FBcUM7eUJBTzFGO3FCQUVGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXJELE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxxQkFBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLEdBQUcsS0FBSyxDQUFDLFdBQVc7WUFDcEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFVBQVUsRUFBRSwyQkFBMkI7WUFDdkMsS0FBSyxFQUFFLE1BQU07WUFDYixPQUFPLEVBQUUsTUFBTTtZQUNmLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUksRUFBRSxJQUFJO1lBQ1YsZUFBZSxFQUFFLElBQUk7WUFDckIsTUFBTSxFQUFFO2dCQUNOLHVCQUF1QjtnQkFDdkIsbUJBQW1CO2dCQUNuQix3QkFBd0I7Z0JBQ3hCLGFBQWE7Z0JBQ2Isd0RBQXdEO2dCQUN4RCwyQkFBMkI7Z0JBQzNCLE9BQU87Z0JBQ1AsS0FBSztnQkFDTCxXQUFXO2dCQUNYLHFDQUFxQztnQkFDckMsMkJBQTJCO2dCQUMzQixLQUFLO2dCQUNMLEdBQUcsRUFBRTtvQkFDSCxRQUFRLEVBQUUsVUFBVTtvQkFDcEIsSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLFlBQVksRUFBRSxLQUFLO29CQUNuQixZQUFZLEVBQUUsZ0JBQWdCLEtBQUssQ0FBQyxpQkFBaUIsaUJBQWlCO29CQUN0RSxnQkFBZ0IsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixpQkFBaUI7b0JBQzFFLGVBQWUsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixnQkFBZ0I7b0JBQ3hFLGNBQWMsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixpQkFBaUI7b0JBQ3hFLGtCQUFrQixFQUFFLGdCQUFnQixLQUFLLENBQUMsaUJBQWlCLGlCQUFpQjtvQkFDNUUsT0FBTyxFQUFFO3dCQUNQLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLEdBQUcsRUFBRSxZQUFZOzZCQUNsQjt5QkFDRjtxQkFDRjtvQkFDRCxXQUFXLEVBQUU7d0JBQ1gsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixJQUFJLEVBQUUsV0FBVztnQ0FDakIsR0FBRyxFQUFFLFlBQVk7NkJBQ2xCO3lCQUNGO3FCQUNGO29CQUNELFdBQVcsRUFBRTt3QkFDWCxTQUFTLEVBQUU7NEJBQ1QsWUFBWSxFQUFFO2dDQUNaLElBQUksRUFBRSxXQUFXO2dDQUNqQixHQUFHLEVBQUUsUUFBUTs2QkFDZDt5QkFDRjtxQkFDRjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixJQUFJLEVBQUUsV0FBVztnQ0FDakIsR0FBRyxFQUFFLFFBQVE7NkJBQ2Q7eUJBQ0Y7cUJBQ0Y7aUJBU0Y7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQLE9BQU8sRUFBRSxJQUFJO29CQUNiLElBQUksRUFBRSxjQUFjO29CQUNwQixHQUFHLEVBQUU7d0JBQ0gsT0FBTyxFQUFFLElBQUk7d0JBQ2IsV0FBVyxFQUFFLEdBQUc7cUJBQ2pCO29CQUNELFdBQVcsRUFBRTt3QkFDWCxzRUFBc0U7d0JBQ3RFLDhEQUE4RCxFQUFFLElBQUk7d0JBQ3BFLHdGQUF3Rjt3QkFDeEYsbURBQW1ELEVBQUUsVUFBVTt3QkFDL0QscURBQXFELEVBQUUsaUJBQWlCO3dCQUN4RSxtREFBbUQsRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxjQUFjO3dCQUMvRiwyQ0FBMkMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLFVBQVU7d0JBQ3ZFLGdGQUFnRixFQUFFLElBQUk7cUJBQ3ZGO2lCQUNGO2dCQUNELGdCQUFnQixFQUFFO29CQUNoQixPQUFPLEVBQUUsSUFBSTtvQkFDYixJQUFJLEVBQUUsY0FBYztvQkFDcEIsR0FBRyxFQUFFO3dCQUNILE9BQU8sRUFBRSxJQUFJO3dCQUNiLFdBQVcsRUFBRSxHQUFHO3FCQUNqQjtvQkFDRCxXQUFXLEVBQUU7d0JBQ1gsc0VBQXNFO3dCQUN0RSw4REFBOEQsRUFBRSxJQUFJO3dCQUNwRSwwRkFBMEY7d0JBQzFGLG1EQUFtRCxFQUFFLFVBQVU7d0JBQy9ELHFEQUFxRCxFQUFFLGlCQUFpQjt3QkFDeEUsMkNBQTJDLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxZQUFZO3dCQUN6RSxtREFBbUQsRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxnQkFBZ0I7d0JBQ2pHLGdGQUFnRixFQUFFLElBQUk7cUJBQ3ZGO2lCQUNGO2dCQUNELEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsS0FBSztpQkFDZjtnQkFDRCxLQUFLLEVBQUU7b0JBQ0wsT0FBTyxFQUFFLElBQUk7b0JBQ2IsSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLE1BQU0sRUFBRTt3QkFDTixnQkFBZ0IsRUFBRSxJQUFJO3FCQUN2QjtvQkFDRCxHQUFHLEVBQUU7d0JBQ0gsV0FBVyxFQUFFLEdBQUc7cUJBRWpCO29CQUNELFdBQVcsRUFBRTt3QkFDWCxzRUFBc0U7d0JBQ3RFLDhEQUE4RCxFQUFFLElBQUk7d0JBQ3BFLHNGQUFzRjt3QkFDdEYsbURBQW1ELEVBQUUsVUFBVTt3QkFDL0QscURBQXFELEVBQUUsVUFBVTt3QkFDakUsdUVBQXVFLEVBQUUsWUFBWTt3QkFDckYsMkNBQTJDLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRO3dCQUNyRSxtREFBbUQsRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxZQUFZO3dCQUM3RixnRkFBZ0YsRUFBRSxJQUFJO3FCQUN2RjtpQkFDRjtnQkFDRCxVQUFVLEVBQ1Y7b0JBQ0UsY0FBYyxFQUFFO3dCQUNkLE1BQU0sRUFBRSxLQUFLO3dCQUNiLElBQUksRUFBRSx1QkFBdUIsQ0FBQyxrQkFBa0I7cUJBQ2pEO29CQUNELGtCQUFrQixFQUNsQjt3QkFDRTs0QkFDRSxJQUFJLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjs0QkFDL0IsR0FBRyxFQUFFO2dDQUNILE1BQU0sRUFBRSwwQkFBMEI7Z0NBQ2xDLFFBQVEsRUFBRSxJQUFJO2dDQUNkLGdCQUFnQixFQUFFO29DQUNoQixtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2lDQUMvQzs2QkFDRjt5QkFDRjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsV0FBVzs0QkFDakIsR0FBRyxFQUFFO2dDQUNILE1BQU0sRUFBRSwwQkFBMEI7Z0NBQ2xDLFFBQVEsRUFBRSxJQUFJO2dDQUNkLGdCQUFnQixFQUFFO29DQUNoQixtQkFBbUIsRUFBRSwwQkFBMEI7aUNBQ2hEOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELHVCQUF1QixFQUFFO3dCQUN2Qjs0QkFDRSxJQUFJLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjs0QkFDL0IsU0FBUyxFQUFFLHNCQUFzQjs0QkFDakMsUUFBUSxFQUFFLElBQUk7eUJBQ2Y7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLFdBQVc7NEJBQ2pCLFNBQVMsRUFBRSxrQkFBa0I7NEJBQzdCLFFBQVEsRUFBRSxJQUFJO3lCQUNmO3FCQUNGO2lCQUNGO2dCQUNELFVBQVUsRUFBRTtvQkFDVixPQUFPLEVBQUUsSUFBSTtvQkFDYixjQUFjLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtpQkFDMUM7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLE9BQU8sRUFBRSxJQUFJO2lCQUNkO2dCQUNELE9BQU8sRUFBRTtvQkFDUCxPQUFPLEVBQUUsSUFBSTtvQkFDYixJQUFJLEVBQUUsY0FBYztvQkFDcEIsR0FBRyxFQUFFO3dCQUNILE9BQU8sRUFBRSxJQUFJO3dCQUNiLFdBQVcsRUFBRSxHQUFHO3FCQUNqQjtvQkFDRCxVQUFVO29CQUNWLG9CQUFvQjtvQkFDcEIsS0FBSztvQkFDTCxXQUFXLEVBQUU7d0JBQ1gsOERBQThELEVBQUUsSUFBSTt3QkFDcEUsc0VBQXNFO3dCQUN0RSx3RkFBd0Y7d0JBQ3hGLG1EQUFtRCxFQUFFLFVBQVU7d0JBQy9ELHFEQUFxRCxFQUFFLFVBQVU7d0JBQ2pFLG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLGNBQWM7d0JBQy9GLDJDQUEyQyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVTt3QkFDdkUsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLE9BQU8sRUFBRSxLQUFLO2lCQUNmO2dCQUNELGFBQWEsRUFBRTtvQkFDYixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsU0FBUztpQkFDcEM7YUFDRjtTQUVGLENBQUMsQ0FBQztRQUVILG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsaUJBQWlCLEVBQUUsY0FBYyxFQUFFLHVCQUF1QixDQUFDLENBQUM7SUFDckcsQ0FBQztDQUNGO0FBdlpELGtEQXVaQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IER1cmF0aW9uLCBhd3NfaWFtLCBhd3NfZWtzLCBhd3NfcmRzLCBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgRG5zUHJvcHMgfSBmcm9tICcuLi8uLi9pbnRlcmZhY2VzL2Rucy1wcm9wcyc7XG4vLyBpbXBvcnQgeyBLb25nQ29udHJvbFBsYW5lIH0gZnJvbSAnLi4vLi4vLi4vS29uZ0NvbnRyb2xQbGFuZSc7XG5cbi8vIGV4cG9ydCBpbnRlcmZhY2UgRG5zUHJvcHMge1xuLy8gICByZWFkb25seSBhZG1pbkRucyA6IHN0cmluZztcbi8vICAgcmVhZG9ubHkgY2x1c3RlckRucyA6IHN0cmluZztcbi8vICAgcmVhZG9ubHkgdGVsZW1ldHJ5RG5zIDogc3RyaW5nO1xuLy8gICByZWFkb25seSBtYW5hZ2VyRG5zIDogc3RyaW5nO1xuLy8gfVxuZXhwb3J0IGludGVyZmFjZSBLb25nQ1BQcm9wcyB7XG4gIHJlYWRvbmx5IGNsdXN0ZXIgOiBhd3NfZWtzLkNsdXN0ZXI7XG4gIHJlYWRvbmx5IHJkczogYXdzX3Jkcy5EYXRhYmFzZUluc3RhbmNlO1xuICByZWFkb25seSBuYW1lc3BhY2U6IHN0cmluZztcbiAgLy8gcmVhZG9ubHkgbm9kZWdyb3VwIDogYXdzX2Vrcy5Ob2RlZ3JvdXA7XG4gIHJlYWRvbmx5IGxpY2Vuc2Vfc2VjcmV0X25hbWUgOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNhY2VydG5hbWUgOiBTdHJpbmc7XG4gIHJlYWRvbmx5IGVuZHBvaW50czogRG5zUHJvcHM7XG4gIC8vIHJlYWRvbmx5IGhvc3RlZFpvbmVOYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNsdXN0ZXJJc3N1ZXJOYW1lOiBTdHJpbmc7XG4gIHJlYWRvbmx5IEhlbG1PcHRpb25zPzogYXdzX2Vrcy5IZWxtQ2hhcnRPcHRpb25zO1xufVxuXG5leHBvcnQgY2xhc3MgS29uZ0NvbnRyb2xQbGFuZUVLUyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBLb25nQ1BQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCByZHNfc2VjcmV0cyA9ICdyZHMtc2VjcmV0cyc7XG4gICAgY29uc3QgcmRzX3NlY3JldHNfcHJvdmlkZXJfY2xhc3MgPSAncmRzLXNlY3JldHMtcHJvdmlkZXItY2xhc3MnO1xuXG4gICAgLy8gY29uc3Qgcm91dGVfNTNfem9uZSA9IGF3c19yb3V0ZTUzLkhvc3RlZFpvbmUuZnJvbUxvb2t1cCh0aGlzLCAnTXlab25lJywge1xuICAgIC8vICAgZG9tYWluTmFtZTogcHJvcHMuZW5kcG9pbnRzLmhvc3RlZFpvbmVOYW1lLFxuICAgIC8vIH0pO1xuICAgIC8vIGNvbnN0IGFkbWluX2NlcnQgPSBuZXcgYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZSh0aGlzLCAnYWRtaW5DZXJ0Jywge1xuICAgIC8vICAgZG9tYWluTmFtZTogcHJvcHMuZW5kcG9pbnRzLmFkbWluRG5zLFxuICAgIC8vICAgdmFsaWRhdGlvbjogYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZVZhbGlkYXRpb24uZnJvbURucyhyb3V0ZV81M196b25lKSxcbiAgICAvLyB9KTtcbiAgICAvLyBjb25zdCBjbHVzdGVyX2NlcnQgPSBuZXcgYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZSh0aGlzLCAnY2x1c3RlckNlcnQnLCB7XG4gICAgLy8gICBkb21haW5OYW1lOiBwcm9wcy5lbmRwb2ludHMuY2x1c3RlckRucyxcbiAgICAvLyAgIHZhbGlkYXRpb246IGF3c19jZXJ0aWZpY2F0ZW1hbmFnZXIuQ2VydGlmaWNhdGVWYWxpZGF0aW9uLmZyb21EbnMocm91dGVfNTNfem9uZSksXG4gICAgLy8gfSk7XG4gICAgLy8gY29uc3QgdGVsZW1ldHJ5X2NlcnQgPSBuZXcgYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZSh0aGlzLCAndGVsZW1ldHJ5Q2VydCcsIHtcbiAgICAvLyAgIGRvbWFpbk5hbWU6IHByb3BzLmVuZHBvaW50cy50ZWxlbWV0cnlEbnMsXG4gICAgLy8gICB2YWxpZGF0aW9uOiBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRG5zKHJvdXRlXzUzX3pvbmUpLFxuICAgIC8vIH0pO1xuICAgIC8vIGNvbnN0IG1hbmFnZXJfY2VydCA9IG5ldyBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlKHRoaXMsICdtYW5hZ2VyQ2VydCcsIHtcbiAgICAvLyAgIGRvbWFpbk5hbWU6IHByb3BzLmVuZHBvaW50cy5tYW5hZ2VyRG5zLFxuICAgIC8vICAgdmFsaWRhdGlvbjogYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZVZhbGlkYXRpb24uZnJvbURucyhyb3V0ZV81M196b25lKSxcbiAgICAvLyB9KTtcblxuICAgIGNvbnN0IGtvbmdfbmFtZXNwYWNlID0gbmV3IGF3c19la3MuS3ViZXJuZXRlc01hbmlmZXN0KHRoaXMsICdLb25nTmFtZXNwYWNlJywge1xuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIG92ZXJ3cml0ZTogdHJ1ZSxcbiAgICAgIG1hbmlmZXN0OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAndjEnLFxuICAgICAgICAgIGtpbmQ6ICdOYW1lc3BhY2UnLFxuICAgICAgICAgIG1ldGFkYXRhOiB7IG5hbWU6IHByb3BzLm5hbWVzcGFjZSB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGtvbmdfY3Bfc2VydmljZV9hY2NvdW50ID0gbmV3IGF3c19la3MuU2VydmljZUFjY291bnQodGhpcywgJ0tvbmdDcFNBJywge1xuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIG5hbWU6ICdrb25nLWNwLXNlcnZpY2UtYWNjb3VudCcsXG4gICAgICBuYW1lc3BhY2U6IHByb3BzLm5hbWVzcGFjZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGF3c19yZWdpb24gPSBTdGFjay5vZih0aGlzKS5yZWdpb247XG4gICAgY29uc3QgYXdzX2FjY291bnQgPSBTdGFjay5vZih0aGlzKS5hY2NvdW50O1xuXG4gICAga29uZ19jcF9zZXJ2aWNlX2FjY291bnQuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGF3c19pYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICBgYXJuOmF3czpzZWNyZXRzbWFuYWdlcjoke2F3c19yZWdpb259OiR7YXdzX2FjY291bnR9OnNlY3JldDoke3Byb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWV9LT8/Pz8/P2AsXG4gICAgICAgIGAke3Byb3BzLnJkcy5zZWNyZXQ/LnNlY3JldEZ1bGxBcm59YCxcbiAgICAgIF0sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIGtvbmdfY3Bfc2VydmljZV9hY2NvdW50Lm5vZGUuYWRkRGVwZW5kZW5jeShrb25nX25hbWVzcGFjZSk7XG5cbiAgICBjb25zdCBrb25nX2NlcnRpZmljYXRlID0gbmV3IGF3c19la3MuS3ViZXJuZXRlc01hbmlmZXN0KHRoaXMsICdLb25nQ2VydCcsIHtcbiAgICAgIGNsdXN0ZXI6IHByb3BzLmNsdXN0ZXIsXG4gICAgICBvdmVyd3JpdGU6IHRydWUsXG4gICAgICBtYW5pZmVzdDogW1xuICAgICAgICB7XG4gICAgICAgICAgYXBpVmVyc2lvbjogJ2NlcnQtbWFuYWdlci5pby92MScsXG4gICAgICAgICAga2luZDogJ0NlcnRpZmljYXRlJyxcbiAgICAgICAgICBtZXRhZGF0YToge1xuICAgICAgICAgICAgbmFtZTogcHJvcHMuY2FjZXJ0bmFtZSxcbiAgICAgICAgICAgIG5hbWVzcGFjZTogcHJvcHMubmFtZXNwYWNlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3BlYzoge1xuICAgICAgICAgICAgY29tbW9uTmFtZTogcHJvcHMuZW5kcG9pbnRzLmhvc3RlZFpvbmVOYW1lLFxuICAgICAgICAgICAgZG5zTmFtZXM6IFtcbiAgICAgICAgICAgICAgcHJvcHMuZW5kcG9pbnRzLmFkbWluRG5zLFxuICAgICAgICAgICAgICBwcm9wcy5lbmRwb2ludHMuY2x1c3RlckRucyxcbiAgICAgICAgICAgICAgcHJvcHMuZW5kcG9pbnRzLm1hbmFnZXJEbnMsXG4gICAgICAgICAgICAgIHByb3BzLmVuZHBvaW50cy50ZWxlbWV0cnlEbnMsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgZHVyYXRpb246ICcyMTYwaDBtMHMnLFxuICAgICAgICAgICAgaXNzdWVyUmVmOiB7XG4gICAgICAgICAgICAgIGdyb3VwOiAnYXdzcGNhLmNlcnQtbWFuYWdlci5pbycsXG4gICAgICAgICAgICAgIGtpbmQ6ICdBV1NQQ0FDbHVzdGVySXNzdWVyJyxcbiAgICAgICAgICAgICAgbmFtZTogcHJvcHMuY2x1c3Rlcklzc3Vlck5hbWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgcmVuZXdCZWZvcmU6ICczNjBoMG0wcycsXG4gICAgICAgICAgICBzZWNyZXROYW1lOiBgJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0YCxcbiAgICAgICAgICAgIHVzYWdlczogW1xuICAgICAgICAgICAgICAnc2VydmVyIGF1dGgnLFxuICAgICAgICAgICAgICAnY2xpZW50IGF1dGgnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHByaXZhdGVLZXk6IHtcbiAgICAgICAgICAgICAgYWxnb3JpdGhtOiAnUlNBJyxcbiAgICAgICAgICAgICAgc2l6ZTogMjA0OCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBrb25nX2NlcnRpZmljYXRlLm5vZGUuYWRkRGVwZW5kZW5jeShrb25nX25hbWVzcGFjZSk7XG5cbiAgICAvLyBjb25zdCBzZWNyZXROYW1lUGFydHMgPSBGbi5zcGxpdCgnLScsIHByb3BzLnJkcy5zZWNyZXQ/LnNlY3JldE5hbWUhKTtcbiAgICAvLyBjb25zdCBzZWNyZXROYW1lV2l0aG91dFN1ZmZpeCA9IEZuLmpvaW4oJy0nLCBbRm4uc2VsZWN0KDAsIHNlY3JldE5hbWVQYXJ0cyksIEZuLnNlbGVjdCgxLCBzZWNyZXROYW1lUGFydHMpXSk7XG5cbiAgICBjb25zdCBzZWNyZXRzX2NyZF9tb3VudCA9IG5ldyBhd3NfZWtzLkt1YmVybmV0ZXNNYW5pZmVzdCh0aGlzLCAnU2VjcmV0c01vdW50Jywge1xuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIG92ZXJ3cml0ZTogdHJ1ZSxcbiAgICAgIG1hbmlmZXN0OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAnc2VjcmV0cy1zdG9yZS5jc2kueC1rOHMuaW8vdjEnLFxuICAgICAgICAgIGtpbmQ6ICdTZWNyZXRQcm92aWRlckNsYXNzJyxcbiAgICAgICAgICBtZXRhZGF0YToge1xuICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHNfcHJvdmlkZXJfY2xhc3MsXG4gICAgICAgICAgICBuYW1lc3BhY2U6IHByb3BzLm5hbWVzcGFjZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHNwZWM6IHtcbiAgICAgICAgICAgIHByb3ZpZGVyOiAnYXdzJyxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgb2JqZWN0czogYC0gb2JqZWN0TmFtZTogXCIke3Byb3BzLnJkcy5zZWNyZXQ/LnNlY3JldEZ1bGxBcm59XCJcXG4gIGptZXNQYXRoOlxcbiAgICAtIHBhdGg6IHVzZXJuYW1lXFxuICAgICAgb2JqZWN0QWxpYXM6IGRidXNlcm5hbWVcXG4gICAgLSBwYXRoOiBwYXNzd29yZFxcbiAgICAgIG9iamVjdEFsaWFzOiBkYnBhc3N3b3JkXFxuICAgIC0gcGF0aDogZGJuYW1lXFxuICAgICAgb2JqZWN0QWxpYXM6IGRibmFtZVxcbiAgICAtIHBhdGg6IGhvc3RcXG4gICAgICBvYmplY3RBbGlhczogZGJob3N0XFxuYCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBzZWNyZXRPYmplY3RzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBzZWNyZXROYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgICB0eXBlOiAnT3BhcXVlJyxcbiAgICAgICAgICAgICAgICBkYXRhOiBbXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6ICdkYnVzZXJuYW1lJyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJ1c2VybmFtZScsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBvYmplY3ROYW1lOiAnZGJwYXNzd29yZCcsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2RicGFzc3dvcmQnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgb2JqZWN0TmFtZTogJ2RibmFtZScsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2RibmFtZScsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBvYmplY3ROYW1lOiAnZGJob3N0JyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJob3N0JyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAvLyBLb25nIGRvZXMgbm90IHN1cHBvcnRzIHBvcnQgb2JmdXNjYXRpb25cbiAgICAgICAgICAgICAgICAgIC8vIHtcbiAgICAgICAgICAgICAgICAgIC8vICAgb2JqZWN0TmFtZTogJ2RicG9ydCcsXG4gICAgICAgICAgICAgICAgICAvLyAgIGtleTogJ2RicG9ydCcsXG4gICAgICAgICAgICAgICAgICAvLyB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAnc2VjcmV0cy1zdG9yZS5jc2kueC1rOHMuaW8vdjEnLFxuICAgICAgICAgIGtpbmQ6ICdTZWNyZXRQcm92aWRlckNsYXNzJyxcbiAgICAgICAgICBtZXRhZGF0YToge1xuICAgICAgICAgICAgbmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgIG5hbWVzcGFjZTogcHJvcHMubmFtZXNwYWNlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3BlYzoge1xuICAgICAgICAgICAgcHJvdmlkZXI6ICdhd3MnLFxuICAgICAgICAgICAgc2VjcmV0T2JqZWN0czogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgc2VjcmV0TmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgICB0eXBlOiAnb3BhcXVlJyxcbiAgICAgICAgICAgICAgICBkYXRhOiBbXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2xpY2Vuc2UnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgb2JqZWN0czogYC0gb2JqZWN0TmFtZTogXCIke3Byb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWV9XCJcXG4gIG9iamVjdFR5cGU6IFwic2VjcmV0c21hbmFnZXJcIlxcbmAsXG4gICAgICAgICAgICAgIC8vIFtcbiAgICAgICAgICAgICAgLy8gICB7XG4gICAgICAgICAgICAgIC8vICAgICBvYmplY3ROYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICAvLyAgICAgb2JqZWN0VHlwZTogJ3NlY3JldHNtYW5hZ2VyJyxcbiAgICAgICAgICAgICAgLy8gICB9LFxuICAgICAgICAgICAgICAvLyBdLFxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgc2VjcmV0c19jcmRfbW91bnQubm9kZS5hZGREZXBlbmRlbmN5KGtvbmdfbmFtZXNwYWNlKTtcblxuICAgIGNvbnN0IGRlcGxveV9rb25nX2NwX2hlbG0gPSBuZXcgYXdzX2Vrcy5IZWxtQ2hhcnQodGhpcywgJ0tvbmdDcEhlbG0nLCB7XG4gICAgICAuLi5wcm9wcy5IZWxtT3B0aW9ucyxcbiAgICAgIGNsdXN0ZXI6IHByb3BzLmNsdXN0ZXIsXG4gICAgICByZXBvc2l0b3J5OiAnaHR0cHM6Ly9jaGFydHMua29uZ2hxLmNvbScsXG4gICAgICBjaGFydDogJ2tvbmcnLFxuICAgICAgcmVsZWFzZTogJ2tvbmcnLFxuICAgICAgbmFtZXNwYWNlOiBwcm9wcy5uYW1lc3BhY2UsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHdhaXQ6IHRydWUsXG4gICAgICBjcmVhdGVOYW1lc3BhY2U6IHRydWUsXG4gICAgICB2YWx1ZXM6IHtcbiAgICAgICAgLy8gaW5ncmVzc0NvbnRyb2xsZXI6IHtcbiAgICAgICAgLy8gICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAvLyAgIGluc3RhbGxDUkRzOiBmYWxzZSxcbiAgICAgICAgLy8gICBpbWFnZToge1xuICAgICAgICAvLyAgICAgcmVwb3NpdG9yeTogJ2tvbmcva3ViZXJuZXRlcy1pbmdyZXNzLWNvbnRyb2xsZXInLFxuICAgICAgICAvLyAgICAgdGFnOiAnMS4zLjEtYWxwaW5lJyxcbiAgICAgICAgLy8gICB9LFxuICAgICAgICAvLyB9LFxuICAgICAgICAvLyBpbWFnZToge1xuICAgICAgICAvLyAgIHJlcG9zaXRvcnk6ICdrb25nL2tvbmctZ2F0ZXdheScsXG4gICAgICAgIC8vICAgdGFnOiAnMi41LjAuMC1hbHBpbmUnLFxuICAgICAgICAvLyB9LFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBkYXRhYmFzZTogJ3Bvc3RncmVzJyxcbiAgICAgICAgICByb2xlOiAnY29udHJvbF9wbGFuZScsXG4gICAgICAgICAgY2x1c3Rlcl9tdGxzOiAncGtpJyxcbiAgICAgICAgICBjbHVzdGVyX2NlcnQ6IGAvZXRjL3NlY3JldHMvJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0L3Rscy5jcnRgLFxuICAgICAgICAgIGNsdXN0ZXJfY2VydF9rZXk6IGAvZXRjL3NlY3JldHMvJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0L3Rscy5rZXlgLFxuICAgICAgICAgIGNsdXN0ZXJfY2FfY2VydDogYC9ldGMvc2VjcmV0cy8ke3Byb3BzLmNsdXN0ZXJJc3N1ZXJOYW1lfS1zZWNyZXQvY2EuY3J0YCxcbiAgICAgICAgICBhZG1pbl9zc2xfY2VydDogYC9ldGMvc2VjcmV0cy8ke3Byb3BzLmNsdXN0ZXJJc3N1ZXJOYW1lfS1zZWNyZXQvdGxzLmNydGAsXG4gICAgICAgICAgYWRtaW5fc3NsX2NlcnRfa2V5OiBgL2V0Yy9zZWNyZXRzLyR7cHJvcHMuY2x1c3Rlcklzc3Vlck5hbWV9LXNlY3JldC90bHMua2V5YCxcbiAgICAgICAgICBwZ191c2VyOiB7XG4gICAgICAgICAgICB2YWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgc2VjcmV0S2V5UmVmOiB7XG4gICAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgICAga2V5OiAnZGJ1c2VybmFtZScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgcGdfcGFzc3dvcmQ6IHtcbiAgICAgICAgICAgIHZhbHVlRnJvbToge1xuICAgICAgICAgICAgICBzZWNyZXRLZXlSZWY6IHtcbiAgICAgICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgICBrZXk6ICdkYnBhc3N3b3JkJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwZ19kYXRhYmFzZToge1xuICAgICAgICAgICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIGtleTogJ2RibmFtZScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgcGdfaG9zdDoge1xuICAgICAgICAgICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIGtleTogJ2RiaG9zdCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgLy8gcGdfcG9ydDoge1xuICAgICAgICAgIC8vICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgLy8gICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgIC8vICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgIC8vICAgICAgIGtleTogJ2RicG9ydCcsXG4gICAgICAgICAgLy8gICAgIH0sXG4gICAgICAgICAgLy8gICB9LFxuICAgICAgICAgIC8vIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGNsdXN0ZXI6IHtcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgIHR5cGU6ICdMb2FkQmFsYW5jZXInLFxuICAgICAgICAgIHRsczoge1xuICAgICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgIHNlcnZpY2VQb3J0OiA0NDMsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXByb3h5LXByb3RvY29sJzogJyonLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5sYi10YXJnZXQtdHlwZSc6ICdpcCcsXG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc3NsLWNlcnQnOiBjbHVzdGVyX2NlcnQuY2VydGlmaWNhdGVBcm4sXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItdHlwZSc6ICdleHRlcm5hbCcsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc2NoZW1lJzogJ2ludGVybmV0LWZhY2luZycsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmFtZSc6IGAke3Byb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWV9LWVrcy1jbHVzdGVyYCxcbiAgICAgICAgICAgICdleHRlcm5hbC1kbnMuYWxwaGEua3ViZXJuZXRlcy5pby9ob3N0bmFtZSc6IHByb3BzLmVuZHBvaW50cy5jbHVzdGVyRG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLWNyb3NzLXpvbmUtbG9hZC1iYWxhbmNpbmctZW5hYmxlZCc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgY2x1c3RlcnRlbGVtZXRyeToge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgc2VydmljZVBvcnQ6IDQ0MyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGFubm90YXRpb25zOiB7XG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItcHJveHktcHJvdG9jb2wnOiAnKicsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmxiLXRhcmdldC10eXBlJzogJ2lwJyxcbiAgICAgICAgICAgIC8vICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1zc2wtY2VydCc6IHRlbGVtZXRyeV9jZXJ0LmNlcnRpZmljYXRlQXJuLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXR5cGUnOiAnZXh0ZXJuYWwnLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNjaGVtZSc6ICdpbnRlcm5ldC1mYWNpbmcnLFxuICAgICAgICAgICAgJ2V4dGVybmFsLWRucy5hbHBoYS5rdWJlcm5ldGVzLmlvL2hvc3RuYW1lJzogcHJvcHMuZW5kcG9pbnRzLnRlbGVtZXRyeURucyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1uYW1lJzogYCR7cHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZX0tZWtzLXRlbGVtZXRyeWAsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItY3Jvc3Mtem9uZS1sb2FkLWJhbGFuY2luZy1lbmFibGVkJzogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBwcm94eToge1xuICAgICAgICAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICB9LFxuICAgICAgICBhZG1pbjoge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgbGFiZWxzOiB7XG4gICAgICAgICAgICAnZW5hYmxlLW1ldHJpY3MnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBzZXJ2aWNlUG9ydDogNDQzLFxuICAgICAgICAgICAgLy8gb3ZlcnJpZGVTZXJ2aWNlVGFyZ2V0UG9ydDogNDQzLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYW5ub3RhdGlvbnM6IHtcbiAgICAgICAgICAgIC8vICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1wcm94eS1wcm90b2NvbCc6ICcqJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1ubGItdGFyZ2V0LXR5cGUnOiAnaXAnLFxuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNzbC1jZXJ0JzogYWRtaW5fY2VydC5jZXJ0aWZpY2F0ZUFybixcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci10eXBlJzogJ2V4dGVybmFsJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1zY2hlbWUnOiAnaW50ZXJuYWwnLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLWFkZGl0aW9uYWwtcmVzb3VyY2UtdGFncyc6ICdUeXBlPWFkbWluJyxcbiAgICAgICAgICAgICdleHRlcm5hbC1kbnMuYWxwaGEua3ViZXJuZXRlcy5pby9ob3N0bmFtZSc6IHByb3BzLmVuZHBvaW50cy5hZG1pbkRucyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1uYW1lJzogYCR7cHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZX0tZWtzLWFkbWluYCxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1jcm9zcy16b25lLWxvYWQtYmFsYW5jaW5nLWVuYWJsZWQnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGRlcGxveW1lbnQ6XG4gICAgICAgIHtcbiAgICAgICAgICBzZXJ2aWNlQWNjb3VudDoge1xuICAgICAgICAgICAgY3JlYXRlOiBmYWxzZSxcbiAgICAgICAgICAgIG5hbWU6IGtvbmdfY3Bfc2VydmljZV9hY2NvdW50LnNlcnZpY2VBY2NvdW50TmFtZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHVzZXJEZWZpbmVkVm9sdW1lczpcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgIGNzaToge1xuICAgICAgICAgICAgICAgIGRyaXZlcjogJ3NlY3JldHMtc3RvcmUuY3NpLms4cy5pbycsXG4gICAgICAgICAgICAgICAgcmVhZE9ubHk6IHRydWUsXG4gICAgICAgICAgICAgICAgdm9sdW1lQXR0cmlidXRlczoge1xuICAgICAgICAgICAgICAgICAgc2VjcmV0UHJvdmlkZXJDbGFzczogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgIGNzaToge1xuICAgICAgICAgICAgICAgIGRyaXZlcjogJ3NlY3JldHMtc3RvcmUuY3NpLms4cy5pbycsXG4gICAgICAgICAgICAgICAgcmVhZE9ubHk6IHRydWUsXG4gICAgICAgICAgICAgICAgdm9sdW1lQXR0cmlidXRlczoge1xuICAgICAgICAgICAgICAgICAgc2VjcmV0UHJvdmlkZXJDbGFzczogcmRzX3NlY3JldHNfcHJvdmlkZXJfY2xhc3MsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICB1c2VyRGVmaW5lZFZvbHVtZU1vdW50czogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBuYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICBtb3VudFBhdGg6ICcvbW50L2xpY2Vuc2Vfc2VjcmV0cycsXG4gICAgICAgICAgICAgIHJlYWRPbmx5OiB0cnVlLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgIG1vdW50UGF0aDogJy9tbnQvcmRzX3NlY3JldHMnLFxuICAgICAgICAgICAgICByZWFkT25seTogdHJ1ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgZW50ZXJwcmlzZToge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgbGljZW5zZV9zZWNyZXQ6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgIH0sXG4gICAgICAgIHNlcnZpY2VNb25pdG9yOiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgbWFuYWdlcjoge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgc2VydmljZVBvcnQ6IDQ0MyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIC8vIGh0dHA6IHtcbiAgICAgICAgICAvLyAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICAgIC8vIH0sXG4gICAgICAgICAgYW5ub3RhdGlvbnM6IHtcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1ubGItdGFyZ2V0LXR5cGUnOiAnaXAnLFxuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXByb3h5LXByb3RvY29sJzogJyonLFxuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNzbC1jZXJ0JzogbWFuYWdlcl9jZXJ0LmNlcnRpZmljYXRlQXJuLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXR5cGUnOiAnZXh0ZXJuYWwnLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNjaGVtZSc6ICdpbnRlcm5hbCcsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmFtZSc6IGAke3Byb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWV9LWVrcy1tYW5hZ2VyYCxcbiAgICAgICAgICAgICdleHRlcm5hbC1kbnMuYWxwaGEua3ViZXJuZXRlcy5pby9ob3N0bmFtZSc6IHByb3BzLmVuZHBvaW50cy5tYW5hZ2VyRG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLWNyb3NzLXpvbmUtbG9hZC1iYWxhbmNpbmctZW5hYmxlZCc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgcG9zdGdyZXNxbDoge1xuICAgICAgICAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICB9LFxuICAgICAgICBzZWNyZXRWb2x1bWVzOiBbXG4gICAgICAgICAgYCR7cHJvcHMuY2x1c3Rlcklzc3Vlck5hbWV9LXNlY3JldGAsXG4gICAgICAgIF0sXG4gICAgICB9LFxuXG4gICAgfSk7XG5cbiAgICBkZXBsb3lfa29uZ19jcF9oZWxtLm5vZGUuYWRkRGVwZW5kZW5jeShzZWNyZXRzX2NyZF9tb3VudCwga29uZ19uYW1lc3BhY2UsIGtvbmdfY3Bfc2VydmljZV9hY2NvdW50KTtcbiAgfVxufSJdfQ==