"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongAdminService = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const aws_servicediscovery_1 = require("aws-cdk-lib/aws-servicediscovery");
class KongAdminService extends aws_ecs_1.FargateService {
    constructor(scope, id, props) {
        super(scope, id, props.kongServiceProps);
        this.zone = aws_cdk_lib_1.aws_route53.HostedZone.fromLookup(this, 'HZ', {
            domainName: props.hostedZoneName,
        });
        // Initialize the port variables or user defined input
        this.adminPort = props.kongFeaturesProps.adminProps.httpPort || 8001;
        this.adminHttpsPort = props.kongFeaturesProps.adminProps.httpsPort || 8444;
        this.managerPort = props.kongFeaturesProps.kongManagerProps.httpPort || 8002;
        this.managerHttpsPort = props.kongFeaturesProps.kongManagerProps.httpsPort || 8445;
        this.telemetryPort = props.kongFeaturesProps.clusterTelemetryProps.port || 8006;
        this.clusterPort = props.kongFeaturesProps.clusterProps.port || 8005;
        this.devPortalPort = props.kongFeaturesProps.devPortalProps.port || 8003;
        // Initialize the endpoints
        this.controlPlaneDnsEp = this.cluster.clusterName;
        this.adminEndpoint = 'admin.' + props.hostedZoneName;
        this.managerEndpoint = 'manager.' + props.hostedZoneName;
        this.devPortalEndpoint = 'devportal.' + props.hostedZoneName;
        this.clusterEndpoint = 'cluster.' + props.hostedZoneName;
        this.telemetryEndpoint = 'telemetry.' + props.hostedZoneName;
        const cloudmap = this.enableCloudMap({
            name: 'kongControlPlaneCloudMapService',
            cloudMapNamespace: new aws_servicediscovery_1.PrivateDnsNamespace(this, 'KongCPNamespace', {
                name: this.controlPlaneDnsEp,
                vpc: props.kongServiceProps.cluster.vpc,
            }),
            dnsRecordType: aws_servicediscovery_1.DnsRecordType.A,
        });
        //Use Cloudmap to expose the https service instead of adding a NLB and map it to a DNS record.
        if (props.kongFeaturesProps.clusterProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongAdminCNameRecord', {
                domainName: this.cloudMapService?.serviceName + '.' + this.controlPlaneDnsEp || '',
                zone: this.zone,
                recordName: this.clusterEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.clusterPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongClusterEndpoint', {
                value: this.clusterEndpoint + ':' + this.clusterPort,
                exportName: 'KongClusterEndpoint',
            });
        }
        if (props.kongFeaturesProps.clusterTelemetryProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongTelemetryCNameRecord', {
                domainName: this.cloudMapService?.serviceName + '.' + this.controlPlaneDnsEp || '',
                zone: this.zone,
                recordName: this.telemetryEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.telemetryPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongTelemetryEndpoint', {
                value: this.telemetryEndpoint + ':' + this.telemetryPort,
                exportName: 'KongTelemetryEndpoint',
            });
        }
        this.registerKongFeatureTargets(443, this.adminHttpsPort, props.adminNlb, 'KongAdminHttps', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.adminEndpoint);
        if (props.kongFeaturesProps.adminProps.enableHttp) {
            this.registerKongFeatureTargets(80, this.adminPort, props.adminNlb, 'KongAdmin', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', false);
        }
        if (props.kongFeaturesProps.kongManagerProps.enabled && props.managerNlb) {
            this.registerKongFeatureTargets(443, this.managerHttpsPort, props.managerNlb, 'KongManager', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.managerEndpoint);
            if (props.kongFeaturesProps.kongManagerProps.enableHttp) {
                this.registerKongFeatureTargets(80, this.managerPort, props.managerNlb, 'KongManagerHttps', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', false);
            }
        }
        ;
        if (props.kongFeaturesProps.devPortalProps.enabled && props.devPortalNlb) {
            this.registerKongFeatureTargets(80, this.devPortalPort, props.devPortalNlb, 'KongDevPortal', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.devPortalEndpoint);
        }
    }
    registerKongFeatureTargets(listenerPort, containerPort, lb, id, containerName, addARecord, recordName) {
        const listener = new aws_elasticloadbalancingv2_1.NetworkListener(this, 'KongCp' + id + 'Listener', {
            port: listenerPort,
            protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            loadBalancer: lb,
        });
        if (addARecord) {
            new aws_route53_1.ARecord(this, 'KongCp' + id + 'ARecord', {
                zone: this.zone,
                recordName: recordName,
                target: aws_route53_1.RecordTarget.fromAlias(new targets.LoadBalancerTarget(lb)),
            });
        }
        this.registerLoadBalancerTargets({
            containerName: containerName,
            containerPort: containerPort,
            newTargetGroupId: id,
            listener: aws_ecs_1.ListenerConfig.networkListener(listener, {
                deregistrationDelay: aws_cdk_lib_1.Duration.seconds(60),
                port: listenerPort,
                healthCheck: {
                    port: containerPort.toString(),
                },
                preserveClientIp: false,
                proxyProtocolV2: false,
                protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            }),
        });
        this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(containerPort));
        if (recordName) {
            new aws_cdk_lib_1.CfnOutput(this, 'KongCp' + id + 'Endpoint', {
                value: recordName,
                exportName: id,
            });
        }
    }
}
exports.KongAdminService = KongAdminService;
//# sourceMappingURL=data:application/json;base64,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