"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Peer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management).
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 *
 * @stability stable
 */
class Peer {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Create an IPv4 peer from a CIDR.
     *
     * @stability stable
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address.
     *
     * @stability stable
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR.
     *
     * @stability stable
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address.
     *
     * @stability stable
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list.
     *
     * @stability stable
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
}
exports.Peer = Peer;
_a = JSII_RTTI_SYMBOL_1;
Peer[_a] = { fqn: "@aws-cdk/aws-ec2.Peer", version: "1.138.0" };
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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