#!/usr/bin/env python3
# coding=utf-8

import os
import sys
import logging
import argparse
import traceback
import sentry_sdk

import moodle_dl.utils.process_lock as process_lock


try:
    # In unix readline needs to be loaded so that
    # arrowkeys work in input
    import readline  # pylint: disable=unused-import
except ImportError:
    pass

from logging.handlers import RotatingFileHandler

from moodle_dl.utils import cutie
from moodle_dl.utils.logger import Log
from moodle_dl.download_service.path_tools import PathTools
from moodle_dl.config_service.config_helper import ConfigHelper
from moodle_dl.config_service.config_service import ConfigService
from moodle_dl.state_recorder.offline_service import OfflineService
from moodle_dl.moodle_connector.moodle_service import MoodleService
from moodle_dl.download_service.download_service import DownloadService
from moodle_dl.notification_services.mail.mail_service import MailService
from moodle_dl.download_service.fake_download_service import FakeDownloadService
from moodle_dl.notification_services.console.console_service import ConsoleService
from moodle_dl.notification_services.telegram.telegram_service import TelegramService

IS_DEBUG = False


class ReRaiseOnError(logging.StreamHandler):
    """
    A logging-handler class which allows the exception-catcher of i.e. PyCharm
    to intervine
    """

    def emit(self, record):
        if hasattr(record, 'exception'):
            raise record.exception


def run_init(storage_path, use_sso=False, skip_cert_verify=False):
    config = ConfigHelper(storage_path)

    if config.is_present():
        do_override_input = cutie.prompt_yes_or_no(Log.error_str('Do you want to override the existing config?'))

        if not do_override_input:
            sys.exit(0)

    MailService(config).interactively_configure()
    TelegramService(config).interactively_configure()

    do_sentry = cutie.prompt_yes_or_no('Do you want to configure Error Reporting via Sentry?')
    if do_sentry:
        sentry_dsn = input('Please enter your Sentry DSN:   ')
        config.set_property('sentry_dsn', sentry_dsn)

    moodle = MoodleService(config, storage_path, skip_cert_verify)

    if use_sso:
        moodle.interactively_acquire_sso_token()
    else:
        moodle.interactively_acquire_token()

    if os.name != 'nt':
        Log.info(
            'On Windows many characters are forbidden in filenames and paths, if you want, these characters can be'
            + ' automatically removed from filenames.'
        )

        Log.warning('If you want to view the downloaded files on Windows this is important!')

        default_windows_map = cutie.prompt_yes_or_no(
            'Do you want to load the default filename character map for windows?'
        )
        if default_windows_map:
            config.set_default_filename_character_map(True)
        else:
            config.set_default_filename_character_map(False)
    else:
        config.set_default_filename_character_map(True)

    Log.success('Configuration finished and saved!')

    if os.name != 'nt':
        if storage_path == '.':
            Log.info(
                '  To set a cron-job for this program on your Unix-System:\n'
                + '    1. `crontab -e`\n'
                + '    2. Add `*/15 * * * * cd %s && python3 %smain.py`\n'
                % (os.getcwd(), os.path.join(os.path.dirname(os.path.realpath(__file__)), ''))
                + '    3. Save and you\'re done!'
            )
        else:
            Log.info(
                '  To set a cron-job for this program on your Unix-System:\n'
                + '    1. `crontab -e`\n'
                + '    2. Add `*/15 * * * *'
                + ' cd %s && python3 %smain.py --path %s`\n'
                % (os.getcwd(), os.path.join(os.path.dirname(os.path.realpath(__file__)), ''), storage_path)
                + '    3. Save and you\'re done!'
            )

    print('')

    Log.info('You can always do the additional configuration later with the --config option.')

    do_config = cutie.prompt_yes_or_no('Do you want to make additional configurations now?')

    if do_config:
        run_configure(storage_path, skip_cert_verify)

    print('')
    Log.success('All set and ready to go!')


def run_configure(storage_path, skip_cert_verify=False):
    config = ConfigHelper(storage_path)
    config.load()  # because we do not want to override the other settings

    ConfigService(config, storage_path, skip_cert_verify).interactively_acquire_config()

    Log.success('Configuration successfully updated!')


def run_new_token(storage_path, use_sso=False, skip_cert_verify=False):
    config = ConfigHelper(storage_path)
    config.load()  # because we do not want to override the other settings

    moodle = MoodleService(config, storage_path, skip_cert_verify)

    if use_sso:
        moodle.interactively_acquire_sso_token(use_stored_url=True)
    else:
        moodle.interactively_acquire_token(use_stored_url=True)

    Log.success('New Token successfully saved!')


def run_manage_database(storage_path):
    config = ConfigHelper(storage_path)
    config.load()  # because we want to only manage configured courses

    offline_service = OfflineService(config, storage_path)
    offline_service.interactively_manage_database()

    Log.success('All done.')


def run_change_notification_mail(storage_path):
    config = ConfigHelper(storage_path)
    config.load()

    MailService(config).interactively_configure()

    Log.success('Configuration successfully updated!')


def run_change_notification_telegram(storage_path):
    config = ConfigHelper(storage_path)
    config.load()

    TelegramService(config).interactively_configure()

    Log.success('Telegram Configuration successfully updated!')


def run_main(storage_path, skip_cert_verify=False, without_downloading_files=False, log_responses=False):

    log_formatter = logging.Formatter('%(asctime)s  %(levelname)s  {%(module)s}  %(message)s', '%Y-%m-%d %H:%M:%S')
    log_file = os.path.join(storage_path, 'MoodleDownloader.log')
    log_handler = RotatingFileHandler(
        log_file, mode='a', maxBytes=1 * 1024 * 1024, backupCount=2, encoding='utf-8', delay=0
    )

    log_handler.setFormatter(log_formatter)
    log_handler.setLevel(logging.DEBUG)

    app_log = logging.getLogger()
    app_log.setLevel(logging.DEBUG)
    app_log.addHandler(log_handler)

    logging.info('--- main started ---------------------')
    Log.info('Moodle Downloader starting...')
    if IS_DEBUG:
        logging.info('Debug-Mode detected. Errors will not be logged but instead re-risen.')
        app_log.addHandler(ReRaiseOnError())

    try:
        logging.debug('Loading config...')
        Log.debug('Loading config...')
        config = ConfigHelper(storage_path)
        config.load()
    except BaseException as e:
        logging.error('Error while trying to load the Configuration! {} Exiting...'.format(e), extra={'exception': e})
        Log.error('Error while trying to load the Configuration!')
        sys.exit(-1)

    r_client = False
    try:
        sentry_dsn = config.get_property('sentry_dsn')
        if sentry_dsn:
            sentry_sdk.init(sentry_dsn)
    except BaseException:
        pass

    mail_service = MailService(config)
    tg_service = TelegramService(config)
    console_service = ConsoleService(config)

    PathTools.filename_character_map = config.get_filename_character_map()

    try:
        if not IS_DEBUG:
            process_lock.lock(storage_path)

        moodle = MoodleService(config, storage_path, skip_cert_verify, log_responses)

        logging.debug('Checking for changes for the configured Moodle-Account....')
        Log.debug('Checking for changes for the configured Moodle-Account...')
        changed_courses = moodle.fetch_state()

        logging.debug('Start downloading changed files...')
        Log.debug('Start downloading changed files...')

        if without_downloading_files:
            downloader = FakeDownloadService(changed_courses, moodle, storage_path)
        else:
            downloader = DownloadService(changed_courses, moodle, storage_path, skip_cert_verify)
        downloader.run()

        changed_courses_to_notify = moodle.recorder.changes_to_notify()

        if len(changed_courses_to_notify) > 0:
            console_service.notify_about_changes_in_moodle(changed_courses_to_notify)
            mail_service.notify_about_changes_in_moodle(changed_courses_to_notify)
            tg_service.notify_about_changes_in_moodle(changed_courses_to_notify)

            moodle.recorder.notified(changed_courses_to_notify)

        else:
            logging.info('No changes found for the configured Moodle-Account.')
            Log.warning('No changes found for the configured Moodle-Account.')

        process_lock.unlock(storage_path)

        logging.debug('All done. Exiting...')
        Log.success('All done. Exiting..')
    except BaseException as e:
        if not isinstance(e, process_lock.LockError):
            process_lock.unlock(storage_path)

        error_formatted = traceback.format_exc()
        logging.error(error_formatted, extra={'exception': e})

        if r_client:
            sentry_sdk.capture_exception(e)

        short_error = '%s\r\n%s' % (str(e), traceback.format_exc(limit=1))
        mail_service.notify_about_error(short_error)
        tg_service.notify_about_error(short_error)

        logging.debug('Exception-Handling completed. Exiting...')
        Log.critical('Exception:\n%s' % (error_formatted))
        Log.error('The following error occurred during execution: %s' % (str(e)))

        sys.exit(-1)


def _dir_path(path):
    if os.path.isdir(path):
        return path
    else:
        raise argparse.ArgumentTypeError('"%s" is not a valid path. Make sure the directory exists.' % (str(path)))


def check_debug():
    global IS_DEBUG
    if 'pydevd' in sys.modules:
        IS_DEBUG = True
        Log.debug('[RUNNING IN DEBUG-MODE!]')


def get_parser():
    """
    Creates a new argument parser.
    """
    parser = argparse.ArgumentParser(
        description=('Moodle Downloader 2 helps you download all the course files  of your Moodle account.')
    )
    group = parser.add_mutually_exclusive_group()
    group.add_argument(
        '-i',
        '--init',
        action='store_true',
        help=(
            'Guides you trough the configuration of the'
            + ' software, including the activation of'
            + ' mail-notifications and obtainment of a'
            + ' login-token for your Moodle-Account. It'
            + ' does not fetch the current state of your'
            + ' Moodle-Account.'
        ),
    )

    group.add_argument(
        '-c',
        '--config',
        action='store_true',
        help=(
            'Guides you through the additional'
            + ' configuration of the software. This'
            + ' includes the selection of the courses to'
            + ' be downloaded and various configuration'
            + ' options for these courses.'
        ),
    )

    group.add_argument(
        '-nt',
        '--new-token',
        action='store_true',
        help=(
            'Overrides the login-token with a newly obtained'
            + ' one. It does not fetch the current state of your'
            + ' Moodle-Account. Use it if at any point in time,'
            + ' for whatever reason, the saved token gets'
            + ' rejected by Moodle. It does not affect the rest'
            + ' of the config.'
        ),
    )

    group.add_argument(
        '-cm',
        '--change-notification-mail',
        action='store_true',
        help=(
            'Activate/deactivate/change the settings for'
            + ' receiving notifications via e-mail. It does not'
            + ' affect the rest of the config.'
        ),
    )

    group.add_argument(
        '-ct',
        '--change-notification-telegram',
        action='store_true',
        help=(
            'Activate/deactivate/change the settings for'
            + ' receiving notifications via Telegram. It does not'
            + ' affect the rest of the config.'
        ),
    )

    group.add_argument(
        '-md',
        '--manage-database',
        action='store_true',
        help=(
            'This option lets you manage the offline database.'
            + ' It allows you to delete entries from the database'
            + ' that are no longer available locally so that they'
            + ' can be downloaded again.'
        ),
    )

    parser.add_argument(
        '-p',
        '--path',
        default='.',
        type=_dir_path,
        help=(
            'Sets the location of the configuration,'
            + ' logs and downloaded files. PATH must be an'
            + ' existing directory in which you have read and'
            + ' write access. (default: current working'
            + ' directory)'
        ),
    )

    parser.add_argument(
        '--skip-cert-verify',
        default=False,
        action='store_true',
        help='If this flag is set, the SSL certificate '
        + 'is not verified. This option should only be used in '
        + 'non production environments.',
    )

    parser.add_argument(
        '--without-downloading-files',
        default=False,
        action='store_true',
        help='If this flag is set, no files are downloaded.'
        + ' This allows the local database to be updated without'
        + ' having to download all files.',
    )

    parser.add_argument(
        '-sso',
        '--sso',
        default=False,
        action='store_true',
        help='This flag can be used together with --init. If'
        + ' this flag is set, you will be guided through the'
        + ' Single Sign On (SSO) login process during'
        + 'initialization.',
    )

    parser.add_argument(
        '--log-responses',
        default=False,
        action='store_true',
        help='If this flag is set, a responses.log file is created'
        + ' in which all JSON responses from Moodles are logged'
        + ' along with the requested URL.',
    )
    return parser


# --- called at the program invocation: -------------------------------------
def main(args=None):
    """The main routine."""

    check_debug()

    parser = get_parser()
    args = parser.parse_args(args)

    use_sso = args.sso
    storage_path = args.path
    skip_cert_verify = args.skip_cert_verify
    without_downloading_files = args.without_downloading_files
    log_responses = args.log_responses

    if args.init:
        run_init(storage_path, use_sso, skip_cert_verify)
    elif args.config:
        run_configure(storage_path, skip_cert_verify)
    elif args.new_token:
        run_new_token(storage_path, use_sso, skip_cert_verify)
    elif args.change_notification_mail:
        run_change_notification_mail(storage_path)
    elif args.change_notification_telegram:

        run_change_notification_telegram(storage_path)
    elif args.manage_database:
        run_manage_database(storage_path)
    else:
        run_main(storage_path, skip_cert_verify, without_downloading_files, log_responses)
