from typing import Any

import httpx
from universal_mcp.applications.application import APIApplication
from universal_mcp.integrations import Integration


class JiraApp(APIApplication):
    def __init__(self, integration: Integration = None, **kwargs) -> None:
        super().__init__(name="jira", integration=integration, **kwargs)
        self._base_url: str | None = None

    def get_base_url(self):
        headers = self._get_headers()
        url = "https://api.atlassian.com/oauth/token/accessible-resources"

        response = httpx.get(url, headers=headers)
        response.raise_for_status()
        resources = response.json()

        if not resources:
            raise ValueError(
                "No accessible Jira resources found for the provided credentials."
            )

        first_resource = resources[0]
        resource_id = first_resource.get("id")

        if not resource_id:
            raise ValueError(
                "Could not determine the resource ID from the first accessible resource."
            )

        return f"https://api.atlassian.com/ex/jira/{resource_id}"

    @property
    def base_url(self):
        """Fetches accessible resources and sets the base_url for the first resource found."""
        if self._base_url:
            return self._base_url
        self._base_url = self.get_base_url()
        return self._base_url

    @base_url.setter
    def base_url(self, value: str) -> None:
        """Sets the base URL for the Confluence API.

        Args:
            value (str): The base URL to set.
        """
        self._base_url = value

    def get_banner(self) -> dict[str, Any]:
        """
        Retrieves the configuration of the announcement banner using the Jira Cloud API.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Announcement banner, important
        """
        url = f"{self.base_url}/rest/api/3/announcementBanner"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_banner(
        self,
        isDismissible: bool | None = None,
        isEnabled: bool | None = None,
        message: str | None = None,
        visibility: str | None = None,
    ) -> Any:
        """
        Updates the announcement banner configuration in Jira Cloud, including message, visibility, and dismissal settings.

        Args:
            isDismissible (boolean): Flag indicating if the announcement banner can be dismissed by the user. Example: False.
            isEnabled (boolean): Flag indicating if the announcement banner is enabled or not. Example: True.
            message (string): The text on the announcement banner. Example: 'This is a public, enabled, non-dismissible banner, set using the API'.
            visibility (string): Visibility of the announcement banner. Can be public or private. Example: 'public'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Announcement banner, important
        """
        request_body_data = None
        request_body_data = {
            "isDismissible": isDismissible,
            "isEnabled": isEnabled,
            "message": message,
            "visibility": visibility,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/announcementBanner"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_custom_fields_configurations(
        self,
        fieldIdsOrKeys: list[str],
        id: list[int] | None = None,
        fieldContextId: list[int] | None = None,
        issueId: int | None = None,
        projectKeyOrId: str | None = None,
        issueTypeId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves and filters a list of custom field context configurations in Jira based on specified criteria like field ID, project, or issue type, returning paginated results.

        Args:
            fieldIdsOrKeys (array): List of IDs or keys of the custom fields. It can be a mix of IDs and keys in the same query. Example: ['customfield_10035', 'customfield_10036'].
            id (array): The list of configuration IDs. To include multiple configurations, separate IDs with an ampersand: `id=10000&id=10001`. Can't be provided with `fieldContextId`, `issueId`, `projectKeyOrId`, or `issueTypeId`.
            fieldContextId (array): The list of field context IDs. To include multiple field contexts, separate IDs with an ampersand: `fieldContextId=10000&fieldContextId=10001`. Can't be provided with `id`, `issueId`, `projectKeyOrId`, or `issueTypeId`.
            issueId (integer): The ID of the issue to filter results by. If the issue doesn't exist, an empty list is returned. Can't be provided with `projectKeyOrId`, or `issueTypeId`.
            projectKeyOrId (string): The ID or key of the project to filter results by. Must be provided with `issueTypeId`. Can't be provided with `issueId`.
            issueTypeId (string): The ID of the issue type to filter results by. Must be provided with `projectKeyOrId`. Can't be provided with `issueId`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field configuration (apps)
        """
        request_body_data = None
        request_body_data = {
            "fieldIdsOrKeys": fieldIdsOrKeys,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/app/field/context/configuration/list"
        query_params = {
            k: v
            for k, v in [
                ("id", id),
                ("fieldContextId", fieldContextId),
                ("issueId", issueId),
                ("projectKeyOrId", projectKeyOrId),
                ("issueTypeId", issueTypeId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_field_value(
        self,
        generateChangelog: bool | None = None,
        updates: list[dict[str, Any]] | None = None,
    ) -> Any:
        """
        Updates the value of a custom field added by a Forge app on one or more Jira issues.

        Args:
            generateChangelog (boolean): Whether to generate a changelog for this update.
            updates (array): updates Example: [{'customField': 'customfield_10010', 'issueIds': [10010, 10011], 'value': 'new value'}, {'customField': 'customfield_10011', 'issueIds': [10010], 'value': 1000}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field values (apps)
        """
        request_body_data = None
        request_body_data = {
            "updates": updates,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/app/field/value"
        query_params = {
            k: v for k, v in [("generateChangelog", generateChangelog)] if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_custom_field_configuration(
        self,
        fieldIdOrKey: str,
        id: list[int] | None = None,
        fieldContextId: list[int] | None = None,
        issueId: int | None = None,
        projectKeyOrId: str | None = None,
        issueTypeId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the configuration of a custom field context in Jira using the provided field ID or key, optionally filtered by specific IDs, field context IDs, issue IDs, project keys or IDs, or issue types, and returns a paginated list of configurations.

        Args:
            fieldIdOrKey (string): fieldIdOrKey
            id (array): The list of configuration IDs. To include multiple configurations, separate IDs with an ampersand: `id=10000&id=10001`. Can't be provided with `fieldContextId`, `issueId`, `projectKeyOrId`, or `issueTypeId`.
            fieldContextId (array): The list of field context IDs. To include multiple field contexts, separate IDs with an ampersand: `fieldContextId=10000&fieldContextId=10001`. Can't be provided with `id`, `issueId`, `projectKeyOrId`, or `issueTypeId`.
            issueId (integer): The ID of the issue to filter results by. If the issue doesn't exist, an empty list is returned. Can't be provided with `projectKeyOrId`, or `issueTypeId`.
            projectKeyOrId (string): The ID or key of the project to filter results by. Must be provided with `issueTypeId`. Can't be provided with `issueId`.
            issueTypeId (string): The ID of the issue type to filter results by. Must be provided with `projectKeyOrId`. Can't be provided with `issueId`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field configuration (apps)
        """
        if fieldIdOrKey is None:
            raise ValueError("Missing required parameter 'fieldIdOrKey'.")
        url = (
            f"{self.base_url}/rest/api/3/app/field/{fieldIdOrKey}/context/configuration"
        )
        query_params = {
            k: v
            for k, v in [
                ("id", id),
                ("fieldContextId", fieldContextId),
                ("issueId", issueId),
                ("projectKeyOrId", projectKeyOrId),
                ("issueTypeId", issueTypeId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_custom_field_configuration(
        self, fieldIdOrKey: str, configurations: list[dict[str, Any]]
    ) -> Any:
        """
        Updates the configuration of a custom field context in Jira using the provided field ID or key.

        Args:
            fieldIdOrKey (string): fieldIdOrKey
            configurations (array): The list of custom field configuration details. Example: [{'id': '10000'}, {'configuration': {'maxValue': 10000, 'minValue': 0}, 'id': '10001', 'schema': {'properties': {'amount': {'type': 'number'}, 'currency': {'type': 'string'}}, 'required': ['amount', 'currency']}}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field configuration (apps)
        """
        if fieldIdOrKey is None:
            raise ValueError("Missing required parameter 'fieldIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "configurations": configurations,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = (
            f"{self.base_url}/rest/api/3/app/field/{fieldIdOrKey}/context/configuration"
        )
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_custom_field_value(
        self,
        fieldIdOrKey: str,
        generateChangelog: bool | None = None,
        updates: list[dict[str, Any]] | None = None,
    ) -> Any:
        """
        Updates the value of a custom field for an issue using the Jira API, but this endpoint is limited to working with fields provided by Forge apps.

        Args:
            fieldIdOrKey (string): fieldIdOrKey
            generateChangelog (boolean): Whether to generate a changelog for this update.
            updates (array): The list of custom field update details. Example: [{'issueIds': [10010], 'value': 'new value'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field values (apps)
        """
        if fieldIdOrKey is None:
            raise ValueError("Missing required parameter 'fieldIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "updates": updates,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/app/field/{fieldIdOrKey}/value"
        query_params = {
            k: v for k, v in [("generateChangelog", generateChangelog)] if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_application_property(
        self,
        key: str | None = None,
        permissionLevel: str | None = None,
        keyFilter: str | None = None,
    ) -> list[Any]:
        """
        Retrieves application properties with optional filtering by key, permission level, or key filter.

        Args:
            key (string): The key of the application property.
            permissionLevel (string): The permission level of all items being returned in the list.
            keyFilter (string): When a `key` isn't provided, this filters the list of results by the application property `key` using a regular expression. For example, using `jira.lf.*` will return all application properties with keys that start with *jira.lf.*.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira settings
        """
        url = f"{self.base_url}/rest/api/3/application-properties"
        query_params = {
            k: v
            for k, v in [
                ("key", key),
                ("permissionLevel", permissionLevel),
                ("keyFilter", keyFilter),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_advanced_settings(self) -> list[Any]:
        """
        The API retrieves Jira's advanced settings properties, returning configuration details displayed on the "General Configuration > Advanced Settings" page.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira settings
        """
        url = f"{self.base_url}/rest/api/3/application-properties/advanced-settings"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_application_property(
        self, id: str, id_body: str | None = None, value: str | None = None
    ) -> dict[str, Any]:
        """
        Updates a specific Jira application property identified by its ID using a PUT request.

        Args:
            id (string): id
            id_body (string): The ID of the application property. Example: 'jira.home'.
            value (string): The new value. Example: '/var/jira/jira-home'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira settings
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "id": id_body,
            "value": value,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/application-properties/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_application_roles(self) -> list[Any]:
        """
        Retrieves information about application roles using the Jira Cloud API and returns data relevant to the specified application roles.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Application roles
        """
        url = f"{self.base_url}/rest/api/3/applicationrole"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_application_role(self, key: str) -> dict[str, Any]:
        """
        Retrieves details of a specific application role in Jira Cloud using the provided `key` parameter via the GET method.

        Args:
            key (string): key

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Application roles
        """
        if key is None:
            raise ValueError("Missing required parameter 'key'.")
        url = f"{self.base_url}/rest/api/3/applicationrole/{key}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_attachment_content(
        self, id: str, redirect: bool | None = None
    ) -> list[Any]:
        """
        Retrieves the contents of a Jira Cloud attachment by ID and returns the binary file data or a redirect URL.

        Args:
            id (string): id
            redirect (boolean): Whether a redirect is provided for the attachment download. Clients that do not automatically follow redirects can set this to `false` to avoid making multiple requests to download the attachment.

        Returns:
            list[Any]: Returned if the request is successful when `redirect` is set to `false`.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/content/{id}"
        query_params = {k: v for k, v in [("redirect", redirect)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_attachment_meta(self) -> dict[str, Any]:
        """
        Retrieves Jira's attachment settings, including whether attachments are enabled and the maximum allowed upload size.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        url = f"{self.base_url}/rest/api/3/attachment/meta"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_attachment_thumbnail(
        self,
        id: str,
        redirect: bool | None = None,
        fallbackToDefault: bool | None = None,
        width: int | None = None,
        height: int | None = None,
    ) -> list[Any]:
        """
        Retrieves a thumbnail image for a specified attachment ID in Jira, supporting optional parameters for dimensions and redirect behavior.

        Args:
            id (string): id
            redirect (boolean): Whether a redirect is provided for the attachment download. Clients that do not automatically follow redirects can set this to `false` to avoid making multiple requests to download the attachment.
            fallbackToDefault (boolean): Whether a default thumbnail is returned when the requested thumbnail is not found.
            width (integer): The maximum width to scale the thumbnail to.
            height (integer): The maximum height to scale the thumbnail to.

        Returns:
            list[Any]: Returned if the request is successful when `redirect` is set to `false`.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/thumbnail/{id}"
        query_params = {
            k: v
            for k, v in [
                ("redirect", redirect),
                ("fallbackToDefault", fallbackToDefault),
                ("width", width),
                ("height", height),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_attachment(self, id: str) -> Any:
        """
        Deletes an attachment from a Jira issue by its ID and returns a success status upon removal.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_attachment(self, id: str) -> dict[str, Any]:
        """
        Retrieves metadata for a specific attachment in Jira using the provided attachment ID.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def expand_attachment_for_humans(self, id: str) -> dict[str, Any]:
        """
        Retrieves human-readable metadata for a specific Jira Cloud attachment in an expanded format.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful. If an empty list is returned in the response, the attachment is empty, corrupt, or not an archive.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/{id}/expand/human"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def expand_attachment_for_machines(self, id: str) -> dict[str, Any]:
        """
        Retrieves the contents metadata for an expanded attachment in Jira Cloud using the "GET /rest/api/3/attachment/{id}/expand/raw" endpoint, suitable for processing without presenting the data to users.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful. If an empty list is returned in the response, the attachment is empty, corrupt, or not an archive.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/attachment/{id}/expand/raw"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_audit_records(
        self,
        offset: int | None = None,
        limit: int | None = None,
        filter: str | None = None,
        from_: str | None = None,
        to: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves Jira audit records with filtering by parameters like date range, text, and category, returning activity logs of administrative actions and changes.

        Args:
            offset (integer): The number of records to skip before returning the first result.
            limit (integer): The maximum number of results to return.
            filter (string): The strings to match with audit field content, space separated.
            from_ (string): The date and time on or after which returned audit records must have been created. If `to` is provided `from` must be before `to` or no audit records are returned.
            to (string): The date and time on or before which returned audit results must have been created. If `from` is provided `to` must be after `from` or no audit records are returned.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Audit records
        """
        url = f"{self.base_url}/rest/api/3/auditing/record"
        query_params = {
            k: v
            for k, v in [
                ("offset", offset),
                ("limit", limit),
                ("filter", filter),
                ("from", from_),
                ("to", to),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_system_avatars(self, type: str) -> dict[str, Any]:
        """
        Retrieves a list of system avatar details for a specified owner type (e.g., user, project, issue type) via the Jira Cloud REST API.

        Args:
            type (string): type

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        url = f"{self.base_url}/rest/api/3/avatar/{type}/system"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_delete(
        self,
        selectedIssueIdsOrKeys: list[str],
        sendBulkNotification: bool | None = None,
    ) -> dict[str, Any]:
        """
        Deletes multiple Jira issues in a single request using the Bulk Delete API.

        Args:
            selectedIssueIdsOrKeys (array): List of issue IDs or keys which are to be bulk deleted. These IDs or keys can be from different projects and issue types. Example: ['10001', '10002'].
            sendBulkNotification (boolean): A boolean value that indicates whether to send a bulk change notification when the issues are being deleted.

        If `true`, dispatches a bulk notification email to users about the updates. Example: False.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "selectedIssueIdsOrKeys": selectedIssueIdsOrKeys,
            "sendBulkNotification": sendBulkNotification,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/delete"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_bulk_editable_fields(
        self,
        issueIdsOrKeys: str,
        searchText: str | None = None,
        endingBefore: str | None = None,
        startingAfter: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of fields that can be edited in bulk for specified Jira issues, allowing for the identification of editable fields for bulk operations using query parameters such as issue IDs or keys, search text, and pagination options.

        Args:
            issueIdsOrKeys (string): The IDs or keys of the issues to get editable fields from.
            searchText (string): (Optional)The text to search for in the editable fields.
            endingBefore (string): (Optional)The end cursor for use in pagination.
            startingAfter (string): (Optional)The start cursor for use in pagination.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        url = f"{self.base_url}/rest/api/3/bulk/issues/fields"
        query_params = {
            k: v
            for k, v in [
                ("issueIdsOrKeys", issueIdsOrKeys),
                ("searchText", searchText),
                ("endingBefore", endingBefore),
                ("startingAfter", startingAfter),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_edit(
        self,
        editedFieldsInput: Any,
        selectedActions: list[str],
        selectedIssueIdsOrKeys: list[str],
        sendBulkNotification: bool | None = None,
    ) -> dict[str, Any]:
        """
        Edits multiple Jira issues in bulk by updating fields using the Jira Cloud API.

        Args:
            editedFieldsInput (string): An object that defines the values to be updated in specified fields of an issue. The structure and content of this parameter vary depending on the type of field being edited. Although the order is not significant, ensure that field IDs align with those in selectedActions.
            selectedActions (array): List of all the field IDs that are to be bulk edited. Each field ID in this list corresponds to a specific attribute of an issue that is set to be modified in the bulk edit operation. The relevant field ID can be obtained by calling the Bulk Edit Get Fields REST API (documentation available on this page itself).
            selectedIssueIdsOrKeys (array): List of issue IDs or keys which are to be bulk edited. These IDs or keys can be from different projects and issue types.
            sendBulkNotification (boolean): A boolean value that indicates whether to send a bulk change notification when the issues are being edited.

        If `true`, dispatches a bulk notification email to users about the updates.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "editedFieldsInput": editedFieldsInput,
            "selectedActions": selectedActions,
            "selectedIssueIdsOrKeys": selectedIssueIdsOrKeys,
            "sendBulkNotification": sendBulkNotification,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/fields"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_move(
        self,
        sendBulkNotification: bool | None = None,
        targetToSourcesMapping: dict[str, dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Moves multiple issues from one Jira project to another using the POST method.

        Args:
            sendBulkNotification (boolean): A boolean value that indicates whether to send a bulk change notification when the issues are being moved.

        If `true`, dispatches a bulk notification email to users about the updates. Example: True.
            targetToSourcesMapping (object): An object representing the mapping of issues and data related to destination entities, like fields and statuses, that are required during a bulk move.

        The key is a string that is created by concatenating the following three entities in order, separated by commas. The format is `<project ID or key>,<issueType ID>,<parent ID or key>`. It should be unique across mappings provided in the payload. If you provide multiple mappings for the same key, only one will be processed. However, the operation won't fail, so the error may be hard to track down.

         *  ***Destination project*** (Required): ID or key of the project to which the issues are being moved.
         *  ***Destination issueType*** (Required): ID of the issueType to which the issues are being moved.
         *  ***Destination parent ID or key*** (Optional): ID or key of the issue which will become the parent of the issues being moved. Only required when the destination issueType is a subtask. Example: {'PROJECT-KEY,10001': {'inferClassificationDefaults': False, 'inferFieldDefaults': False, 'inferStatusDefaults': False, 'inferSubtaskTypeDefault': True, 'issueIdsOrKeys': ['ISSUE-1'], 'targetClassification': [{'classifications': {'5bfa70f7-4af1-44f5-9e12-1ce185f15a38': ['bd58e74c-c31b-41a7-ba69-9673ebd9dae9', '-1']}}], 'targetMandatoryFields': [{'fields': {'customfield_10000': {'retain': False, 'type': 'raw', 'value': ['value-1', 'value-2']}, 'description': {'retain': True, 'type': 'adf', 'value': {'content': [{'content': [{'text': 'New description value', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}}, 'fixVersions': {'retain': False, 'type': 'raw', 'value': ['10009']}, 'labels': {'retain': False, 'type': 'raw', 'value': ['label-1', 'label-2']}}}], 'targetStatus': [{'statuses': {'10001': ['10002', '10003']}}]}}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "sendBulkNotification": sendBulkNotification,
            "targetToSourcesMapping": targetToSourcesMapping,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/move"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_available_transitions(
        self,
        issueIdsOrKeys: str,
        endingBefore: str | None = None,
        startingAfter: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves valid workflow transitions for multiple Jira issues based on provided issue IDs or keys.

        Args:
            issueIdsOrKeys (string): Comma (,) separated Ids or keys of the issues to get transitions available for them.
            endingBefore (string): (Optional)The end cursor for use in pagination.
            startingAfter (string): (Optional)The start cursor for use in pagination.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        url = f"{self.base_url}/rest/api/3/bulk/issues/transition"
        query_params = {
            k: v
            for k, v in [
                ("issueIdsOrKeys", issueIdsOrKeys),
                ("endingBefore", endingBefore),
                ("startingAfter", startingAfter),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_transition(
        self,
        bulkTransitionInputs: list[dict[str, Any]],
        sendBulkNotification: bool | None = None,
    ) -> dict[str, Any]:
        """
        Transitions multiple Jira issues to a specified status in bulk using the Jira REST API, allowing for streamlined workflow management and automation of repetitive tasks.

        Args:
            bulkTransitionInputs (array): List of objects and each object has two properties:

         *  Issues that will be bulk transitioned.
         *  TransitionId that corresponds to a specific transition of issues that share the same workflow. Example: [{'selectedIssueIdsOrKeys': ['10001', '10002'], 'transitionId': '11'}, {'selectedIssueIdsOrKeys': ['TEST-1'], 'transitionId': '2'}].
            sendBulkNotification (boolean): A boolean value that indicates whether to send a bulk change notification when the issues are being transitioned.

        If `true`, dispatches a bulk notification email to users about the updates. Example: False.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "bulkTransitionInputs": bulkTransitionInputs,
            "sendBulkNotification": sendBulkNotification,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/transition"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_unwatch(self, selectedIssueIdsOrKeys: list[str]) -> dict[str, Any]:
        """
        Unwatches up to 1,000 specified Jira issues in a single bulk operation via POST request, requiring write permissions and returning success/error responses.

        Args:
            selectedIssueIdsOrKeys (array): List of issue IDs or keys which are to be bulk watched or unwatched. These IDs or keys can be from different projects and issue types. Example: ['10001', '10002'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "selectedIssueIdsOrKeys": selectedIssueIdsOrKeys,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/unwatch"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def submit_bulk_watch(self, selectedIssueIdsOrKeys: list[str]) -> dict[str, Any]:
        """
        Adds watchers to multiple Jira issues in bulk through a single operation.

        Args:
            selectedIssueIdsOrKeys (array): List of issue IDs or keys which are to be bulk watched or unwatched. These IDs or keys can be from different projects and issue types. Example: ['10001', '10002'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        request_body_data = None
        request_body_data = {
            "selectedIssueIdsOrKeys": selectedIssueIdsOrKeys,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/bulk/issues/watch"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_bulk_operation_progress(self, taskId: str) -> dict[str, Any]:
        """
        Retrieves the status of a bulk operation task identified by the specified taskId.

        Args:
            taskId (string): taskId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue bulk operations
        """
        if taskId is None:
            raise ValueError("Missing required parameter 'taskId'.")
        url = f"{self.base_url}/rest/api/3/bulk/queue/{taskId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_bulk_changelogs(
        self,
        issueIdsOrKeys: list[str],
        fieldIds: list[str] | None = None,
        maxResults: int | None = None,
        nextPageToken: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves changelog data for multiple Jira issues in a single request, eliminating the need for individual API calls per issue.

        Args:
            issueIdsOrKeys (array): List of issue IDs/keys to fetch changelogs for
            fieldIds (array): List of field IDs to filter changelogs
            maxResults (integer): The maximum number of items to return per page
            nextPageToken (string): The cursor for pagination

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "fieldIds": fieldIds,
            "issueIdsOrKeys": issueIdsOrKeys,
            "maxResults": maxResults,
            "nextPageToken": nextPageToken,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/changelog/bulkfetch"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_classification_levels(
        self, status: list[str] | None = None, orderBy: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a list of all classification levels in Jira Cloud, supporting optional filtering by status and ordering using the "orderBy" parameter.

        Args:
            status (array): Optional set of statuses to filter by.
            orderBy (string): Ordering of the results by a given field. If not provided, values will not be sorted.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Classification levels
        """
        url = f"{self.base_url}/rest/api/3/classification-levels"
        query_params = {
            k: v for k, v in [("status", status), ("orderBy", orderBy)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_comments_by_ids(
        self, ids: list[int], expand: str | None = None
    ) -> dict[str, Any]:
        """
        Fetches a paginated list of Jira comments by their IDs using a POST request.

        Args:
            ids (array): The list of comment IDs. A maximum of 1000 IDs can be specified. Example: [1, 2, 5, 10].
            expand (string): Use [expand](#expansion) to include additional information about comments in the response. This parameter accepts a comma-separated list. Expand options include: * `renderedBody` Returns the comment body rendered in HTML. * `properties` Returns the comment's properties.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        request_body_data = None
        request_body_data = {
            "ids": ids,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/comment/list"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_comment_property_keys(self, commentId: str) -> dict[str, Any]:
        """
        Retrieves the keys of all properties associated with a specified issue comment in Jira Cloud using the REST API.

        Args:
            commentId (string): commentId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comment properties
        """
        if commentId is None:
            raise ValueError("Missing required parameter 'commentId'.")
        url = f"{self.base_url}/rest/api/3/comment/{commentId}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_comment_property(self, commentId: str, propertyKey: str) -> Any:
        """
        Deletes a specific property from a comment in Jira using the Jira Cloud REST API and returns a status code upon successful deletion.

        Args:
            commentId (string): commentId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comment properties
        """
        if commentId is None:
            raise ValueError("Missing required parameter 'commentId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/comment/{commentId}/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_comment_property(self, commentId: str, propertyKey: str) -> dict[str, Any]:
        """
        Retrieves the value of a specific property for an issue comment in Jira using the comment ID and property key.

        Args:
            commentId (string): commentId
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comment properties
        """
        if commentId is None:
            raise ValueError("Missing required parameter 'commentId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/comment/{commentId}/properties/{propertyKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_comment_property(self, commentId: str, propertyKey: str) -> Any:
        """
        Updates the value of a specific property for a Jira comment using the PUT method, storing custom data against a comment identified by its ID and property key.

        Args:
            commentId (string): commentId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the comment property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comment properties
        """
        if commentId is None:
            raise ValueError("Missing required parameter 'commentId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/comment/{commentId}/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_components_for_projects(
        self,
        projectIdsOrKeys: list[str] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
        query: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of Jira components using the GET method at the "/rest/api/3/component" path, allowing filtering by project IDs, pagination, and sorting, and returns the results in a paginated format.

        Args:
            projectIdsOrKeys (array): The project IDs and/or project keys (case sensitive).
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field: * `description` Sorts by the component description. * `name` Sorts by component name.
            query (string): Filter the results using a literal string. Components with a matching `name` or `description` are returned (case insensitive).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        url = f"{self.base_url}/rest/api/3/component"
        query_params = {
            k: v
            for k, v in [
                ("projectIdsOrKeys", projectIdsOrKeys),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
                ("query", query),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_component(
        self,
        ari: str | None = None,
        assignee: Any | None = None,
        assigneeType: str | None = None,
        description: str | None = None,
        id: str | None = None,
        isAssigneeTypeValid: bool | None = None,
        lead: Any | None = None,
        leadAccountId: str | None = None,
        leadUserName: str | None = None,
        metadata: dict[str, str] | None = None,
        name: str | None = None,
        project: str | None = None,
        projectId: int | None = None,
        realAssignee: Any | None = None,
        realAssigneeType: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new component in Jira, providing a container for issues within a project, using the POST method on the "/rest/api/3/component" endpoint.

        Args:
            ari (string): Compass component's ID. Can't be updated. Not required for creating a Project Component.
            assignee (string): The details of the user associated with `assigneeType`, if any. See `realAssignee` for details of the user assigned to issues created with this component.
            assigneeType (string): The nominal user type used to determine the assignee for issues created with this component. See `realAssigneeType` for details on how the type of the user, and hence the user, assigned to issues is determined. Can take the following values:

         *  `PROJECT_LEAD` the assignee to any issues created with this component is nominally the lead for the project the component is in.
         *  `COMPONENT_LEAD` the assignee to any issues created with this component is nominally the lead for the component.
         *  `UNASSIGNED` an assignee is not set for issues created with this component.
         *  `PROJECT_DEFAULT` the assignee to any issues created with this component is nominally the default assignee for the project that the component is in.

        Default value: `PROJECT_DEFAULT`.
        Optional when creating or updating a component. Example: 'PROJECT_LEAD'.
            description (string): The description for the component. Optional when creating or updating a component. Example: 'This is a Jira component'.
            id (string): The unique identifier for the component.
            isAssigneeTypeValid (boolean): Whether a user is associated with `assigneeType`. For example, if the `assigneeType` is set to `COMPONENT_LEAD` but the component lead is not set, then `false` is returned. Example: False.
            lead (string): The user details for the component's lead user.
            leadAccountId (string): The accountId of the component's lead user. The accountId uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10a2844c20165700ede21g'.
            leadUserName (string): This property is no longer available and will be removed from the documentation soon. See the [deprecation notice](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            metadata (object): Compass component's metadata. Can't be updated. Not required for creating a Project Component.
            name (string): The unique name for the component in the project. Required when creating a component. Optional when updating a component. The maximum length is 255 characters. Example: 'Component 1'.
            project (string): The key of the project the component is assigned to. Required when creating a component. Can't be updated. Example: 'HSP'.
            projectId (integer): The ID of the project the component is assigned to.
            realAssignee (string): The user assigned to issues created with this component, when `assigneeType` does not identify a valid assignee.
            realAssigneeType (string): The type of the assignee that is assigned to issues created with this component, when an assignee cannot be set from the `assigneeType`. For example, `assigneeType` is set to `COMPONENT_LEAD` but no component lead is set. This property is set to one of the following values:

         *  `PROJECT_LEAD` when `assigneeType` is `PROJECT_LEAD` and the project lead has permission to be assigned issues in the project that the component is in.
         *  `COMPONENT_LEAD` when `assignee`Type is `COMPONENT_LEAD` and the component lead has permission to be assigned issues in the project that the component is in.
         *  `UNASSIGNED` when `assigneeType` is `UNASSIGNED` and Jira is configured to allow unassigned issues.
         *  `PROJECT_DEFAULT` when none of the preceding cases are true.
            self_arg_body (string): The URL of the component.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        request_body_data = None
        request_body_data = {
            "ari": ari,
            "assignee": assignee,
            "assigneeType": assigneeType,
            "description": description,
            "id": id,
            "isAssigneeTypeValid": isAssigneeTypeValid,
            "lead": lead,
            "leadAccountId": leadAccountId,
            "leadUserName": leadUserName,
            "metadata": metadata,
            "name": name,
            "project": project,
            "projectId": projectId,
            "realAssignee": realAssignee,
            "realAssigneeType": realAssigneeType,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/component"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_component(self, id: str, moveIssuesTo: str | None = None) -> Any:
        """
        Deletes a specific component in Jira by ID, optionally reassigning its issues to another component.

        Args:
            id (string): id
            moveIssuesTo (string): The ID of the component to replace the deleted component. If this value is null no replacement is made.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/component/{id}"
        query_params = {
            k: v for k, v in [("moveIssuesTo", moveIssuesTo)] if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_component(self, id: str) -> dict[str, Any]:
        """
        Retrieves detailed information about a specific Jira component, identified by its ID, using the Jira REST API.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/component/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_component(
        self,
        id: str,
        ari: str | None = None,
        assignee: Any | None = None,
        assigneeType: str | None = None,
        description: str | None = None,
        id_body: str | None = None,
        isAssigneeTypeValid: bool | None = None,
        lead: Any | None = None,
        leadAccountId: str | None = None,
        leadUserName: str | None = None,
        metadata: dict[str, str] | None = None,
        name: str | None = None,
        project: str | None = None,
        projectId: int | None = None,
        realAssignee: Any | None = None,
        realAssigneeType: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates the specified component's details using the provided ID.

        Args:
            id (string): id
            ari (string): Compass component's ID. Can't be updated. Not required for creating a Project Component.
            assignee (string): The details of the user associated with `assigneeType`, if any. See `realAssignee` for details of the user assigned to issues created with this component.
            assigneeType (string): The nominal user type used to determine the assignee for issues created with this component. See `realAssigneeType` for details on how the type of the user, and hence the user, assigned to issues is determined. Can take the following values:

         *  `PROJECT_LEAD` the assignee to any issues created with this component is nominally the lead for the project the component is in.
         *  `COMPONENT_LEAD` the assignee to any issues created with this component is nominally the lead for the component.
         *  `UNASSIGNED` an assignee is not set for issues created with this component.
         *  `PROJECT_DEFAULT` the assignee to any issues created with this component is nominally the default assignee for the project that the component is in.

        Default value: `PROJECT_DEFAULT`.
        Optional when creating or updating a component. Example: 'PROJECT_LEAD'.
            description (string): The description for the component. Optional when creating or updating a component. Example: 'This is a Jira component'.
            id_body (string): The unique identifier for the component.
            isAssigneeTypeValid (boolean): Whether a user is associated with `assigneeType`. For example, if the `assigneeType` is set to `COMPONENT_LEAD` but the component lead is not set, then `false` is returned. Example: False.
            lead (string): The user details for the component's lead user.
            leadAccountId (string): The accountId of the component's lead user. The accountId uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10a2844c20165700ede21g'.
            leadUserName (string): This property is no longer available and will be removed from the documentation soon. See the [deprecation notice](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            metadata (object): Compass component's metadata. Can't be updated. Not required for creating a Project Component.
            name (string): The unique name for the component in the project. Required when creating a component. Optional when updating a component. The maximum length is 255 characters. Example: 'Component 1'.
            project (string): The key of the project the component is assigned to. Required when creating a component. Can't be updated.
            projectId (integer): The ID of the project the component is assigned to.
            realAssignee (string): The user assigned to issues created with this component, when `assigneeType` does not identify a valid assignee.
            realAssigneeType (string): The type of the assignee that is assigned to issues created with this component, when an assignee cannot be set from the `assigneeType`. For example, `assigneeType` is set to `COMPONENT_LEAD` but no component lead is set. This property is set to one of the following values:

         *  `PROJECT_LEAD` when `assigneeType` is `PROJECT_LEAD` and the project lead has permission to be assigned issues in the project that the component is in.
         *  `COMPONENT_LEAD` when `assignee`Type is `COMPONENT_LEAD` and the component lead has permission to be assigned issues in the project that the component is in.
         *  `UNASSIGNED` when `assigneeType` is `UNASSIGNED` and Jira is configured to allow unassigned issues.
         *  `PROJECT_DEFAULT` when none of the preceding cases are true.
            self_arg_body (string): The URL of the component.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "ari": ari,
            "assignee": assignee,
            "assigneeType": assigneeType,
            "description": description,
            "id": id_body,
            "isAssigneeTypeValid": isAssigneeTypeValid,
            "lead": lead,
            "leadAccountId": leadAccountId,
            "leadUserName": leadUserName,
            "metadata": metadata,
            "name": name,
            "project": project,
            "projectId": projectId,
            "realAssignee": realAssignee,
            "realAssigneeType": realAssigneeType,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/component/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_component_related_issues(self, id: str) -> dict[str, Any]:
        """
        Retrieves the issue counts related to a specific Jira component identified by its ID using the "GET" method.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/component/{id}/relatedIssueCounts"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_configuration(self) -> dict[str, Any]:
        """
        Retrieves configuration details from Jira using the GET method and returns the result in the response.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira settings
        """
        url = f"{self.base_url}/rest/api/3/configuration"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_time_tracking_config(self) -> dict[str, Any]:
        """
        Retrieves the time tracking settings in Jira, including time format and default time unit, using the GET method at "/rest/api/3/configuration/timetracking".

        Returns:
            dict[str, Any]: Returned if the request is successful and time tracking is enabled.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Time tracking
        """
        url = f"{self.base_url}/rest/api/3/configuration/timetracking"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_time_tracking_config(
        self, key: str, name: str | None = None, url: str | None = None
    ) -> Any:
        """
        Updates time tracking settings in Jira using the Jira Cloud platform REST API at "/rest/api/3/configuration/timetracking" with a "PUT" method, allowing configurations such as time format and default time unit.

        Args:
            key (string): The key for the time tracking provider. For example, *JIRA*. Example: 'Jira'.
            name (string): The name of the time tracking provider. For example, *JIRA provided time tracking*.
            url (string): The URL of the configuration page for the time tracking provider app. For example, */example/config/url*. This property is only returned if the `adminPageKey` property is set in the module descriptor of the time tracking provider app.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Time tracking
        """
        request_body_data = None
        request_body_data = {
            "key": key,
            "name": name,
            "url": url,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/configuration/timetracking"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_time_tracking_configs(self) -> list[Any]:
        """
        Retrieves a list of all configured time tracking providers in Jira, including the active provider if time tracking is enabled.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Time tracking
        """
        url = f"{self.base_url}/rest/api/3/configuration/timetracking/list"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_time_tracking_options(self) -> dict[str, Any]:
        """
        Retrieves time tracking configuration settings such as time format and default units using the Jira Cloud REST API.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Time tracking
        """
        url = f"{self.base_url}/rest/api/3/configuration/timetracking/options"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_time_tracking_options(
        self,
        defaultUnit: str,
        timeFormat: str,
        workingDaysPerWeek: float,
        workingHoursPerDay: float,
    ) -> dict[str, Any]:
        """
        Updates Jira time tracking configuration settings including time format, working hours, and default time unit.

        Args:
            defaultUnit (string): The default unit of time applied to logged time. Example: 'hour'.
            timeFormat (string): The format that will appear on an issue's *Time Spent* field. Example: 'pretty'.
            workingDaysPerWeek (number): The number of days in a working week. Example: 5.5.
            workingHoursPerDay (number): The number of hours in a working day. Example: 7.6.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Time tracking
        """
        request_body_data = None
        request_body_data = {
            "defaultUnit": defaultUnit,
            "timeFormat": timeFormat,
            "workingDaysPerWeek": workingDaysPerWeek,
            "workingHoursPerDay": workingHoursPerDay,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/configuration/timetracking/options"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_custom_field_option(self, id: str) -> dict[str, Any]:
        """
        Retrieves a full representation of a custom field option by its ID using the Jira REST API.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/customFieldOption/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_dashboards(
        self,
        filter: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of Jira dashboards using the GET method, allowing for optional filtering, pagination with start and max results parameters.

        Args:
            filter (string): The filter applied to the list of dashboards. Valid values are: * `favourite` Returns dashboards the user has marked as favorite. * `my` Returns dashboards owned by the user.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        url = f"{self.base_url}/rest/api/3/dashboard"
        query_params = {
            k: v
            for k, v in [
                ("filter", filter),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_dashboard(
        self,
        editPermissions: list[dict[str, Any]],
        name: str,
        sharePermissions: list[dict[str, Any]],
        extendAdminPermissions: bool | None = None,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new dashboard in Jira Cloud using the REST API and returns a response indicating the success or failure of the operation.

        Args:
            editPermissions (array): The edit permissions for the dashboard. Example: [].
            name (string): The name of the dashboard. Example: 'Auditors dashboard'.
            sharePermissions (array): The share permissions for the dashboard. Example: [{'type': 'global'}].
            extendAdminPermissions (boolean): Whether admin level permissions are used. It should only be true if the user has *Administer Jira* [global permission](
            description (string): The description of the dashboard. Example: 'A dashboard to help auditors identify sample of issues to check.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "editPermissions": editPermissions,
            "name": name,
            "sharePermissions": sharePermissions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard"
        query_params = {
            k: v
            for k, v in [("extendAdminPermissions", extendAdminPermissions)]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_edit_dashboards(
        self,
        action: str,
        entityIds: list[int],
        changeOwnerDetails: Any | None = None,
        extendAdminPermissions: bool | None = None,
        permissionDetails: Any | None = None,
    ) -> dict[str, Any]:
        """
        Bulk updates permissions and settings for multiple Jira dashboards in a single operation.

        Args:
            action (string): Allowed action for bulk edit shareable entity Example: 'changePermission'.
            entityIds (array): The id list of shareable entities to be changed. Example: [10001, 10002].
            changeOwnerDetails (string): The details of change owner action.
            extendAdminPermissions (boolean): Whether the actions are executed by users with Administer Jira global permission. Example: True.
            permissionDetails (string): The permission details to be changed. Example: {'editPermissions': [{'group': {'groupId': '276f955c-63d7-42c8-9520-92d01dca0625', 'name': 'jira-administrators', 'self': 'https://your-domain.atlassian.net/rest/api/~ver~/group?groupId=276f955c-63d7-42c8-9520-92d01dca0625'}, 'id': 10010, 'type': 'group'}], 'sharePermissions': [{'id': 10000, 'type': 'global'}]}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        request_body_data = None
        request_body_data = {
            "action": action,
            "changeOwnerDetails": changeOwnerDetails,
            "entityIds": entityIds,
            "extendAdminPermissions": extendAdminPermissions,
            "permissionDetails": permissionDetails,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard/bulk/edit"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_gadgets(self) -> dict[str, Any]:
        """
        Retrieves a list of available gadgets that can be added to Jira dashboards using the Jira Cloud API.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        url = f"{self.base_url}/rest/api/3/dashboard/gadgets"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_dashboards_paginated(
        self,
        dashboardName: str | None = None,
        accountId: str | None = None,
        owner: str | None = None,
        groupname: str | None = None,
        groupId: str | None = None,
        projectId: int | None = None,
        orderBy: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        status: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Searches for Jira dashboards using specified criteria, such as dashboard name, account ID, owner, group name, group ID, project ID, and other parameters, and returns a list of matching dashboards.

        Args:
            dashboardName (string): String used to perform a case-insensitive partial match with `name`.
            accountId (string): User account ID used to return dashboards with the matching `owner.accountId`. This parameter cannot be used with the `owner` parameter.
            owner (string): This parameter is deprecated because of privacy changes. Use `accountId` instead. See the [migration guide]( for details. User name used to return dashboards with the matching `owner.name`. This parameter cannot be used with the `accountId` parameter.
            groupname (string): As a group's name can change, use of `groupId` is recommended. Group name used to return dashboards that are shared with a group that matches `sharePermissions.group.name`. This parameter cannot be used with the `groupId` parameter.
            groupId (string): Group ID used to return dashboards that are shared with a group that matches `sharePermissions.group.groupId`. This parameter cannot be used with the `groupname` parameter.
            projectId (integer): Project ID used to returns dashboards that are shared with a project that matches `sharePermissions.project.id`.
            orderBy (string): [Order](#ordering) the results by a field: * `description` Sorts by dashboard description. Note that this sort works independently of whether the expand to display the description field is in use. * `favourite_count` Sorts by dashboard popularity. * `id` Sorts by dashboard ID. * `is_favourite` Sorts by whether the dashboard is marked as a favorite. * `name` Sorts by dashboard name. * `owner` Sorts by dashboard owner name.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            status (string): The status to filter by. It may be active, archived or deleted.
            expand (string): Use [expand](#expansion) to include additional information about dashboard in the response. This parameter accepts a comma-separated list. Expand options include: * `description` Returns the description of the dashboard. * `owner` Returns the owner of the dashboard. * `viewUrl` Returns the URL that is used to view the dashboard. * `favourite` Returns `isFavourite`, an indicator of whether the user has set the dashboard as a favorite. * `favouritedCount` Returns `popularity`, a count of how many users have set this dashboard as a favorite. * `sharePermissions` Returns details of the share permissions defined for the dashboard. * `editPermissions` Returns details of the edit permissions defined for the dashboard. * `isWritable` Returns whether the current user has permission to edit the dashboard.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        url = f"{self.base_url}/rest/api/3/dashboard/search"
        query_params = {
            k: v
            for k, v in [
                ("dashboardName", dashboardName),
                ("accountId", accountId),
                ("owner", owner),
                ("groupname", groupname),
                ("groupId", groupId),
                ("projectId", projectId),
                ("orderBy", orderBy),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("status", status),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_gadgets(
        self,
        dashboardId: str,
        moduleKey: list[str] | None = None,
        uri: list[str] | None = None,
        gadgetId: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a specific gadget or all gadgets from a Jira dashboard.

        Args:
            dashboardId (string): dashboardId
            moduleKey (array): The list of gadgets module keys. To include multiple module keys, separate module keys with ampersand: `moduleKey=key:one&moduleKey=key:two`.
            uri (array): The list of gadgets URIs. To include multiple URIs, separate URIs with ampersand: `uri=/rest/example/uri/1&uri=/rest/example/uri/2`.
            gadgetId (array): The list of gadgets IDs. To include multiple IDs, separate IDs with ampersand: `gadgetId=10000&gadgetId=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/gadget"
        query_params = {
            k: v
            for k, v in [("moduleKey", moduleKey), ("uri", uri), ("gadgetId", gadgetId)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_gadget(
        self,
        dashboardId: str,
        color: str | None = None,
        ignoreUriAndModuleKeyValidation: bool | None = None,
        moduleKey: str | None = None,
        position: Any | None = None,
        title: str | None = None,
        uri: str | None = None,
    ) -> dict[str, Any]:
        """
        Adds a gadget to a specified Jira dashboard using the provided configuration and returns the created gadget details upon success.

        Args:
            dashboardId (string): dashboardId
            color (string): The color of the gadget. Should be one of `blue`, `red`, `yellow`, `green`, `cyan`, `purple`, `gray`, or `white`. Example: 'blue'.
            ignoreUriAndModuleKeyValidation (boolean): Whether to ignore the validation of module key and URI. For example, when a gadget is created that is a part of an application that isn't installed. Example: False.
            moduleKey (string): The module key of the gadget type. Can't be provided with `uri`. Example: 'com.atlassian.plugins.atlassian-connect-plugin:com.atlassian.connect.node.sample-addon__sample-dashboard-item'.
            position (string): The position of the gadget. When the gadget is placed into the position, other gadgets in the same column are moved down to accommodate it. Example: {'column': 1, 'row': 0}.
            title (string): The title of the gadget. Example: 'Issue statistics'.
            uri (string): The URI of the gadget type. Can't be provided with `moduleKey`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        request_body_data = None
        request_body_data = {
            "color": color,
            "ignoreUriAndModuleKeyValidation": ignoreUriAndModuleKeyValidation,
            "moduleKey": moduleKey,
            "position": position,
            "title": title,
            "uri": uri,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/gadget"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_gadget(self, dashboardId: str, gadgetId: str) -> Any:
        """
        Removes a specific gadget from a Jira Cloud dashboard using the DELETE method, where other gadgets in the same column are automatically moved up to fill the emptied position.

        Args:
            dashboardId (string): dashboardId
            gadgetId (string): gadgetId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if gadgetId is None:
            raise ValueError("Missing required parameter 'gadgetId'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/gadget/{gadgetId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_gadget(
        self,
        dashboardId: str,
        gadgetId: str,
        color: str | None = None,
        position: Any | None = None,
        title: str | None = None,
    ) -> Any:
        """
        Updates a gadget's configuration (e.g., color, position, title) on a specified Jira dashboard.

        Args:
            dashboardId (string): dashboardId
            gadgetId (string): gadgetId
            color (string): The color of the gadget. Should be one of `blue`, `red`, `yellow`, `green`, `cyan`, `purple`, `gray`, or `white`. Example: 'red'.
            position (string): The position of the gadget. Example: {'column': 1, 'row': 1}.
            title (string): The title of the gadget. Example: 'My new gadget title'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if gadgetId is None:
            raise ValueError("Missing required parameter 'gadgetId'.")
        request_body_data = None
        request_body_data = {
            "color": color,
            "position": position,
            "title": title,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/gadget/{gadgetId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_dashboard_item_property_keys(
        self, dashboardId: str, itemId: str
    ) -> dict[str, Any]:
        """
        Retrieves all property keys for a specific dashboard item in Jira using provided dashboard and item IDs.

        Args:
            dashboardId (string): dashboardId
            itemId (string): itemId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if itemId is None:
            raise ValueError("Missing required parameter 'itemId'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/items/{itemId}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_dashboard_item_property(
        self, dashboardId: str, itemId: str, propertyKey: str
    ) -> Any:
        """
        Deletes a dashboard item property (identified by propertyKey) from a specific dashboard item, accessible anonymously but requiring dashboard ownership for successful deletion.

        Args:
            dashboardId (string): dashboardId
            itemId (string): itemId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the dashboard item property is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if itemId is None:
            raise ValueError("Missing required parameter 'itemId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/items/{itemId}/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_dashboard_item_property(
        self, dashboardId: str, itemId: str, propertyKey: str
    ) -> dict[str, Any]:
        """
        Retrieves a specific property of a dashboard item using the Jira API, returning the property value associated with the given dashboard ID, item ID, and property key.

        Args:
            dashboardId (string): dashboardId
            itemId (string): itemId
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if itemId is None:
            raise ValueError("Missing required parameter 'itemId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/items/{itemId}/properties/{propertyKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_dashboard_item_property(
        self, dashboardId: str, itemId: str, propertyKey: str
    ) -> Any:
        """
        Updates or creates a specific property for an item on a dashboard using the PUT method, identified by dashboardId, itemId, and propertyKey, returning success or creation status.

        Args:
            dashboardId (string): dashboardId
            itemId (string): itemId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the dashboard item property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if dashboardId is None:
            raise ValueError("Missing required parameter 'dashboardId'.")
        if itemId is None:
            raise ValueError("Missing required parameter 'itemId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/dashboard/{dashboardId}/items/{itemId}/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_dashboard(self, id: str) -> Any:
        """
        Deletes a dashboard identified by its ID using the "DELETE" method and returns a successful status if the operation is completed without errors.

        Args:
            id (string): id

        Returns:
            Any: Returned if the dashboard is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_dashboard(self, id: str) -> dict[str, Any]:
        """
        Retrieves the details of a specific Jira dashboard by ID.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/dashboard/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_dashboard(
        self,
        id: str,
        editPermissions: list[dict[str, Any]],
        name: str,
        sharePermissions: list[dict[str, Any]],
        extendAdminPermissions: bool | None = None,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates a dashboard with the specified ID using the PUT method, optionally extending admin permissions, and returns a status message if successful.

        Args:
            id (string): id
            editPermissions (array): The edit permissions for the dashboard. Example: [].
            name (string): The name of the dashboard. Example: 'Auditors dashboard'.
            sharePermissions (array): The share permissions for the dashboard. Example: [{'type': 'global'}].
            extendAdminPermissions (boolean): Whether admin level permissions are used. It should only be true if the user has *Administer Jira* [global permission](
            description (string): The description of the dashboard. Example: 'A dashboard to help auditors identify sample of issues to check.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "editPermissions": editPermissions,
            "name": name,
            "sharePermissions": sharePermissions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard/{id}"
        query_params = {
            k: v
            for k, v in [("extendAdminPermissions", extendAdminPermissions)]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def copy_dashboard(
        self,
        id: str,
        editPermissions: list[dict[str, Any]],
        name: str,
        sharePermissions: list[dict[str, Any]],
        extendAdminPermissions: bool | None = None,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Copies a dashboard and replaces specified parameters in the copied dashboard, returning the new dashboard details.

        Args:
            id (string): id
            editPermissions (array): The edit permissions for the dashboard. Example: [].
            name (string): The name of the dashboard. Example: 'Auditors dashboard'.
            sharePermissions (array): The share permissions for the dashboard. Example: [{'type': 'global'}].
            extendAdminPermissions (boolean): Whether admin level permissions are used. It should only be true if the user has *Administer Jira* [global permission](
            description (string): The description of the dashboard. Example: 'A dashboard to help auditors identify sample of issues to check.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Dashboards
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "editPermissions": editPermissions,
            "name": name,
            "sharePermissions": sharePermissions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/dashboard/{id}/copy"
        query_params = {
            k: v
            for k, v in [("extendAdminPermissions", extendAdminPermissions)]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_policy(self) -> dict[str, Any]:
        """
        Retrieves details about the data policy for a workspace using the Jira Cloud REST API, returning information on whether data policies are enabled for the workspace.

        Returns:
            dict[str, Any]: Returned if the request is successful

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            App data policies
        """
        url = f"{self.base_url}/rest/api/3/data-policy"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_policies(self, ids: str | None = None) -> dict[str, Any]:
        """
        Retrieves data policies affecting specific projects in Jira using the "/rest/api/3/data-policy/project" endpoint, returning details about which projects are impacted by data security policies.

        Args:
            ids (string): A list of project identifiers. This parameter accepts a comma-separated list.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            App data policies
        """
        url = f"{self.base_url}/rest/api/3/data-policy/project"
        query_params = {k: v for k, v in [("ids", ids)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_events(self) -> list[Any]:
        """
        Retrieves a list of events using the Jira Cloud API by sending a GET request to "/rest/api/3/events," providing a paginated response based on the specified parameters.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        url = f"{self.base_url}/rest/api/3/events"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def analyse_expression(
        self,
        expressions: list[str],
        check: str | None = None,
        contextVariables: dict[str, str] | None = None,
    ) -> dict[str, Any]:
        """
        Analyzes a Jira expression to statically check its characteristics, such as complexity, without evaluating it, using a POST method at "/rest/api/3/expression/analyse" with the option to specify what to check via a query parameter.

        Args:
            expressions (array): The list of Jira expressions to analyse. Example: "issues.map(issue => issue.properties['property_key'])".
            check (string): The check to perform: * `syntax` Each expression's syntax is checked to ensure the expression can be parsed. Also, syntactic limits are validated. For example, the expression's length. * `type` EXPERIMENTAL. Each expression is type checked and the final type of the expression inferred. Any type errors that would result in the expression failure at runtime are reported. For example, accessing properties that don't exist or passing the wrong number of arguments to functions. Also performs the syntax check. * `complexity` EXPERIMENTAL. Determines the formulae for how many [expensive operations]( each expression may execute.
            contextVariables (object): Context variables and their types. The type checker assumes that [common context variables](https://developer.atlassian.com/cloud/jira/platform/jira-expressions/#context-variables), such as `issue` or `project`, are available in context and sets their type. Use this property to override the default types or provide details of new variables. Example: {'listOfStrings': 'List<String>', 'record': '{ a: Number, b: String }', 'value': 'User'}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira expressions
        """
        request_body_data = None
        request_body_data = {
            "contextVariables": contextVariables,
            "expressions": expressions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/expression/analyse"
        query_params = {k: v for k, v in [("check", check)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def evaluate_jira_expression(
        self,
        expression: str,
        expand: str | None = None,
        context: Any | None = None,
    ) -> dict[str, Any]:
        """
        Evaluates Jira expressions using an enhanced search API for scalable processing of JQL queries and returns primitive values, lists, or objects.

        Args:
            expression (string): The Jira expression to evaluate. Example: '{ key: issue.key, type: issue.issueType.name, links: issue.links.map(link => link.linkedIssue.id) }'.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts `meta.complexity` that returns information about the expression complexity. For example, the number of expensive operations used by the expression and how close the expression is to reaching the [complexity limit]( Useful when designing and debugging your expressions.
            context (string): The context in which the Jira expression is evaluated. Example: {'board': 10100, 'custom': {'config': {'type': 'json', 'value': {'userId': '10002'}}, 'issuesList': [{'key': 'ACJIRA-1471', 'type': 'issue'}, {'id': 100001, 'type': 'issue'}], 'myUser': {'accountId': '100001', 'type': 'user'}, 'nullField': {'type': 'json'}}, 'customerRequest': 1450, 'issue': {'key': 'ACJIRA-1470'}, 'issues': {'jql': {'maxResults': 100, 'query': 'project = HSP', 'startAt': 0, 'validation': 'strict'}}, 'project': {'key': 'ACJIRA'}, 'serviceDesk': 10023, 'sprint': 10001}.

        Returns:
            dict[str, Any]: Returned if the evaluation results in a value. The result is a JSON primitive value, list, or object.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira expressions
        """
        request_body_data = None
        request_body_data = {
            "context": context,
            "expression": expression,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/expression/eval"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def evaluate_jsisjira_expression(
        self,
        expression: str,
        expand: str | None = None,
        context: Any | None = None,
    ) -> dict[str, Any]:
        """
        Evaluates Jira expressions using the enhanced search API with support for pagination and eventually consistent JQL queries, returning primitive values, lists, or objects.

        Args:
            expression (string): The Jira expression to evaluate. Example: '{ key: issue.key, type: issue.issueType.name, links: issue.links.map(link => link.linkedIssue.id) }'.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts `meta.complexity` that returns information about the expression complexity. For example, the number of expensive operations used by the expression and how close the expression is to reaching the [complexity limit]( Useful when designing and debugging your expressions.
            context (string): The context in which the Jira expression is evaluated. Example: {'board': 10100, 'custom': {'config': {'type': 'json', 'value': {'userId': '10002'}}, 'issuesList': [{'key': 'ACJIRA-1471', 'type': 'issue'}, {'id': 100001, 'type': 'issue'}], 'myUser': {'accountId': '100001', 'type': 'user'}, 'nullField': {'type': 'json'}}, 'customerRequest': 1450, 'issue': {'key': 'ACJIRA-1470'}, 'issues': {'jql': {'maxResults': 100, 'nextPageToken': 'EgQIlMIC', 'query': 'project = HSP'}}, 'project': {'key': 'ACJIRA'}, 'serviceDesk': 10023, 'sprint': 10001}.

        Returns:
            dict[str, Any]: Returned if the evaluation results in a value. The result is a JSON primitive value, list, or object.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Jira expressions
        """
        request_body_data = None
        request_body_data = {
            "context": context,
            "expression": expression,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/expression/evaluate"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_fields(self) -> list[Any]:
        """
        Retrieves a list of available fields in Jira using the GET method at the "/rest/api/3/field" path.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        url = f"{self.base_url}/rest/api/3/field"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_custom_field(
        self,
        name: str,
        type: str,
        description: str | None = None,
        searcherKey: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a custom field using a definition and returns a successful creation message when the operation is completed successfully.

        Args:
            name (string): The name of the custom field, which is displayed in Jira. This is not the unique identifier. Example: 'New custom field'.
            type (string): The type of the custom field. These built-in custom field types are available:

         *  `cascadingselect`: Enables values to be selected from two levels of select lists (value: `com.atlassian.jira.plugin.system.customfieldtypes:cascadingselect`)
         *  `datepicker`: Stores a date using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:datepicker`)
         *  `datetime`: Stores a date with a time component (value: `com.atlassian.jira.plugin.system.customfieldtypes:datetime`)
         *  `float`: Stores and validates a numeric (floating point) input (value: `com.atlassian.jira.plugin.system.customfieldtypes:float`)
         *  `grouppicker`: Stores a user group using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:grouppicker`)
         *  `importid`: A read-only field that stores the ID the issue had in the system it was imported from (value: `com.atlassian.jira.plugin.system.customfieldtypes:importid`)
         *  `labels`: Stores labels (value: `com.atlassian.jira.plugin.system.customfieldtypes:labels`)
         *  `multicheckboxes`: Stores multiple values using checkboxes (value: ``)
         *  `multigrouppicker`: Stores multiple user groups using a picker control (value: ``)
         *  `multiselect`: Stores multiple values using a select list (value: `com.atlassian.jira.plugin.system.customfieldtypes:multicheckboxes`)
         *  `multiuserpicker`: Stores multiple users using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:multigrouppicker`)
         *  `multiversion`: Stores multiple versions from the versions available in a project using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:multiversion`)
         *  `project`: Stores a project from a list of projects that the user is permitted to view (value: `com.atlassian.jira.plugin.system.customfieldtypes:project`)
         *  `radiobuttons`: Stores a value using radio buttons (value: `com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons`)
         *  `readonlyfield`: Stores a read-only text value, which can only be populated via the API (value: `com.atlassian.jira.plugin.system.customfieldtypes:readonlyfield`)
         *  `select`: Stores a value from a configurable list of options (value: `com.atlassian.jira.plugin.system.customfieldtypes:select`)
         *  `textarea`: Stores a long text string using a multiline text area (value: `com.atlassian.jira.plugin.system.customfieldtypes:textarea`)
         *  `textfield`: Stores a text string using a single-line text box (value: `com.atlassian.jira.plugin.system.customfieldtypes:textfield`)
         *  `url`: Stores a URL (value: `com.atlassian.jira.plugin.system.customfieldtypes:url`)
         *  `userpicker`: Stores a user using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:userpicker`)
         *  `version`: Stores a version using a picker control (value: `com.atlassian.jira.plugin.system.customfieldtypes:version`)

        To create a field based on a [Forge custom field type](https://developer.atlassian.com/platform/forge/manifest-reference/modules/#jira-custom-field-type--beta-), use the ID of the Forge custom field type as the value. For example, `ari:cloud:ecosystem::extension/e62f20a2-4b61-4dbe-bfb9-9a88b5e3ac84/548c5df1-24aa-4f7c-bbbb-3038d947cb05/static/my-cf-type-key`. Example: 'com.atlassian.jira.plugin.system.customfieldtypes:grouppicker'.
            description (string): The description of the custom field, which is displayed in Jira. Example: 'Custom field for picking groups'.
            searcherKey (string): The searcher defines the way the field is searched in Jira. For example, *com.atlassian.jira.plugin.system.customfieldtypes:grouppickersearcher*.
        The search UI (basic search and JQL search) will display different operations and values for the field, based on the field searcher. You must specify a searcher that is valid for the field type, as listed below (abbreviated values shown):

         *  `cascadingselect`: `cascadingselectsearcher`
         *  `datepicker`: `daterange`
         *  `datetime`: `datetimerange`
         *  `float`: `exactnumber` or `numberrange`
         *  `grouppicker`: `grouppickersearcher`
         *  `importid`: `exactnumber` or `numberrange`
         *  `labels`: `labelsearcher`
         *  `multicheckboxes`: `multiselectsearcher`
         *  `multigrouppicker`: `multiselectsearcher`
         *  `multiselect`: `multiselectsearcher`
         *  `multiuserpicker`: `userpickergroupsearcher`
         *  `multiversion`: `versionsearcher`
         *  `project`: `projectsearcher`
         *  `radiobuttons`: `multiselectsearcher`
         *  `readonlyfield`: `textsearcher`
         *  `select`: `multiselectsearcher`
         *  `textarea`: `textsearcher`
         *  `textfield`: `textsearcher`
         *  `url`: `exacttextsearcher`
         *  `userpicker`: `userpickergroupsearcher`
         *  `version`: `versionsearcher`

        If no searcher is provided, the field isn't searchable. However, [Forge custom fields](https://developer.atlassian.com/platform/forge/manifest-reference/modules/#jira-custom-field-type--beta-) have a searcher set automatically, so are always searchable. Example: 'com.atlassian.jira.plugin.system.customfieldtypes:grouppickersearcher'.

        Returns:
            dict[str, Any]: Returned if the custom field is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
            "searcherKey": searcherKey,
            "type": type,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_associations(
        self, associationContexts: list[dict[str, Any]], fields: list[dict[str, Any]]
    ) -> Any:
        """
        Deletes an association between a field and its related entities, returning response codes for success or error conditions.

        Args:
            associationContexts (array): Contexts to associate/unassociate the fields with. Example: [{'identifier': 10000, 'type': 'PROJECT_ID'}, {'identifier': 10001, 'type': 'PROJECT_ID'}].
            fields (array): Fields to associate/unassociate with projects. Example: [{'identifier': 'customfield_10000', 'type': 'FIELD_ID'}, {'identifier': 'customfield_10001', 'type': 'FIELD_ID'}].

        Returns:
            Any: Returned if the field association validation passes.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field associations
        """
        request_body_data = {
            "associationContexts": associationContexts,
            "fields": fields,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/association"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_associations(
        self, associationContexts: list[dict[str, Any]], fields: list[dict[str, Any]]
    ) -> Any:
        """
        Associates or unassociates custom fields with projects and issue types in Jira using the PUT method.

        Args:
            associationContexts (array): Contexts to associate/unassociate the fields with. Example: [{'identifier': 10000, 'type': 'PROJECT_ID'}, {'identifier': 10001, 'type': 'PROJECT_ID'}].
            fields (array): Fields to associate/unassociate with projects. Example: [{'identifier': 'customfield_10000', 'type': 'FIELD_ID'}, {'identifier': 'customfield_10001', 'type': 'FIELD_ID'}].

        Returns:
            Any: Returned if the field association validation passes.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field associations
        """
        request_body_data = None
        request_body_data = {
            "associationContexts": associationContexts,
            "fields": fields,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/association"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_fields_paginated(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        type: list[str] | None = None,
        id: list[str] | None = None,
        query: str | None = None,
        orderBy: str | None = None,
        expand: str | None = None,
        projectIds: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves and filters Jira fields by criteria such as ID, type, or name, supporting pagination and field expansion.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            type (array): The type of fields to search.
            id (array): The IDs of the custom fields to return or, where `query` is specified, filter.
            query (string): String used to perform a case-insensitive partial match with field names or descriptions.
            orderBy (string): [Order](#ordering) the results by: * `contextsCount` sorts by the number of contexts related to a field * `lastUsed` sorts by the date when the value of the field last changed * `name` sorts by the field name * `screensCount` sorts by the number of screens related to a field
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `key` returns the key for each field * `stableId` returns the stableId for each field * `lastUsed` returns the date when the value of the field last changed * `screensCount` returns the number of screens related to a field * `contextsCount` returns the number of contexts related to a field * `isLocked` returns information about whether the field is locked * `searcherKey` returns the searcher key for each custom field
            projectIds (array): Comma-separated list of project IDs to filter the field search results by, restricting them to fields used in the specified projects.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        url = f"{self.base_url}/rest/api/3/field/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("type", type),
                ("id", id),
                ("query", query),
                ("orderBy", orderBy),
                ("expand", expand),
                ("projectIds", projectIds),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_trashed_fields_paginated(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[str] | None = None,
        query: str | None = None,
        expand: str | None = None,
        orderBy: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of custom fields that have been trashed in Jira, allowing filtering by field ID, name, or description.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): **id**: The parameter used to filter trashed fields by their unique identifier.
            query (string): String used to perform a case-insensitive partial match with field names or descriptions.
            expand (string): The "expand" parameter allows you to specify which additional fields or entities of the response should be expanded with more detailed information, using a comma-separated list of entity names.
            orderBy (string): [Order](#ordering) the results by a field: * `name` sorts by the field name * `trashDate` sorts by the date the field was moved to the trash * `plannedDeletionDate` sorts by the planned deletion date

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        url = f"{self.base_url}/rest/api/3/field/search/trashed"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("query", query),
                ("expand", expand),
                ("orderBy", orderBy),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_custom_field(
        self,
        fieldId: str,
        description: str | None = None,
        name: str | None = None,
        searcherKey: str | None = None,
    ) -> Any:
        """
        Updates a Jira custom field using the `PUT` method, specifying the field ID in the path, and returns a status response upon successful modification.

        Args:
            fieldId (string): fieldId
            description (string): The description of the custom field. The maximum length is 40000 characters. Example: 'Select the manager and the corresponding employee.'.
            name (string): The name of the custom field. It doesn't have to be unique. The maximum length is 255 characters. Example: 'Managers and employees list'.
            searcherKey (string): The searcher that defines the way the field is searched in Jira. It can be set to `null`, otherwise you must specify the valid searcher for the field type, as listed below (abbreviated values shown):

         *  `cascadingselect`: `cascadingselectsearcher`
         *  `datepicker`: `daterange`
         *  `datetime`: `datetimerange`
         *  `float`: `exactnumber` or `numberrange`
         *  `grouppicker`: `grouppickersearcher`
         *  `importid`: `exactnumber` or `numberrange`
         *  `labels`: `labelsearcher`
         *  `multicheckboxes`: `multiselectsearcher`
         *  `multigrouppicker`: `multiselectsearcher`
         *  `multiselect`: `multiselectsearcher`
         *  `multiuserpicker`: `userpickergroupsearcher`
         *  `multiversion`: `versionsearcher`
         *  `project`: `projectsearcher`
         *  `radiobuttons`: `multiselectsearcher`
         *  `readonlyfield`: `textsearcher`
         *  `select`: `multiselectsearcher`
         *  `textarea`: `textsearcher`
         *  `textfield`: `textsearcher`
         *  `url`: `exacttextsearcher`
         *  `userpicker`: `userpickergroupsearcher`
         *  `version`: `versionsearcher` Example: 'com.atlassian.jira.plugin.system.customfieldtypes:cascadingselectsearcher'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
            "searcherKey": searcherKey,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_contexts_for_field(
        self,
        fieldId: str,
        isAnyIssueType: bool | None = None,
        isGlobalContext: bool | None = None,
        contextId: list[int] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of custom field contexts for a specified field in Jira using the path "/rest/api/3/field/{fieldId}/context" with optional filters for issue type and global context.

        Args:
            fieldId (string): fieldId
            isAnyIssueType (boolean): Whether to return contexts that apply to all issue types.
            isGlobalContext (boolean): Whether to return contexts that apply to all projects.
            contextId (array): The list of context IDs. To include multiple contexts, separate IDs with ampersand: `contextId=10000&contextId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context"
        query_params = {
            k: v
            for k, v in [
                ("isAnyIssueType", isAnyIssueType),
                ("isGlobalContext", isGlobalContext),
                ("contextId", contextId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_custom_field_context(
        self,
        fieldId: str,
        name: str,
        description: str | None = None,
        id: str | None = None,
        issueTypeIds: list[str] | None = None,
        projectIds: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new custom field context for the specified field ID, defining its project and issue type associations.

        Args:
            fieldId (string): fieldId
            name (string): The name of the context. Example: 'Bug fields context'.
            description (string): The description of the context. Example: 'A context used to define the custom field options for bugs.'.
            id (string): The ID of the context.
            issueTypeIds (array): The list of issue types IDs for the context. If the list is empty, the context refers to all issue types. Example: ['10010'].
            projectIds (array): The list of project IDs associated with the context. If the list is empty, the context is global. Example: [].

        Returns:
            dict[str, Any]: Returned if the custom field context is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "id": id,
            "issueTypeIds": issueTypeIds,
            "name": name,
            "projectIds": projectIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_default_values(
        self,
        fieldId: str,
        contextId: list[int] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the default values for contexts of a specified custom field in Jira, including optional pagination parameters for larger datasets.

        Args:
            fieldId (string): fieldId
            contextId (array): The IDs of the contexts.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/defaultValue"
        query_params = {
            k: v
            for k, v in [
                ("contextId", contextId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_default_values(
        self, fieldId: str, defaultValues: list[dict[str, Any]] | None = None
    ) -> Any:
        """
        Sets the default value for a specified custom field context via the Jira REST API.

        Args:
            fieldId (string): fieldId
            defaultValues (array): defaultValues Example: [{'contextId': '10100', 'optionId': '10001', 'type': 'option.single'}, {'contextId': '10101', 'optionId': '10003', 'type': 'option.single'}, {'contextId': '10103', 'optionId': '10005', 'type': 'option.single'}].

        Returns:
            Any: Returned if operation is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        request_body_data = None
        request_body_data = {
            "defaultValues": defaultValues,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/defaultValue"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_field_issue_type_mappings(
        self,
        fieldId: str,
        contextId: list[int] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of context to issue type mappings for a specified custom field using the Jira Cloud API, allowing for filters by context ID, start index, and maximum results.

        Args:
            fieldId (string): fieldId
            contextId (array): The ID of the context. To include multiple contexts, provide an ampersand-separated list. For example, `contextId=10001&contextId=10002`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if operation is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/issuetypemapping"
        query_params = {
            k: v
            for k, v in [
                ("contextId", contextId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def post_field_context_mapping(
        self,
        fieldId: str,
        mappings: list[dict[str, Any]],
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves and maps custom field contexts to specific projects and issue types for a given custom field ID.

        Args:
            fieldId (string): fieldId
            mappings (array): The project and issue type mappings. Example: [{'issueTypeId': '10000', 'projectId': '10000'}, {'issueTypeId': '10001', 'projectId': '10000'}, {'issueTypeId': '10002', 'projectId': '10001'}].
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        request_body_data = None
        request_body_data = {
            "mappings": mappings,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/mapping"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_context_mapping(
        self,
        fieldId: str,
        contextId: list[int] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves paginated mappings between projects and custom field contexts, optionally filtered by context ID.

        Args:
            fieldId (string): fieldId
            contextId (array): The list of context IDs. To include multiple context, separate IDs with ampersand: `contextId=10000&contextId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/projectmapping"
        query_params = {
            k: v
            for k, v in [
                ("contextId", contextId),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_custom_field_context(self, fieldId: str, contextId: str) -> Any:
        """
        Deletes a custom field context in Jira using the provided `fieldId` and `contextId`, removing it from the system.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId

        Returns:
            Any: Returned if the context is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_custom_field_context(
        self,
        fieldId: str,
        contextId: str,
        description: str | None = None,
        name: str | None = None,
    ) -> Any:
        """
        Updates a custom field context's configuration in Jira, including associated projects and issue types.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            description (string): The description of the custom field context. The maximum length is 255 characters. Example: 'A context used to define the custom field options for bugs.'.
            name (string): The name of the custom field context. The name must be unique. The maximum length is 255 characters. Example: 'Bug fields context'.

        Returns:
            Any: Returned if the context is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_issue_types_to_context(
        self, fieldId: str, contextId: str, issueTypeIds: list[str]
    ) -> Any:
        """
        Updates the issue type associations for a specific custom field context in Jira using the specified field and context IDs.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            issueTypeIds (array): The list of issue type IDs. Example: ['10001', '10005', '10006'].

        Returns:
            Any: Returned if operation is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "issueTypeIds": issueTypeIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = (
            f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/issuetype"
        )
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_issue_types_from_context(
        self, fieldId: str, contextId: str, issueTypeIds: list[str]
    ) -> Any:
        """
        Removes issue types from a custom field context in Jira, reverting them to apply to all issue types if none remain.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            issueTypeIds (array): The list of issue type IDs. Example: ['10001', '10005', '10006'].

        Returns:
            Any: Returned if operation is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "issueTypeIds": issueTypeIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/issuetype/remove"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_options_for_context(
        self,
        fieldId: str,
        contextId: str,
        optionId: int | None = None,
        onlyOptions: bool | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves and paginates custom field options (single/multiple choice values) for a specific field and context in Jira, including filtering by option ID.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            optionId (integer): The ID of the option.
            onlyOptions (boolean): Whether only options are returned.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option"
        query_params = {
            k: v
            for k, v in [
                ("optionId", optionId),
                ("onlyOptions", onlyOptions),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_custom_field_option(
        self,
        fieldId: str,
        contextId: str,
        options: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Creates new custom field options for a specific context in Jira using the POST method, allowing for the addition of options to select lists or similar fields.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            options (array): Details of options to create. Example: [{'disabled': False, 'value': 'Scranton'}, {'disabled': True, 'optionId': '10000', 'value': 'Manhattan'}, {'disabled': False, 'value': 'The Electric City'}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "options": options,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_custom_field_option(
        self,
        fieldId: str,
        contextId: str,
        options: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Updates options for a custom field context in Jira using the provided field and context IDs, but this endpoint is not explicitly documented for a PUT method; typically, such endpoints involve updating or adding options to a select field within a specific context.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            options (array): Details of the options to update. Example: [{'disabled': False, 'id': '10001', 'value': 'Scranton'}, {'disabled': True, 'id': '10002', 'value': 'Manhattan'}, {'disabled': False, 'id': '10003', 'value': 'The Electric City'}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "options": options,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def reorder_custom_field_options(
        self,
        fieldId: str,
        contextId: str,
        customFieldOptionIds: list[str],
        after: str | None = None,
        position: str | None = None,
    ) -> Any:
        """
        Reorders custom field options or cascading options within a specified context using the provided IDs and position parameters.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            customFieldOptionIds (array): A list of IDs of custom field options to move. The order of the custom field option IDs in the list is the order they are given after the move. The list must contain custom field options or cascading options, but not both. Example: ['10001', '10002'].
            after (string): The ID of the custom field option or cascading option to place the moved options after. Required if `position` isn't provided.
            position (string): The position the custom field options should be moved to. Required if `after` isn't provided. Example: 'First'.

        Returns:
            Any: Returned if options are reordered.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "after": after,
            "customFieldOptionIds": customFieldOptionIds,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option/move"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_custom_field_option(
        self, fieldId: str, contextId: str, optionId: str
    ) -> Any:
        """
        Deletes a specific custom field option within a designated custom field context in Jira.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            optionId (string): optionId

        Returns:
            Any: Returned if the option is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option/{optionId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def replace_custom_field_option(
        self,
        fieldId: str,
        contextId: str,
        optionId: str,
        replaceWith: int | None = None,
        jql: str | None = None,
    ) -> Any:
        """
        Deletes a specific custom field option within a context for a Jira field, allowing optional replacement via query parameters.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            optionId (string): optionId
            replaceWith (integer): The ID of the option that will replace the currently selected option.
            jql (string): A JQL query that specifies the issues to be updated. For example, *project=10000*.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/option/{optionId}/issue"
        query_params = {
            k: v
            for k, v in [("replaceWith", replaceWith), ("jql", jql)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def assign_project_field_context(
        self, fieldId: str, contextId: str, projectIds: list[str]
    ) -> Any:
        """
        Updates a custom field context by adding a project to it, using the Jira Cloud platform REST API, and returns a status message based on the operation's success or failure.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            projectIds (array): The IDs of projects. Example: ['10001', '10005', '10006'].

        Returns:
            Any: Returned if operation is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "projectIds": projectIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/project"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_project_from_field_context(
        self, fieldId: str, contextId: str, projectIds: list[str]
    ) -> Any:
        """
        Removes specified projects from a custom field context in Jira, causing it to apply to all projects if no projects remain.

        Args:
            fieldId (string): fieldId
            contextId (string): contextId
            projectIds (array): The IDs of projects. Example: ['10001', '10005', '10006'].

        Returns:
            Any: Returned if the custom field context is removed from the projects.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field contexts
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        if contextId is None:
            raise ValueError("Missing required parameter 'contextId'.")
        request_body_data = None
        request_body_data = {
            "projectIds": projectIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/context/{contextId}/project/remove"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_contexts_for_field_deprecated(
        self,
        fieldId: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of contexts for a specified custom field in Jira, allowing filtering by start index and maximum number of results.

        Args:
            fieldId (string): fieldId
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/contexts"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_screens_for_field(
        self,
        fieldId: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of screens that include a specified field, identified by the `fieldId` parameter, allowing for pagination and expansion of results.

        Args:
            fieldId (string): fieldId
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            expand (string): Use [expand](#expansion) to include additional information about screens in the response. This parameter accepts `tab` which returns details about the screen tabs the field is used in.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldId}/screens"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_issue_field_options(
        self,
        fieldKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of all custom field options for a specified field, supporting pagination via startAt and maxResults parameters.

        Args:
            fieldKey (string): fieldKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_field_option(
        self,
        fieldKey: str,
        value: str,
        config: dict[str, Any] | None = None,
        properties: dict[str, Any] | None = None,
    ) -> dict[str, Any]:
        """
        Adds a new option to a specified custom field in Jira using the POST method, requiring the field's key as a path parameter.

        Args:
            fieldKey (string): fieldKey
            value (string): The option's name, which is displayed in Jira. Example: 'Team 1'.
            config (object): Details of the projects the option is available in. Example: {'attributes': [], 'scope': {'global': {}, 'projects': [], 'projects2': [{'attributes': ['notSelectable'], 'id': 1001}, {'attributes': ['notSelectable'], 'id': 1002}]}}.
            properties (object): The properties of the option as arbitrary key-value pairs. These properties can be searched using JQL, if the extractions (see https://developer.atlassian.com/cloud/jira/platform/modules/issue-field-option-property-index/) are defined in the descriptor for the issue field module. Example: {'description': "The team's description", 'founded': '2016-06-06', 'leader': {'email': 'lname@example.com', 'name': 'Leader Name'}, 'members': 42}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        request_body_data = None
        request_body_data = {
            "config": config,
            "properties": properties,
            "value": value,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_selectable_issue_field_options(
        self,
        fieldKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        projectId: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves paginated suggestions for editable options of a specific custom field, filtered by project ID if provided, in Jira Cloud.

        Args:
            fieldKey (string): fieldKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            projectId (integer): Filters the results to options that are only available in the specified project.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/suggestions/edit"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_visible_issue_field_options(
        self,
        fieldKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        projectId: int | None = None,
    ) -> dict[str, Any]:
        """
        Searches for and returns a list of option suggestions for a specific custom field in Jira, based on the provided field key, allowing for pagination using query parameters like `startAt` and `maxResults`.

        Args:
            fieldKey (string): fieldKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            projectId (integer): Filters the results to options that are only available in the specified project.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/suggestions/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_field_option(self, fieldKey: str, optionId: str) -> Any:
        """
        Deletes a specific custom field option in Jira and initiates asynchronous cleanup of associated issue data.

        Args:
            fieldKey (string): fieldKey
            optionId (string): optionId

        Returns:
            Any: Returned if the field option is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/{optionId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_field_option(self, fieldKey: str, optionId: str) -> dict[str, Any]:
        """
        Retrieves a specific custom field option's details for a given field key and option ID in Jira.

        Args:
            fieldKey (string): fieldKey
            optionId (string): optionId

        Returns:
            dict[str, Any]: Returned if the requested option is returned.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/{optionId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_field_option(
        self,
        fieldKey: str,
        optionId: str,
        id: int,
        value: str,
        config: dict[str, Any] | None = None,
        properties: dict[str, Any] | None = None,
    ) -> dict[str, Any]:
        """
        Updates a custom field option identified by its ID in Jira using the PUT method, allowing for modification of existing option details such as value or status.

        Args:
            fieldKey (string): fieldKey
            optionId (string): optionId
            id (integer): The unique identifier for the option. This is only unique within the select field's set of options. Example: 1.
            value (string): The option's name, which is displayed in Jira. Example: 'Team 1'.
            config (object): Details of the projects the option is available in. Example: {'attributes': [], 'scope': {'global': {}, 'projects': [], 'projects2': [{'attributes': ['notSelectable'], 'id': 1001}, {'attributes': ['notSelectable'], 'id': 1002}]}}.
            properties (object): The properties of the object, as arbitrary key-value pairs. These properties can be searched using JQL, if the extractions (see [Issue Field Option Property Index](https://developer.atlassian.com/cloud/jira/platform/modules/issue-field-option-property-index/)) are defined in the descriptor for the issue field module. Example: {'description': "The team's description", 'founded': '2016-06-06', 'leader': {'email': 'lname@example.com', 'name': 'Leader Name'}, 'members': 42}.

        Returns:
            dict[str, Any]: Returned if the option is updated or created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        request_body_data = None
        request_body_data = {
            "config": config,
            "id": id,
            "properties": properties,
            "value": value,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/{optionId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def replace_issue_field_option(
        self,
        fieldKey: str,
        optionId: str,
        replaceWith: int | None = None,
        jql: str | None = None,
        overrideScreenSecurity: bool | None = None,
        overrideEditableFlag: bool | None = None,
    ) -> Any:
        """
        Deletes a custom field option from specific issues using the Jira API, allowing for parameters such as replacing the option, filtering by JQL, and overriding screen security and editable flags.

        Args:
            fieldKey (string): fieldKey
            optionId (string): optionId
            replaceWith (integer): The ID of the option that will replace the currently selected option.
            jql (string): A JQL query that specifies the issues to be updated. For example, *project=10000*.
            overrideScreenSecurity (boolean): Whether screen security is overridden to enable hidden fields to be edited. Available to Connect and Forge app users with admin permission.
            overrideEditableFlag (boolean): Whether screen security is overridden to enable uneditable fields to be edited. Available to Connect and Forge app users with *Administer Jira* [global permission](

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue custom field options (apps)
        """
        if fieldKey is None:
            raise ValueError("Missing required parameter 'fieldKey'.")
        if optionId is None:
            raise ValueError("Missing required parameter 'optionId'.")
        url = f"{self.base_url}/rest/api/3/field/{fieldKey}/option/{optionId}/issue"
        query_params = {
            k: v
            for k, v in [
                ("replaceWith", replaceWith),
                ("jql", jql),
                ("overrideScreenSecurity", overrideScreenSecurity),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_custom_field(self, id: str) -> Any:
        """
        Deletes a custom field in Jira Cloud using the REST API with the "DELETE" method at the path "/rest/api/3/field/{id}", where "{id}" is the identifier of the field to be deleted.

        Args:
            id (string): id

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/field/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def restore_custom_field(self, id: str) -> Any:
        """
        Restores a custom field from trash in Jira using the specified field ID.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/field/{id}/restore"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def trash_custom_field(self, id: str) -> Any:
        """
        Moves a custom field to trash in Jira using the specified field ID, requiring admin permissions.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue fields
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/field/{id}/trash"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_field_configurations(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
        isDefault: bool | None = None,
        query: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves field configurations from Jira using the GET method at the "/rest/api/3/fieldconfiguration" path, allowing filtering by parameters such as startAt, maxResults, id, isDefault, and query.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of field configuration IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.
            isDefault (boolean): If *true* returns default field configurations only.
            query (string): The query string used to match against field configuration names and descriptions.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        url = f"{self.base_url}/rest/api/3/fieldconfiguration"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("isDefault", isDefault),
                ("query", query),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_field_configuration(
        self, name: str, description: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a new field configuration in Jira to manage field visibility and behavior, returning the configuration details upon success.

        Args:
            name (string): The name of the field configuration. Must be unique. Example: 'My Field Configuration'.
            description (string): The description of the field configuration. Example: 'My field configuration description'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfiguration"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_field_configuration(self, id: str) -> Any:
        """
        Deletes a Jira field configuration by ID and returns a success status upon completion.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/fieldconfiguration/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_field_configuration(
        self, id: str, name: str, description: str | None = None
    ) -> Any:
        """
        Updates a field configuration (name and description) in company-managed Jira projects, requiring Administer Jira permissions.

        Args:
            id (string): id
            name (string): The name of the field configuration. Must be unique. Example: 'My Modified Field Configuration'.
            description (string): The description of the field configuration. Example: 'A brand new description'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfiguration/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_field_configuration_items(
        self, id: str, startAt: int | None = None, maxResults: int | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a list of fields associated with a field configuration specified by its ID, allowing pagination via optional startAt and maxResults parameters.

        Args:
            id (string): id
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/fieldconfiguration/{id}/fields"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_field_configuration_items(
        self, id: str, fieldConfigurationItems: list[dict[str, Any]]
    ) -> Any:
        """
        Updates the fields of a field configuration in Jira using the PUT method with the specified configuration ID.

        Args:
            id (string): id
            fieldConfigurationItems (array): Details of fields in a field configuration. Example: [{'description': 'The new description of this item.', 'id': 'customfield_10012', 'isHidden': False}, {'id': 'customfield_10011', 'isRequired': True}, {'description': 'Another new description.', 'id': 'customfield_10010', 'isHidden': False, 'isRequired': False, 'renderer': 'wiki-renderer'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "fieldConfigurationItems": fieldConfigurationItems,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfiguration/{id}/fields"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_field_configs(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves field configuration schemes in Jira with support for pagination and optional ID filtering.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of field configuration scheme IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults), ("id", id)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_field_configuration_scheme(
        self, name: str, description: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a field configuration scheme using the Jira Cloud platform REST API.

        Args:
            name (string): The name of the field configuration scheme. The name must be unique. Example: 'Field Configuration Scheme for software related projects'.
            description (string): The description of the field configuration scheme. Example: 'We can use this one for software projects.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_field_mapping(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        fieldConfigurationSchemeId: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves mappings for a specified field configuration scheme using the Jira API, providing details of how fields are configured across projects.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            fieldConfigurationSchemeId (array): The list of field configuration scheme IDs. To include multiple field configuration schemes separate IDs with ampersand: `fieldConfigurationSchemeId=10000&fieldConfigurationSchemeId=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/mapping"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("fieldConfigurationSchemeId", fieldConfigurationSchemeId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_field_configs_for_project(
        self,
        projectId: list[int],
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of field configuration schemes for a specified project, including the projects that use each scheme, using the `GET` method at the `/rest/api/3/fieldconfigurationscheme/project` path.

        Args:
            projectId (array): The list of project IDs. To include multiple projects, separate IDs with ampersand: `projectId=10000&projectId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_field_config_scheme_project(
        self, projectId: str, fieldConfigurationSchemeId: str | None = None
    ) -> Any:
        """
        Updates a field configuration scheme associated with a project using the Jira Cloud API, specifically for company-managed (classic) projects.

        Args:
            projectId (string): The ID of the project. Example: '10000'.
            fieldConfigurationSchemeId (string): The ID of the field configuration scheme. If the field configuration scheme ID is `null`, the operation assigns the default field configuration scheme. Example: '10000'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        request_body_data = None
        request_body_data = {
            "fieldConfigurationSchemeId": fieldConfigurationSchemeId,
            "projectId": projectId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/project"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_field_configuration_scheme(self, id: str) -> Any:
        """
        Deletes a field configuration scheme from Jira by ID, requiring Administer Jira permissions.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_field_configuration_scheme(
        self, id: str, name: str, description: str | None = None
    ) -> Any:
        """
        Updates a field configuration scheme using its ID, applicable only to company-managed projects, requiring the *Administer Jira* global permission.

        Args:
            id (string): id
            name (string): The name of the field configuration scheme. The name must be unique. Example: 'Field Configuration Scheme for software related projects'.
            description (string): The description of the field configuration scheme. Example: 'We can use this one for software projects.'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_field_config_scheme_mapping(
        self, id: str, mappings: list[dict[str, Any]]
    ) -> Any:
        """
        Updates the field configuration scheme mapping using the Jira Cloud API and returns a status message.

        Args:
            id (string): id
            mappings (array): Field configuration to issue type mappings. Example: [{'fieldConfigurationId': '10000', 'issueTypeId': 'default'}, {'fieldConfigurationId': '10002', 'issueTypeId': '10001'}, {'fieldConfigurationId': '10001', 'issueTypeId': '10002'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "mappings": mappings,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/{id}/mapping"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_field_config_mapping(self, id: str, issueTypeIds: list[str]) -> Any:
        """
        Removes specified issue types from a field configuration scheme in Jira via a POST request.

        Args:
            id (string): id
            issueTypeIds (array): The list of issue type IDs. Must contain unique values not longer than 255 characters and not be empty. Maximum of 100 IDs. Example: ['10000', '10001', '10002'].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue field configurations
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "issueTypeIds": issueTypeIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/fieldconfigurationscheme/{id}/mapping/delete"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_filter(
        self,
        name: str,
        expand: str | None = None,
        overrideSharePermissions: bool | None = None,
        approximateLastUsed: str | None = None,
        description: str | None = None,
        editPermissions: list[dict[str, Any]] | None = None,
        favourite: bool | None = None,
        favouritedCount: int | None = None,
        id: str | None = None,
        jql: str | None = None,
        owner: Any | None = None,
        searchUrl: str | None = None,
        self_arg_body: str | None = None,
        sharePermissions: list[dict[str, Any]] | None = None,
        sharedUsers: Any | None = None,
        subscriptions: Any | None = None,
        viewUrl: str | None = None,
    ) -> dict[str, Any]:
        r"""
        Creates a Jira filter with specified parameters such as name, JQL query, and visibility, returning the newly created filter details.

        Args:
            name (string): The name of the filter. Must be unique. Example: 'All Open Bugs'.
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.
            overrideSharePermissions (boolean): EXPERIMENTAL: Whether share permissions are overridden to enable filters with any share permissions to be created. Available to users with *Administer Jira* [global permission](
            approximateLastUsed (string): \[Experimental\] Approximate last used time. Returns the date and time when the filter was last used. Returns `null` if the filter hasn't been used after tracking was enabled. For performance reasons, timestamps aren't updated in real time and therefore may not be exactly accurate.
            description (string): A description of the filter. Example: 'Lists all open bugs'.
            editPermissions (array): The groups and projects that can edit the filter.
            favourite (boolean): Whether the filter is selected as a favorite.
            favouritedCount (integer): The count of how many users have selected this filter as a favorite, including the filter owner.
            id (string): The unique identifier for the filter.
            jql (string): The JQL query for the filter. For example, *project = SSP AND issuetype = Bug*. Example: 'type = Bug and resolution is empty'.
            owner (string): The user who owns the filter. This is defaulted to the creator of the filter, however Jira administrators can change the owner of a shared filter in the admin settings.
            searchUrl (string): A URL to view the filter results in Jira, using the [Search for issues using JQL](#api-rest-api-3-filter-search-get) operation with the filter's JQL string to return the filter results. For example, *https://your-domain.atlassian.net/rest/api/3/search?jql=project+%3D+SSP+AND+issuetype+%3D+Bug*.
            self_arg_body (string): The URL of the filter.
            sharePermissions (array): The groups and projects that the filter is shared with.
            sharedUsers (string): A paginated list of the users that the filter is shared with. This includes users that are members of the groups or can browse the projects that the filter is shared with.
            subscriptions (string): A paginated list of the users that are subscribed to the filter.
            viewUrl (string): A URL to view the filter results in Jira, using the ID of the filter. For example, *https://your-domain.atlassian.net/issues/?filter=10100*.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        request_body_data = None
        request_body_data = {
            "approximateLastUsed": approximateLastUsed,
            "description": description,
            "editPermissions": editPermissions,
            "favourite": favourite,
            "favouritedCount": favouritedCount,
            "id": id,
            "jql": jql,
            "name": name,
            "owner": owner,
            "searchUrl": searchUrl,
            "self": self_arg_body,
            "sharePermissions": sharePermissions,
            "sharedUsers": sharedUsers,
            "subscriptions": subscriptions,
            "viewUrl": viewUrl,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/filter"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("overrideSharePermissions", overrideSharePermissions),
            ]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_default_share_scope(self) -> dict[str, Any]:
        """
        Retrieves the default sharing scope setting for Jira filters.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        url = f"{self.base_url}/rest/api/3/filter/defaultShareScope"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_default_share_scope(self, scope: str) -> dict[str, Any]:
        """
        Sets the default share scope for new filters and dashboards using the Jira Cloud REST API, allowing users to set the scope to either GLOBAL or PRIVATE.

        Args:
            scope (string): The scope of the default sharing for new filters and dashboards:

         *  `AUTHENTICATED` Shared with all logged-in users.
         *  `GLOBAL` Shared with all logged-in users. This shows as `AUTHENTICATED` in the response.
         *  `PRIVATE` Not shared with any users. Example: 'GLOBAL'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        request_body_data = None
        request_body_data = {
            "scope": scope,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/filter/defaultShareScope"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_favourite_filters(self, expand: str | None = None) -> list[Any]:
        """
        Retrieves the user's visible favorite filters with optional expansion details.

        Args:
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        url = f"{self.base_url}/rest/api/3/filter/favourite"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_my_filters(
        self, expand: str | None = None, includeFavourites: bool | None = None
    ) -> list[Any]:
        """
        Retrieves a list of filters accessible by the user, with options to expand and include favorite filters, using the Jira REST API.

        Args:
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.
            includeFavourites (boolean): Include the user's favorite filters in the response.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        url = f"{self.base_url}/rest/api/3/filter/my"
        query_params = {
            k: v
            for k, v in [("expand", expand), ("includeFavourites", includeFavourites)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_filters_paginated(
        self,
        filterName: str | None = None,
        accountId: str | None = None,
        owner: str | None = None,
        groupname: str | None = None,
        groupId: str | None = None,
        projectId: int | None = None,
        id: list[int] | None = None,
        orderBy: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        expand: str | None = None,
        overrideSharePermissions: bool | None = None,
        isSubstringMatch: bool | None = None,
    ) -> dict[str, Any]:
        r"""
        Retrieves a list of filters accessible to the user based on parameters like name, owner, project, or group, supporting pagination and substring matching.

        Args:
            filterName (string): String used to perform a case-insensitive partial match with `name`.
            accountId (string): User account ID used to return filters with the matching `owner.accountId`. This parameter cannot be used with `owner`.
            owner (string): This parameter is deprecated because of privacy changes. Use `accountId` instead. See the [migration guide]( for details. User name used to return filters with the matching `owner.name`. This parameter cannot be used with `accountId`.
            groupname (string): As a group's name can change, use of `groupId` is recommended to identify a group. Group name used to returns filters that are shared with a group that matches `sharePermissions.group.groupname`. This parameter cannot be used with the `groupId` parameter.
            groupId (string): Group ID used to returns filters that are shared with a group that matches `sharePermissions.group.groupId`. This parameter cannot be used with the `groupname` parameter.
            projectId (integer): Project ID used to returns filters that are shared with a project that matches `sharePermissions.project.id`.
            id (array): The list of filter IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`. Do not exceed 200 filter IDs.
            orderBy (string): [Order](#ordering) the results by a field: * `description` Sorts by filter description. Note that this sorting works independently of whether the expand to display the description field is in use. * `favourite_count` Sorts by the count of how many users have this filter as a favorite. * `is_favourite` Sorts by whether the filter is marked as a favorite. * `id` Sorts by filter ID. * `name` Sorts by filter name. * `owner` Sorts by the ID of the filter owner. * `is_shared` Sorts by whether the filter is shared.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `description` Returns the description of the filter. * `favourite` Returns an indicator of whether the user has set the filter as a favorite. * `favouritedCount` Returns a count of how many users have set this filter as a favorite. * `jql` Returns the JQL query that the filter uses. * `owner` Returns the owner of the filter. * `searchUrl` Returns a URL to perform the filter's JQL query. * `sharePermissions` Returns the share permissions defined for the filter. * `editPermissions` Returns the edit permissions defined for the filter. * `isWritable` Returns whether the current user has permission to edit the filter. * `approximateLastUsed` \[Experimental\] Returns the approximate date and time when the filter was last evaluated. * `subscriptions` Returns the users that are subscribed to the filter. * `viewUrl` Returns a URL to view the filter.
            overrideSharePermissions (boolean): EXPERIMENTAL: Whether share permissions are overridden to enable filters with any share permissions to be returned. Available to users with *Administer Jira* [global permission](
            isSubstringMatch (boolean): When `true` this will perform a case-insensitive substring match for the provided `filterName`. When `false` the filter name will be searched using [full text search syntax](

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        url = f"{self.base_url}/rest/api/3/filter/search"
        query_params = {
            k: v
            for k, v in [
                ("filterName", filterName),
                ("accountId", accountId),
                ("owner", owner),
                ("groupname", groupname),
                ("groupId", groupId),
                ("projectId", projectId),
                ("id", id),
                ("orderBy", orderBy),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("expand", expand),
                ("overrideSharePermissions", overrideSharePermissions),
                ("isSubstringMatch", isSubstringMatch),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_filter(self, id: str) -> Any:
        """
        Deletes a specific Jira filter by its ID using the Jira API and returns a success status if the operation is successful.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_filter(
        self,
        id: str,
        expand: str | None = None,
        overrideSharePermissions: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a specific filter's details by ID from Jira, optionally expanding fields or overriding share permissions.

        Args:
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.
            overrideSharePermissions (boolean): EXPERIMENTAL: Whether share permissions are overridden to enable filters with any share permissions to be returned. Available to users with *Administer Jira* [global permission](

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("overrideSharePermissions", overrideSharePermissions),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_filter(
        self,
        id: str,
        name: str,
        expand: str | None = None,
        overrideSharePermissions: bool | None = None,
        approximateLastUsed: str | None = None,
        description: str | None = None,
        editPermissions: list[dict[str, Any]] | None = None,
        favourite: bool | None = None,
        favouritedCount: int | None = None,
        id_body: str | None = None,
        jql: str | None = None,
        owner: Any | None = None,
        searchUrl: str | None = None,
        self_arg_body: str | None = None,
        sharePermissions: list[dict[str, Any]] | None = None,
        sharedUsers: Any | None = None,
        subscriptions: Any | None = None,
        viewUrl: str | None = None,
    ) -> dict[str, Any]:
        r"""
        Updates an existing Jira filter (including permissions if overrideSharePermissions is specified) and returns the modified filter.

        Args:
            id (string): id
            name (string): The name of the filter. Must be unique. Example: 'All Open Bugs'.
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.
            overrideSharePermissions (boolean): EXPERIMENTAL: Whether share permissions are overridden to enable the addition of any share permissions to filters. Available to users with *Administer Jira* [global permission](
            approximateLastUsed (string): \[Experimental\] Approximate last used time. Returns the date and time when the filter was last used. Returns `null` if the filter hasn't been used after tracking was enabled. For performance reasons, timestamps aren't updated in real time and therefore may not be exactly accurate.
            description (string): A description of the filter. Example: 'Lists all open bugs'.
            editPermissions (array): The groups and projects that can edit the filter.
            favourite (boolean): Whether the filter is selected as a favorite.
            favouritedCount (integer): The count of how many users have selected this filter as a favorite, including the filter owner.
            id_body (string): The unique identifier for the filter.
            jql (string): The JQL query for the filter. For example, *project = SSP AND issuetype = Bug*. Example: 'type = Bug and resolution is empty'.
            owner (string): The user who owns the filter. This is defaulted to the creator of the filter, however Jira administrators can change the owner of a shared filter in the admin settings.
            searchUrl (string): A URL to view the filter results in Jira, using the [Search for issues using JQL](#api-rest-api-3-filter-search-get) operation with the filter's JQL string to return the filter results. For example, *https://your-domain.atlassian.net/rest/api/3/search?jql=project+%3D+SSP+AND+issuetype+%3D+Bug*.
            self_arg_body (string): The URL of the filter.
            sharePermissions (array): The groups and projects that the filter is shared with.
            sharedUsers (string): A paginated list of the users that the filter is shared with. This includes users that are members of the groups or can browse the projects that the filter is shared with.
            subscriptions (string): A paginated list of the users that are subscribed to the filter.
            viewUrl (string): A URL to view the filter results in Jira, using the ID of the filter. For example, *https://your-domain.atlassian.net/issues/?filter=10100*.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "approximateLastUsed": approximateLastUsed,
            "description": description,
            "editPermissions": editPermissions,
            "favourite": favourite,
            "favouritedCount": favouritedCount,
            "id": id_body,
            "jql": jql,
            "name": name,
            "owner": owner,
            "searchUrl": searchUrl,
            "self": self_arg_body,
            "sharePermissions": sharePermissions,
            "sharedUsers": sharedUsers,
            "subscriptions": subscriptions,
            "viewUrl": viewUrl,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/filter/{id}"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("overrideSharePermissions", overrideSharePermissions),
            ]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def reset_columns(self, id: str) -> Any:
        """
        Deletes the columns configuration for a specific filter in Jira using the filter ID.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/columns"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_columns(self, id: str) -> list[Any]:
        """
        Retrieves the column configuration for a specified filter in Jira using the filter ID.

        Args:
            id (string): id

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/columns"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_columns(self, id: str, columns: list[str] | None = None) -> Any:
        """
        Updates the columns of a specific filter in Jira using the REST API and returns a response indicating the status of the update operation.

        Args:
            id (string): id
            columns (array): columns

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        files_data = None
        request_body_data = {}
        files_data = {}
        if columns is not None:
            request_body_data["columns"] = columns
        files_data = {k: v for k, v in files_data.items() if v is not None}
        if not files_data:
            files_data = None
        url = f"{self.base_url}/rest/api/3/filter/{id}/columns"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            files=files_data,
            params=query_params,
            content_type="multipart/form-data",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_favourite_for_filter(
        self, id: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Removes a filter with the specified ID from the user's favorites list using the Jira Cloud API.

        Args:
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/favourite"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_favourite_for_filter(
        self, id: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Adds a filter to the user's favorites list in Jira and returns the updated filter details.

        Args:
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about filter in the response. This parameter accepts a comma-separated list. Expand options include: * `sharedUsers` Returns the users that the filter is shared with. This includes users that can browse projects that the filter is shared with. If you don't specify `sharedUsers`, then the `sharedUsers` object is returned but it doesn't list any users. The list of users returned is limited to 1000, to access additional users append `[start-index:end-index]` to the expand request. For example, to access the next 1000 users, use `?expand=sharedUsers[1001:2000]`. * `subscriptions` Returns the users that are subscribed to the filter. If you don't specify `subscriptions`, the `subscriptions` object is returned but it doesn't list any subscriptions. The list of subscriptions returned is limited to 1000, to access additional subscriptions append `[start-index:end-index]` to the expand request. For example, to access the next 1000 subscriptions, use `?expand=subscriptions[1001:2000]`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/filter/{id}/favourite"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def change_filter_owner(self, id: str, accountId: str) -> Any:
        """
        Updates the owner of a Jira filter specified by ID via PUT request, requiring admin rights or ownership and returning a 204 status on success.

        Args:
            id (string): id
            accountId (string): The account ID of the new owner. Example: '0000-0000-0000-0000'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filters
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "accountId": accountId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/filter/{id}/owner"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_share_permissions(self, id: str) -> list[Any]:
        """
        Retrieves the permissions for a specific Jira filter identified by its ID using the GET method, returning details about the permissions assigned to the filter.

        Args:
            id (string): id

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/permission"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_share_permission(
        self,
        id: str,
        type: str,
        accountId: str | None = None,
        groupId: str | None = None,
        groupname: str | None = None,
        projectId: str | None = None,
        projectRoleId: str | None = None,
        rights: int | None = None,
    ) -> list[Any]:
        """
        Adds permissions to an existing filter using the filter ID, allowing control over who can access or modify the filter via the API.

        Args:
            id (string): id
            type (string): The type of the share permission.Specify the type as follows:

         *  `user` Share with a user.
         *  `group` Share with a group. Specify `groupname` as well.
         *  `project` Share with a project. Specify `projectId` as well.
         *  `projectRole` Share with a project role in a project. Specify `projectId` and `projectRoleId` as well.
         *  `global` Share globally, including anonymous users. If set, this type overrides all existing share permissions and must be deleted before any non-global share permissions is set.
         *  `authenticated` Share with all logged-in users. This shows as `loggedin` in the response. If set, this type overrides all existing share permissions and must be deleted before any non-global share permissions is set. Example: 'group'.
            accountId (string): The user account ID that the filter is shared with. For a request, specify the `accountId` property for the user.
            groupId (string): The ID of the group, which uniquely identifies the group across all Atlassian products.For example, *952d12c3-5b5b-4d04-bb32-44d383afc4b2*. Cannot be provided with `groupname`.
            groupname (string): The name of the group to share the filter with. Set `type` to `group`. Please note that the name of a group is mutable, to reliably identify a group use `groupId`. Example: 'jira-administrators'.
            projectId (string): The ID of the project to share the filter with. Set `type` to `project`.
            projectRoleId (string): The ID of the project role to share the filter with. Set `type` to `projectRole` and the `projectId` for the project that the role is in.
            rights (integer): The rights for the share permission. Example: 1.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "accountId": accountId,
            "groupId": groupId,
            "groupname": groupname,
            "projectId": projectId,
            "projectRoleId": projectRoleId,
            "rights": rights,
            "type": type,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/filter/{id}/permission"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_share_permission(self, id: str, permissionId: str) -> Any:
        """
        Deletes a specific permission associated with a filter in Jira using the REST API.

        Args:
            id (string): id
            permissionId (string): permissionId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        if permissionId is None:
            raise ValueError("Missing required parameter 'permissionId'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/permission/{permissionId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_share_permission(self, id: str, permissionId: str) -> dict[str, Any]:
        """
        Retrieves the specified permission details for a given filter ID in the API, allowing access to specific permission information.

        Args:
            id (string): id
            permissionId (string): permissionId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Filter sharing
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        if permissionId is None:
            raise ValueError("Missing required parameter 'permissionId'.")
        url = f"{self.base_url}/rest/api/3/filter/{id}/permission/{permissionId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_group(
        self,
        groupname: str | None = None,
        groupId: str | None = None,
        swapGroup: str | None = None,
        swapGroupId: str | None = None,
    ) -> Any:
        """
        Deletes a group from the organization's directory using the `DELETE` method, allowing for optional group swapping, and returns a status message based on the operation's success or failure.

        Args:
            groupname (string): The name of the group to be deleted, specified as a query parameter.
            groupId (string): The ID of the group. This parameter cannot be used with the `groupname` parameter.
            swapGroup (string): As a group's name can change, use of `swapGroupId` is recommended to identify a group. The group to transfer restrictions to. Only comments and worklogs are transferred. If restrictions are not transferred, comments and worklogs are inaccessible after the deletion. This parameter cannot be used with the `swapGroupId` parameter.
            swapGroupId (string): The ID of the group to transfer restrictions to. Only comments and worklogs are transferred. If restrictions are not transferred, comments and worklogs are inaccessible after the deletion. This parameter cannot be used with the `swapGroup` parameter.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/group"
        query_params = {
            k: v
            for k, v in [
                ("groupname", groupname),
                ("groupId", groupId),
                ("swapGroup", swapGroup),
                ("swapGroupId", swapGroupId),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_group(
        self,
        groupname: str | None = None,
        groupId: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves group information from Jira by group name or ID, optionally expanding the response with additional details.

        Args:
            groupname (string): As a group's name can change, use of `groupId` is recommended to identify a group. The name of the group. This parameter cannot be used with the `groupId` parameter.
            groupId (string): The ID of the group. This parameter cannot be used with the `groupName` parameter.
            expand (string): List of fields to expand.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/group"
        query_params = {
            k: v
            for k, v in [
                ("groupname", groupname),
                ("groupId", groupId),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_group(self, name: str) -> dict[str, Any]:
        """
        Creates a new group using the Jira Cloud REST API and returns a response indicating the creation status.

        Args:
            name (string): The name of the group. Example: 'power-users'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        request_body_data = None
        request_body_data = {
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/group"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_get_groups(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        groupId: list[str] | None = None,
        groupName: list[str] | None = None,
        accessType: str | None = None,
        applicationKey: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves multiple groups in bulk from Jira Cloud based on query parameters such as group IDs, names, and access types.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            groupId (array): The ID of a group. To specify multiple IDs, pass multiple `groupId` parameters. For example, `groupId=5b10a2844c20165700ede21g&groupId=5b10ac8d82e05b22cc7d4ef5`. Example: '3571b9a7-348f-414a-9087-8e1ea03a7df8'.
            groupName (array): The name of a group. To specify multiple names, pass multiple `groupName` parameters. For example, `groupName=administrators&groupName=jira-software-users`.
            accessType (string): The access level of a group. Valid values: 'site-admin', 'admin', 'user'.
            applicationKey (string): The application key of the product user groups to search for. Valid values: 'jira-servicedesk', 'jira-software', 'jira-product-discovery', 'jira-core'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/group/bulk"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("groupId", groupId),
                ("groupName", groupName),
                ("accessType", accessType),
                ("applicationKey", applicationKey),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_users_from_group(
        self,
        groupname: str | None = None,
        groupId: str | None = None,
        includeInactiveUsers: bool | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves paginated members of a Jira group with optional filtering for inactive users, supporting group identification via name or ID.

        Args:
            groupname (string): As a group's name can change, use of `groupId` is recommended to identify a group. The name of the group. This parameter cannot be used with the `groupId` parameter.
            groupId (string): The ID of the group. This parameter cannot be used with the `groupName` parameter.
            includeInactiveUsers (boolean): Include inactive users.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page (number should be between 1 and 50).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/group/member"
        query_params = {
            k: v
            for k, v in [
                ("groupname", groupname),
                ("groupId", groupId),
                ("includeInactiveUsers", includeInactiveUsers),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_user_from_group(
        self,
        accountId: str,
        groupname: str | None = None,
        groupId: str | None = None,
        username: str | None = None,
    ) -> Any:
        """
        Removes a specified user from a group in Jira Cloud using account ID, username, groupname, or groupId as parameters.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            groupname (string): As a group's name can change, use of `groupId` is recommended to identify a group. The name of the group. This parameter cannot be used with the `groupId` parameter.
            groupId (string): The ID of the group. This parameter cannot be used with the `groupName` parameter.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/group/user"
        query_params = {
            k: v
            for k, v in [
                ("groupname", groupname),
                ("groupId", groupId),
                ("username", username),
                ("accountId", accountId),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_user_to_group(
        self,
        groupname: str | None = None,
        groupId: str | None = None,
        accountId: str | None = None,
        name: str | None = None,
    ) -> dict[str, Any]:
        """
        Adds a user to a specified Jira group using the "POST" method at the "/rest/api/3/group/user" endpoint, requiring a group ID or name to identify the target group.

        Args:
            groupname (string): As a group's name can change, use of `groupId` is recommended to identify a group. The name of the group. This parameter cannot be used with the `groupId` parameter.
            groupId (string): The ID of the group. This parameter cannot be used with the `groupName` parameter.
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            name (string): This property is no longer available. See the [deprecation notice](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        request_body_data = None
        request_body_data = {
            "accountId": accountId,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/group/user"
        query_params = {
            k: v
            for k, v in [("groupname", groupname), ("groupId", groupId)]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_groups(
        self,
        accountId: str | None = None,
        query: str | None = None,
        exclude: list[str] | None = None,
        excludeId: list[str] | None = None,
        maxResults: int | None = None,
        caseInsensitive: bool | None = None,
        userName: str | None = None,
    ) -> dict[str, Any]:
        """
        Searches for groups matching query parameters and returns results with highlighted matches and a picker-friendly header indicating the number of matching groups.

        Args:
            accountId (string): This parameter is deprecated, setting it does not affect the results. To find groups containing a particular user, use [Get user groups](#api-rest-api-3-user-groups-get).
            query (string): The string to find in group names. Example: 'query'.
            exclude (array): As a group's name can change, use of `excludeGroupIds` is recommended to identify a group. A group to exclude from the result. To exclude multiple groups, provide an ampersand-separated list. For example, `exclude=group1&exclude=group2`. This parameter cannot be used with the `excludeGroupIds` parameter.
            excludeId (array): A group ID to exclude from the result. To exclude multiple groups, provide an ampersand-separated list. For example, `excludeId=group1-id&excludeId=group2-id`. This parameter cannot be used with the `excludeGroups` parameter.
            maxResults (integer): The maximum number of groups to return. The maximum number of groups that can be returned is limited by the system property `jira.ajax.autocomplete.limit`.
            caseInsensitive (boolean): Whether the search for groups should be case insensitive.
            userName (string): This parameter is no longer available. See the [deprecation notice]( for details.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Groups
        """
        url = f"{self.base_url}/rest/api/3/groups/picker"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("query", query),
                ("exclude", exclude),
                ("excludeId", excludeId),
                ("maxResults", maxResults),
                ("caseInsensitive", caseInsensitive),
                ("userName", userName),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users_and_groups(
        self,
        query: str,
        maxResults: int | None = None,
        showAvatar: bool | None = None,
        fieldId: str | None = None,
        projectId: list[str] | None = None,
        issueTypeId: list[str] | None = None,
        avatarSize: str | None = None,
        caseInsensitive: bool | None = None,
        excludeConnectAddons: bool | None = None,
    ) -> dict[str, Any]:
        """
        Searches for users and groups matching a query string in Jira Cloud and returns results with HTML highlighting for picker fields.

        Args:
            query (string): The search string.
            maxResults (integer): The maximum number of items to return in each list.
            showAvatar (boolean): Whether the user avatar should be returned. If an invalid value is provided, the default value is used.
            fieldId (string): The custom field ID of the field this request is for.
            projectId (array): The ID of a project that returned users and groups must have permission to view. To include multiple projects, provide an ampersand-separated list. For example, `projectId=10000&projectId=10001`. This parameter is only used when `fieldId` is present.
            issueTypeId (array): The ID of an issue type that returned users and groups must have permission to view. To include multiple issue types, provide an ampersand-separated list. For example, `issueTypeId=10000&issueTypeId=10001`. Special values, such as `-1` (all standard issue types) and `-2` (all subtask issue types), are supported. This parameter is only used when `fieldId` is present.
            avatarSize (string): The size of the avatar to return. If an invalid value is provided, the default value is used.
            caseInsensitive (boolean): Whether the search for groups should be case insensitive.
            excludeConnectAddons (boolean): Whether Connect app users and groups should be excluded from the search results. If an invalid value is provided, the default value is used.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Group and user picker
        """
        url = f"{self.base_url}/rest/api/3/groupuserpicker"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("maxResults", maxResults),
                ("showAvatar", showAvatar),
                ("fieldId", fieldId),
                ("projectId", projectId),
                ("issueTypeId", issueTypeId),
                ("avatarSize", avatarSize),
                ("caseInsensitive", caseInsensitive),
                ("excludeConnectAddons", excludeConnectAddons),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_license(self) -> dict[str, Any]:
        """
        Retrieves licensing information about a Jira instance, returning details such as license metrics using the Jira Cloud REST API.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            License metrics
        """
        url = f"{self.base_url}/rest/api/3/instance/license"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue(
        self,
        updateHistory: bool | None = None,
        fields: dict[str, Any] | None = None,
        historyMetadata: Any | None = None,
        properties: list[dict[str, Any]] | None = None,
        transition: Any | None = None,
        update: dict[str, list[dict[str, Any]]] | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new Jira issue using the specified project and issue type, returning a successful creation response if valid.

        Args:
            updateHistory (boolean): Whether the project in which the issue is created is added to the user's **Recently viewed** project list, as shown under **Projects** in Jira. When provided, the issue type and request type are added to the user's history for a project. These values are then used to provide defaults on the issue create screen.
            fields (object): List of issue screen fields to update, specifying the sub-field to update and its value for each field. This field provides a straightforward option when setting a sub-field. When multiple sub-fields or other operations are required, use `update`. Fields included in here cannot be included in `update`. Example: {'assignee': {'id': '5b109f2e9729b51b54dc274d'}, 'components': [{'id': '10000'}], 'customfield_10000': '09/Jun/19', 'customfield_20000': '06/Jul/19 3:25 PM', 'customfield_30000': ['10000', '10002'], 'customfield_40000': {'content': [{'content': [{'text': 'Occurs on all orders', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_50000': {'content': [{'content': [{'text': 'Could impact day-to-day work.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_60000': 'jira-software-users', 'customfield_70000': ['jira-administrators', 'jira-software-users'], 'customfield_80000': {'value': 'red'}, 'description': {'content': [{'content': [{'text': 'Order entry fails when selecting supplier.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'duedate': '2019-05-11', 'environment': {'content': [{'content': [{'text': 'UAT', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'fixVersions': [{'id': '10001'}], 'issuetype': {'id': '10000'}, 'labels': ['bugfix', 'blitz_test'], 'parent': {'key': 'PROJ-123'}, 'priority': {'id': '20000'}, 'project': {'id': '10000'}, 'reporter': {'id': '5b10a2844c20165700ede21g'}, 'security': {'id': '10000'}, 'summary': 'Main order flow broken', 'timetracking': {'originalEstimate': '10', 'remainingEstimate': '5'}, 'versions': [{'id': '10000'}]}.
            historyMetadata (string): Additional issue history details.
            properties (array): Details of issue properties to be add or update.
            transition (string): Details of a transition. Required when performing a transition, optional when creating or editing an issue.
            update (object): A Map containing the field field name and a list of operations to perform on the issue screen field. Note that fields included in here cannot be included in `fields`. Example: {}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "fields": fields,
            "historyMetadata": historyMetadata,
            "properties": properties,
            "transition": transition,
            "update": update,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue"
        query_params = {
            k: v for k, v in [("updateHistory", updateHistory)] if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def archive_issues_async(self, jql: str | None = None) -> Any:
        """
        Archives Jira issues via ID/key using a POST request, returning async status codes for success/failure.

        Args:
            jql (string): jql Example: 'project = FOO AND updated < -2y'.

        Returns:
            Any: Returns the URL to check the status of the submitted request.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "jql": jql,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/archive"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def archive_issues(self, issueIdsOrKeys: list[str] | None = None) -> dict[str, Any]:
        """
        Archives Jira issues via the specified issue IDs/keys using the PUT method, handling bulk operations and returning status/error details.

        Args:
            issueIdsOrKeys (array): issueIdsOrKeys Example: ['PR-1', '1001', 'PROJECT-2'].

        Returns:
            dict[str, Any]: Returned if there is at least one valid issue to archive in the request. The return message will include the count of archived issues and subtasks, as well as error details for issues which failed to get archived.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "issueIdsOrKeys": issueIdsOrKeys,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/archive"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issues(
        self, issueUpdates: list[dict[str, Any]] | None = None
    ) -> dict[str, Any]:
        """
        Performs bulk operations on Jira issues, such as moving or editing multiple issues at once, using the POST method at the "/rest/api/3/issue/bulk" endpoint.

        Args:
            issueUpdates (array): issueUpdates Example: [{'fields': {'assignee': {'id': '5b109f2e9729b51b54dc274d'}, 'components': [{'id': '10000'}], 'customfield_10000': '09/Jun/19', 'customfield_20000': '06/Jul/19 3:25 PM', 'customfield_30000': ['10000', '10002'], 'customfield_40000': {'content': [{'content': [{'text': 'Occurs on all orders', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_50000': {'content': [{'content': [{'text': 'Could impact day-to-day work.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_60000': 'jira-software-users', 'customfield_70000': ['jira-administrators', 'jira-software-users'], 'customfield_80000': {'value': 'red'}, 'description': {'content': [{'content': [{'text': 'Order entry fails when selecting supplier.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'duedate': '2011-03-11', 'environment': {'content': [{'content': [{'text': 'UAT', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'fixVersions': [{'id': '10001'}], 'issuetype': {'id': '10000'}, 'labels': ['bugfix', 'blitz_test'], 'priority': {'id': '20000'}, 'project': {'id': '10000'}, 'reporter': {'id': '5b10a2844c20165700ede21g'}, 'security': {'id': '10000'}, 'summary': 'Main order flow broken', 'timetracking': {'originalEstimate': '10', 'remainingEstimate': '5'}, 'versions': [{'id': '10000'}]}, 'update': {'worklog': [{'add': {'started': '2019-07-05T11:05:00.000+0000', 'timeSpent': '60m'}}]}}, {'fields': {'assignee': {'id': '5b109f2e9729b51b54dc274d'}, 'components': [{'id': '10000'}], 'customfield_10000': '09/Jun/19', 'customfield_20000': '06/Jul/19 3:25 PM', 'customfield_30000': ['10000', '10002'], 'customfield_40000': {'content': [{'content': [{'text': 'Occurs on all orders', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_50000': {'content': [{'content': [{'text': 'Could impact day-to-day work.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_60000': 'jira-software-users', 'customfield_70000': ['jira-administrators', 'jira-software-users'], 'customfield_80000': {'value': 'red'}, 'description': {'content': [{'content': [{'text': 'Order remains pending after approved.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'duedate': '2019-04-16', 'environment': {'content': [{'content': [{'text': 'UAT', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'fixVersions': [{'id': '10001'}], 'issuetype': {'id': '10000'}, 'labels': ['new_release'], 'priority': {'id': '20000'}, 'project': {'id': '1000'}, 'reporter': {'id': '5b10a2844c20165700ede21g'}, 'security': {'id': '10000'}, 'summary': 'Order stuck in pending', 'timetracking': {'originalEstimate': '15', 'remainingEstimate': '5'}, 'versions': [{'id': '10000'}]}, 'update': {}}].

        Returns:
            dict[str, Any]: Returned if any of the issue or subtask creation requests were successful. A request may be unsuccessful when it:

         *  is missing required fields.
         *  contains invalid field values.
         *  contains fields that cannot be set for the issue type.
         *  is by a user who does not have the necessary permission.
         *  is to create a subtype in a project different that of the parent issue.
         *  is for a subtask when the option to create subtasks is disabled.
         *  is invalid for any other reason.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "issueUpdates": issueUpdates,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/bulk"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_fetch_issues(
        self,
        issueIdsOrKeys: list[str],
        expand: list[str] | None = None,
        fields: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        properties: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Fetches multiple issues in bulk from Jira using the POST method at "/rest/api/3/issue/bulkfetch", returning the specified issues based on provided issue IDs or keys.

        Args:
            issueIdsOrKeys (array): An array of issue IDs or issue keys to fetch. You can mix issue IDs and keys in the same query. Example: ['EX-1', 'EX-2', '10005'].
            expand (array): Use [expand](#expansion) to include additional information about issues in the response. Note that, unlike the majority of instances where `expand` is specified, `expand` is defined as a list of values. The expand options are:

         *  `renderedFields` Returns field values rendered in HTML format.
         *  `names` Returns the display name of each field.
         *  `schema` Returns the schema describing a field type.
         *  `transitions` Returns all possible transitions for the issue.
         *  `operations` Returns all possible operations for the issue.
         *  `editmeta` Returns information about how each field can be edited.
         *  `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent.
         *  `versionedRepresentations` Instead of `fields`, returns `versionedRepresentations` a JSON array containing each version of a field's value, with the highest numbered item representing the most recent version. Example: ['names'].
            fields (array): A list of fields to return for each issue, use it to retrieve a subset of fields. This parameter accepts a comma-separated list. Expand options include:

         *  `*all` Returns all fields.
         *  `*navigable` Returns navigable fields.
         *  Any issue field, prefixed with a minus to exclude.

        The default is `*navigable`.

        Examples:

         *  `summary,comment` Returns the summary and comments fields only.
         *  `-description` Returns all navigable (default) fields except description.
         *  `*all,-comment` Returns all fields except comments.

        Multiple `fields` parameters can be included in a request.

        Note: All navigable fields are returned by default. This differs from [GET issue](#api-rest-api-3-issue-issueIdOrKey-get) where the default is all fields. Example: ['summary', 'project', 'assignee'].
            fieldsByKeys (boolean): Reference fields by their key (rather than ID). The default is `false`. Example: False.
            properties (array): A list of issue property keys of issue properties to be included in the results. A maximum of 5 issue property keys can be specified. Example: [].

        Returns:
            dict[str, Any]: Returned if the request is successful. A response may contain both successful issues and issue errors.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "expand": expand,
            "fields": fields,
            "fieldsByKeys": fieldsByKeys,
            "issueIdsOrKeys": issueIdsOrKeys,
            "properties": properties,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/bulkfetch"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_create_issue_meta(
        self,
        projectIds: list[str] | None = None,
        projectKeys: list[str] | None = None,
        issuetypeIds: list[str] | None = None,
        issuetypeNames: list[str] | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves metadata including required fields, default values, and allowed configurations for creating Jira issues based on specified projects and issue types.

        Args:
            projectIds (array): List of project IDs. This parameter accepts a comma-separated list. Multiple project IDs can also be provided using an ampersand-separated list. For example, `projectIds=10000,10001&projectIds=10020,10021`. This parameter may be provided with `projectKeys`.
            projectKeys (array): List of project keys. This parameter accepts a comma-separated list. Multiple project keys can also be provided using an ampersand-separated list. For example, `projectKeys=proj1,proj2&projectKeys=proj3`. This parameter may be provided with `projectIds`.
            issuetypeIds (array): List of issue type IDs. This parameter accepts a comma-separated list. Multiple issue type IDs can also be provided using an ampersand-separated list. For example, `issuetypeIds=10000,10001&issuetypeIds=10020,10021`. This parameter may be provided with `issuetypeNames`.
            issuetypeNames (array): List of issue type names. This parameter accepts a comma-separated list. Multiple issue type names can also be provided using an ampersand-separated list. For example, `issuetypeNames=name1,name2&issuetypeNames=name3`. This parameter may be provided with `issuetypeIds`.
            expand (string): Use [expand](#expansion) to include additional information about issue metadata in the response. This parameter accepts `projects.issuetypes.fields`, which returns information about the fields in the issue creation screen for each issue type. Fields hidden from the screen are not returned. Use the information to populate the `fields` and `update` fields in [Create issue](#api-rest-api-3-issue-post) and [Create issues](#api-rest-api-3-issue-bulk-post).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        url = f"{self.base_url}/rest/api/3/issue/createmeta"
        query_params = {
            k: v
            for k, v in [
                ("projectIds", projectIds),
                ("projectKeys", projectKeys),
                ("issuetypeIds", issuetypeIds),
                ("issuetypeNames", issuetypeNames),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_create_issue_meta_issue_types(
        self,
        projectIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves metadata for creating issues in Jira for a specific project's issue types, including available fields and mandatory requirements.

        Args:
            projectIdOrKey (string): projectIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/createmeta/{projectIdOrKey}/issuetypes"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_create_issue_meta_issue_type_id(
        self,
        projectIdOrKey: str,
        issueTypeId: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves metadata for specific issue types within a project in Jira using the "GET" method, returning details such as available fields and their schemas based on the project and issue type identifiers.

        Args:
            projectIdOrKey (string): projectIdOrKey
            issueTypeId (string): issueTypeId
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        url = f"{self.base_url}/rest/api/3/issue/createmeta/{projectIdOrKey}/issuetypes/{issueTypeId}"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_limit_report(
        self, isReturningKeys: bool | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a report of issues approaching their worklog limit thresholds using the specified parameters.

        Args:
            isReturningKeys (boolean): Return issue keys instead of issue ids in the response. Usage: Add `?isReturningKeys=true` to the end of the path to request issue keys.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        url = f"{self.base_url}/rest/api/3/issue/limit/report"
        query_params = {
            k: v for k, v in [("isReturningKeys", isReturningKeys)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_picker_resource(
        self,
        query: str | None = None,
        currentJQL: str | None = None,
        currentIssueKey: str | None = None,
        currentProjectId: str | None = None,
        showSubTasks: bool | None = None,
        showSubTaskParent: bool | None = None,
    ) -> dict[str, Any]:
        """
        Provides auto-completion suggestions for Jira issues based on search queries and JQL filters, returning matching issues from user history and current searches.

        Args:
            query (string): A string to match against text fields in the issue such as title, description, or comments. Example: 'query'.
            currentJQL (string): A JQL query defining a list of issues to search for the query term. Note that `username` and `userkey` cannot be used as search terms for this parameter, due to privacy reasons. Use `accountId` instead.
            currentIssueKey (string): The key of an issue to exclude from search results. For example, the issue the user is viewing when they perform this query.
            currentProjectId (string): The ID of a project that suggested issues must belong to.
            showSubTasks (boolean): Indicate whether to include subtasks in the suggestions list.
            showSubTaskParent (boolean): When `currentIssueKey` is a subtask, whether to include the parent issue in the suggestions if it matches the query.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        url = f"{self.base_url}/rest/api/3/issue/picker"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("currentJQL", currentJQL),
                ("currentIssueKey", currentIssueKey),
                ("currentProjectId", currentProjectId),
                ("showSubTasks", showSubTasks),
                ("showSubTaskParent", showSubTaskParent),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_set_issues_properties_list(
        self,
        entitiesIds: list[int] | None = None,
        properties: dict[str, dict[str, Any]] | None = None,
    ) -> Any:
        """
        Sets or updates multiple issue properties for specified issues using JIRA's REST API, supporting bulk operations on custom data storage.

        Args:
            entitiesIds (array): A list of entity property IDs.
            properties (object): A list of entity property keys and values.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        request_body_data = None
        request_body_data = {
            "entitiesIds": entitiesIds,
            "properties": properties,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/properties"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_set_issue_properties_by_issue(
        self, issues: list[dict[str, Any]] | None = None
    ) -> Any:
        """
        Sets or updates custom properties on multiple Jira issues in a single request and returns the task status for asynchronous processing.

        Args:
            issues (array): A list of issue IDs and their respective properties. Example: [{'issueID': 1000, 'properties': {'myProperty': {'owner': 'admin', 'weight': 100}}}, {'issueID': 1001, 'properties': {'myOtherProperty': {'cost': 150, 'transportation': 'car'}}}].

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        request_body_data = None
        request_body_data = {
            "issues": issues,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/properties/multi"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_delete_issue_property(
        self,
        propertyKey: str,
        currentValue: Any | None = None,
        entityIds: list[int] | None = None,
    ) -> Any:
        """
        Deletes a specified issue property from multiple Jira issues using filter criteria including entity IDs or property values.

        Args:
            propertyKey (string): propertyKey
            currentValue (string): The value of properties to perform the bulk operation on. Example: 'deprecated value'.
            entityIds (array): List of issues to perform the bulk delete operation on. Example: [10100, 100010].

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = {
            "currentValue": currentValue,
            "entityIds": entityIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_set_issue_property(
        self,
        propertyKey: str,
        expression: str | None = None,
        filter: Any | None = None,
        value: Any | None = None,
    ) -> Any:
        """
        Updates or sets a custom property value for a Jira issue identified by the property key, returning a status reference for asynchronous processing.

        Args:
            propertyKey (string): propertyKey
            expression (string): EXPERIMENTAL. The Jira expression to calculate the value of the property. The value of the expression must be an object that can be converted to JSON, such as a number, boolean, string, list, or map. The context variables available to the expression are `issue` and `user`. Issues for which the expression returns a value whose JSON representation is longer than 32768 characters are ignored.
            filter (string): The bulk operation filter. Example: {'currentValue': {'owner': 'admin', 'weight': 50}, 'entityIds': [10100, 100010], 'hasProperty': True}.
            value (string): The value of the property. The value must be a [valid](https://tools.ietf.org/html/rfc4627), non-empty JSON blob. The maximum length is 32768 characters. Example: {'owner': 'admin', 'weight': 100}.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        request_body_data = {
            "expression": expression,
            "filter": filter,
            "value": value,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def unarchive_issues(
        self, issueIdsOrKeys: list[str] | None = None
    ) -> dict[str, Any]:
        """
        Unarchives up to 1000 Jira issues in a single request using their IDs or keys, returning the count of unarchived issues and any errors encountered.

        Args:
            issueIdsOrKeys (array): issueIdsOrKeys Example: ['PR-1', '1001', 'PROJECT-2'].

        Returns:
            dict[str, Any]: Returned if there is at least one valid issue to unarchive in the request. It will return the count of unarchived issues, which also includes the count of the subtasks unarchived, and it will show the detailed errors for those issues which are not unarchived.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "issueIdsOrKeys": issueIdsOrKeys,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/unarchive"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_is_watching_issue_bulk(self, issueIds: list[str]) -> dict[str, Any]:
        """
        Determines whether the current user is watching specific issues using the Jira Cloud API, returning a status of whether the user is watching each provided issue.

        Args:
            issueIds (array): The list of issue IDs. Example: ['10001', '10002', '10005'].

        Returns:
            dict[str, Any]: Returned if the request is successful

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue watchers
        """
        request_body_data = None
        request_body_data = {
            "issueIds": issueIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/watching"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue(self, issueIdOrKey: str, deleteSubtasks: str | None = None) -> Any:
        """
        Deletes a Jira issue identified by its ID or key, optionally deleting associated subtasks if the `deleteSubtasks` query parameter is set to `true`.

        Args:
            issueIdOrKey (string): issueIdOrKey
            deleteSubtasks (string): Whether the issue's subtasks are deleted when the issue is deleted.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}"
        query_params = {
            k: v for k, v in [("deleteSubtasks", deleteSubtasks)] if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue(
        self,
        issueIdOrKey: str,
        fields: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        expand: str | None = None,
        properties: list[str] | None = None,
        updateHistory: bool | None = None,
        failFast: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves detailed information about a Jira issue using its ID or key, allowing optional parameters to specify fields, expansions, and additional data.

        Args:
            issueIdOrKey (string): issueIdOrKey
            fields (array): A list of fields to return for the issue. This parameter accepts a comma-separated list. Use it to retrieve a subset of fields. Allowed values: * `*all` Returns all fields. * `*navigable` Returns navigable fields. * Any issue field, prefixed with a minus to exclude. Examples: * `summary,comment` Returns only the summary and comments fields. * `-description` Returns all (default) fields except description. * `*navigable,-comment` Returns all navigable fields except comment. This parameter may be specified multiple times. For example, `fields=field1,field2& fields=field3`. Note: All fields are returned by default. This differs from [Search for issues using JQL (GET)](#api-rest-api-3-search-get) and [Search for issues using JQL (POST)](#api-rest-api-3-search-post) where the default is all navigable fields.
            fieldsByKeys (boolean): Whether fields in `fields` are referenced by keys rather than IDs. This parameter is useful where fields have been added by a connect app and a field's key may differ from its ID.
            expand (string): Use [expand](#expansion) to include additional information about the issues in the response. This parameter accepts a comma-separated list. Expand options include: * `renderedFields` Returns field values rendered in HTML format. * `names` Returns the display name of each field. * `schema` Returns the schema describing a field type. * `transitions` Returns all possible transitions for the issue. * `editmeta` Returns information about how each field can be edited. * `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent. * `versionedRepresentations` Returns a JSON array for each version of a field's value, with the highest number representing the most recent version. Note: When included in the request, the `fields` parameter is ignored.
            properties (array): A list of issue properties to return for the issue. This parameter accepts a comma-separated list. Allowed values: * `*all` Returns all issue properties. * Any issue property key, prefixed with a minus to exclude. Examples: * `*all` Returns all properties. * `*all,-prop1` Returns all properties except `prop1`. * `prop1,prop2` Returns `prop1` and `prop2` properties. This parameter may be specified multiple times. For example, `properties=prop1,prop2& properties=prop3`.
            updateHistory (boolean): Whether the project in which the issue is created is added to the user's **Recently viewed** project list, as shown under **Projects** in Jira. This also populates the [JQL issues search](#api-rest-api-3-search-get) `lastViewed` field.
            failFast (boolean): Whether to fail the request quickly in case of an error while loading fields for an issue. For `failFast=true`, if one field fails, the entire operation fails. For `failFast=false`, the operation will continue even if a field fails. It will return a valid response, but without values for the failed field(s).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}"
        query_params = {
            k: v
            for k, v in [
                ("fields", fields),
                ("fieldsByKeys", fieldsByKeys),
                ("expand", expand),
                ("properties", properties),
                ("updateHistory", updateHistory),
                ("failFast", failFast),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def edit_issue(
        self,
        issueIdOrKey: str,
        notifyUsers: bool | None = None,
        overrideScreenSecurity: bool | None = None,
        overrideEditableFlag: bool | None = None,
        returnIssue: bool | None = None,
        expand: str | None = None,
        fields: dict[str, Any] | None = None,
        historyMetadata: Any | None = None,
        properties: list[dict[str, Any]] | None = None,
        transition: Any | None = None,
        update: dict[str, list[dict[str, Any]]] | None = None,
    ) -> Any:
        """
        Updates an issue in Jira using the specified issue ID or key, allowing modification of issue fields, with optional parameters to control notification, screen security, editable flags, and response details.

        Args:
            issueIdOrKey (string): issueIdOrKey
            notifyUsers (boolean): Whether a notification email about the issue update is sent to all watchers. To disable the notification, administer Jira or administer project permissions are required. If the user doesn't have the necessary permission the request is ignored.
            overrideScreenSecurity (boolean): Whether screen security is overridden to enable hidden fields to be edited. Available to Connect app users with *Administer Jira* [global permission]( and Forge apps acting on behalf of users with *Administer Jira* [global permission](
            overrideEditableFlag (boolean): Whether screen security is overridden to enable uneditable fields to be edited. Available to Connect app users with *Administer Jira* [global permission]( and Forge apps acting on behalf of users with *Administer Jira* [global permission](
            returnIssue (boolean): Whether the response should contain the issue with fields edited in this request. The returned issue will have the same format as in the [Get issue API](#api-rest-api-3-issue-issueidorkey-get).
            expand (string): The Get issue API expand parameter to use in the response if the `returnIssue` parameter is `true`.
            fields (object): List of issue screen fields to update, specifying the sub-field to update and its value for each field. This field provides a straightforward option when setting a sub-field. When multiple sub-fields or other operations are required, use `update`. Fields included in here cannot be included in `update`. Example: {'customfield_10000': {'content': [{'content': [{'text': 'Investigation underway', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'customfield_10010': 1, 'summary': 'Completed orders still displaying in pending'}.
            historyMetadata (string): Additional issue history details. Example: {'activityDescription': 'Complete order processing', 'actor': {'avatarUrl': 'http://mysystem/avatar/tony.jpg', 'displayName': 'Tony', 'id': 'tony', 'type': 'mysystem-user', 'url': 'http://mysystem/users/tony'}, 'cause': {'id': 'myevent', 'type': 'mysystem-event'}, 'description': 'From the order testing process', 'extraData': {'Iteration': '10a', 'Step': '4'}, 'generator': {'id': 'mysystem-1', 'type': 'mysystem-application'}, 'type': 'myplugin:type'}.
            properties (array): Details of issue properties to be add or update. Example: [{'key': 'key1', 'value': 'Order number 10784'}, {'key': 'key2', 'value': 'Order number 10923'}].
            transition (string): Details of a transition. Required when performing a transition, optional when creating or editing an issue.
            update (object): A Map containing the field field name and a list of operations to perform on the issue screen field. Note that fields included in here cannot be included in `fields`. Example: {'components': [{'set': ''}], 'labels': [{'add': 'triaged'}, {'remove': 'blocker'}], 'summary': [{'set': 'Bug in business logic'}], 'timetracking': [{'edit': {'originalEstimate': '1w 1d', 'remainingEstimate': '4d'}}]}.

        Returns:
            Any: Returned if the request is successful and the `returnIssue` parameter is `true`

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "fields": fields,
            "historyMetadata": historyMetadata,
            "properties": properties,
            "transition": transition,
            "update": update,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}"
        query_params = {
            k: v
            for k, v in [
                ("notifyUsers", notifyUsers),
                ("overrideScreenSecurity", overrideScreenSecurity),
                ("overrideEditableFlag", overrideEditableFlag),
                ("returnIssue", returnIssue),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def assign_issue(
        self,
        issueIdOrKey: str,
        accountId: str | None = None,
        accountType: str | None = None,
        active: bool | None = None,
        applicationRoles: Any | None = None,
        avatarUrls: Any | None = None,
        displayName: str | None = None,
        emailAddress: str | None = None,
        expand: str | None = None,
        groups: Any | None = None,
        key: str | None = None,
        locale: str | None = None,
        name: str | None = None,
        self_arg_body: str | None = None,
        timeZone: str | None = None,
    ) -> Any:
        """
        Assigns or unassigns a Jira issue to a specific user, sets it to unassigned, or assigns it to the project's default assignee using the provided account ID or null value.

        Args:
            issueIdOrKey (string): issueIdOrKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Required in requests. Example: '5b10ac8d82e05b22cc7d4ef5'.
            accountType (string): The user account type. Can take the following values:

         *  `atlassian` regular Atlassian user account
         *  `app` system account used for Connect applications and OAuth to represent external systems
         *  `customer` Jira Service Desk account representing an external service desk
            active (boolean): Whether the user is active.
            applicationRoles (string): The application roles the user is assigned to.
            avatarUrls (string): The avatars of the user.
            displayName (string): The display name of the user. Depending on the user’s privacy setting, this may return an alternative value.
            emailAddress (string): The email address of the user. Depending on the user’s privacy setting, this may be returned as null.
            expand (string): Expand options that include additional user details in the response.
            groups (string): The groups that the user belongs to.
            key (string): This property is no longer available and will be removed from the documentation soon. See the [deprecation notice](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            locale (string): The locale of the user. Depending on the user’s privacy setting, this may be returned as null.
            name (string): This property is no longer available and will be removed from the documentation soon. See the [deprecation notice](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            self_arg_body (string): The URL of the user.
            timeZone (string): The time zone specified in the user's profile. If the user's time zone is not visible to the current user (due to user's profile setting), or if a time zone has not been set, the instance's default time zone will be returned.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "accountId": accountId,
            "accountType": accountType,
            "active": active,
            "applicationRoles": applicationRoles,
            "avatarUrls": avatarUrls,
            "displayName": displayName,
            "emailAddress": emailAddress,
            "expand": expand,
            "groups": groups,
            "key": key,
            "locale": locale,
            "name": name,
            "self": self_arg_body,
            "timeZone": timeZone,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/assignee"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_attachment(
        self, issueIdOrKey: str, items: list[dict[str, Any]]
    ) -> list[Any]:
        """
        Adds one or more attachments to a specified Jira issue using the "POST" method, with the issue identified by its ID or key.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue attachments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        files_data = None
        # Using array parameter 'items' directly as request body
        request_body_data = items
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/attachments"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            files=files_data,
            params=query_params,
            content_type="multipart/form-data",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_change_logs(
        self,
        issueIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves paginated changelog history for a specified Jira issue, including parameters for result pagination.

        Args:
            issueIdOrKey (string): issueIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/changelog"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_change_logs_by_ids(
        self, issueIdOrKey: str, changelogIds: list[int]
    ) -> dict[str, Any]:
        """
        Retrieves the full changelog history for a specified Jira issue using its ID or key, allowing for pagination and retrieval of all changes.

        Args:
            issueIdOrKey (string): issueIdOrKey
            changelogIds (array): The list of changelog IDs. Example: [10001, 10002].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "changelogIds": changelogIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/changelog/list"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_comments(
        self,
        issueIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves all comments for a specified Jira issue using pagination parameters.

        Args:
            issueIdOrKey (string): issueIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field. Accepts *created* to sort comments by their created date.
            expand (string): Use [expand](#expansion) to include additional information about comments in the response. This parameter accepts `renderedBody`, which returns the comment body rendered in HTML.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/comment"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_comment(
        self,
        issueIdOrKey: str,
        expand: str | None = None,
        author: Any | None = None,
        body: Any | None = None,
        created: str | None = None,
        id: str | None = None,
        jsdAuthorCanSeeRequest: bool | None = None,
        jsdPublic: bool | None = None,
        properties: list[dict[str, Any]] | None = None,
        renderedBody: str | None = None,
        self_arg_body: str | None = None,
        updateAuthor: Any | None = None,
        updated: str | None = None,
        visibility: Any | None = None,
    ) -> dict[str, Any]:
        """
        Adds a comment to a Jira issue with support for visibility settings and returns the created comment.

        Args:
            issueIdOrKey (string): issueIdOrKey
            expand (string): Use [expand](#expansion) to include additional information about comments in the response. This parameter accepts `renderedBody`, which returns the comment body rendered in HTML.
            author (string): The ID of the user who created the comment.
            body (string): The comment text in [Atlassian Document Format](https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/). Example: {'content': [{'content': [{'text': 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Pellentesque eget venenatis elit. Duis eu justo eget augue iaculis fermentum. Sed semper quam laoreet nisi egestas at posuere augue semper.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}.
            created (string): The date and time at which the comment was created.
            id (string): The ID of the comment.
            jsdAuthorCanSeeRequest (boolean): Whether the comment was added from an email sent by a person who is not part of the issue. See [Allow external emails to be added as comments on issues](https://support.atlassian.com/jira-service-management-cloud/docs/allow-external-emails-to-be-added-as-comments-on-issues/)for information on setting up this feature.
            jsdPublic (boolean): Whether the comment is visible in Jira Service Desk. Defaults to true when comments are created in the Jira Cloud Platform. This includes when the site doesn't use Jira Service Desk or the project isn't a Jira Service Desk project and, therefore, there is no Jira Service Desk for the issue to be visible on. To create a comment with its visibility in Jira Service Desk set to false, use the Jira Service Desk REST API [Create request comment](https://developer.atlassian.com/cloud/jira/service-desk/rest/#api-rest-servicedeskapi-request-issueIdOrKey-comment-post) operation.
            properties (array): A list of comment properties. Optional on create and update.
            renderedBody (string): The rendered version of the comment.
            self_arg_body (string): The URL of the comment.
            updateAuthor (string): The ID of the user who updated the comment last.
            updated (string): The date and time at which the comment was updated last.
            visibility (string): The group or role to which this comment is visible. Optional on create and update. Example: {'identifier': 'Administrators', 'type': 'role', 'value': 'Administrators'}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "author": author,
            "body": body,
            "created": created,
            "id": id,
            "jsdAuthorCanSeeRequest": jsdAuthorCanSeeRequest,
            "jsdPublic": jsdPublic,
            "properties": properties,
            "renderedBody": renderedBody,
            "self": self_arg_body,
            "updateAuthor": updateAuthor,
            "updated": updated,
            "visibility": visibility,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/comment"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_comment(self, issueIdOrKey: str, id: str) -> Any:
        """
        Deletes a specific comment from a Jira issue using the comment ID and issue identifier.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/comment/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_comment(
        self, issueIdOrKey: str, id: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a specific comment from a Jira issue using its ID and returns the comment details.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about comments in the response. This parameter accepts `renderedBody`, which returns the comment body rendered in HTML.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/comment/{id}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_comment(
        self,
        issueIdOrKey: str,
        id: str,
        notifyUsers: bool | None = None,
        overrideEditableFlag: bool | None = None,
        expand: str | None = None,
        author: Any | None = None,
        body: Any | None = None,
        created: str | None = None,
        id_body: str | None = None,
        jsdAuthorCanSeeRequest: bool | None = None,
        jsdPublic: bool | None = None,
        properties: list[dict[str, Any]] | None = None,
        renderedBody: str | None = None,
        self_arg_body: str | None = None,
        updateAuthor: Any | None = None,
        updated: str | None = None,
        visibility: Any | None = None,
    ) -> dict[str, Any]:
        """
        Updates an existing comment on a Jira issue and returns the modified comment details.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id
            notifyUsers (boolean): Whether users are notified when a comment is updated.
            overrideEditableFlag (boolean): Whether screen security is overridden to enable uneditable fields to be edited. Available to Connect app users with the *Administer Jira* [global permission]( and Forge apps acting on behalf of users with *Administer Jira* [global permission](
            expand (string): Use [expand](#expansion) to include additional information about comments in the response. This parameter accepts `renderedBody`, which returns the comment body rendered in HTML.
            author (string): The ID of the user who created the comment.
            body (string): The comment text in [Atlassian Document Format](https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/). Example: {'content': [{'content': [{'text': 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Pellentesque eget venenatis elit. Duis eu justo eget augue iaculis fermentum. Sed semper quam laoreet nisi egestas at posuere augue semper.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}.
            created (string): The date and time at which the comment was created.
            id_body (string): The ID of the comment.
            jsdAuthorCanSeeRequest (boolean): Whether the comment was added from an email sent by a person who is not part of the issue. See [Allow external emails to be added as comments on issues](https://support.atlassian.com/jira-service-management-cloud/docs/allow-external-emails-to-be-added-as-comments-on-issues/)for information on setting up this feature.
            jsdPublic (boolean): Whether the comment is visible in Jira Service Desk. Defaults to true when comments are created in the Jira Cloud Platform. This includes when the site doesn't use Jira Service Desk or the project isn't a Jira Service Desk project and, therefore, there is no Jira Service Desk for the issue to be visible on. To create a comment with its visibility in Jira Service Desk set to false, use the Jira Service Desk REST API [Create request comment](https://developer.atlassian.com/cloud/jira/service-desk/rest/#api-rest-servicedeskapi-request-issueIdOrKey-comment-post) operation.
            properties (array): A list of comment properties. Optional on create and update.
            renderedBody (string): The rendered version of the comment.
            self_arg_body (string): The URL of the comment.
            updateAuthor (string): The ID of the user who updated the comment last.
            updated (string): The date and time at which the comment was updated last.
            visibility (string): The group or role to which this comment is visible. Optional on create and update. Example: {'identifier': 'Administrators', 'type': 'role', 'value': 'Administrators'}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue comments
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "author": author,
            "body": body,
            "created": created,
            "id": id_body,
            "jsdAuthorCanSeeRequest": jsdAuthorCanSeeRequest,
            "jsdPublic": jsdPublic,
            "properties": properties,
            "renderedBody": renderedBody,
            "self": self_arg_body,
            "updateAuthor": updateAuthor,
            "updated": updated,
            "visibility": visibility,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/comment/{id}"
        query_params = {
            k: v
            for k, v in [
                ("notifyUsers", notifyUsers),
                ("overrideEditableFlag", overrideEditableFlag),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_edit_issue_meta(
        self,
        issueIdOrKey: str,
        overrideScreenSecurity: bool | None = None,
        overrideEditableFlag: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves editable field metadata and supported operations for a specific Jira issue to guide modifications via the API.

        Args:
            issueIdOrKey (string): issueIdOrKey
            overrideScreenSecurity (boolean): Whether hidden fields are returned. Available to Connect app users with *Administer Jira* [global permission]( and Forge apps acting on behalf of users with *Administer Jira* [global permission](
            overrideEditableFlag (boolean): Whether non-editable fields are returned. Available to Connect app users with *Administer Jira* [global permission]( and Forge apps acting on behalf of users with *Administer Jira* [global permission](

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/editmeta"
        query_params = {
            k: v
            for k, v in [
                ("overrideScreenSecurity", overrideScreenSecurity),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def notify(
        self,
        issueIdOrKey: str,
        htmlBody: str | None = None,
        restrict: Any | None = None,
        subject: str | None = None,
        textBody: str | None = None,
        to: Any | None = None,
    ) -> Any:
        """
        Sends notifications related to a specific Jira issue, identified by its ID or key, allowing customization of the notification content and recipients.

        Args:
            issueIdOrKey (string): issueIdOrKey
            htmlBody (string): The HTML body of the email notification for the issue. Example: 'The <strong>latest</strong> test results for this ticket are now available.'.
            restrict (string): Restricts the notifications to users with the specified permissions. Example: {'groupIds': [], 'groups': [{'name': 'notification-group'}], 'permissions': [{'key': 'BROWSE'}]}.
            subject (string): The subject of the email notification for the issue. If this is not specified, then the subject is set to the issue key and summary. Example: 'Latest test results'.
            textBody (string): The plain text body of the email notification for the issue. Example: 'The latest test results for this ticket are now available.'.
            to (string): The recipients of the email notification for the issue. Example: {'assignee': False, 'groupIds': [], 'groups': [{'name': 'notification-group'}], 'reporter': False, 'users': [{'accountId': '5b10a2844c20165700ede21g', 'active': False}], 'voters': True, 'watchers': True}.

        Returns:
            Any: Returned if the email is queued for sending.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "htmlBody": htmlBody,
            "restrict": restrict,
            "subject": subject,
            "textBody": textBody,
            "to": to,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/notify"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_property_keys(self, issueIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves the URLs and keys of all properties associated with a specified Jira issue using the issue ID or key.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_property(self, issueIdOrKey: str, propertyKey: str) -> Any:
        """
        Deletes a specific property from an issue in Jira, identified by its issue ID or key and the property key, using the Jira API.

        Args:
            issueIdOrKey (string): issueIdOrKey
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = (
            f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/properties/{propertyKey}"
        )
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_property(self, issueIdOrKey: str, propertyKey: str) -> dict[str, Any]:
        """
        Retrieves the value of a specific property associated with a Jira issue using the provided issue ID or key and property key.

        Args:
            issueIdOrKey (string): issueIdOrKey
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = (
            f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/properties/{propertyKey}"
        )
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_issue_property(self, issueIdOrKey: str, propertyKey: str) -> Any:
        """
        Updates an issue property in Jira using the PUT method, allowing users to set or modify custom data associated with an issue by issue ID or key and property key.

        Args:
            issueIdOrKey (string): issueIdOrKey
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the issue property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = (
            f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/properties/{propertyKey}"
        )
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_remote_link(self, issueIdOrKey: str, globalId: str) -> Any:
        """
        Deletes a remote issue link from a Jira issue using either the link's internal ID or its global ID.

        Args:
            issueIdOrKey (string): issueIdOrKey
            globalId (string): The global ID of a remote issue link. Example: 'system=http://www.mycompany.com/support&id=1'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink"
        query_params = {k: v for k, v in [("globalId", globalId)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_remote_issue_links(
        self, issueIdOrKey: str, globalId: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a list of remote links associated with a specified Jira issue, identified by its ID or key, using the GET method at the "/rest/api/3/issue/{issueIdOrKey}/remotelink" path, allowing for optional filtering by global ID.

        Args:
            issueIdOrKey (string): issueIdOrKey
            globalId (string): The global ID of the remote issue link.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink"
        query_params = {k: v for k, v in [("globalId", globalId)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_or_update_remote_issue_link(
        self,
        issueIdOrKey: str,
        object: Any,
        application: Any | None = None,
        globalId: str | None = None,
        relationship: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a remote link to an external object for a specified Jira issue, allowing users to associate external resources with issue tracking in Jira.

        Args:
            issueIdOrKey (string): issueIdOrKey
            object (string): Details of the item linked to. Example: {'icon': {'title': 'Support Ticket', 'url16x16': 'http://www.mycompany.com/support/ticket.png'}, 'status': {'icon': {'link': 'http://www.mycompany.com/support?id=1&details=closed', 'title': 'Case Closed', 'url16x16': 'http://www.mycompany.com/support/resolved.png'}, 'resolved': True}, 'summary': 'Customer support issue', 'title': 'TSTSUP-111', 'url': 'http://www.mycompany.com/support?id=1'}.
            application (string): Details of the remote application the linked item is in. For example, trello. Example: {'name': 'My Acme Tracker', 'type': 'com.acme.tracker'}.
            globalId (string): An identifier for the remote item in the remote system. For example, the global ID for a remote item in Confluence would consist of the app ID and page ID, like this: `appId=456&pageId=123`.

        Setting this field enables the remote issue link details to be updated or deleted using remote system and item details as the record identifier, rather than using the record's Jira ID.

        The maximum length is 255 characters. Example: 'system=http://www.mycompany.com/support&id=1'.
            relationship (string): Description of the relationship between the issue and the linked item. If not set, the relationship description "links to" is used in Jira. Example: 'causes'.

        Returns:
            dict[str, Any]: Returned if the remote issue link is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "application": application,
            "globalId": globalId,
            "object": object,
            "relationship": relationship,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_remote_issue_link_by_id(self, issueIdOrKey: str, linkId: str) -> Any:
        """
        Deletes a remote issue link from a specified Jira issue using the link's internal ID.

        Args:
            issueIdOrKey (string): issueIdOrKey
            linkId (string): linkId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if linkId is None:
            raise ValueError("Missing required parameter 'linkId'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink/{linkId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_remote_issue_link_by_id(
        self, issueIdOrKey: str, linkId: str
    ) -> dict[str, Any]:
        """
        Retrieves a specific remote link by its ID associated with a Jira issue, identified by the issue ID or key.

        Args:
            issueIdOrKey (string): issueIdOrKey
            linkId (string): linkId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if linkId is None:
            raise ValueError("Missing required parameter 'linkId'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink/{linkId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_remote_issue_link(
        self,
        issueIdOrKey: str,
        linkId: str,
        object: Any,
        application: Any | None = None,
        globalId: str | None = None,
        relationship: str | None = None,
    ) -> Any:
        """
        Updates a specific remote issue link by ID using a PUT request for the specified Jira issue.

        Args:
            issueIdOrKey (string): issueIdOrKey
            linkId (string): linkId
            object (string): Details of the item linked to. Example: {'icon': {'title': 'Support Ticket', 'url16x16': 'http://www.mycompany.com/support/ticket.png'}, 'status': {'icon': {'link': 'http://www.mycompany.com/support?id=1&details=closed', 'title': 'Case Closed', 'url16x16': 'http://www.mycompany.com/support/resolved.png'}, 'resolved': True}, 'summary': 'Customer support issue', 'title': 'TSTSUP-111', 'url': 'http://www.mycompany.com/support?id=1'}.
            application (string): Details of the remote application the linked item is in. For example, trello. Example: {'name': 'My Acme Tracker', 'type': 'com.acme.tracker'}.
            globalId (string): An identifier for the remote item in the remote system. For example, the global ID for a remote item in Confluence would consist of the app ID and page ID, like this: `appId=456&pageId=123`.

        Setting this field enables the remote issue link details to be updated or deleted using remote system and item details as the record identifier, rather than using the record's Jira ID.

        The maximum length is 255 characters. Example: 'system=http://www.mycompany.com/support&id=1'.
            relationship (string): Description of the relationship between the issue and the linked item. If not set, the relationship description "links to" is used in Jira. Example: 'causes'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue remote links
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if linkId is None:
            raise ValueError("Missing required parameter 'linkId'.")
        request_body_data = None
        request_body_data = {
            "application": application,
            "globalId": globalId,
            "object": object,
            "relationship": relationship,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/remotelink/{linkId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_transitions(
        self,
        issueIdOrKey: str,
        expand: str | None = None,
        transitionId: str | None = None,
        skipRemoteOnlyCondition: bool | None = None,
        includeUnavailableTransitions: bool | None = None,
        sortByOpsBarAndStatus: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves all available transitions for a Jira issue in its current status, including optional details like required fields and validation rules.

        Args:
            issueIdOrKey (string): issueIdOrKey
            expand (string): Use [expand](#expansion) to include additional information about transitions in the response. This parameter accepts `transitions.fields`, which returns information about the fields in the transition screen for each transition. Fields hidden from the screen are not returned. Use this information to populate the `fields` and `update` fields in [Transition issue](#api-rest-api-3-issue-issueIdOrKey-transitions-post).
            transitionId (string): The ID of the transition.
            skipRemoteOnlyCondition (boolean): Whether transitions with the condition *Hide From User Condition* are included in the response.
            includeUnavailableTransitions (boolean): Whether details of transitions that fail a condition are included in the response
            sortByOpsBarAndStatus (boolean): Whether the transitions are sorted by ops-bar sequence value first then category order (Todo, In Progress, Done) or only by ops-bar sequence value.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/transitions"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("transitionId", transitionId),
                ("skipRemoteOnlyCondition", skipRemoteOnlyCondition),
                ("includeUnavailableTransitions", includeUnavailableTransitions),
                ("sortByOpsBarAndStatus", sortByOpsBarAndStatus),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def do_transition(
        self,
        issueIdOrKey: str,
        fields: dict[str, Any] | None = None,
        historyMetadata: Any | None = None,
        properties: list[dict[str, Any]] | None = None,
        transition: Any | None = None,
        update: dict[str, list[dict[str, Any]]] | None = None,
    ) -> Any:
        """
        Transitions a Jira issue to a new workflow status using the specified transition ID.

        Args:
            issueIdOrKey (string): issueIdOrKey
            fields (object): List of issue screen fields to update, specifying the sub-field to update and its value for each field. This field provides a straightforward option when setting a sub-field. When multiple sub-fields or other operations are required, use `update`. Fields included in here cannot be included in `update`. Example: {'assignee': {'name': 'bob'}, 'resolution': {'name': 'Fixed'}}.
            historyMetadata (string): Additional issue history details. Example: {'activityDescription': 'Complete order processing', 'actor': {'avatarUrl': 'http://mysystem/avatar/tony.jpg', 'displayName': 'Tony', 'id': 'tony', 'type': 'mysystem-user', 'url': 'http://mysystem/users/tony'}, 'cause': {'id': 'myevent', 'type': 'mysystem-event'}, 'description': 'From the order testing process', 'extraData': {'Iteration': '10a', 'Step': '4'}, 'generator': {'id': 'mysystem-1', 'type': 'mysystem-application'}, 'type': 'myplugin:type'}.
            properties (array): Details of issue properties to be add or update.
            transition (string): Details of a transition. Required when performing a transition, optional when creating or editing an issue. Example: {'id': '5'}.
            update (object): A Map containing the field field name and a list of operations to perform on the issue screen field. Note that fields included in here cannot be included in `fields`. Example: {'comment': [{'add': {'body': {'content': [{'content': [{'text': 'Bug has been fixed', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}}}]}.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "fields": fields,
            "historyMetadata": historyMetadata,
            "properties": properties,
            "transition": transition,
            "update": update,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/transitions"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_vote(self, issueIdOrKey: str) -> Any:
        """
        Deletes a user's vote from a specified Jira issue, identified by its ID or key, using the Jira REST API.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue votes
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/votes"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_votes(self, issueIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves details about the votes on a specific Jira issue, identified by its issue ID or key.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue votes
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/votes"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_vote(self, issueIdOrKey: str) -> Any:
        """
        Casts a vote on a Jira issue and returns no content on success.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue votes
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/votes"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_watcher(
        self,
        issueIdOrKey: str,
        username: str | None = None,
        accountId: str | None = None,
    ) -> Any:
        """
        Removes a specified user as a watcher from a Jira issue via their username or account ID and returns a success status upon completion.

        Args:
            issueIdOrKey (string): issueIdOrKey
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Required. Example: '5b10ac8d82e05b22cc7d4ef5'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue watchers
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/watchers"
        query_params = {
            k: v
            for k, v in [("username", username), ("accountId", accountId)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_watchers(self, issueIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves the list of watchers for a specific Jira issue using the provided issue ID or key.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue watchers
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/watchers"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_watcher(self, issueIdOrKey: str) -> Any:
        """
        Adds a user as a watcher to a specified Jira issue by passing the user's account ID, returning a status message upon successful execution.

        Args:
            issueIdOrKey (string): issueIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue watchers
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/watchers"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_delete_worklogs(
        self,
        issueIdOrKey: str,
        ids: list[int],
        adjustEstimate: str | None = None,
        overrideEditableFlag: bool | None = None,
    ) -> Any:
        """
        Deletes a worklog from a specific issue in Jira using the provided issue ID or key, allowing for optional adjustments to the estimate and overriding of editable flags.

        Args:
            issueIdOrKey (string): issueIdOrKey
            ids (array): A list of worklog IDs. Example: [1, 2, 5, 10].
            adjustEstimate (string): Defines how to update the issue's time estimate, the options are: * `leave` Leaves the estimate unchanged. * `auto` Reduces the estimate by the aggregate value of `timeSpent` across all worklogs being deleted.
            overrideEditableFlag (boolean): Whether the work log entries should be removed to the issue even if the issue is not editable, because jira.issue.editable set to false or missing. For example, the issue is closed. Connect and Forge app users with admin permission can use this flag.

        Returns:
            Any: Returned if the bulk deletion request was partially successful, with a message indicating partial success.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = {
            "ids": ids,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog"
        query_params = {
            k: v
            for k, v in [
                ("adjustEstimate", adjustEstimate),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_worklog(
        self,
        issueIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        startedAfter: int | None = None,
        startedBefore: int | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of worklogs for a specific Jira issue using the "GET" method, allowing filtering by start date and other parameters.

        Args:
            issueIdOrKey (string): issueIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            startedAfter (integer): The worklog start date and time, as a UNIX timestamp in milliseconds, after which worklogs are returned.
            startedBefore (integer): The worklog start date and time, as a UNIX timestamp in milliseconds, before which worklogs are returned.
            expand (string): Use [expand](#expansion) to include additional information about worklogs in the response. This parameter accepts`properties`, which returns worklog properties.

        Returns:
            dict[str, Any]: Returned if the request is successful

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("startedAfter", startedAfter),
                ("startedBefore", startedBefore),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_worklog(
        self,
        issueIdOrKey: str,
        notifyUsers: bool | None = None,
        adjustEstimate: str | None = None,
        newEstimate: str | None = None,
        reduceBy: str | None = None,
        expand: str | None = None,
        overrideEditableFlag: bool | None = None,
        author: Any | None = None,
        comment: Any | None = None,
        created: str | None = None,
        id: str | None = None,
        issueId: str | None = None,
        properties: list[dict[str, Any]] | None = None,
        self_arg_body: str | None = None,
        started: str | None = None,
        timeSpent: str | None = None,
        timeSpentSeconds: int | None = None,
        updateAuthor: Any | None = None,
        updated: str | None = None,
        visibility: Any | None = None,
    ) -> dict[str, Any]:
        r"""
        Adds a worklog entry to a Jira issue for time tracking and returns the created worklog details.

        Args:
            issueIdOrKey (string): issueIdOrKey
            notifyUsers (boolean): Whether users watching the issue are notified by email.
            adjustEstimate (string): Defines how to update the issue's time estimate, the options are: * `new` Sets the estimate to a specific value, defined in `newEstimate`. * `leave` Leaves the estimate unchanged. * `manual` Reduces the estimate by amount specified in `reduceBy`. * `auto` Reduces the estimate by the value of `timeSpent` in the worklog.
            newEstimate (string): The value to set as the issue's remaining time estimate, as days (\#d), hours (\#h), or minutes (\#m or \#). For example, *2d*. Required when `adjustEstimate` is `new`.
            reduceBy (string): The amount to reduce the issue's remaining estimate by, as days (\#d), hours (\#h), or minutes (\#m). For example, *2d*. Required when `adjustEstimate` is `manual`.
            expand (string): Use [expand](#expansion) to include additional information about work logs in the response. This parameter accepts `properties`, which returns worklog properties.
            overrideEditableFlag (boolean): Whether the worklog entry should be added to the issue even if the issue is not editable, because jira.issue.editable set to false or missing. For example, the issue is closed. Connect and Forge app users with *Administer Jira* [global permission]( can use this flag.
            author (string): Details of the user who created the worklog.
            comment (string): A comment about the worklog in [Atlassian Document Format](https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/). Optional when creating or updating a worklog. Example: {'content': [{'content': [{'text': 'I did some work here.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}.
            created (string): The datetime on which the worklog was created.
            id (string): The ID of the worklog record.
            issueId (string): The ID of the issue this worklog is for.
            properties (array): Details of properties for the worklog. Optional when creating or updating a worklog.
            self_arg_body (string): The URL of the worklog item.
            started (string): The datetime on which the worklog effort was started. Required when creating a worklog. Optional when updating a worklog. Example: '2021-01-17T12:34:00.000+0000'.
            timeSpent (string): The time spent working on the issue as days (\#d), hours (\#h), or minutes (\#m or \#). Required when creating a worklog if `timeSpentSeconds` isn't provided. Optional when updating a worklog. Cannot be provided if `timeSpentSecond` is provided.
            timeSpentSeconds (integer): The time in seconds spent working on the issue. Required when creating a worklog if `timeSpent` isn't provided. Optional when updating a worklog. Cannot be provided if `timeSpent` is provided. Example: 12000.
            updateAuthor (string): Details of the user who last updated the worklog.
            updated (string): The datetime on which the worklog was last updated.
            visibility (string): Details about any restrictions in the visibility of the worklog. Optional when creating or updating a worklog. Example: {'identifier': '276f955c-63d7-42c8-9520-92d01dca0625', 'type': 'group'}.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "author": author,
            "comment": comment,
            "created": created,
            "id": id,
            "issueId": issueId,
            "properties": properties,
            "self": self_arg_body,
            "started": started,
            "timeSpent": timeSpent,
            "timeSpentSeconds": timeSpentSeconds,
            "updateAuthor": updateAuthor,
            "updated": updated,
            "visibility": visibility,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog"
        query_params = {
            k: v
            for k, v in [
                ("notifyUsers", notifyUsers),
                ("adjustEstimate", adjustEstimate),
                ("newEstimate", newEstimate),
                ("reduceBy", reduceBy),
                ("expand", expand),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_move_worklogs(
        self,
        issueIdOrKey: str,
        adjustEstimate: str | None = None,
        overrideEditableFlag: bool | None = None,
        ids: list[int] | None = None,
        issueIdOrKey_body: str | None = None,
    ) -> Any:
        """
        Moves worklogs from one Jira issue to another using the `POST` method, allowing adjustments to estimates and overriding editable flags if necessary.

        Args:
            issueIdOrKey (string): issueIdOrKey
            adjustEstimate (string): Defines how to update the issues' time estimate, the options are: * `leave` Leaves the estimate unchanged. * `auto` Reduces the estimate by the aggregate value of `timeSpent` across all worklogs being moved in the source issue, and increases it in the destination issue.
            overrideEditableFlag (boolean): Whether the work log entry should be moved to and from the issues even if the issues are not editable, because jira.issue.editable set to false or missing. For example, the issue is closed. Connect and Forge app users with admin permission can use this flag.
            ids (array): A list of worklog IDs. Example: [1, 2, 5, 10].
            issueIdOrKey_body (string): The issue id or key of the destination issue Example: 'ABC-1234'.

        Returns:
            Any: Returned if the request is partially successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "ids": ids,
            "issueIdOrKey": issueIdOrKey_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/move"
        query_params = {
            k: v
            for k, v in [
                ("adjustEstimate", adjustEstimate),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_worklog(
        self,
        issueIdOrKey: str,
        id: str,
        notifyUsers: bool | None = None,
        adjustEstimate: str | None = None,
        newEstimate: str | None = None,
        increaseBy: str | None = None,
        overrideEditableFlag: bool | None = None,
    ) -> Any:
        r"""
        Deletes a specific worklog entry from a Jira issue using the worklog ID and returns a success status upon removal.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id
            notifyUsers (boolean): Whether users watching the issue are notified by email.
            adjustEstimate (string): Defines how to update the issue's time estimate, the options are: * `new` Sets the estimate to a specific value, defined in `newEstimate`. * `leave` Leaves the estimate unchanged. * `manual` Increases the estimate by amount specified in `increaseBy`. * `auto` Reduces the estimate by the value of `timeSpent` in the worklog.
            newEstimate (string): The value to set as the issue's remaining time estimate, as days (\#d), hours (\#h), or minutes (\#m or \#). For example, *2d*. Required when `adjustEstimate` is `new`.
            increaseBy (string): The amount to increase the issue's remaining estimate by, as days (\#d), hours (\#h), or minutes (\#m or \#). For example, *2d*. Required when `adjustEstimate` is `manual`.
            overrideEditableFlag (boolean): Whether the work log entry should be added to the issue even if the issue is not editable, because jira.issue.editable set to false or missing. For example, the issue is closed. Connect and Forge app users with admin permission can use this flag.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{id}"
        query_params = {
            k: v
            for k, v in [
                ("notifyUsers", notifyUsers),
                ("adjustEstimate", adjustEstimate),
                ("newEstimate", newEstimate),
                ("increaseBy", increaseBy),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_worklog(
        self, issueIdOrKey: str, id: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a specific worklog by its ID for a given Jira issue using the GET method.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about work logs in the response. This parameter accepts `properties`, which returns worklog properties.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{id}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_worklog(
        self,
        issueIdOrKey: str,
        id: str,
        notifyUsers: bool | None = None,
        adjustEstimate: str | None = None,
        newEstimate: str | None = None,
        expand: str | None = None,
        overrideEditableFlag: bool | None = None,
        author: Any | None = None,
        comment: Any | None = None,
        created: str | None = None,
        id_body: str | None = None,
        issueId: str | None = None,
        properties: list[dict[str, Any]] | None = None,
        self_arg_body: str | None = None,
        started: str | None = None,
        timeSpent: str | None = None,
        timeSpentSeconds: int | None = None,
        updateAuthor: Any | None = None,
        updated: str | None = None,
        visibility: Any | None = None,
    ) -> dict[str, Any]:
        r"""
        Updates a specific worklog for an issue in Jira using the PUT method, allowing modifications to attributes such as the worklog comments, time spent, and start date, while requiring permissions to access and edit issue worklogs.

        Args:
            issueIdOrKey (string): issueIdOrKey
            id (string): id
            notifyUsers (boolean): Whether users watching the issue are notified by email.
            adjustEstimate (string): Defines how to update the issue's time estimate, the options are: * `new` Sets the estimate to a specific value, defined in `newEstimate`. * `leave` Leaves the estimate unchanged. * `auto` Updates the estimate by the difference between the original and updated value of `timeSpent` or `timeSpentSeconds`.
            newEstimate (string): The value to set as the issue's remaining time estimate, as days (\#d), hours (\#h), or minutes (\#m or \#). For example, *2d*. Required when `adjustEstimate` is `new`.
            expand (string): Use [expand](#expansion) to include additional information about worklogs in the response. This parameter accepts `properties`, which returns worklog properties.
            overrideEditableFlag (boolean): Whether the worklog should be added to the issue even if the issue is not editable. For example, because the issue is closed. Connect and Forge app users with *Administer Jira* [global permission]( can use this flag.
            author (string): Details of the user who created the worklog.
            comment (string): A comment about the worklog in [Atlassian Document Format](https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/). Optional when creating or updating a worklog. Example: {'content': [{'content': [{'text': 'I did some work here.', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}.
            created (string): The datetime on which the worklog was created.
            id_body (string): The ID of the worklog record.
            issueId (string): The ID of the issue this worklog is for.
            properties (array): Details of properties for the worklog. Optional when creating or updating a worklog.
            self_arg_body (string): The URL of the worklog item.
            started (string): The datetime on which the worklog effort was started. Required when creating a worklog. Optional when updating a worklog. Example: '2021-01-17T12:34:00.000+0000'.
            timeSpent (string): The time spent working on the issue as days (\#d), hours (\#h), or minutes (\#m or \#). Required when creating a worklog if `timeSpentSeconds` isn't provided. Optional when updating a worklog. Cannot be provided if `timeSpentSecond` is provided.
            timeSpentSeconds (integer): The time in seconds spent working on the issue. Required when creating a worklog if `timeSpent` isn't provided. Optional when updating a worklog. Cannot be provided if `timeSpent` is provided. Example: 12000.
            updateAuthor (string): Details of the user who last updated the worklog.
            updated (string): The datetime on which the worklog was last updated.
            visibility (string): Details about any restrictions in the visibility of the worklog. Optional when creating or updating a worklog. Example: {'identifier': '276f955c-63d7-42c8-9520-92d01dca0625', 'type': 'group'}.

        Returns:
            dict[str, Any]: Returned if the request is successful

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklogs
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "author": author,
            "comment": comment,
            "created": created,
            "id": id_body,
            "issueId": issueId,
            "properties": properties,
            "self": self_arg_body,
            "started": started,
            "timeSpent": timeSpent,
            "timeSpentSeconds": timeSpentSeconds,
            "updateAuthor": updateAuthor,
            "updated": updated,
            "visibility": visibility,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{id}"
        query_params = {
            k: v
            for k, v in [
                ("notifyUsers", notifyUsers),
                ("adjustEstimate", adjustEstimate),
                ("newEstimate", newEstimate),
                ("expand", expand),
                ("overrideEditableFlag", overrideEditableFlag),
            ]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_worklog_property_keys(
        self, issueIdOrKey: str, worklogId: str
    ) -> dict[str, Any]:
        """
        Retrieves the keys of all custom properties stored against a specific worklog entry in Jira issues.

        Args:
            issueIdOrKey (string): issueIdOrKey
            worklogId (string): worklogId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklog properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if worklogId is None:
            raise ValueError("Missing required parameter 'worklogId'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{worklogId}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_worklog_property(
        self, issueIdOrKey: str, worklogId: str, propertyKey: str
    ) -> Any:
        """
        Deletes a specific property from a Jira issue's worklog entry.

        Args:
            issueIdOrKey (string): issueIdOrKey
            worklogId (string): worklogId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the worklog property is removed.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklog properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if worklogId is None:
            raise ValueError("Missing required parameter 'worklogId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{worklogId}/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_worklog_property(
        self, issueIdOrKey: str, worklogId: str, propertyKey: str
    ) -> dict[str, Any]:
        """
        Retrieves the value of a specific property associated with a worklog for a given issue in Jira using the specified issue ID/key, worklog ID, and property key.

        Args:
            issueIdOrKey (string): issueIdOrKey
            worklogId (string): worklogId
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklog properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if worklogId is None:
            raise ValueError("Missing required parameter 'worklogId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{worklogId}/properties/{propertyKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_worklog_property(
        self, issueIdOrKey: str, worklogId: str, propertyKey: str
    ) -> Any:
        """
        Updates a specific property of a worklog in Jira using the PUT method, allowing for custom data storage against the worklog.

        Args:
            issueIdOrKey (string): issueIdOrKey
            worklogId (string): worklogId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the worklog property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue worklog properties
        """
        if issueIdOrKey is None:
            raise ValueError("Missing required parameter 'issueIdOrKey'.")
        if worklogId is None:
            raise ValueError("Missing required parameter 'worklogId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/issue/{issueIdOrKey}/worklog/{worklogId}/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def link_issues(
        self,
        inwardIssue: dict[str, Any],
        outwardIssue: dict[str, Any],
        type: dict[str, Any],
        comment: dict[str, Any] | None = None,
    ) -> Any:
        """
        Creates a link between two Jira issues, allowing you to define the relationship type and optionally include a comment, using the POST method at the "/rest/api/3/issueLink" endpoint.

        Args:
            inwardIssue (object): The ID or key of a linked issue. Example: {'key': 'HSP-1'}.
            outwardIssue (object): The ID or key of a linked issue. Example: {'key': 'MKY-1'}.
            type (object): This object is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it defines and reports on the type of link between the issues. Find a list of issue link types with [Get issue link types](#api-rest-api-3-issueLinkType-get).
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it defines and reports on issue link types. Example: {'name': 'Duplicate'}.
            comment (object): A comment. Example: {'body': {'content': [{'content': [{'text': 'Linked related issue!', 'type': 'text'}], 'type': 'paragraph'}], 'type': 'doc', 'version': 1}, 'visibility': {'identifier': '276f955c-63d7-42c8-9520-92d01dca0625', 'type': 'group', 'value': 'jira-software-users'}}.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue links
        """
        request_body_data = None
        request_body_data = {
            "comment": comment,
            "inwardIssue": inwardIssue,
            "outwardIssue": outwardIssue,
            "type": type,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issueLink"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_link(self, linkId: str) -> Any:
        """
        Deletes a specific issue link in Jira by its link ID and returns a success status.

        Args:
            linkId (string): linkId

        Returns:
            Any: 200 response

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue links
        """
        if linkId is None:
            raise ValueError("Missing required parameter 'linkId'.")
        url = f"{self.base_url}/rest/api/3/issueLink/{linkId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_link(self, linkId: str) -> dict[str, Any]:
        """
        Retrieves details of a specific issue link in Jira by its unique identifier using the Jira REST API.

        Args:
            linkId (string): linkId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue links
        """
        if linkId is None:
            raise ValueError("Missing required parameter 'linkId'.")
        url = f"{self.base_url}/rest/api/3/issueLink/{linkId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_link_types(self) -> dict[str, Any]:
        """
        Retrieves information about an issue link type in Jira using the provided ID.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue link types
        """
        url = f"{self.base_url}/rest/api/3/issueLinkType"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_link_type(
        self,
        id: str | None = None,
        inward: str | None = None,
        name: str | None = None,
        outward: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new issue link type in Jira to define relationships between linked issues.

        Args:
            id (string): The ID of the issue link type and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is the type of issue link. Required on create when `name` isn't provided. Otherwise, read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is read only.
            inward (string): The description of the issue link type inward link and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicated by'.
            name (string): The name of the issue link type and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is the type of issue link. Required on create when `id` isn't provided. Otherwise, read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicate'.
            outward (string): The description of the issue link type outward link and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicates'.
            self_arg_body (string): The URL of the issue link type. Read only.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue link types
        """
        request_body_data = None
        request_body_data = {
            "id": id,
            "inward": inward,
            "name": name,
            "outward": outward,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issueLinkType"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_link_type(self, issueLinkTypeId: str) -> Any:
        """
        Deletes a specified issue link type in Jira and returns a success status upon removal.

        Args:
            issueLinkTypeId (string): issueLinkTypeId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue link types
        """
        if issueLinkTypeId is None:
            raise ValueError("Missing required parameter 'issueLinkTypeId'.")
        url = f"{self.base_url}/rest/api/3/issueLinkType/{issueLinkTypeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_link_type(self, issueLinkTypeId: str) -> dict[str, Any]:
        """
        Retrieves a specific issue link type by its ID from Jira, including relationship descriptions for inward and outward links.

        Args:
            issueLinkTypeId (string): issueLinkTypeId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue link types
        """
        if issueLinkTypeId is None:
            raise ValueError("Missing required parameter 'issueLinkTypeId'.")
        url = f"{self.base_url}/rest/api/3/issueLinkType/{issueLinkTypeId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_link_type(
        self,
        issueLinkTypeId: str,
        id: str | None = None,
        inward: str | None = None,
        name: str | None = None,
        outward: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates the specified issue link type (e.g., Duplicate, Blocks) by ID to modify its name and relationship descriptions.

        Args:
            issueLinkTypeId (string): issueLinkTypeId
            id (string): The ID of the issue link type and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is the type of issue link. Required on create when `name` isn't provided. Otherwise, read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is read only.
            inward (string): The description of the issue link type inward link and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicated by'.
            name (string): The name of the issue link type and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is the type of issue link. Required on create when `id` isn't provided. Otherwise, read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicate'.
            outward (string): The description of the issue link type outward link and is used as follows:

         *  In the [ issueLink](#api-rest-api-3-issueLink-post) resource it is read only.
         *  In the [ issueLinkType](#api-rest-api-3-issueLinkType-post) resource it is required on create and optional on update. Otherwise, read only. Example: 'Duplicates'.
            self_arg_body (string): The URL of the issue link type. Read only.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue link types
        """
        if issueLinkTypeId is None:
            raise ValueError("Missing required parameter 'issueLinkTypeId'.")
        request_body_data = None
        request_body_data = {
            "id": id,
            "inward": inward,
            "name": name,
            "outward": outward,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issueLinkType/{issueLinkTypeId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def export_archived_issues(
        self,
        archivedBy: list[str] | None = None,
        archivedDateRange: dict[str, Any] | None = None,
        issueTypes: list[str] | None = None,
        projects: list[str] | None = None,
        reporters: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Exports archived issues using the Jira API, initiating a task that sends an email with a link to download a CSV file containing the issue details upon completion.

        Args:
            archivedBy (array): List archived issues archived by a specified account ID. Example: ['uuid-rep-001', 'uuid-rep-002'].
            archivedDateRange (object): List issues archived within a specified date range. Example: {'dateAfter': '2023-01-01', 'dateBefore': '2023-01-12'}.
            issueTypes (array): List archived issues with a specified issue type ID. Example: ['10001', '10002'].
            projects (array): List archived issues with a specified project key. Example: ['FOO', 'BAR'].
            reporters (array): List archived issues where the reporter is a specified account ID. Example: ['uuid-rep-001', 'uuid-rep-002'].

        Returns:
            dict[str, Any]: Returns the details of your export task. You can use the [get task](https://developer.atlassian.com/cloud/jira/platform/rest/v3/api-group-tasks/#api-rest-api-3-task-taskid-get) API to view the progress of your request.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issues
        """
        request_body_data = None
        request_body_data = {
            "archivedBy": archivedBy,
            "archivedDateRange": archivedDateRange,
            "issueTypes": issueTypes,
            "projects": projects,
            "reporters": reporters,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issues/archive/export"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_security_schemes(self) -> dict[str, Any]:
        """
        Retrieves a list of all issue security schemes available in a Jira instance, allowing administrators to manage which users or groups can view issues.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_security_scheme(
        self,
        name: str,
        description: str | None = None,
        levels: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Creates an issue security scheme in Jira Cloud using the POST method, allowing administrators to define security levels and members, and returns the ID of the newly created scheme upon success.

        Args:
            name (string): The name of the issue security scheme. Must be unique (case-insensitive). Example: 'New security scheme'.
            description (string): The description of the issue security scheme. Example: 'Newly created issue security scheme'.
            levels (array): The list of scheme levels which should be added to the security scheme. Example: [{'description': 'Newly created level', 'isDefault': True, 'members': [{'parameter': 'administrators', 'type': 'group'}], 'name': 'New level'}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "levels": levels,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_security_levels(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        schemeId: list[str] | None = None,
        onlyDefault: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves details of issue security levels within a scheme, including pagination support and filtering by scheme ID or default status.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of issue security scheme level IDs. To include multiple issue security levels, separate IDs with an ampersand: `id=10000&id=10001`.
            schemeId (array): The list of issue security scheme IDs. To include multiple issue security schemes, separate IDs with an ampersand: `schemeId=10000&schemeId=10001`.
            onlyDefault (boolean): When set to true, returns multiple default levels for each security scheme containing a default. If you provide scheme and level IDs not associated with the default, returns an empty page. The default value is false.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/level"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("schemeId", schemeId),
                ("onlyDefault", onlyDefault),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_default_levels(self, defaultValues: list[dict[str, Any]]) -> Any:
        """
        Sets default issue security levels for schemes, allowing administrators to configure which security levels are applied by default across specified issue security schemes.

        Args:
            defaultValues (array): List of objects with issue security scheme ID and new default level ID. Example: [{'defaultLevelId': '20000', 'issueSecuritySchemeId': '10000'}, {'defaultLevelId': '30000', 'issueSecuritySchemeId': '12000'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        request_body_data = None
        request_body_data = {
            "defaultValues": defaultValues,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/level/default"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_security_level_members(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        schemeId: list[str] | None = None,
        levelId: list[str] | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the members of a specific issue security level using the Jira Cloud API, allowing for pagination and expansion of details by specifying parameters such as start index, maximum results, and expansion options.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of issue security level member IDs. To include multiple issue security level members separate IDs with an ampersand: `id=10000&id=10001`.
            schemeId (array): The list of issue security scheme IDs. To include multiple issue security schemes separate IDs with an ampersand: `schemeId=10000&schemeId=10001`.
            levelId (array): The list of issue security level IDs. To include multiple issue security levels separate IDs with an ampersand: `levelId=10000&levelId=10001`.
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `all` Returns all expandable information * `field` Returns information about the custom field granted the permission * `group` Returns information about the group that is granted the permission * `projectRole` Returns information about the project role granted the permission * `user` Returns information about the user who is granted the permission

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/level/member"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("schemeId", schemeId),
                ("levelId", levelId),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_security_schemes_by_project(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        issueSecuritySchemeId: list[str] | None = None,
        projectId: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves projects associated with specific issue security schemes based on scheme ID or project ID query parameters.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            issueSecuritySchemeId (array): The list of security scheme IDs to be filtered out.
            projectId (array): The list of project IDs to be filtered out.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("issueSecuritySchemeId", issueSecuritySchemeId),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def associate_schemes_to_projects(
        self,
        projectId: str,
        schemeId: str,
        oldToNewSecurityLevelMappings: list[dict[str, Any]] | None = None,
    ) -> Any:
        """
        Associates an issue security scheme with a project using the Jira Cloud API, allowing for the remapping of security levels for issues, with the operation being asynchronous.

        Args:
            projectId (string): The ID of the project. Example: '10000'.
            schemeId (string): The ID of the issue security scheme. Providing null will clear the association with the issue security scheme. Example: '20000'.
            oldToNewSecurityLevelMappings (array): The list of scheme levels which should be remapped to new levels of the issue security scheme. Example: [{'newLevelId': '30001', 'oldLevelId': '30000'}].

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        request_body_data = None
        request_body_data = {
            "oldToNewSecurityLevelMappings": oldToNewSecurityLevelMappings,
            "projectId": projectId,
            "schemeId": schemeId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/project"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_security_schemes(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        projectId: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Searches for and returns issue security schemes in Jira Cloud, allowing filtering by start index, maximum results, ID, or project ID.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of issue security scheme IDs. To include multiple issue security scheme IDs, separate IDs with an ampersand: `id=10000&id=10001`.
            projectId (array): The list of project IDs. To include multiple project IDs, separate IDs with an ampersand: `projectId=10000&projectId=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_security_scheme(self, id: str) -> dict[str, Any]:
        """
        Retrieves the details of a specific issue security scheme by its ID, including associated security levels and project mappings.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_security_scheme(
        self, id: str, description: str | None = None, name: str | None = None
    ) -> Any:
        """
        Updates an existing issue security scheme by specifying the ID in the path using the Jira Cloud API.

        Args:
            id (string): id
            description (string): The description of the security scheme scheme. Example: 'My issue security scheme description'.
            name (string): The name of the security scheme scheme. Must be unique. Example: 'My issue security scheme name'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_security_level_members(
        self,
        issueSecuritySchemeId: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        issueSecurityLevelId: list[str] | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of members associated with issue security levels in a specified issue security scheme using the Jira API.

        Args:
            issueSecuritySchemeId (string): issueSecuritySchemeId
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            issueSecurityLevelId (array): The list of issue security level IDs. To include multiple issue security levels separate IDs with ampersand: `issueSecurityLevelId=10000&issueSecurityLevelId=10001`.
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security level
        """
        if issueSecuritySchemeId is None:
            raise ValueError("Missing required parameter 'issueSecuritySchemeId'.")
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{issueSecuritySchemeId}/members"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("issueSecurityLevelId", issueSecurityLevelId),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_security_scheme(self, schemeId: str) -> Any:
        """
        Deletes an issue security scheme in Jira and disassociates it from all projects.

        Args:
            schemeId (string): schemeId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_security_level(
        self, schemeId: str, levels: list[dict[str, Any]] | None = None
    ) -> Any:
        """
        Updates an issue security level within a specified security scheme in Jira.

        Args:
            schemeId (string): schemeId
            levels (array): The list of scheme levels which should be added to the security scheme. Example: [{'description': 'First Level Description', 'isDefault': True, 'members': [{'type': 'reporter'}, {'parameter': 'jira-administrators', 'type': 'group'}], 'name': 'First Level'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        request_body_data = None
        request_body_data = {
            "levels": levels,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}/level"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_level(
        self, schemeId: str, levelId: str, replaceWith: str | None = None
    ) -> Any:
        """
        Deletes an issue security level with the specified `levelId` from an issue security scheme identified by `schemeId`, optionally allowing replacement with another level using the `replaceWith` query parameter.

        Args:
            schemeId (string): schemeId
            levelId (string): levelId
            replaceWith (string): The ID of the issue security level that will replace the currently selected level.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if levelId is None:
            raise ValueError("Missing required parameter 'levelId'.")
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}/level/{levelId}"
        query_params = {
            k: v for k, v in [("replaceWith", replaceWith)] if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_security_level(
        self,
        schemeId: str,
        levelId: str,
        description: str | None = None,
        name: str | None = None,
    ) -> Any:
        """
        Updates an issue security level in a Jira issue security scheme by modifying its name and description using the `PUT` method.

        Args:
            schemeId (string): schemeId
            levelId (string): levelId
            description (string): The description of the issue security scheme level. Example: 'New level description'.
            name (string): The name of the issue security scheme level. Must be unique. Example: 'New level name'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if levelId is None:
            raise ValueError("Missing required parameter 'levelId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}/level/{levelId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_security_level_members(
        self,
        schemeId: str,
        levelId: str,
        members: list[dict[str, Any]] | None = None,
    ) -> Any:
        """
        Adds members to a specific security level within an issue security scheme in Jira.

        Args:
            schemeId (string): schemeId
            levelId (string): levelId
            members (array): The list of level members which should be added to the issue security scheme level. Example: [{'type': 'reporter'}, {'parameter': 'jira-administrators', 'type': 'group'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if levelId is None:
            raise ValueError("Missing required parameter 'levelId'.")
        request_body_data = None
        request_body_data = {
            "members": members,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}/level/{levelId}/member"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_member_from_security_level(
        self, schemeId: str, levelId: str, memberId: str
    ) -> Any:
        """
        Removes a specified member from an issue security level within a Jira issue security scheme.

        Args:
            schemeId (string): schemeId
            levelId (string): levelId
            memberId (string): memberId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if levelId is None:
            raise ValueError("Missing required parameter 'levelId'.")
        if memberId is None:
            raise ValueError("Missing required parameter 'memberId'.")
        url = f"{self.base_url}/rest/api/3/issuesecurityschemes/{schemeId}/level/{levelId}/member/{memberId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_all_types(self) -> list[Any]:
        """
        Retrieves a list of all issue types available in Jira using the GET method at the "/rest/api/3/issuetype" endpoint.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        url = f"{self.base_url}/rest/api/3/issuetype"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_type(
        self,
        name: str,
        description: str | None = None,
        hierarchyLevel: int | None = None,
        type: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new issue type in Jira and adds it to the default issue type scheme.

        Args:
            name (string): The unique name for the issue type. The maximum length is 60 characters. Example: 'name'.
            description (string): The description of the issue type. Example: 'description'.
            hierarchyLevel (integer): The hierarchy level of the issue type. Use:

         *  `-1` for Subtask.
         *  `0` for Base.

        Defaults to `0`.
            type (string): Deprecated. Use `hierarchyLevel` instead. See the [deprecation notice](https://community.developer.atlassian.com/t/deprecation-of-the-epic-link-parent-link-and-other-related-fields-in-rest-apis-and-webhooks/54048) for details.

        Whether the issue type is `subtype` or `standard`. Defaults to `standard`. Example: 'standard'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "hierarchyLevel": hierarchyLevel,
            "name": name,
            "type": type,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetype"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_types_for_project(
        self, projectId: int, level: int | None = None
    ) -> list[Any]:
        """
        Retrieves issue types associated with a specified project in Jira using the "GET" method at the "/rest/api/3/issuetype/project" endpoint.

        Args:
            projectId (integer): The ID of the project.
            level (integer): The level of the issue type to filter by. Use: * `-1` for Subtask. * `0` for Base. * `1` for Epic.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        url = f"{self.base_url}/rest/api/3/issuetype/project"
        query_params = {
            k: v
            for k, v in [("projectId", projectId), ("level", level)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_type(
        self, id: str, alternativeIssueTypeId: str | None = None
    ) -> Any:
        """
        Deletes the specified Jira issue type and migrates associated issues to an alternative type if provided.

        Args:
            id (string): id
            alternativeIssueTypeId (string): The ID of the replacement issue type.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{id}"
        query_params = {
            k: v
            for k, v in [("alternativeIssueTypeId", alternativeIssueTypeId)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type(self, id: str) -> dict[str, Any]:
        """
        Retrieves detailed information about a specific issue type in Jira by its ID using the "GET" method.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_type(
        self,
        id: str,
        avatarId: int | None = None,
        description: str | None = None,
        name: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates an existing Jira issue type by its ID, returning the modified issue type details or relevant error responses.

        Args:
            id (string): id
            avatarId (integer): The ID of an issue type avatar. Example: 1.
            description (string): The description of the issue type. Example: 'description'.
            name (string): The unique name for the issue type. The maximum length is 60 characters. Example: 'name'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "avatarId": avatarId,
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetype/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_alternative_issue_types(self, id: str) -> list[Any]:
        """
        Retrieves alternative issue types for a specified issue type ID using the Jira Cloud REST API.

        Args:
            id (string): id

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{id}/alternatives"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_type_avatar(
        self,
        id: str,
        size: int,
        body_content: bytes,
        x: int | None = None,
        y: int | None = None,
    ) -> dict[str, Any]:
        """
        Generates an avatar for a specific issue type in Jira using the "POST" method at the path "/rest/api/3/issuetype/{id}/avatar2", allowing parameters such as size and other query parameters.

        Args:
            id (string): id
            size (integer): The length of each side of the crop region.
            body_content (bytes | None): Raw binary content for the request body.
            x (integer): The X coordinate of the top-left corner of the crop region.
            y (integer): The Y coordinate of the top-left corner of the crop region.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue types
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = body_content
        url = f"{self.base_url}/rest/api/3/issuetype/{id}/avatar2"
        query_params = {
            k: v for k, v in [("x", x), ("y", y), ("size", size)] if v is not None
        }
        response = self._post(
            url, data=request_body_data, params=query_params, content_type="*/*"
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type_property_keys(self, issueTypeId: str) -> dict[str, Any]:
        """
        Retrieves all property keys for a specific Jira issue type using the issueTypeId path parameter.

        Args:
            issueTypeId (string): issueTypeId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type properties
        """
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{issueTypeId}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_type_property(self, issueTypeId: str, propertyKey: str) -> Any:
        """
        Deletes a specific property from an issue type in Jira using the specified property key and issue type ID.

        Args:
            issueTypeId (string): issueTypeId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the issue type property is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type properties
        """
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{issueTypeId}/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type_property(
        self, issueTypeId: str, propertyKey: str
    ) -> dict[str, Any]:
        """
        Retrieves a specific custom property associated with an issue type using its unique identifier and property key.

        Args:
            issueTypeId (string): issueTypeId
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type properties
        """
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/issuetype/{issueTypeId}/properties/{propertyKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_issue_type_property(self, issueTypeId: str, propertyKey: str) -> Any:
        """
        Creates or updates a custom property value for a specific Jira issue type, requiring admin permissions and returning success codes for creation/update.

        Args:
            issueTypeId (string): issueTypeId
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the issue type property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type properties
        """
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/issuetype/{issueTypeId}/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_issue_type_schemes(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
        orderBy: str | None = None,
        expand: str | None = None,
        queryString: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of issue type schemes with optional filtering by ID, ordering, and expansion of related entities.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of issue type schemes IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.
            orderBy (string): [Order](#ordering) the results by a field: * `name` Sorts by issue type scheme name. * `id` Sorts by issue type scheme ID.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `projects` For each issue type schemes, returns information about the projects the issue type scheme is assigned to. * `issueTypes` For each issue type schemes, returns information about the issueTypes the issue type scheme have.
            queryString (string): String used to perform a case-insensitive partial match with issue type scheme name.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescheme"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("orderBy", orderBy),
                ("expand", expand),
                ("queryString", queryString),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_type_scheme(
        self,
        issueTypeIds: list[str],
        name: str,
        defaultIssueTypeId: str | None = None,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new issue type scheme in Jira and returns the created resource.

        Args:
            issueTypeIds (array): The list of issue types IDs of the issue type scheme. At least one standard issue type ID is required. Example: ['10001', '10002', '10003'].
            name (string): The name of the issue type scheme. The name must be unique. The maximum length is 255 characters. Example: 'Kanban Issue Type Scheme'.
            defaultIssueTypeId (string): The ID of the default issue type of the issue type scheme. This ID must be included in `issueTypeIds`. Example: '10002'.
            description (string): The description of the issue type scheme. The maximum length is 4000 characters. Example: 'A collection of issue types suited to use in a kanban style project.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        request_body_data = None
        request_body_data = {
            "defaultIssueTypeId": defaultIssueTypeId,
            "description": description,
            "issueTypeIds": issueTypeIds,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type_schemes_mapping(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        issueTypeSchemeId: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of issue type scheme mappings for classic Jira projects, filtered by scheme ID.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            issueTypeSchemeId (array): The list of issue type scheme IDs. To include multiple IDs, provide an ampersand-separated list. For example, `issueTypeSchemeId=10000&issueTypeSchemeId=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescheme/mapping"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("issueTypeSchemeId", issueTypeSchemeId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type_scheme_for_projects(
        self,
        projectId: list[int],
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of issue type schemes associated with specific Jira projects using query parameters for pagination (startAt, maxResults) and project filtering (projectId).

        Args:
            projectId (array): The list of project IDs. To include multiple project IDs, provide an ampersand-separated list. For example, `projectId=10000&projectId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescheme/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def assign_issue_type_scheme_to_project(
        self, issueTypeSchemeId: str, projectId: str
    ) -> Any:
        """
        Assigns an issue type scheme to a Jira Cloud project, requiring global Administer Jira permissions and validating all project issues use scheme types.

        Args:
            issueTypeSchemeId (string): The ID of the issue type scheme. Example: '10000'.
            projectId (string): The ID of the project. Example: '10000'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        request_body_data = None
        request_body_data = {
            "issueTypeSchemeId": issueTypeSchemeId,
            "projectId": projectId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescheme/project"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_type_scheme(self, issueTypeSchemeId: str) -> Any:
        """
        Deletes an issue type scheme by its ID, reassigning any associated projects to the default issue type scheme.

        Args:
            issueTypeSchemeId (string): issueTypeSchemeId

        Returns:
            Any: Returned if the issue type scheme is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        if issueTypeSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeSchemeId'.")
        url = f"{self.base_url}/rest/api/3/issuetypescheme/{issueTypeSchemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_type_scheme(
        self,
        issueTypeSchemeId: str,
        defaultIssueTypeId: str | None = None,
        description: str | None = None,
        name: str | None = None,
    ) -> Any:
        """
        Updates an issue type scheme by modifying its configuration (such as associated issue types) for the specified scheme ID.

        Args:
            issueTypeSchemeId (string): issueTypeSchemeId
            defaultIssueTypeId (string): The ID of the default issue type of the issue type scheme. Example: '10002'.
            description (string): The description of the issue type scheme. The maximum length is 4000 characters. Example: 'A collection of issue types suited to use in a kanban style project.'.
            name (string): The name of the issue type scheme. The name must be unique. The maximum length is 255 characters. Example: 'Kanban Issue Type Scheme'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        if issueTypeSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeSchemeId'.")
        request_body_data = None
        request_body_data = {
            "defaultIssueTypeId": defaultIssueTypeId,
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescheme/{issueTypeSchemeId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_issue_types_to_issue_type_scheme(
        self, issueTypeSchemeId: str, issueTypeIds: list[str]
    ) -> Any:
        """
        Adds issue types to an existing issue type scheme in Jira Cloud, appending them to the current list and returning a success status if the operation completes without conflicts.

        Args:
            issueTypeSchemeId (string): issueTypeSchemeId
            issueTypeIds (array): The list of issue type IDs. Example: ['10000', '10002', '10003'].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        if issueTypeSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeSchemeId'.")
        request_body_data = None
        request_body_data = {
            "issueTypeIds": issueTypeIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = (
            f"{self.base_url}/rest/api/3/issuetypescheme/{issueTypeSchemeId}/issuetype"
        )
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_issue_type_in_scheme(
        self,
        issueTypeSchemeId: str,
        issueTypeIds: list[str],
        after: str | None = None,
        position: str | None = None,
    ) -> Any:
        """
        Moves issue types within a specified issue type scheme in Jira and returns an empty response on success.

        Args:
            issueTypeSchemeId (string): issueTypeSchemeId
            issueTypeIds (array): A list of the issue type IDs to move. The order of the issue type IDs in the list is the order they are given after the move. Example: ['10001', '10004', '10002'].
            after (string): The ID of the issue type to place the moved issue types after. Required if `position` isn't provided. Example: '10008'.
            position (string): The position the issue types should be moved to. Required if `after` isn't provided.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        if issueTypeSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeSchemeId'.")
        request_body_data = None
        request_body_data = {
            "after": after,
            "issueTypeIds": issueTypeIds,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescheme/{issueTypeSchemeId}/issuetype/move"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_issue_type_from_scheme_by_id(
        self, issueTypeSchemeId: str, issueTypeId: str
    ) -> Any:
        """
        Deletes an issue type from an issue type scheme using the Jira Cloud API by specifying the `issueTypeSchemeId` and `issueTypeId`, allowing for removal of issue types from schemes.

        Args:
            issueTypeSchemeId (string): issueTypeSchemeId
            issueTypeId (string): issueTypeId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type schemes
        """
        if issueTypeSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeSchemeId'.")
        if issueTypeId is None:
            raise ValueError("Missing required parameter 'issueTypeId'.")
        url = f"{self.base_url}/rest/api/3/issuetypescheme/{issueTypeSchemeId}/issuetype/{issueTypeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_issue_type_screen_schemes(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
        queryString: str | None = None,
        orderBy: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of issue type screen schemes in Jira with options to filter, paginate, and expand results.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of issue type screen scheme IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.
            queryString (string): String used to perform a case-insensitive partial match with issue type screen scheme name.
            orderBy (string): [Order](#ordering) the results by a field: * `name` Sorts by issue type screen scheme name. * `id` Sorts by issue type screen scheme ID.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts `projects` that, for each issue type screen schemes, returns information about the projects the issue type screen scheme is assigned to.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("queryString", queryString),
                ("orderBy", orderBy),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_issue_type_screen_scheme(
        self,
        issueTypeMappings: list[dict[str, Any]],
        name: str,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates an issue type screen scheme using the Jira Cloud API, allowing administrators to map issue types to specific screen schemes for organizing project workflows.

        Args:
            issueTypeMappings (array): The IDs of the screen schemes for the issue type IDs and *default*. A *default* entry is required to create an issue type screen scheme, it defines the mapping for all issue types without a screen scheme. Example: [{'issueTypeId': 'default', 'screenSchemeId': '10001'}, {'issueTypeId': '10001', 'screenSchemeId': '10002'}, {'issueTypeId': '10002', 'screenSchemeId': '10002'}].
            name (string): The name of the issue type screen scheme. The name must be unique. The maximum length is 255 characters. Example: 'Scrum issue type screen scheme'.
            description (string): The description of the issue type screen scheme. The maximum length is 255 characters.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "issueTypeMappings": issueTypeMappings,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_mappings(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        issueTypeScreenSchemeId: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of issue type to screen scheme mappings associated with a specified issue type screen scheme using the Jira Cloud API.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            issueTypeScreenSchemeId (array): The list of issue type screen scheme IDs. To include multiple issue type screen schemes, separate IDs with ampersand: `issueTypeScreenSchemeId=10000&issueTypeScreenSchemeId=10001`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/mapping"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("issueTypeScreenSchemeId", issueTypeScreenSchemeId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_screen_schemes(
        self,
        projectId: list[int],
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of issue type screen schemes and their associated projects using the specified query parameters.

        Args:
            projectId (array): The list of project IDs. To include multiple projects, separate IDs with ampersand: `projectId=10000&projectId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project_scheme(
        self,
        issueTypeScreenSchemeId: str | None = None,
        projectId: str | None = None,
    ) -> Any:
        """
        Assigns an issue type screen scheme to a project using the Jira API, requiring *Administer Jira* global permission to update project configurations.

        Args:
            issueTypeScreenSchemeId (string): The ID of the issue type screen scheme. Example: '10001'.
            projectId (string): The ID of the project. Example: '10002'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        request_body_data = None
        request_body_data = {
            "issueTypeScreenSchemeId": issueTypeScreenSchemeId,
            "projectId": projectId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/project"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_issue_type_screen_scheme(self, issueTypeScreenSchemeId: str) -> Any:
        """
        Deletes an issue type screen scheme in Jira and returns a success status upon removal.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId

        Returns:
            Any: Returned if the issue type screen scheme is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_type_screen_scheme(
        self,
        issueTypeScreenSchemeId: str,
        description: str | None = None,
        name: str | None = None,
    ) -> Any:
        """
        Updates the default screen scheme for unmapped issue types in the specified issue type screen scheme.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId
            description (string): The description of the issue type screen scheme. The maximum length is 255 characters. Example: 'Screens for scrum issue types.'.
            name (string): The name of the issue type screen scheme. The name must be unique. The maximum length is 255 characters. Example: 'Scrum scheme'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_issue_type_screen_mapping(
        self, issueTypeScreenSchemeId: str, issueTypeMappings: list[dict[str, Any]]
    ) -> Any:
        """
        Updates the mappings of an issue type screen scheme using the PUT method, specifically allowing administrators to append or modify issue type to screen scheme mappings by providing the necessary `issueTypeScreenSchemeId` in the path.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId
            issueTypeMappings (array): The list of issue type to screen scheme mappings. A *default* entry cannot be specified because a default entry is added when an issue type screen scheme is created. Example: [{'issueTypeId': '10000', 'screenSchemeId': '10001'}, {'issueTypeId': '10001', 'screenSchemeId': '10002'}, {'issueTypeId': '10002', 'screenSchemeId': '10002'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        request_body_data = None
        request_body_data = {
            "issueTypeMappings": issueTypeMappings,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}/mapping"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_default_screen_scheme(
        self, issueTypeScreenSchemeId: str, screenSchemeId: str
    ) -> Any:
        """
        Updates the default screen scheme mapping for an issue type screen scheme identified by the `{issueTypeScreenSchemeId}` using the PUT method, which is used for all unmapped issue types in Jira.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId
            screenSchemeId (string): The ID of the screen scheme. Example: '10010'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        request_body_data = None
        request_body_data = {
            "screenSchemeId": screenSchemeId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}/mapping/default"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_issue_type_mapping(
        self, issueTypeScreenSchemeId: str, issueTypeIds: list[str]
    ) -> Any:
        """
        Removes issue type to screen scheme mappings from an issue type screen scheme in Jira using the provided issue type IDs.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId
            issueTypeIds (array): The list of issue type IDs. Example: ['10000', '10001', '10004'].

        Returns:
            Any: Returned if the screen scheme mappings are removed from the issue type screen scheme.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        request_body_data = None
        request_body_data = {
            "issueTypeIds": issueTypeIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}/mapping/remove"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def fetch_project_by_scheme(
        self,
        issueTypeScreenSchemeId: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        query: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of projects associated with a specific issue type screen scheme.

        Args:
            issueTypeScreenSchemeId (string): issueTypeScreenSchemeId
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            query (string): Filter issues by a JQL (Jira Query Language) statement to narrow down the results from the specified issue type screen scheme and project.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue type screen schemes
        """
        if issueTypeScreenSchemeId is None:
            raise ValueError("Missing required parameter 'issueTypeScreenSchemeId'.")
        url = f"{self.base_url}/rest/api/3/issuetypescreenscheme/{issueTypeScreenSchemeId}/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("query", query),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_auto_complete(self) -> dict[str, Any]:
        """
        Retrieves JQL search auto-complete data including field references, operators, and suggestions to assist in programmatic query construction.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        url = f"{self.base_url}/rest/api/3/jql/autocompletedata"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_auto_complete_post(
        self,
        includeCollapsedFields: bool | None = None,
        projectIds: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Provides JQL search auto-complete data and field reference information to assist in programmatic query construction or validation.

        Args:
            includeCollapsedFields (boolean): Include collapsed fields for fields that have non-unique names. Example: True.
            projectIds (array): List of project IDs used to filter the visible field details returned. Example: [10000, 10001, 10002].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        request_body_data = None
        request_body_data = {
            "includeCollapsedFields": includeCollapsedFields,
            "projectIds": projectIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/autocompletedata"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_jql_suggestions(
        self,
        fieldName: str | None = None,
        fieldValue: str | None = None,
        predicateName: str | None = None,
        predicateValue: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves JQL search autocomplete suggestions for specific fields, values, predicates, or predicate values to assist in query construction.

        Args:
            fieldName (string): The name of the field. Example: 'reporter'.
            fieldValue (string): The partial field item name entered by the user.
            predicateName (string): The name of the [ CHANGED operator predicate]( for which the suggestions are generated. The valid predicate operators are *by*, *from*, and *to*.
            predicateValue (string): The partial predicate item name entered by the user.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        url = f"{self.base_url}/rest/api/3/jql/autocompletedata/suggestions"
        query_params = {
            k: v
            for k, v in [
                ("fieldName", fieldName),
                ("fieldValue", fieldValue),
                ("predicateName", predicateName),
                ("predicateValue", predicateValue),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_precomputations(
        self,
        functionKey: list[str] | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of precomputations for a specified JQL function, including when they were created, updated, and last used, allowing apps to inspect their own functions.

        Args:
            functionKey (array): The function key in format: * Forge: `ari:cloud:ecosystem::extension/[App ID]/[Environment ID]/static/[Function key from manifest]` * Connect: `[App key]__[Module key]`
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field: * `functionKey` Sorts by the functionKey. * `used` Sorts by the used timestamp. * `created` Sorts by the created timestamp. * `updated` Sorts by the updated timestamp.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL functions (apps)
        """
        url = f"{self.base_url}/rest/api/3/jql/function/computation"
        query_params = {
            k: v
            for k, v in [
                ("functionKey", functionKey),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_precomputations(
        self,
        skipNotFoundPrecomputations: bool | None = None,
        values: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Updates precomputations (JQL fragments mapped to custom functions) and optionally skips invalid entries based on query parameters.

        Args:
            skipNotFoundPrecomputations (boolean): Specifies whether to skip precomputations when referenced entities (e.g., users, projects) are not found during JQL function processing.
            values (array): values Example: [{'id': 'f2ef228b-367f-4c6b-bd9d-0d0e96b5bd7b', 'value': 'issue in (TEST-1, TEST-2, TEST-3)'}, {'error': 'Error message to be displayed to the user', 'id': '2a854f11-d0e1-4260-aea8-64a562a7062a'}].

        Returns:
            dict[str, Any]: 200 response

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL functions (apps)
        """
        request_body_data = None
        request_body_data = {
            "values": values,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/function/computation"
        query_params = {
            k: v
            for k, v in [("skipNotFoundPrecomputations", skipNotFoundPrecomputations)]
            if v is not None
        }
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_precomputations_by_id(
        self,
        orderBy: str | None = None,
        precomputationIDs: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Performs a computation search using JQL functions in Jira Cloud, allowing users to specify an **orderBy** parameter and returns the results of the computation search via a POST request.

        Args:
            orderBy (string): [Order](#ordering) the results by a field: * `functionKey` Sorts by the functionKey. * `used` Sorts by the used timestamp. * `created` Sorts by the created timestamp. * `updated` Sorts by the updated timestamp.
            precomputationIDs (array): precomputationIDs Example: ['f2ef228b-367f-4c6b-bd9d-0d0e96b5bd7b', '2a854f11-d0e1-4260-aea8-64a562a7062a'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL functions (apps)
        """
        request_body_data = None
        request_body_data = {
            "precomputationIDs": precomputationIDs,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/function/computation/search"
        query_params = {k: v for k, v in [("orderBy", orderBy)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def match_issues(self, issueIds: list[int], jqls: list[str]) -> dict[str, Any]:
        """
        Checks which issues from a provided list match specified JQL queries and returns matched issues for each query.

        Args:
            issueIds (array): A list of issue IDs. Example: [10001, 1000, 10042].
            jqls (array): A list of JQL queries. Example: ['project = FOO', 'issuetype = Bug', 'summary ~ "some text" AND project in (FOO, BAR)'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        request_body_data = None
        request_body_data = {
            "issueIds": issueIds,
            "jqls": jqls,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/match"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def parse_jql_queries(self, validation: str, queries: list[str]) -> dict[str, Any]:
        """
        Parses a JQL query using the POST method at "/rest/api/3/jql/parse" and returns its abstract syntax tree, allowing for analysis or processing of JQL queries, with optional validation configuration.

        Args:
            validation (string): How to validate the JQL query and treat the validation results. Validation options include: * `strict` Returns all errors. If validation fails, the query structure is not returned. * `warn` Returns all errors. If validation fails but the JQL query is correctly formed, the query structure is returned. * `none` No validation is performed. If JQL query is correctly formed, the query structure is returned.
            queries (array): A list of queries to parse. Example: ['summary ~ test AND (labels in (urgent, blocker) OR lastCommentedBy = currentUser()) AND status CHANGED AFTER startOfMonth(-1M) ORDER BY updated DESC', 'issue.property["spaces here"].value in ("Service requests", Incidents)', 'invalid query', 'summary = test', 'summary in test', 'project = INVALID', 'universe = 42'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        request_body_data = None
        request_body_data = {
            "queries": queries,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/parse"
        query_params = {k: v for k, v in [("validation", validation)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def migrate_queries(self, queryStrings: list[str] | None = None) -> dict[str, Any]:
        """
        Converts JQL queries containing usernames or user keys to equivalent queries with account IDs, handling unknown users appropriately.

        Args:
            queryStrings (array): A list of queries with user identifiers. Maximum of 100 queries. Example: ['assignee = mia', 'issuetype = Bug AND assignee in (mia) AND reporter in (alana) order by lastViewed DESC'].

        Returns:
            dict[str, Any]: Returned if the request is successful. Note that the JQL queries are returned in the same order that they were passed.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        request_body_data = None
        request_body_data = {
            "queryStrings": queryStrings,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/pdcleaner"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def sanitise_jql_queries(self, queries: list[dict[str, Any]]) -> dict[str, Any]:
        """
        Sanitizes one or more JQL queries by converting readable details into IDs where a user lacks permission to view the entity, ensuring that unauthorized project names are replaced with project IDs.

        Args:
            queries (array): The list of JQL queries to sanitize. Must contain unique values. Maximum of 20 queries. Example: [{'query': "project = 'Sample project'"}, {'accountId': '5b10ac8d82e05b22cc7d4ef5', 'query': "project = 'Sample project'"}, {'accountId': 'cda2aa1395ac195d951b3387', 'query': "project = 'Sample project'"}, {'accountId': '5b10ac8d82e05b22cc7d4ef5', 'query': 'invalid query'}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            JQL
        """
        request_body_data = None
        request_body_data = {
            "queries": queries,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/jql/sanitize"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_labels(
        self, startAt: int | None = None, maxResults: int | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of labels starting from a specified index and limited by a maximum number of results using the Jira API.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Labels
        """
        url = f"{self.base_url}/rest/api/3/label"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_approximate_license_count(self) -> dict[str, Any]:
        """
        Retrieves the approximate user license count for a Jira instance, which may be cached for up to 7 days.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            License metrics
        """
        url = f"{self.base_url}/rest/api/3/license/approximateLicenseCount"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_license_count_by_product_key(self, applicationKey: str) -> dict[str, Any]:
        """
        Retrieves the approximate license count for a specific application in Jira Cloud using the provided application key.

        Args:
            applicationKey (string): applicationKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            License metrics
        """
        if applicationKey is None:
            raise ValueError("Missing required parameter 'applicationKey'.")
        url = f"{self.base_url}/rest/api/3/license/approximateLicenseCount/product/{applicationKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_my_permissions(
        self,
        projectKey: str | None = None,
        projectId: str | None = None,
        issueKey: str | None = None,
        issueId: str | None = None,
        permissions: str | None = None,
        projectUuid: str | None = None,
        projectConfigurationUuid: str | None = None,
        commentId: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the current user's permissions in Jira, optionally filtered by project, issue, or specific permission keys, and indicates whether each permission is granted.

        Args:
            projectKey (string): The key of project. Ignored if `projectId` is provided.
            projectId (string): The ID of project.
            issueKey (string): The key of the issue. Ignored if `issueId` is provided.
            issueId (string): The ID of the issue.
            permissions (string): A list of permission keys. (Required) This parameter accepts a comma-separated list. To get the list of available permissions, use [Get all permissions](#api-rest-api-3-permissions-get). Example: 'BROWSE_PROJECTS,EDIT_ISSUES'.
            projectUuid (string): Specifies the UUID of the project to check permissions for.
            projectConfigurationUuid (string): A UUID identifying the specific project configuration to check permissions against.
            commentId (string): The ID of the comment.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permissions
        """
        url = f"{self.base_url}/rest/api/3/mypermissions"
        query_params = {
            k: v
            for k, v in [
                ("projectKey", projectKey),
                ("projectId", projectId),
                ("issueKey", issueKey),
                ("issueId", issueId),
                ("permissions", permissions),
                ("projectUuid", projectUuid),
                ("projectConfigurationUuid", projectConfigurationUuid),
                ("commentId", commentId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_preference(self, key: str) -> Any:
        """
        Deletes a user's Jira preference specified by the key query parameter and returns a 204 status code on success.

        Args:
            key (string): The key of the preference.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        url = f"{self.base_url}/rest/api/3/mypreferences"
        query_params = {k: v for k, v in [("key", key)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_preference(self, key: str) -> Any:
        """
        Retrieves the specified user preference value for the currently authenticated user using the provided key parameter.

        Args:
            key (string): The key of the preference.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        url = f"{self.base_url}/rest/api/3/mypreferences"
        query_params = {k: v for k, v in [("key", key)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_preference(self, key: str) -> Any:
        """
        Creates or updates a user preference in Jira by setting or modifying a specific key-value pair using the PUT method.

        Args:
            key (string): The key of the preference. The maximum length is 255 characters.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/mypreferences"
        query_params = {k: v for k, v in [("key", key)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_locale(self) -> Any:
        """
        Deletes the locale preference for a user, restoring the default locale setting, using the Jira Cloud platform REST API.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        url = f"{self.base_url}/rest/api/3/mypreferences/locale"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_locale(self) -> dict[str, Any]:
        """
        Retrieves the locale preference for the currently authenticated user in Jira using the GET method.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        url = f"{self.base_url}/rest/api/3/mypreferences/locale"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_locale(self, locale: str | None = None) -> Any:
        r"""
        Updates the user's locale preference in Jira, restoring the default if no value is specified (deprecated, use user management API instead).

        Args:
            locale (string): The locale code. The Java the locale format is used: a two character language code (ISO 639), an underscore, and two letter country code (ISO 3166). For example, en\_US represents a locale of English (United States). Required on create. Example: 'en_US'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        request_body_data = None
        request_body_data = {
            "locale": locale,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/mypreferences/locale"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_current_user(self, expand: str | None = None) -> dict[str, Any]:
        """
        Retrieves the authenticated user's profile details (with privacy-based limitations on sensitive fields) from Jira Cloud.

        Args:
            expand (string): Use [expand](#expansion) to include additional information about user in the response. This parameter accepts a comma-separated list. Expand options include: * `groups` Returns all groups, including nested groups, the user belongs to. * `applicationRoles` Returns the application roles the user is assigned to.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Myself
        """
        url = f"{self.base_url}/rest/api/3/myself"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_notification_schemes(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        projectId: list[str] | None = None,
        onlyDefault: bool | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves notification schemes listing configured events and their notification recipients for Jira issues, supporting filtering by project, ID, and pagination.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of notification schemes IDs to be filtered by
            projectId (array): The list of projects IDs to be filtered by
            onlyDefault (boolean): When set to true, returns only the default notification scheme. If you provide project IDs not associated with the default, returns an empty page. The default value is false.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `all` Returns all expandable information * `field` Returns information about any custom fields assigned to receive an event * `group` Returns information about any groups assigned to receive an event * `notificationSchemeEvents` Returns a list of event associations. This list is returned for all expandable information * `projectRole` Returns information about any project roles assigned to receive an event * `user` Returns information about any users assigned to receive an event

        Returns:
            dict[str, Any]: Returned if the request is successful. Only returns notification schemes that the user has permission to access. An empty list is returned if the user lacks permission to access all notification schemes.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        url = f"{self.base_url}/rest/api/3/notificationscheme"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("projectId", projectId),
                ("onlyDefault", onlyDefault),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_notification_scheme(
        self,
        name: str,
        description: str | None = None,
        notificationSchemeEvents: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new notification scheme using the "POST" method at the "/rest/api/3/notificationscheme" endpoint, returning a successful creation response when the operation is completed.

        Args:
            name (string): The name of the notification scheme. Must be unique (case-insensitive). Example: 'My new notification scheme'.
            description (string): The description of the notification scheme. Example: 'My new scheme description'.
            notificationSchemeEvents (array): The list of notifications which should be added to the notification scheme. Example: [{'event': {'id': '1'}, 'notifications': [{'notificationType': 'Group', 'parameter': 'jira-administrators'}]}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
            "notificationSchemeEvents": notificationSchemeEvents,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/notificationscheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_notification_scheme_projects(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        notificationSchemeId: list[str] | None = None,
        projectId: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the association between notification schemes and projects in Jira, including scheme IDs and project IDs, based on query parameters such as notificationSchemeId and projectId.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            notificationSchemeId (array): The list of notifications scheme IDs to be filtered out
            projectId (array): The list of project IDs to be filtered out

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        url = f"{self.base_url}/rest/api/3/notificationscheme/project"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("notificationSchemeId", notificationSchemeId),
                ("projectId", projectId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_notification_scheme(
        self, id: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves details of a specific notification scheme by its ID using the Jira API, optionally expanding the response with additional details.

        Args:
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `all` Returns all expandable information * `field` Returns information about any custom fields assigned to receive an event * `group` Returns information about any groups assigned to receive an event * `notificationSchemeEvents` Returns a list of event associations. This list is returned for all expandable information * `projectRole` Returns information about any project roles assigned to receive an event * `user` Returns information about any users assigned to receive an event

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/notificationscheme/{id}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_notification_scheme(
        self, id: str, description: str | None = None, name: str | None = None
    ) -> Any:
        """
        Updates a notification scheme using its ID, allowing modifications to the scheme's configuration, such as events and recipients.

        Args:
            id (string): id
            description (string): The description of the notification scheme. Example: 'My updated notification scheme description'.
            name (string): The name of the notification scheme. Must be unique. Example: 'My updated notification scheme'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/notificationscheme/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_notifications(
        self, id: str, notificationSchemeEvents: list[dict[str, Any]]
    ) -> Any:
        """
        Updates notifications for a specific notification scheme in Jira by adding or modifying event-based notification rules.

        Args:
            id (string): id
            notificationSchemeEvents (array): The list of notifications which should be added to the notification scheme. Example: [{'event': {'id': '1'}, 'notifications': [{'notificationType': 'Group', 'parameter': 'jira-administrators'}]}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "notificationSchemeEvents": notificationSchemeEvents,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/notificationscheme/{id}/notification"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_notification_scheme(self, notificationSchemeId: str) -> Any:
        """
        Deletes a specific notification scheme in Jira using its ID, returning appropriate status codes for success or error conditions.

        Args:
            notificationSchemeId (string): notificationSchemeId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        if notificationSchemeId is None:
            raise ValueError("Missing required parameter 'notificationSchemeId'.")
        url = f"{self.base_url}/rest/api/3/notificationscheme/{notificationSchemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_notification_from_scheme(
        self, notificationSchemeId: str, notificationId: str
    ) -> Any:
        """
        Deletes a specific notification from a notification scheme in Jira using the specified notification scheme and notification IDs.

        Args:
            notificationSchemeId (string): notificationSchemeId
            notificationId (string): notificationId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue notification schemes
        """
        if notificationSchemeId is None:
            raise ValueError("Missing required parameter 'notificationSchemeId'.")
        if notificationId is None:
            raise ValueError("Missing required parameter 'notificationId'.")
        url = f"{self.base_url}/rest/api/3/notificationscheme/{notificationSchemeId}/notification/{notificationId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_permissions(self) -> dict[str, Any]:
        """
        Retrieves details of global and project permissions granted to a user using the Jira Cloud REST API.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permissions
        """
        url = f"{self.base_url}/rest/api/3/permissions"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_bulk_permissions(
        self,
        accountId: str | None = None,
        globalPermissions: list[str] | None = None,
        projectPermissions: list[dict[str, Any]] | None = None,
    ) -> dict[str, Any]:
        """
        Checks user permissions in Jira projects and returns global and project-specific permission details.

        Args:
            accountId (string): The account ID of a user. Example: '5b10a2844c20165700ede21g'.
            globalPermissions (array): Global permissions to look up. Example: ['ADMINISTER'].
            projectPermissions (array): Project permissions with associated projects and issues to look up. Example: [{'issues': [10010, 10011, 10012, 10013, 10014], 'permissions': ['EDIT_ISSUES'], 'projects': [10001]}].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permissions
        """
        request_body_data = None
        request_body_data = {
            "accountId": accountId,
            "globalPermissions": globalPermissions,
            "projectPermissions": projectPermissions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/permissions/check"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_permitted_projects(self, permissions: list[str]) -> dict[str, Any]:
        """
        Retrieves all projects where a user has specified project permissions and returns the list of projects with granted access.

        Args:
            permissions (array): A list of permission keys.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permissions
        """
        request_body_data = None
        request_body_data = {
            "permissions": permissions,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/permissions/project"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_permission_schemes(self, expand: str | None = None) -> dict[str, Any]:
        """
        Retrieves a list of all permission schemes in Jira Cloud, optionally expanding the response to include additional details such as groups by using the "expand" query parameter.

        Args:
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        url = f"{self.base_url}/rest/api/3/permissionscheme"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_permission_scheme(
        self,
        name: str,
        expand: str | None = None,
        description: str | None = None,
        expand_body: str | None = None,
        id: int | None = None,
        permissions: list[dict[str, Any]] | None = None,
        scope: Any | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new permission scheme in Jira using the "POST" method at the "/rest/api/3/permissionscheme" path, allowing for the definition of a permission set with various grants.

        Args:
            name (string): The name of the permission scheme. Must be unique. Example: 'Example permission scheme'.
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are always included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.
            description (string): A description for the permission scheme. Example: 'description'.
            expand_body (string): The expand options available for the permission scheme.
            id (integer): The ID of the permission scheme.
            permissions (array): The permission scheme to create or update. See [About permission schemes and grants](../api-group-permission-schemes/#about-permission-schemes-and-grants) for more information. Example: [{'holder': {'parameter': 'jira-core-users', 'type': 'group', 'value': 'ca85fac0-d974-40ca-a615-7af99c48d24f'}, 'permission': 'ADMINISTER_PROJECTS'}].
            scope (string): The scope of the permission scheme.
            self_arg_body (string): The URL of the permission scheme.

        Returns:
            dict[str, Any]: Returned if the permission scheme is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "expand": expand_body,
            "id": id,
            "name": name,
            "permissions": permissions,
            "scope": scope,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/permissionscheme"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_permission_scheme(self, schemeId: str) -> Any:
        """
        Deletes a permission scheme specified by the provided `schemeId` using the Jira REST API, removing it from the system.

        Args:
            schemeId (string): schemeId

        Returns:
            Any: Returned if the permission scheme is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_permission_scheme(
        self, schemeId: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a specific permission scheme in Jira, identified by its scheme ID, and optionally expands its details with certain information based on the provided expand parameter.

        Args:
            schemeId (string): schemeId
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_permission_scheme(
        self,
        schemeId: str,
        name: str,
        expand: str | None = None,
        description: str | None = None,
        expand_body: str | None = None,
        id: int | None = None,
        permissions: list[dict[str, Any]] | None = None,
        scope: Any | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates a permission scheme identified by `{schemeId}` using the PUT method, allowing modifications to its permissions and settings.

        Args:
            schemeId (string): schemeId
            name (string): The name of the permission scheme. Must be unique. Example: 'Example permission scheme'.
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are always included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.
            description (string): A description for the permission scheme. Example: 'description'.
            expand_body (string): The expand options available for the permission scheme.
            id (integer): The ID of the permission scheme.
            permissions (array): The permission scheme to create or update. See [About permission schemes and grants](../api-group-permission-schemes/#about-permission-schemes-and-grants) for more information. Example: [{'holder': {'parameter': 'jira-core-users', 'type': 'group', 'value': 'ca85fac0-d974-40ca-a615-7af99c48d24f'}, 'permission': 'ADMINISTER_PROJECTS'}].
            scope (string): The scope of the permission scheme.
            self_arg_body (string): The URL of the permission scheme.

        Returns:
            dict[str, Any]: Returned if the scheme is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "expand": expand_body,
            "id": id,
            "name": name,
            "permissions": permissions,
            "scope": scope,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_permission_scheme_grants(
        self, schemeId: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves the permissions granted by a specific permission scheme in Jira, including details about each permission grant within the scheme.

        Args:
            schemeId (string): schemeId
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are always included when you specify any value. Expand options include: * `permissions` Returns all permission grants for each permission scheme. * `user` Returns information about the user who is granted the permission. * `group` Returns information about the group that is granted the permission. * `projectRole` Returns information about the project role granted the permission. * `field` Returns information about the custom field granted the permission. * `all` Returns all expandable information.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}/permission"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_permission_grant(
        self,
        schemeId: str,
        expand: str | None = None,
        holder: Any | None = None,
        id: int | None = None,
        permission: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Adds permissions to a specific permission scheme in Jira, enabling access control configuration for users, groups, or roles.

        Args:
            schemeId (string): schemeId
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are always included when you specify any value. Expand options include: * `permissions` Returns all permission grants for each permission scheme. * `user` Returns information about the user who is granted the permission. * `group` Returns information about the group that is granted the permission. * `projectRole` Returns information about the project role granted the permission. * `field` Returns information about the custom field granted the permission. * `all` Returns all expandable information.
            holder (string): The user or group being granted the permission. It consists of a `type`, a type-dependent `parameter` and a type-dependent `value`. See [Holder object](../api-group-permission-schemes/#holder-object) in *Get all permission schemes* for more information. Example: {'parameter': 'jira-core-users', 'type': 'group', 'value': 'ca85fac0-d974-40ca-a615-7af99c48d24f'}.
            id (integer): The ID of the permission granted details.
            permission (string): The permission to grant. This permission can be one of the built-in permissions or a custom permission added by an app. See [Built-in permissions](../api-group-permission-schemes/#built-in-permissions) in *Get all permission schemes* for more information about the built-in permissions. See the [project permission](https://developer.atlassian.com/cloud/jira/platform/modules/project-permission/) and [global permission](https://developer.atlassian.com/cloud/jira/platform/modules/global-permission/) module documentation for more information about custom permissions. Example: 'ADMINISTER_PROJECTS'.
            self_arg_body (string): The URL of the permission granted details.

        Returns:
            dict[str, Any]: Returned if the scheme permission is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        request_body_data = None
        request_body_data = {
            "holder": holder,
            "id": id,
            "permission": permission,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}/permission"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_permission_scheme_entity(self, schemeId: str, permissionId: str) -> Any:
        """
        Deletes a permission grant from a permission scheme in Jira using the specified `schemeId` and `permissionId`.

        Args:
            schemeId (string): schemeId
            permissionId (string): permissionId

        Returns:
            Any: Returned if the permission grant is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if permissionId is None:
            raise ValueError("Missing required parameter 'permissionId'.")
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}/permission/{permissionId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_permission_scheme_grant(
        self, schemeId: str, permissionId: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a specific permission grant's details within a Jira permission scheme identified by the scheme ID and permission ID.

        Args:
            schemeId (string): schemeId
            permissionId (string): permissionId
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are always included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Permission schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        if permissionId is None:
            raise ValueError("Missing required parameter 'permissionId'.")
        url = f"{self.base_url}/rest/api/3/permissionscheme/{schemeId}/permission/{permissionId}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_plans(
        self,
        includeTrashed: bool | None = None,
        includeArchived: bool | None = None,
        cursor: str | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves plan details using pagination and optional filters for trashed or archived items.

        Args:
            includeTrashed (boolean): Whether to include trashed plans in the results.
            includeArchived (boolean): Whether to include archived plans in the results.
            cursor (string): The cursor to start from. If not provided, the first page will be returned.
            maxResults (integer): The maximum number of plans to return per page. The maximum value is 50. The default value is 50.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        url = f"{self.base_url}/rest/api/3/plans/plan"
        query_params = {
            k: v
            for k, v in [
                ("includeTrashed", includeTrashed),
                ("includeArchived", includeArchived),
                ("cursor", cursor),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_plan(
        self,
        issueSources: list[dict[str, Any]],
        name: str,
        scheduling: Any,
        useGroupId: bool | None = None,
        crossProjectReleases: list[dict[str, Any]] | None = None,
        customFields: list[dict[str, Any]] | None = None,
        exclusionRules: Any | None = None,
        leadAccountId: str | None = None,
        permissions: list[dict[str, Any]] | None = None,
    ) -> Any:
        """
        Creates a new plan resource via the specified endpoint, requiring a request body with plan details and supporting optional useGroupId query parameter for group association.

        Args:
            issueSources (array): The issue sources to include in the plan. Example: [{'type': 'Project', 'value': 12}, {'type': 'Board', 'value': 462}].
            name (string): The plan name. Example: 'ABC Quaterly plan'.
            scheduling (string): The scheduling settings for the plan. Example: {'dependencies': 'Sequential', 'endDate': {'type': 'DueDate'}, 'estimation': 'Days', 'inferredDates': 'ReleaseDates', 'startDate': {'type': 'TargetStartDate'}}.
            useGroupId (boolean): Whether to accept group IDs instead of group names. Group names are deprecated.
            crossProjectReleases (array): The cross-project releases to include in the plan. Example: [{'name': 'AB and BC merge', 'releaseIds': [29, 39]}].
            customFields (array): The custom fields for the plan. Example: [{'customFieldId': 2335, 'filter': True}].
            exclusionRules (string): The exclusion rules for the plan. Example: {'issueIds': [2, 3], 'issueTypeIds': [32, 33], 'numberOfDaysToShowCompletedIssues': 50, 'releaseIds': [42, 43], 'workStatusCategoryIds': [22, 23], 'workStatusIds': [12, 13]}.
            leadAccountId (string): The account ID of the plan lead. Example: 'abc-12-rbji'.
            permissions (array): The permissions for the plan. Example: [{'holder': {'type': 'AccountId', 'value': '234-tgj-343'}, 'type': 'Edit'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        request_body_data = None
        request_body_data = {
            "crossProjectReleases": crossProjectReleases,
            "customFields": customFields,
            "exclusionRules": exclusionRules,
            "issueSources": issueSources,
            "leadAccountId": leadAccountId,
            "name": name,
            "permissions": permissions,
            "scheduling": scheduling,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/plans/plan"
        query_params = {k: v for k, v in [("useGroupId", useGroupId)] if v is not None}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_plan(self, planId: str, useGroupId: bool | None = None) -> dict[str, Any]:
        """
        Retrieves the details of a specific plan identified by its planId using a GET request.

        Args:
            planId (string): planId
            useGroupId (boolean): Whether to return group IDs instead of group names. Group names are deprecated.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}"
        query_params = {k: v for k, v in [("useGroupId", useGroupId)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_plan(
        self, planId: str, body_content: bytes, useGroupId: bool | None = None
    ) -> Any:
        """
        Updates a plan with the specified ID in "/rest/api/3/plans/plan/{planId}" using the PUT method, potentially modifying plan details based on provided parameters.

        Args:
            planId (string): planId
            body_content (bytes | None): Raw binary content for the request body.
            useGroupId (boolean): Whether to accept group IDs instead of group names. Group names are deprecated.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        request_body_data = body_content
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}"
        query_params = {k: v for k, v in [("useGroupId", useGroupId)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json-patch+json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def archive_plan(self, planId: str) -> Any:
        """
        Archives a specific plan in Jira using the PUT method at the "/rest/api/3/plans/plan/{planId}/archive" endpoint, identified by the planId parameter.

        Args:
            planId (string): planId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/archive"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def duplicate_plan(self, planId: str, name: str) -> Any:
        """
        Creates a duplicate of the specified Jira plan using the provided plan ID and returns the new plan's details.

        Args:
            planId (string): planId
            name (string): The plan name. Example: 'Copied Plan'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        request_body_data = {
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/duplicate"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_teams(
        self,
        planId: str,
        cursor: str | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of teams associated with a specific plan in Jira Cloud.

        Args:
            planId (string): planId
            cursor (string): The cursor to start from. If not provided, the first page will be returned.
            maxResults (integer): The maximum number of plan teams to return per page. The maximum value is 50. The default value is 50.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team"
        query_params = {
            k: v
            for k, v in [("cursor", cursor), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_atlassian_team(
        self,
        planId: str,
        id: str,
        planningStyle: str,
        capacity: float | None = None,
        issueSourceId: int | None = None,
        sprintLength: int | None = None,
    ) -> Any:
        """
        Adds an Atlassian team to a plan using the Jira Cloud API and returns a status message, allowing for the management of team configurations within plans.

        Args:
            planId (string): planId
            id (string): The Atlassian team ID. Example: 'AtlassianTeamId'.
            planningStyle (string): The planning style for the Atlassian team. This must be "Scrum" or "Kanban". Example: 'Scrum'.
            capacity (number): The capacity for the Atlassian team. Example: 200.
            issueSourceId (integer): The ID of the issue source for the Atlassian team. Example: 0.
            sprintLength (integer): The sprint length for the Atlassian team. Example: 2.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        request_body_data = {
            "capacity": capacity,
            "id": id,
            "issueSourceId": issueSourceId,
            "planningStyle": planningStyle,
            "sprintLength": sprintLength,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/atlassian"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_atlassian_team(self, planId: str, atlassianTeamId: str) -> Any:
        """
        Deletes an Atlassian team from a specified plan in Jira Cloud using the "DELETE" method, requiring plan ID and Atlassian team ID as path parameters.

        Args:
            planId (string): planId
            atlassianTeamId (string): atlassianTeamId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if atlassianTeamId is None:
            raise ValueError("Missing required parameter 'atlassianTeamId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/atlassian/{atlassianTeamId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_atlassian_team(self, planId: str, atlassianTeamId: str) -> dict[str, Any]:
        """
        Retrieves planning settings for an Atlassian team within a specific plan in Jira.

        Args:
            planId (string): planId
            atlassianTeamId (string): atlassianTeamId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan, important
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if atlassianTeamId is None:
            raise ValueError("Missing required parameter 'atlassianTeamId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/atlassian/{atlassianTeamId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_atlassian_team(
        self, planId: str, atlassianTeamId: str, body_content: bytes
    ) -> Any:
        """
        Associates a specified Atlassian team with a plan identified by a plan ID, using the "PUT" method to update the team assignment.

        Args:
            planId (string): planId
            atlassianTeamId (string): atlassianTeamId
            body_content (bytes | None): Raw binary content for the request body.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan, important
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if atlassianTeamId is None:
            raise ValueError("Missing required parameter 'atlassianTeamId'.")
        request_body_data = None
        request_body_data = body_content
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/atlassian/{atlassianTeamId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json-patch+json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_plan_only_team(
        self,
        planId: str,
        name: str,
        planningStyle: str,
        capacity: float | None = None,
        issueSourceId: int | None = None,
        memberAccountIds: list[str] | None = None,
        sprintLength: int | None = None,
    ) -> Any:
        """
        Creates a plan-only team in a Jira Cloud plan with specified planning settings and returns the configuration.

        Args:
            planId (string): planId
            name (string): The plan-only team name. Example: 'Team1'.
            planningStyle (string): The planning style for the plan-only team. This must be "Scrum" or "Kanban". Example: 'Scrum'.
            capacity (number): The capacity for the plan-only team. Example: 200.
            issueSourceId (integer): The ID of the issue source for the plan-only team. Example: 0.
            memberAccountIds (array): The account IDs of the plan-only team members. Example: ['member1AccountId', 'member2AccountId'].
            sprintLength (integer): The sprint length for the plan-only team. Example: 2.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        request_body_data = {
            "capacity": capacity,
            "issueSourceId": issueSourceId,
            "memberAccountIds": memberAccountIds,
            "name": name,
            "planningStyle": planningStyle,
            "sprintLength": sprintLength,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/planonly"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_plan_only_team(self, planId: str, planOnlyTeamId: str) -> Any:
        """
        Deletes a specific team associated with a plan in a REST API (likely related to project management or issue tracking).

        Args:
            planId (string): planId
            planOnlyTeamId (string): planOnlyTeamId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if planOnlyTeamId is None:
            raise ValueError("Missing required parameter 'planOnlyTeamId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/planonly/{planOnlyTeamId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_plan_only_team(self, planId: str, planOnlyTeamId: str) -> dict[str, Any]:
        """
        Retrieves planning settings for a specific plan-only team in a Jira plan using the Jira Cloud REST API.

        Args:
            planId (string): planId
            planOnlyTeamId (string): planOnlyTeamId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if planOnlyTeamId is None:
            raise ValueError("Missing required parameter 'planOnlyTeamId'.")
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/planonly/{planOnlyTeamId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_plan_only_team(
        self, planId: str, planOnlyTeamId: str, body_content: bytes
    ) -> Any:
        """
        Updates planning settings for a specific team in a Jira plan using the provided parameters and returns a success status upon completion.

        Args:
            planId (string): planId
            planOnlyTeamId (string): planOnlyTeamId
            body_content (bytes | None): Raw binary content for the request body.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Teams in plan
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        if planOnlyTeamId is None:
            raise ValueError("Missing required parameter 'planOnlyTeamId'.")
        request_body_data = None
        request_body_data = body_content
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/team/planonly/{planOnlyTeamId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json-patch+json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def trash_plan(self, planId: str) -> Any:
        """
        Moves a specified plan to trash using the Jira API and returns an empty response on success.

        Args:
            planId (string): planId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Plans
        """
        if planId is None:
            raise ValueError("Missing required parameter 'planId'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/plans/plan/{planId}/trash"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_priorities(self) -> list[Any]:
        """
        Retrieves a list of all issue priorities in Jira using the "/rest/api/3/priority" endpoint with the GET method.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        url = f"{self.base_url}/rest/api/3/priority"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_priority(
        self,
        name: str,
        statusColor: str,
        avatarId: int | None = None,
        description: str | None = None,
        iconUrl: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new priority in Jira with specified properties and returns the generated ID.

        Args:
            name (string): The name of the priority. Must be unique. Example: 'My new priority'.
            statusColor (string): The status color of the priority in 3-digit or 6-digit hexadecimal format. Example: '#ABCDEF'.
            avatarId (integer): The ID for the avatar for the priority. Either the iconUrl or avatarId must be defined, but not both. This parameter is nullable and will become mandatory once the iconUrl parameter is deprecated.
            description (string): The description of the priority. Example: 'My priority description'.
            iconUrl (string): The URL of an icon for the priority. Accepted protocols are HTTP and HTTPS. Built in icons can also be used. Either the iconUrl or avatarId must be defined, but not both. Example: 'images/icons/priorities/major.png'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        request_body_data = None
        request_body_data = {
            "avatarId": avatarId,
            "description": description,
            "iconUrl": iconUrl,
            "name": name,
            "statusColor": statusColor,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priority"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_default_priority(self, id: str) -> Any:
        """
        Sets the default issue priority in Jira using the PUT method at the "/rest/api/3/priority/default" path.

        Args:
            id (string): The ID of the new default issue priority. Must be an existing ID or null. Setting this to null erases the default priority setting. Example: '3'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        request_body_data = None
        request_body_data = {
            "id": id,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priority/default"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_priorities(
        self,
        ids: list[str],
        after: str | None = None,
        position: str | None = None,
    ) -> Any:
        """
        Reorders the priority of issues in Jira by updating their sequence using a PUT request.

        Args:
            ids (array): The list of issue IDs to be reordered. Cannot contain duplicates nor after ID. Example: ['10004', '10005'].
            after (string): The ID of the priority. Required if `position` isn't provided. Example: '10003'.
            position (string): The position for issue priorities to be moved to. Required if `after` isn't provided.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        request_body_data = None
        request_body_data = {
            "after": after,
            "ids": ids,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priority/move"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_priorities(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        projectId: list[str] | None = None,
        priorityName: str | None = None,
        onlyDefault: bool | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of priorities from Jira using the GET method at "/rest/api/3/priority/search", allowing for customizable queries with parameters like start index, maximum results, ID, project ID, priority name, and expansion options.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of priority IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=2&id=3`.
            projectId (array): The list of projects IDs. To include multiple IDs, provide an ampersand-separated list. For example, `projectId=10010&projectId=10111`.
            priorityName (string): The name of priority to search for.
            onlyDefault (boolean): Whether only the default priority is returned.
            expand (string): Use `schemes` to return the associated priority schemes for each priority. Limited to returning first 15 priority schemes per priority.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        url = f"{self.base_url}/rest/api/3/priority/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("projectId", projectId),
                ("priorityName", priorityName),
                ("onlyDefault", onlyDefault),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_priority(self, id: str) -> Any:
        """
        Deletes an issue priority asynchronously by its ID using the Jira Cloud REST API, requiring administrative permissions and returning various status messages based on the outcome.

        Args:
            id (string): id

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/priority/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_priority(self, id: str) -> dict[str, Any]:
        """
        Retrieves details of a specific issue priority by its ID in Jira.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/priority/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_priority(
        self,
        id: str,
        avatarId: int | None = None,
        description: str | None = None,
        iconUrl: str | None = None,
        name: str | None = None,
        statusColor: str | None = None,
    ) -> Any:
        """
        Updates an issue priority with the specified ID using the Jira Cloud API, allowing modifications to priority details such as name or icon URL.

        Args:
            id (string): id
            avatarId (integer): The ID for the avatar for the priority. This parameter is nullable and both iconUrl and avatarId cannot be defined.
            description (string): The description of the priority. Example: 'My updated priority description'.
            iconUrl (string): The URL of an icon for the priority. Accepted protocols are HTTP and HTTPS. Built in icons can also be used. Both iconUrl and avatarId cannot be defined. Example: 'images/icons/priorities/minor.png'.
            name (string): The name of the priority. Must be unique. Example: 'My updated priority'.
            statusColor (string): The status color of the priority in 3-digit or 6-digit hexadecimal format. Example: '#123456'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue priorities
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "avatarId": avatarId,
            "description": description,
            "iconUrl": iconUrl,
            "name": name,
            "statusColor": statusColor,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priority/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_priority_schemes(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        priorityId: list[int] | None = None,
        schemeId: list[int] | None = None,
        schemeName: str | None = None,
        onlyDefault: bool | None = None,
        orderBy: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves priority schemes and their associated priorities from a Jira instance using various filters such as priority ID, scheme ID, scheme name, and more.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            priorityId (array): A set of priority IDs to filter by. To include multiple IDs, provide an ampersand-separated list. For example, `priorityId=10000&priorityId=10001`.
            schemeId (array): A set of priority scheme IDs. To include multiple IDs, provide an ampersand-separated list. For example, `schemeId=10000&schemeId=10001`.
            schemeName (string): The name of scheme to search for.
            onlyDefault (boolean): Whether only the default priority is returned.
            orderBy (string): The ordering to return the priority schemes by.
            expand (string): A comma separated list of additional information to return. "priorities" will return priorities associated with the priority scheme. "projects" will return projects associated with the priority scheme. `expand=priorities,projects`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        url = f"{self.base_url}/rest/api/3/priorityscheme"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("priorityId", priorityId),
                ("schemeId", schemeId),
                ("schemeName", schemeName),
                ("onlyDefault", onlyDefault),
                ("orderBy", orderBy),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_priority_scheme(
        self,
        defaultPriorityId: int,
        name: str,
        priorityIds: list[int],
        description: str | None = None,
        mappings: Any | None = None,
        projectIds: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new priority scheme with configurable priority mappings and project associations in Jira.

        Args:
            defaultPriorityId (integer): The ID of the default priority for the priority scheme. Example: 10001.
            name (string): The name of the priority scheme. Must be unique. Example: 'My new priority scheme'.
            priorityIds (array): The IDs of priorities in the scheme. Example: [10000, 10001, 10003].
            description (string): The description of the priority scheme. Example: 'My priority scheme description'.
            mappings (string): Instructions to migrate the priorities of issues.

        `in` mappings are used to migrate the priorities of issues to priorities used within the priority scheme.

        `out` mappings are used to migrate the priorities of issues to priorities not used within the priority scheme.

         *  When **priorities** are **added** to the new priority scheme, no mapping needs to be provided as the new priorities are not used by any issues.
         *  When **priorities** are **removed** from the new priority scheme, no mapping needs to be provided as the removed priorities are not used by any issues.
         *  When **projects** are **added** to the priority scheme, the priorities of issues in those projects might need to be migrated to new priorities used by the priority scheme. This can occur when the current scheme does not use all the priorities in the project(s)' priority scheme(s).

             *  An `in` mapping must be provided for each of these priorities.
         *  When **projects** are **removed** from the priority scheme, no mapping needs to be provided as the removed projects are not using the priorities of the new priority scheme.

        For more information on `in` and `out` mappings, see the child properties documentation for the `PriorityMapping` object below. Example: {'in': {'10002': 10000, '10005': 10001, '10006': 10001, '10008': 10003}, 'out': {}}.
            projectIds (array): The IDs of projects that will use the priority scheme. Example: [10005, 10006, 10007].

        Returns:
            dict[str, Any]: Returned if the request is completed.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        request_body_data = None
        request_body_data = {
            "defaultPriorityId": defaultPriorityId,
            "description": description,
            "mappings": mappings,
            "name": name,
            "priorityIds": priorityIds,
            "projectIds": projectIds,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priorityscheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def suggested_priorities_for_mappings(
        self,
        maxResults: int | None = None,
        priorities: Any | None = None,
        projects: Any | None = None,
        schemeId: int | None = None,
        startAt: int | None = None,
    ) -> dict[str, Any]:
        """
        Submits priority mappings for a scheme and returns the updated configuration upon completion.

        Args:
            maxResults (integer): The maximum number of results that could be on the page. Example: 50.
            priorities (string): The priority changes in the scheme. Example: {'add': [10001, 10002], 'remove': [10003]}.
            projects (string): The project changes in the scheme. Example: {'add': [10021]}.
            schemeId (integer): The id of the priority scheme. Example: 10005.
            startAt (integer): The index of the first item returned on the page. Example: 0.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        request_body_data = None
        request_body_data = {
            "maxResults": maxResults,
            "priorities": priorities,
            "projects": projects,
            "schemeId": schemeId,
            "startAt": startAt,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priorityscheme/mappings"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_priorities(
        self,
        schemeId: str,
        startAt: str | None = None,
        maxResults: str | None = None,
        query: str | None = None,
        exclude: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of available priorities for a specified priority scheme or across all schemes, supporting filtering by query and exclusion criteria.

        Args:
            schemeId (string): The priority scheme ID.
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            query (string): The string to query priorities on by name.
            exclude (array): A list of priority IDs to exclude from the results.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        url = f"{self.base_url}/rest/api/3/priorityscheme/priorities/available"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("query", query),
                ("schemeId", schemeId),
                ("exclude", exclude),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_priority_scheme(self, schemeId: str) -> Any:
        """
        Deletes a specific priority scheme identified by its scheme ID, causing projects that were using it to default to the standard priority scheme.

        Args:
            schemeId (string): schemeId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/priorityscheme/{schemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_priority_scheme(
        self,
        schemeId: str,
        defaultPriorityId: int | None = None,
        description: str | None = None,
        mappings: Any | None = None,
        name: str | None = None,
        priorities: Any | None = None,
        projects: Any | None = None,
    ) -> dict[str, Any]:
        """
        Updates a Jira priority scheme configuration with the given ID, rejecting updates if they would require issue migrations.

        Args:
            schemeId (string): schemeId
            defaultPriorityId (integer): The default priority of the scheme. Example: 10001.
            description (string): The description of the priority scheme. Example: 'My priority scheme description'.
            mappings (string): Instructions to migrate the priorities of issues.

        `in` mappings are used to migrate the priorities of issues to priorities used within the priority scheme.

        `out` mappings are used to migrate the priorities of issues to priorities not used within the priority scheme.

         *  When **priorities** are **added** to the priority scheme, no mapping needs to be provided as the new priorities are not used by any issues.
         *  When **priorities** are **removed** from the priority scheme, issues that are using those priorities must be migrated to new priorities used by the priority scheme.

             *  An `in` mapping must be provided for each of these priorities.
         *  When **projects** are **added** to the priority scheme, the priorities of issues in those projects might need to be migrated to new priorities used by the priority scheme. This can occur when the current scheme does not use all the priorities in the project(s)' priority scheme(s).

             *  An `in` mapping must be provided for each of these priorities.
         *  When **projects** are **removed** from the priority scheme, the priorities of issues in those projects might need to be migrated to new priorities within the **Default Priority Scheme** that are not used by the priority scheme. This can occur when the **Default Priority Scheme** does not use all the priorities within the current scheme.

             *  An `out` mapping must be provided for each of these priorities.

        For more information on `in` and `out` mappings, see the child properties documentation for the `PriorityMapping` object below. Example: {'in': {'10003': 10002, '10004': 10001}, 'out': {'10001': 10005, '10002': 10006}}.
            name (string): The name of the priority scheme. Must be unique. Example: 'My new priority scheme'.
            priorities (string): The priorities in the scheme. Example: {'add': {'ids': [10001, 10002]}, 'remove': {'ids': [10003, 10004]}}.
            projects (string): The projects in the scheme. Example: {'add': {'ids': [10101, 10102]}, 'remove': {'ids': [10103, 10104]}}.

        Returns:
            dict[str, Any]: Returned if the request is accepted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        request_body_data = None
        request_body_data = {
            "defaultPriorityId": defaultPriorityId,
            "description": description,
            "mappings": mappings,
            "name": name,
            "priorities": priorities,
            "projects": projects,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/priorityscheme/{schemeId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_priorities_by_priority_scheme(
        self,
        schemeId: str,
        startAt: str | None = None,
        maxResults: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of priorities associated with a specific priority scheme in Jira, supporting startAt and maxResults parameters.

        Args:
            schemeId (string): schemeId
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/priorityscheme/{schemeId}/priorities"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_projects_by_priority_scheme(
        self,
        schemeId: str,
        startAt: str | None = None,
        maxResults: str | None = None,
        projectId: list[int] | None = None,
        query: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of projects associated with a specific priority scheme in Jira.

        Args:
            schemeId (string): schemeId
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            projectId (array): The project IDs to filter by. For example, `projectId=10000&projectId=10001`.
            query (string): The string to query projects on by name.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Priority schemes
        """
        if schemeId is None:
            raise ValueError("Missing required parameter 'schemeId'.")
        url = f"{self.base_url}/rest/api/3/priorityscheme/{schemeId}/projects"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("projectId", projectId),
                ("query", query),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_projects(
        self,
        expand: str | None = None,
        recent: int | None = None,
        properties: list[str] | None = None,
    ) -> list[Any]:
        """
        Retrieves project details from Jira with optional parameters to expand properties, limit to recent projects, or include specific properties.

        Args:
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expanded options include: * `description` Returns the project description. * `issueTypes` Returns all issue types associated with the project. * `lead` Returns information about the project lead. * `projectKeys` Returns all project keys associated with the project.
            recent (integer): Returns the user's most recently accessed projects. You may specify the number of results to return up to a maximum of 20. If access is anonymous, then the recently accessed projects are based on the current HTTP session.
            properties (array): A list of project properties to return for the project. This parameter accepts a comma-separated list.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        url = f"{self.base_url}/rest/api/3/project"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("recent", recent),
                ("properties", properties),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_project(
        self,
        key: str,
        name: str,
        assigneeType: str | None = None,
        avatarId: int | None = None,
        categoryId: int | None = None,
        description: str | None = None,
        fieldConfigurationScheme: int | None = None,
        issueSecurityScheme: int | None = None,
        issueTypeScheme: int | None = None,
        issueTypeScreenScheme: int | None = None,
        lead: str | None = None,
        leadAccountId: str | None = None,
        notificationScheme: int | None = None,
        permissionScheme: int | None = None,
        projectTemplateKey: str | None = None,
        projectTypeKey: str | None = None,
        url: str | None = None,
        workflowScheme: int | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new Jira project using the REST API, allowing the specification of project details such as key, name, type, and description.

        Args:
            key (string): Project keys must be unique and start with an uppercase letter followed by one or more uppercase alphanumeric characters. The maximum length is 10 characters. Example: 'EX'.
            name (string): The name of the project. Example: 'Example'.
            assigneeType (string): The default assignee when creating issues for this project. Example: 'PROJECT_LEAD'.
            avatarId (integer): An integer value for the project's avatar. Example: 10200.
            categoryId (integer): The ID of the project's category. A complete list of category IDs is found using the [Get all project categories](#api-rest-api-3-projectCategory-get) operation. Example: 10120.
            description (string): A brief description of the project. Example: 'Cloud migration initiative'.
            fieldConfigurationScheme (integer): The ID of the field configuration scheme for the project. Use the [Get all field configuration schemes](#api-rest-api-3-fieldconfigurationscheme-get) operation to get a list of field configuration scheme IDs. If you specify the field configuration scheme you cannot specify the project template key.
            issueSecurityScheme (integer): The ID of the issue security scheme for the project, which enables you to control who can and cannot view issues. Use the [Get issue security schemes](#api-rest-api-3-issuesecurityschemes-get) resource to get all issue security scheme IDs. Example: 10001.
            issueTypeScheme (integer): The ID of the issue type scheme for the project. Use the [Get all issue type schemes](#api-rest-api-3-issuetypescheme-get) operation to get a list of issue type scheme IDs. If you specify the issue type scheme you cannot specify the project template key.
            issueTypeScreenScheme (integer): The ID of the issue type screen scheme for the project. Use the [Get all issue type screen schemes](#api-rest-api-3-issuetypescreenscheme-get) operation to get a list of issue type screen scheme IDs. If you specify the issue type screen scheme you cannot specify the project template key.
            lead (string): This parameter is deprecated because of privacy changes. Use `leadAccountId` instead. See the [migration guide](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details. The user name of the project lead. Either `lead` or `leadAccountId` must be set when creating a project. Cannot be provided with `leadAccountId`.
            leadAccountId (string): The account ID of the project lead. Either `lead` or `leadAccountId` must be set when creating a project. Cannot be provided with `lead`. Example: '5b10a0effa615349cb016cd8'.
            notificationScheme (integer): The ID of the notification scheme for the project. Use the [Get notification schemes](#api-rest-api-3-notificationscheme-get) resource to get a list of notification scheme IDs. Example: 10021.
            permissionScheme (integer): The ID of the permission scheme for the project. Use the [Get all permission schemes](#api-rest-api-3-permissionscheme-get) resource to see a list of all permission scheme IDs. Example: 10011.
            projectTemplateKey (string): A predefined configuration for a project. The type of the `projectTemplateKey` must match with the type of the `projectTypeKey`. Example: 'com.atlassian.jira-core-project-templates:jira-core-simplified-process-control'.
            projectTypeKey (string): The [project type](https://confluence.atlassian.com/x/GwiiLQ#Jiraapplicationsoverview-Productfeaturesandprojecttypes), which defines the application-specific feature set. If you don't specify the project template you have to specify the project type. Example: 'business'.
            url (string): A link to information about this project, such as project documentation Example: 'http://atlassian.com'.
            workflowScheme (integer): The ID of the workflow scheme for the project. Use the [Get all workflow schemes](#api-rest-api-3-workflowscheme-get) operation to get a list of workflow scheme IDs. If you specify the workflow scheme you cannot specify the project template key.

        Returns:
            dict[str, Any]: Returned if the project is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        request_body_data = None
        request_body_data = {
            "assigneeType": assigneeType,
            "avatarId": avatarId,
            "categoryId": categoryId,
            "description": description,
            "fieldConfigurationScheme": fieldConfigurationScheme,
            "issueSecurityScheme": issueSecurityScheme,
            "issueTypeScheme": issueTypeScheme,
            "issueTypeScreenScheme": issueTypeScreenScheme,
            "key": key,
            "lead": lead,
            "leadAccountId": leadAccountId,
            "name": name,
            "notificationScheme": notificationScheme,
            "permissionScheme": permissionScheme,
            "projectTemplateKey": projectTemplateKey,
            "projectTypeKey": projectTypeKey,
            "url": url,
            "workflowScheme": workflowScheme,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_project_template(
        self,
        details: dict[str, Any] | None = None,
        template: dict[str, Any] | None = None,
    ) -> Any:
        """
        Creates a project in Jira from a specified project template and returns a redirect response to the new project.

        Args:
            details (object): Project Details
            template (object): The specific request object for creating a project with template.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project templates
        """
        request_body_data = None
        request_body_data = {
            "details": details,
            "template": template,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project-template"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_recent(
        self,
        expand: str | None = None,
        properties: list[dict[str, Any]] | None = None,
    ) -> list[Any]:
        """
        Retrieves a list of up to 20 recently viewed projects still visible to the user.

        Args:
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expanded options include: * `description` Returns the project description. * `projectKeys` Returns all project keys associated with a project. * `lead` Returns information about the project lead. * `issueTypes` Returns all issue types associated with the project. * `url` Returns the URL associated with the project. * `permissions` Returns the permissions associated with the project. * `insight` EXPERIMENTAL. Returns the insight details of total issue count and last issue update time for the project. * `*` Returns the project with all available expand options.
            properties (array): EXPERIMENTAL. A list of project properties to return for the project. This parameter accepts a comma-separated list. Invalid property names are ignored.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        url = f"{self.base_url}/rest/api/3/project/recent"
        query_params = {
            k: v
            for k, v in [("expand", expand), ("properties", properties)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_projects(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
        id: list[int] | None = None,
        keys: list[str] | None = None,
        query: str | None = None,
        typeKey: str | None = None,
        categoryId: int | None = None,
        action: str | None = None,
        expand: str | None = None,
        status: list[str] | None = None,
        properties: list[dict[str, Any]] | None = None,
        propertyQuery: str | None = None,
    ) -> dict[str, Any]:
        """
        Searches for Jira projects using various criteria such as project ID, keys, category, and more, returning a list of matching projects based on the specified parameters using the Jira Cloud REST API.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field. * `category` Sorts by project category. A complete list of category IDs is found using [Get all project categories](#api-rest-api-3-projectCategory-get). * `issueCount` Sorts by the total number of issues in each project. * `key` Sorts by project key. * `lastIssueUpdatedTime` Sorts by the last issue update time. * `name` Sorts by project name. * `owner` Sorts by project lead. * `archivedDate` EXPERIMENTAL. Sorts by project archived date. * `deletedDate` EXPERIMENTAL. Sorts by project deleted date.
            id (array): The project IDs to filter the results by. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`. Up to 50 project IDs can be provided.
            keys (array): The project keys to filter the results by. To include multiple keys, provide an ampersand-separated list. For example, `keys=PA&keys=PB`. Up to 50 project keys can be provided.
            query (string): Filter the results using a literal string. Projects with a matching `key` or `name` are returned (case insensitive).
            typeKey (string): Orders results by the [project type]( This parameter accepts a comma-separated list. Valid values are `business`, `service_desk`, and `software`.
            categoryId (integer): The ID of the project's category. A complete list of category IDs is found using the [Get all project categories](#api-rest-api-3-projectCategory-get) operation.
            action (string): Filter results by projects for which the user can: * `view` the project, meaning that they have one of the following permissions: * *Browse projects* [project permission]( for the project. * *Administer projects* [project permission]( for the project. * *Administer Jira* [global permission]( * `browse` the project, meaning that they have the *Browse projects* [project permission]( for the project. * `edit` the project, meaning that they have one of the following permissions: * *Administer projects* [project permission]( for the project. * *Administer Jira* [global permission]( * `create` the project, meaning that they have the *Create issues* [project permission]( for the project in which the issue is created.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expanded options include: * `description` Returns the project description. * `projectKeys` Returns all project keys associated with a project. * `lead` Returns information about the project lead. * `issueTypes` Returns all issue types associated with the project. * `url` Returns the URL associated with the project. * `insight` EXPERIMENTAL. Returns the insight details of total issue count and last issue update time for the project.
            status (array): EXPERIMENTAL. Filter results by project status: * `live` Search live projects. * `archived` Search archived projects. * `deleted` Search deleted projects, those in the recycle bin.
            properties (array): EXPERIMENTAL. A list of project properties to return for the project. This parameter accepts a comma-separated list.
            propertyQuery (string): EXPERIMENTAL. A query string used to search properties. The query string cannot be specified using a JSON object. For example, to search for the value of `nested` from `{"something":{"nested":1,"other":2}}` use `[thepropertykey].something.nested=1`. Note that the propertyQuery key is enclosed in square brackets to enable searching where the propertyQuery key includes dot (.) or equals (=) characters. Note that `thepropertykey` is only returned when included in `properties`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        url = f"{self.base_url}/rest/api/3/project/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
                ("id", id),
                ("keys", keys),
                ("query", query),
                ("typeKey", typeKey),
                ("categoryId", categoryId),
                ("action", action),
                ("expand", expand),
                ("status", status),
                ("properties", properties),
                ("propertyQuery", propertyQuery),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_project_types(self) -> list[Any]:
        """
        Retrieves all project types available in Jira Cloud, including those without valid licenses, and can be accessed anonymously without permissions.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project types
        """
        url = f"{self.base_url}/rest/api/3/project/type"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_accessible_project_types(self) -> list[Any]:
        """
        Retrieves a list of project types accessible to the calling user via the Jira Cloud REST API.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project types
        """
        url = f"{self.base_url}/rest/api/3/project/type/accessible"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_type_by_key(self, projectTypeKey: str) -> dict[str, Any]:
        """
        Retrieves the project type details for a specified project type key using the "GET" method.

        Args:
            projectTypeKey (string): projectTypeKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project types
        """
        if projectTypeKey is None:
            raise ValueError("Missing required parameter 'projectTypeKey'.")
        url = f"{self.base_url}/rest/api/3/project/type/{projectTypeKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_accessible_project_type_by_key(self, projectTypeKey: str) -> dict[str, Any]:
        """
        Retrieves details of a specific project type accessible to the user in Jira Cloud based on the provided project type key.

        Args:
            projectTypeKey (string): projectTypeKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project types
        """
        if projectTypeKey is None:
            raise ValueError("Missing required parameter 'projectTypeKey'.")
        url = f"{self.base_url}/rest/api/3/project/type/{projectTypeKey}/accessible"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_project(
        self, projectIdOrKey: str, enableUndo: bool | None = None
    ) -> Any:
        """
        Deletes a Jira project identified by its ID or key, with an optional undo capability via query parameter.

        Args:
            projectIdOrKey (string): projectIdOrKey
            enableUndo (boolean): Whether this project is placed in the Jira recycle bin where it will be available for restoration.

        Returns:
            Any: Returned if the project is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}"
        query_params = {k: v for k, v in [("enableUndo", enableUndo)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project(
        self,
        projectIdOrKey: str,
        expand: str | None = None,
        properties: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves details about a specific Jira project by its ID or key, allowing optional expansion of additional details and properties, using the Jira REST API.

        Args:
            projectIdOrKey (string): projectIdOrKey
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Note that the project description, issue types, and project lead are included in all responses by default. Expand options include: * `description` The project description. * `issueTypes` The issue types associated with the project. * `lead` The project lead. * `projectKeys` All project keys associated with the project. * `issueTypeHierarchy` The project issue type hierarchy.
            properties (array): A list of project properties to return for the project. This parameter accepts a comma-separated list.

        Returns:
            dict[str, Any]: Returned if successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}"
        query_params = {
            k: v
            for k, v in [("expand", expand), ("properties", properties)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project(
        self,
        projectIdOrKey: str,
        expand: str | None = None,
        assigneeType: str | None = None,
        avatarId: int | None = None,
        categoryId: int | None = None,
        description: str | None = None,
        issueSecurityScheme: int | None = None,
        key: str | None = None,
        lead: str | None = None,
        leadAccountId: str | None = None,
        name: str | None = None,
        notificationScheme: int | None = None,
        permissionScheme: int | None = None,
        releasedProjectKeys: list[str] | None = None,
        url: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates or replaces a Jira project identified by the projectIdOrKey and returns the modified project.

        Args:
            projectIdOrKey (string): projectIdOrKey
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Note that the project description, issue types, and project lead are included in all responses by default. Expand options include: * `description` The project description. * `issueTypes` The issue types associated with the project. * `lead` The project lead. * `projectKeys` All project keys associated with the project.
            assigneeType (string): The default assignee when creating issues for this project. Example: 'PROJECT_LEAD'.
            avatarId (integer): An integer value for the project's avatar. Example: 10200.
            categoryId (integer): The ID of the project's category. A complete list of category IDs is found using the [Get all project categories](#api-rest-api-3-projectCategory-get) operation. To remove the project category from the project, set the value to `-1.` Example: 10120.
            description (string): A brief description of the project. Example: 'Cloud migration initiative'.
            issueSecurityScheme (integer): The ID of the issue security scheme for the project, which enables you to control who can and cannot view issues. Use the [Get issue security schemes](#api-rest-api-3-issuesecurityschemes-get) resource to get all issue security scheme IDs. Example: 10001.
            key (string): Project keys must be unique and start with an uppercase letter followed by one or more uppercase alphanumeric characters. The maximum length is 10 characters. Example: 'EX'.
            lead (string): This parameter is deprecated because of privacy changes. Use `leadAccountId` instead. See the [migration guide](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details. The user name of the project lead. Cannot be provided with `leadAccountId`.
            leadAccountId (string): The account ID of the project lead. Cannot be provided with `lead`. Example: '5b10a0effa615349cb016cd8'.
            name (string): The name of the project. Example: 'Example'.
            notificationScheme (integer): The ID of the notification scheme for the project. Use the [Get notification schemes](#api-rest-api-3-notificationscheme-get) resource to get a list of notification scheme IDs. Example: 10021.
            permissionScheme (integer): The ID of the permission scheme for the project. Use the [Get all permission schemes](#api-rest-api-3-permissionscheme-get) resource to see a list of all permission scheme IDs. Example: 10011.
            releasedProjectKeys (array): Previous project keys to be released from the current project. Released keys must belong to the current project and not contain the current project key
            url (string): A link to information about this project, such as project documentation Example: 'http://atlassian.com'.

        Returns:
            dict[str, Any]: Returned if the project is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "assigneeType": assigneeType,
            "avatarId": avatarId,
            "categoryId": categoryId,
            "description": description,
            "issueSecurityScheme": issueSecurityScheme,
            "key": key,
            "lead": lead,
            "leadAccountId": leadAccountId,
            "name": name,
            "notificationScheme": notificationScheme,
            "permissionScheme": permissionScheme,
            "releasedProjectKeys": releasedProjectKeys,
            "url": url,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def archive_project(self, projectIdOrKey: str) -> Any:
        """
        Archives a Jira project using the "POST" method by specifying the project ID or key in the path "/rest/api/3/project/{projectIdOrKey}/archive".

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/archive"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project_avatar(
        self,
        projectIdOrKey: str,
        id: str,
        fileName: str | None = None,
        isDeletable: bool | None = None,
        isSelected: bool | None = None,
        isSystemAvatar: bool | None = None,
        owner: str | None = None,
        urls: dict[str, str] | None = None,
    ) -> Any:
        """
        Sets the displayed avatar for a Jira project using the specified project ID or key.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): The ID of the avatar. Example: '10010'.
            fileName (string): The file name of the avatar icon. Returned for system avatars.
            isDeletable (boolean): Whether the avatar can be deleted.
            isSelected (boolean): Whether the avatar is used in Jira. For example, shown as a project's avatar.
            isSystemAvatar (boolean): Whether the avatar is a system avatar.
            owner (string): The owner of the avatar. For a system avatar the owner is null (and nothing is returned). For non-system avatars this is the appropriate identifier, such as the ID for a project or the account ID for a user.
            urls (object): The list of avatar icon URLs.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project avatars
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "fileName": fileName,
            "id": id,
            "isDeletable": isDeletable,
            "isSelected": isSelected,
            "isSystemAvatar": isSystemAvatar,
            "owner": owner,
            "urls": urls,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/avatar"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_project_avatar(self, projectIdOrKey: str, id: str) -> Any:
        """
        Deletes a custom project avatar (system avatars cannot be deleted) using the Jira REST API.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project avatars
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/avatar/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_project_avatar(
        self,
        projectIdOrKey: str,
        body_content: bytes,
        x: int | None = None,
        y: int | None = None,
        size: int | None = None,
    ) -> dict[str, Any]:
        """
        Loads a custom avatar for a Jira project using the specified parameters and returns the avatar details upon success.

        Args:
            projectIdOrKey (string): projectIdOrKey
            body_content (bytes | None): Raw binary content for the request body.
            x (integer): The X coordinate of the top-left corner of the crop region.
            y (integer): The Y coordinate of the top-left corner of the crop region.
            size (integer): The length of each side of the crop region.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project avatars
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        request_body_data = body_content
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/avatar2"
        query_params = {
            k: v for k, v in [("x", x), ("y", y), ("size", size)] if v is not None
        }
        response = self._post(
            url, data=request_body_data, params=query_params, content_type="*/*"
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_project_avatars(self, projectIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves the list of avatars associated with a specified Jira project, including system and custom avatars.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            dict[str, Any]: Returned if request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project avatars
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/avatars"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_classification_level(self, projectIdOrKey: str) -> Any:
        """
        Removes the default data classification level from a Jira project using the Jira Cloud API, returning a status code indicating success or failure.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project classification levels
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/classification-level/default"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_classification_level(self, projectIdOrKey: str) -> Any:
        """
        Retrieves the default data classification level configured for a specified Jira project.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project classification levels
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/classification-level/default"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project_class_default(self, projectIdOrKey: str, id: str) -> Any:
        """
        Updates the default data classification level for a Jira project.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): The ID of the project classification. Example: 'ari:cloud:platform::classification-tag/dec24c48-5073-4c25-8fef-9d81a992c30c'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project classification levels
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        request_body_data = {
            "id": id,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/classification-level/default"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_components_paginated(
        self,
        projectIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
        componentSource: str | None = None,
        query: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of components associated with a specified Jira project, optionally filtered and ordered by query parameters.

        Args:
            projectIdOrKey (string): projectIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field: * `description` Sorts by the component description. * `issueCount` Sorts by the count of issues associated with the component. * `lead` Sorts by the user key of the component's project lead. * `name` Sorts by component name.
            componentSource (string): The source of the components to return. Can be `jira` (default), `compass` or `auto`. When `auto` is specified, the API will return connected Compass components if the project is opted into Compass, otherwise it will return Jira components. Defaults to `jira`.
            query (string): Filter the results using a literal string. Components with a matching `name` or `description` are returned (case insensitive).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/component"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
                ("componentSource", componentSource),
                ("query", query),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_components(
        self, projectIdOrKey: str, componentSource: str | None = None
    ) -> list[Any]:
        """
        Retrieves a list of components for a specified Jira project using its ID or key, with optional filtering by component source.

        Args:
            projectIdOrKey (string): projectIdOrKey
            componentSource (string): The source of the components to return. Can be `jira` (default), `compass` or `auto`. When `auto` is specified, the API will return connected Compass components if the project is opted into Compass, otherwise it will return Jira components. Defaults to `jira`.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project components
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/components"
        query_params = {
            k: v for k, v in [("componentSource", componentSource)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_project_asynchronously(self, projectIdOrKey: str) -> Any:
        """
        Deletes a Jira project specified by its ID or key via a POST request and returns relevant status codes.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/delete"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_features_for_project(self, projectIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves the list of features for a specified Jira project using the project ID or key.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project features
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/features"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def toggle_feature_for_project(
        self, projectIdOrKey: str, featureKey: str, state: str | None = None
    ) -> dict[str, Any]:
        """
        Updates the configuration of a specific feature for a Jira project using the project identifier and feature key.

        Args:
            projectIdOrKey (string): projectIdOrKey
            featureKey (string): featureKey
            state (string): The feature state. Example: 'ENABLED'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project features
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if featureKey is None:
            raise ValueError("Missing required parameter 'featureKey'.")
        request_body_data = None
        request_body_data = {
            "state": state,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = (
            f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/features/{featureKey}"
        )
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_property_keys(self, projectIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves a list of project property keys for a specified project in Jira Cloud using the provided project ID or key.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project properties
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/properties"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_project_property(self, projectIdOrKey: str, propertyKey: str) -> Any:
        """
        Deletes a specific project property from a Jira project using the project ID or key and property key, requiring administrative permissions.

        Args:
            projectIdOrKey (string): projectIdOrKey
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the project property is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project properties
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/properties/{propertyKey}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_property(
        self, projectIdOrKey: str, propertyKey: str
    ) -> dict[str, Any]:
        """
        Retrieves the value of a specific project property using the Jira Cloud API and returns it based on the provided project ID or key and property key.

        Args:
            projectIdOrKey (string): projectIdOrKey
            propertyKey (string): propertyKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project properties
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/properties/{propertyKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_project_property(self, projectIdOrKey: str, propertyKey: str) -> Any:
        """
        Updates a project property using the Jira Cloud REST API, allowing custom data to be stored against a specific project by setting the value of a specified property key.

        Args:
            projectIdOrKey (string): projectIdOrKey
            propertyKey (string): propertyKey

        Returns:
            Any: Returned if the project property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project properties
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/properties/{propertyKey}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def restore(self, projectIdOrKey: str) -> dict[str, Any]:
        """
        Restores a deleted or archived Jira project identified by its project ID or key.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/restore"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_roles(self, projectIdOrKey: str) -> dict[str, Any]:
        """
        Retrieves a list of project roles (including names, IDs, and self URLs) for a specific Jira project using its ID or key.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/role"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_actor(
        self,
        projectIdOrKey: str,
        id: str,
        user: str | None = None,
        group: str | None = None,
        groupId: str | None = None,
    ) -> Any:
        """
        Deletes a user or group from a specific project role in Jira, returning a success status if removed.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): id
            user (string): The user account ID of the user to remove from the project role. Example: '5b10ac8d82e05b22cc7d4ef5'.
            group (string): The name of the group to remove from the project role. This parameter cannot be used with the `groupId` parameter. As a group's name can change, use of `groupId` is recommended.
            groupId (string): The ID of the group to remove from the project role. This parameter cannot be used with the `group` parameter.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/role/{id}"
        query_params = {
            k: v
            for k, v in [("user", user), ("group", group), ("groupId", groupId)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_role(
        self, projectIdOrKey: str, id: str, excludeInactiveUsers: bool | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a specific project role's details and associated actors for a Jira project using the provided project identifier and role ID.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): id
            excludeInactiveUsers (boolean): Exclude inactive users.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles, important
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/role/{id}"
        query_params = {
            k: v
            for k, v in [("excludeInactiveUsers", excludeInactiveUsers)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_actor_users(
        self,
        projectIdOrKey: str,
        id: str,
        group: list[str] | None = None,
        groupId: list[str] | None = None,
        user: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Partially updates a project role's name or description using the Jira REST API.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): id
            group (array): The name of the group to add. This parameter cannot be used with the `groupId` parameter. As a group's name can change, use of `groupId` is recommended.
            groupId (array): The ID of the group to add. This parameter cannot be used with the `group` parameter. Example: ['952d12c3-5b5b-4d04-bb32-44d383afc4b2'].
            user (array): The user account ID of the user to add.

        Returns:
            dict[str, Any]: Returned if the request is successful. The complete list of actors for the project is returned.

        For example, the cURL request above adds a group, *jira-developers*. For the response below to be returned as a result of that request, the user *Mia Krystof* would have previously been added as a `user` actor for this project.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "group": group,
            "groupId": groupId,
            "user": user,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/role/{id}"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_actors(
        self,
        projectIdOrKey: str,
        id: str,
        categorisedActors: dict[str, list[str]] | None = None,
        id_body: int | None = None,
    ) -> dict[str, Any]:
        """
        Fully updates a project role by ID for a specified Jira project using the provided parameters.

        Args:
            projectIdOrKey (string): projectIdOrKey
            id (string): id
            categorisedActors (object): The actors to add to the project role.

        Add groups using:

         *  `atlassian-group-role-actor` and a list of group names.
         *  `atlassian-group-role-actor-id` and a list of group IDs.

        As a group's name can change, use of `atlassian-group-role-actor-id` is recommended. For example, `"atlassian-group-role-actor-id":["eef79f81-0b89-4fca-a736-4be531a10869","77f6ab39-e755-4570-a6ae-2d7a8df0bcb8"]`.

        Add users using `atlassian-user-role-actor` and a list of account IDs. For example, `"atlassian-user-role-actor":["12345678-9abc-def1-2345-6789abcdef12", "abcdef12-3456-789a-bcde-f123456789ab"]`. Example: {'atlassian-group-role-actor-id': ['952d12c3-5b5b-4d04-bb32-44d383afc4b2'], 'atlassian-user-role-actor': ['12345678-9abc-def1-2345-6789abcdef12']}.
            id_body (integer): The ID of the project role. Use [Get all project roles](#api-rest-api-3-role-get) to get a list of project role IDs.

        Returns:
            dict[str, Any]: Returned if the request is successful. The complete list of actors for the project is returned.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "categorisedActors": categorisedActors,
            "id": id_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/role/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_role_details(
        self,
        projectIdOrKey: str,
        currentMember: bool | None = None,
        excludeConnectAddons: bool | None = None,
    ) -> list[Any]:
        """
        Retrieves role details for a specified project, including information about project roles and their associated members, using a GET request.

        Args:
            projectIdOrKey (string): projectIdOrKey
            currentMember (boolean): Whether the roles should be filtered to include only those the user is assigned to.
            excludeConnectAddons (boolean): Excludes Connect add-ons from the role details response when set to true.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/roledetails"
        query_params = {
            k: v
            for k, v in [
                ("currentMember", currentMember),
                ("excludeConnectAddons", excludeConnectAddons),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_statuses(self, projectIdOrKey: str) -> list[Any]:
        """
        Retrieves a list of statuses available for a specific Jira project, identified by its ID or key, using the GET method.

        Args:
            projectIdOrKey (string): projectIdOrKey

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/statuses"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_versions_paginated(
        self,
        projectIdOrKey: str,
        startAt: int | None = None,
        maxResults: int | None = None,
        orderBy: str | None = None,
        query: str | None = None,
        status: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of versions for a specified Jira project, allowing for pagination, filtering, and expansion of details using various query parameters.

        Args:
            projectIdOrKey (string): projectIdOrKey
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            orderBy (string): [Order](#ordering) the results by a field: * `description` Sorts by version description. * `name` Sorts by version name. * `releaseDate` Sorts by release date, starting with the oldest date. Versions with no release date are listed last. * `sequence` Sorts by the order of appearance in the user interface. * `startDate` Sorts by start date, starting with the oldest date. Versions with no start date are listed last.
            query (string): Filter the results using a literal string. Versions with matching `name` or `description` are returned (case insensitive).
            status (string): A list of status values used to filter the results by version status. This parameter accepts a comma-separated list. The status values are `released`, `unreleased`, and `archived`.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `issuesstatus` Returns the number of issues in each status category for each version. * `operations` Returns actions that can be performed on the specified version. * `driver` Returns the Atlassian account ID of the version driver. * `approvers` Returns a list containing the approvers for this version.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/version"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("orderBy", orderBy),
                ("query", query),
                ("status", status),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_versions(
        self, projectIdOrKey: str, expand: str | None = None
    ) -> list[Any]:
        """
        Retrieves all versions of a specified Jira project, including version details like names, descriptions, and issue status counts.

        Args:
            projectIdOrKey (string): projectIdOrKey
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts `operations`, which returns actions that can be performed on the version.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if projectIdOrKey is None:
            raise ValueError("Missing required parameter 'projectIdOrKey'.")
        url = f"{self.base_url}/rest/api/3/project/{projectIdOrKey}/versions"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_email(self, projectId: str) -> dict[str, Any]:
        """
        Retrieves email-related information for a specified project using its unique identifier.

        Args:
            projectId (string): projectId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project email
        """
        if projectId is None:
            raise ValueError("Missing required parameter 'projectId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectId}/email"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project_email(
        self,
        projectId: str,
        emailAddress: str | None = None,
        emailAddressStatus: list[str] | None = None,
    ) -> Any:
        """
        Updates the sender email address for a specific project's notifications and returns a success status upon completion.

        Args:
            projectId (string): projectId
            emailAddress (string): The email address. Example: 'jira@example.atlassian.net'.
            emailAddressStatus (array): When using a custom domain, the status of the email address.

        Returns:
            Any: Returned if the project's sender email address is successfully set.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project email
        """
        if projectId is None:
            raise ValueError("Missing required parameter 'projectId'.")
        request_body_data = None
        request_body_data = {
            "emailAddress": emailAddress,
            "emailAddressStatus": emailAddressStatus,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectId}/email"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_hierarchy(self, projectId: str) -> dict[str, Any]:
        """
        Retrieves the hierarchy details for a specific project using the `GET` method at path "/rest/api/3/project/{projectId}/hierarchy".

        Args:
            projectId (string): projectId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectId is None:
            raise ValueError("Missing required parameter 'projectId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectId}/hierarchy"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_issue_security_scheme(self, projectKeyOrId: str) -> dict[str, Any]:
        """
        Retrieves the issue security level scheme associated with a specified project in Jira.

        Args:
            projectKeyOrId (string): projectKeyOrId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project permission schemes
        """
        if projectKeyOrId is None:
            raise ValueError("Missing required parameter 'projectKeyOrId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectKeyOrId}/issuesecuritylevelscheme"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_notification_scheme_by_project(
        self, projectKeyOrId: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves the notification scheme associated with a specific project in Jira, including event configurations and recipient details.

        Args:
            projectKeyOrId (string): projectKeyOrId
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `all` Returns all expandable information * `field` Returns information about any custom fields assigned to receive an event * `group` Returns information about any groups assigned to receive an event * `notificationSchemeEvents` Returns a list of event associations. This list is returned for all expandable information * `projectRole` Returns information about any project roles assigned to receive an event * `user` Returns information about any users assigned to receive an event

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Projects
        """
        if projectKeyOrId is None:
            raise ValueError("Missing required parameter 'projectKeyOrId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectKeyOrId}/notificationscheme"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_assigned_permission_scheme(
        self, projectKeyOrId: str, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves the permission scheme associated with a specified Jira project by its key or ID, allowing optional expansion of certain details.

        Args:
            projectKeyOrId (string): projectKeyOrId
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project permission schemes
        """
        if projectKeyOrId is None:
            raise ValueError("Missing required parameter 'projectKeyOrId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectKeyOrId}/permissionscheme"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def assign_permission_scheme(
        self, projectKeyOrId: str, id: int, expand: str | None = None
    ) -> dict[str, Any]:
        """
        Assigns a permission scheme to a project using the Jira Cloud API, allowing administrators to manage project permissions by associating a specific permission scheme with a given project.

        Args:
            projectKeyOrId (string): projectKeyOrId
            id (integer): The ID of the permission scheme to associate with the project. Use the [Get all permission schemes](#api-rest-api-3-permissionscheme-get) resource to get a list of permission scheme IDs. Example: 10000.
            expand (string): Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Note that permissions are included when you specify any value. Expand options include: * `all` Returns all expandable information. * `field` Returns information about the custom field granted the permission. * `group` Returns information about the group that is granted the permission. * `permissions` Returns all permission grants for each permission scheme. * `projectRole` Returns information about the project role granted the permission. * `user` Returns information about the user who is granted the permission.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project permission schemes
        """
        if projectKeyOrId is None:
            raise ValueError("Missing required parameter 'projectKeyOrId'.")
        request_body_data = None
        request_body_data = {
            "id": id,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/project/{projectKeyOrId}/permissionscheme"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_security_levels_for_project(self, projectKeyOrId: str) -> dict[str, Any]:
        """
        Retrieves issue security levels for a specified project using the provided project key or ID, returning details about the security levels associated with the project.

        Args:
            projectKeyOrId (string): projectKeyOrId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project permission schemes
        """
        if projectKeyOrId is None:
            raise ValueError("Missing required parameter 'projectKeyOrId'.")
        url = f"{self.base_url}/rest/api/3/project/{projectKeyOrId}/securitylevel"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_project_categories(self) -> list[Any]:
        """
        Retrieves a project category from Jira using its ID and returns the category details.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project categories
        """
        url = f"{self.base_url}/rest/api/3/projectCategory"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_project_category(
        self,
        description: str | None = None,
        id: str | None = None,
        name: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new project category in Jira and returns the created category details.

        Args:
            description (string): The description of the project category. Example: 'Created Project Category'.
            id (string): The ID of the project category.
            name (string): The name of the project category. Required on create, optional on update. Example: 'CREATED'.
            self_arg_body (string): The URL of the project category.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project categories
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "id": id,
            "name": name,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/projectCategory"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_project_category(self, id: str) -> Any:
        """
        Deletes a project category by its ID using the Jira Cloud API, requiring admin permissions and returning a successful response without content if the operation is completed.

        Args:
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project categories
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/projectCategory/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_category_by_id(self, id: str) -> dict[str, Any]:
        """
        Retrieves a specific project category by ID from Jira using the REST API.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project categories
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/projectCategory/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_project_category(
        self,
        id: str,
        description: str | None = None,
        id_body: str | None = None,
        name: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates a specific project category by ID in Jira using the PUT method, allowing modifications to the category details such as name and description.

        Args:
            id (string): id
            description (string): The description of the project category. Example: 'Updated Project Category'.
            id_body (string): The ID of the project category.
            name (string): The name of the project category. Required on create, optional on update. Example: 'UPDATED'.
            self_arg_body (string): The URL of the project category.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project categories
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "id": id_body,
            "name": name,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/projectCategory/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def validate_project_key(self, key: str | None = None) -> dict[str, Any]:
        """
        Validates a project key by confirming the key's validity and checking for existing usage in Jira Cloud.

        Args:
            key (string): The project key. Example: 'HSP'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project key and name validation
        """
        url = f"{self.base_url}/rest/api/3/projectvalidate/key"
        query_params = {k: v for k, v in [("key", key)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_valid_project_key(self, key: str | None = None) -> Any:
        """
        Validates a project key by confirming it is a valid string and not in use, returning success or error messages using the Jira Cloud REST API.

        Args:
            key (string): The project key. Example: 'HSP'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project key and name validation
        """
        url = f"{self.base_url}/rest/api/3/projectvalidate/validProjectKey"
        query_params = {k: v for k, v in [("key", key)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_valid_project_name(self, name: str) -> Any:
        """
        Validates a project name's availability, returning the original name if available or generating a new valid name if unavailable.

        Args:
            name (string): The project name.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project key and name validation
        """
        url = f"{self.base_url}/rest/api/3/projectvalidate/validProjectName"
        query_params = {k: v for k, v in [("name", name)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_resolutions(self) -> list[Any]:
        """
        Retrieves a list of available resolution statuses for Jira issues.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        url = f"{self.base_url}/rest/api/3/resolution"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_resolution(
        self, name: str, description: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a new issue resolution in Jira using the REST API and returns the created resolution details.

        Args:
            name (string): The name of the resolution. Must be unique (case-insensitive). Example: 'My new resolution'.
            description (string): The description of the resolution. Example: 'My resolution description'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/resolution"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_default_resolution(self, id: str) -> Any:
        """
        Sets the default issue resolution in Jira using the REST API, requiring administrative permissions.

        Args:
            id (string): The ID of the new default issue resolution. Must be an existing ID or null. Setting this to null erases the default resolution setting. Example: '3'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        request_body_data = None
        request_body_data = {
            "id": id,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/resolution/default"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_resolutions(
        self,
        ids: list[str],
        after: str | None = None,
        position: str | None = None,
    ) -> Any:
        """
        Moves issue resolutions using the Jira Cloud API with a PUT request to the "/rest/api/3/resolution/move" endpoint.

        Args:
            ids (array): The list of resolution IDs to be reordered. Cannot contain duplicates nor after ID. Example: ['10000', '10001'].
            after (string): The ID of the resolution. Required if `position` isn't provided. Example: '10002'.
            position (string): The position for issue resolutions to be moved to. Required if `after` isn't provided.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        request_body_data = None
        request_body_data = {
            "after": after,
            "ids": ids,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/resolution/move"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_resolutions(
        self,
        startAt: str | None = None,
        maxResults: str | None = None,
        id: list[str] | None = None,
        onlyDefault: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of Jira issue resolutions using the GET method at the "/rest/api/3/resolution/search" path, allowing for filtering by parameters such as start position, maximum results, resolution ID, and default-only options.

        Args:
            startAt (string): The index of the first item to return in a page of results (page offset).
            maxResults (string): The maximum number of items to return per page.
            id (array): The list of resolutions IDs to be filtered out
            onlyDefault (boolean): When set to true, return default only, when IDs provided, if none of them is default, return empty page. Default value is false

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        url = f"{self.base_url}/rest/api/3/resolution/search"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("onlyDefault", onlyDefault),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_resolution(self, id: str, replaceWith: str) -> Any:
        """
        Deletes an issue resolution by ID using the Jira API and optionally replaces it with another resolution if specified.

        Args:
            id (string): id
            replaceWith (string): The ID of the issue resolution that will replace the currently selected resolution.

        Returns:
            Any: API response data.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/resolution/{id}"
        query_params = {
            k: v for k, v in [("replaceWith", replaceWith)] if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_resolution(self, id: str) -> dict[str, Any]:
        """
        Retrieves the details of a specific resolution by its ID using the Jira Cloud Platform REST API and returns the resolution information.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/resolution/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_resolution(
        self, id: str, name: str, description: str | None = None
    ) -> Any:
        """
        Updates the specified resolution's details in Jira using the provided ID, returning a success status upon completion.

        Args:
            id (string): id
            name (string): The name of the resolution. Must be unique. Example: 'My updated resolution'.
            description (string): The description of the resolution. Example: 'My updated resolution description'.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue resolutions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/resolution/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_project_roles(self) -> list[Any]:
        """
        Retrieves role details in Jira projects using the specified REST API endpoint.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        url = f"{self.base_url}/rest/api/3/role"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_project_role(
        self, description: str | None = None, name: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a new role using the Jira API and returns a status message, handling responses for various HTTP status codes including successful creation, bad requests, unauthorized access, forbidden actions, and conflicts.

        Args:
            description (string): A description of the project role. Required when fully updating a project role. Optional when creating or partially updating a project role. Example: 'A project role that represents developers in a project'.
            name (string): The name of the project role. Must be unique. Cannot begin or end with whitespace. The maximum length is 255 characters. Required when creating a project role. Optional when partially updating a project role. Example: 'Developers'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/role"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_project_role(self, id: str, swap: int | None = None) -> Any:
        """
        Deletes a role by its ID using the REST API.

        Args:
            id (string): id
            swap (integer): The ID of the project role that will replace the one being deleted. The swap will attempt to swap the role in schemes (notifications, permissions, issue security), workflows, worklogs and comments.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/role/{id}"
        query_params = {k: v for k, v in [("swap", swap)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_role_by_id(self, id: str) -> dict[str, Any]:
        """
        Retrieves a specific role in Jira by its ID using the GET method and returns the role details.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/role/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def partial_update_project_role(
        self, id: str, description: str | None = None, name: str | None = None
    ) -> dict[str, Any]:
        """
        Updates a specific project role's configuration and returns the modified role details.

        Args:
            id (string): id
            description (string): A description of the project role. Required when fully updating a project role. Optional when creating or partially updating a project role. Example: 'A project role that represents developers in a project'.
            name (string): The name of the project role. Must be unique. Cannot begin or end with whitespace. The maximum length is 255 characters. Required when creating a project role. Optional when partially updating a project role. Example: 'Developers'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/role/{id}"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def fully_update_project_role(
        self, id: str, description: str | None = None, name: str | None = None
    ) -> dict[str, Any]:
        """
        Updates or replaces an existing role's configuration via specified ID and returns the operation status.

        Args:
            id (string): id
            description (string): A description of the project role. Required when fully updating a project role. Optional when creating or partially updating a project role. Example: 'A project role that represents developers in a project'.
            name (string): The name of the project role. Must be unique. Cannot begin or end with whitespace. The maximum length is 255 characters. Required when creating a project role. Optional when partially updating a project role. Example: 'Developers'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project roles
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/role/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_role_actor_by_id(
        self,
        id: str,
        user: str | None = None,
        groupId: str | None = None,
        group: str | None = None,
    ) -> dict[str, Any]:
        """
        Deletes actors from a role using the "DELETE" method with options to specify a user or group ID, and returns corresponding status codes based on the success or failure of the operation.

        Args:
            id (string): id
            user (string): The user account ID of the user to remove as a default actor. Example: '5b10ac8d82e05b22cc7d4ef5'.
            groupId (string): The group ID of the group to be removed as a default actor. This parameter cannot be used with the `group` parameter.
            group (string): The group name of the group to be removed as a default actor.This parameter cannot be used with the `groupId` parameter. As a group's name can change, use of `groupId` is recommended.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/role/{id}/actors"
        query_params = {
            k: v
            for k, v in [("user", user), ("groupId", groupId), ("group", group)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_role_actors_for_role(self, id: str) -> dict[str, Any]:
        """
        Retrieves a list of actors associated with a specified role ID using the Jira REST API.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/role/{id}/actors"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_project_role_actors_to_role(
        self,
        id: str,
        group: list[str] | None = None,
        groupId: list[Any] | None = None,
        user: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Adds or modifies actors (users/groups) for a specific project role ID and returns the updated role details.

        Args:
            id (string): id
            group (array): The name of the group to add as a default actor. This parameter cannot be used with the `groupId` parameter. As a group's name can change,use of `groupId` is recommended. This parameter accepts a comma-separated list. For example, `"group":["project-admin", "jira-developers"]`.
            groupId (array): The ID of the group to add as a default actor. This parameter cannot be used with the `group` parameter This parameter accepts a comma-separated list. For example, `"groupId":["77f6ab39-e755-4570-a6ae-2d7a8df0bcb8", "0c011f85-69ed-49c4-a801-3b18d0f771bc"]`.
            user (array): The account IDs of the users to add as default actors. This parameter accepts a comma-separated list. For example, `"user":["5b10a2844c20165700ede21g", "5b109f2e9729b51b54dc274d"]`. Example: ['admin'].

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project role actors
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "group": group,
            "groupId": groupId,
            "user": user,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/role/{id}/actors"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_screens(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
        queryString: str | None = None,
        scope: list[str] | None = None,
        orderBy: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of all screens or specified screens by ID in Jira, allowing for optional filtering by query parameters such as start position, maximum results, and query string.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of screen IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.
            queryString (string): String used to perform a case-insensitive partial match with screen name.
            scope (array): The scope filter string. To filter by multiple scope, provide an ampersand-separated list. For example, `scope=GLOBAL&scope=PROJECT`.
            orderBy (string): [Order](#ordering) the results by a field: * `id` Sorts by screen ID. * `name` Sorts by screen name.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        url = f"{self.base_url}/rest/api/3/screens"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("queryString", queryString),
                ("scope", scope),
                ("orderBy", orderBy),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_screen(
        self, name: str, description: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a new screen in Jira using the REST API and returns a successful response if the operation is completed without errors.

        Args:
            name (string): The name of the screen. The name must be unique. The maximum length is 255 characters. Example: 'Resolve Security Issue Screen'.
            description (string): The description of the screen. The maximum length is 255 characters. Example: 'Enables changes to resolution and linked issues.'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_field_to_default_screen(self, fieldId: str) -> Any:
        """
        Adds a custom field to the default tab of the default screen using the Jira Cloud REST API.

        Args:
            fieldId (string): fieldId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        if fieldId is None:
            raise ValueError("Missing required parameter 'fieldId'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/screens/addToDefault/{fieldId}"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_bulk_screen_tabs(
        self,
        screenId: list[int] | None = None,
        tabId: list[int] | None = None,
        startAt: int | None = None,
        maxResult: int | None = None,
    ) -> Any:
        """
        Retrieves the list of tabs for a specified screen in Jira Cloud, including pagination support through startAt and maxResult parameters.

        Args:
            screenId (array): The list of screen IDs. To include multiple screen IDs, provide an ampersand-separated list. For example, `screenId=10000&screenId=10001`.
            tabId (array): The list of tab IDs. To include multiple tab IDs, provide an ampersand-separated list. For example, `tabId=10000&tabId=10001`.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResult (integer): The maximum number of items to return per page. The maximum number is 100,

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        url = f"{self.base_url}/rest/api/3/screens/tabs"
        query_params = {
            k: v
            for k, v in [
                ("screenId", screenId),
                ("tabId", tabId),
                ("startAt", startAt),
                ("maxResult", maxResult),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_screen(self, screenId: str) -> Any:
        """
        Deletes a specified screen in Jira if not used in screen schemes, workflows, or workflow drafts, returning success if the operation completes.

        Args:
            screenId (string): screenId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_screen(
        self,
        screenId: str,
        description: str | None = None,
        name: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates or replaces a screen resource identified by the specified `screenId` using the PUT method.

        Args:
            screenId (string): screenId
            description (string): The description of the screen. The maximum length is 255 characters. Example: 'Enables changes to resolution and linked issues for accessibility related issues.'.
            name (string): The name of the screen. The name must be unique. The maximum length is 255 characters. Example: 'Resolve Accessibility Issue Screen'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens/{screenId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_available_screen_fields(self, screenId: str) -> list[Any]:
        """
        Retrieves available fields for a specified screen in Jira, including both system and custom fields.

        Args:
            screenId (string): screenId

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screens
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/availableFields"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_screen_tabs(
        self, screenId: str, projectKey: str | None = None
    ) -> list[Any]:
        """
        Retrieves the list of tabs configured for a specific screen in Jira using the provided screen ID.

        Args:
            screenId (string): screenId
            projectKey (string): The key of the project.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs"
        query_params = {k: v for k, v in [("projectKey", projectKey)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_screen_tab(
        self, screenId: str, name: str, id: int | None = None
    ) -> dict[str, Any]:
        """
        Creates a new tab in a Jira screen using the specified screen ID and returns the created screen tab.

        Args:
            screenId (string): screenId
            name (string): The name of the screen tab. The maximum length is 255 characters. Example: 'Fields Tab'.
            id (integer): The ID of the screen tab.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        request_body_data = None
        request_body_data = {
            "id": id,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_screen_tab(self, screenId: str, tabId: str) -> Any:
        """
        Deletes a specified screen tab from a Jira screen and returns a success status upon completion.

        Args:
            screenId (string): screenId
            tabId (string): tabId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def rename_screen_tab(
        self, screenId: str, tabId: str, name: str, id: int | None = None
    ) -> dict[str, Any]:
        """
        Updates the details of a specific screen tab identified by `screenId` and `tabId` using the Jira API, returning a status message upon successful modification.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            name (string): The name of the screen tab. The maximum length is 255 characters.
            id (integer): The ID of the screen tab.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        request_body_data = None
        request_body_data = {
            "id": id,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_screen_tab_fields(
        self, screenId: str, tabId: str, projectKey: str | None = None
    ) -> list[Any]:
        """
        Retrieves all fields associated with a specific screen tab in Jira Cloud.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            projectKey (string): The key of the project.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tab fields
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}/fields"
        query_params = {k: v for k, v in [("projectKey", projectKey)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def add_screen_tab_field(
        self, screenId: str, tabId: str, fieldId: str
    ) -> dict[str, Any]:
        """
        Adds a field to a specified screen tab in Jira and returns the field configuration upon success.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            fieldId (string): The ID of the field to add. Example: 'summary'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tab fields
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        request_body_data = None
        request_body_data = {
            "fieldId": fieldId,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}/fields"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_screen_tab_field(self, screenId: str, tabId: str, id: str) -> Any:
        """
        Removes a field from a specific screen tab in Jira and returns an empty response upon success.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tab fields
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}/fields/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_screen_tab_field(
        self,
        screenId: str,
        tabId: str,
        id: str,
        after: str | None = None,
        position: str | None = None,
    ) -> Any:
        """
        Moves a screen tab field to a new position using the Jira Cloud platform REST API, allowing for reorganization of issue details fields on a specific screen tab.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            id (string): id
            after (string): The ID of the screen tab field after which to place the moved screen tab field. Required if `position` isn't provided.
            position (string): The named position to which the screen tab field should be moved. Required if `after` isn't provided.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tab fields
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "after": after,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}/fields/{id}/move"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_screen_tab(self, screenId: str, tabId: str, pos: str) -> Any:
        """
        Moves a tab to a specified position within a screen using the "POST" method.

        Args:
            screenId (string): screenId
            tabId (string): tabId
            pos (string): pos

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen tabs
        """
        if screenId is None:
            raise ValueError("Missing required parameter 'screenId'.")
        if tabId is None:
            raise ValueError("Missing required parameter 'tabId'.")
        if pos is None:
            raise ValueError("Missing required parameter 'pos'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/screens/{screenId}/tabs/{tabId}/move/{pos}"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_screen_schemes(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        id: list[int] | None = None,
        expand: str | None = None,
        queryString: str | None = None,
        orderBy: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of screen schemes used in classic projects in Jira Cloud using the GET method, allowing for optional filtering by query parameters such as startAt, maxResults, id, expand, queryString, and orderBy.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            id (array): The list of screen scheme IDs. To include multiple IDs, provide an ampersand-separated list. For example, `id=10000&id=10001`.
            expand (string): Use [expand](#expansion) include additional information in the response. This parameter accepts `issueTypeScreenSchemes` that, for each screen schemes, returns information about the issue type screen scheme the screen scheme is assigned to.
            queryString (string): String used to perform a case-insensitive partial match with screen scheme name.
            orderBy (string): [Order](#ordering) the results by a field: * `id` Sorts by screen scheme ID. * `name` Sorts by screen scheme name.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen schemes
        """
        url = f"{self.base_url}/rest/api/3/screenscheme"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("id", id),
                ("expand", expand),
                ("queryString", queryString),
                ("orderBy", orderBy),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_screen_scheme(
        self, name: str, screens: Any, description: str | None = None
    ) -> dict[str, Any]:
        """
        Creates a new screen scheme in Jira for defining screen configurations associated with workflows and returns the created resource upon success.

        Args:
            name (string): The name of the screen scheme. The name must be unique. The maximum length is 255 characters. Example: 'Employee screen scheme'.
            screens (string): The IDs of the screens for the screen types of the screen scheme. Only screens used in classic projects are accepted. Example: {'default': 10017, 'edit': 10019, 'view': 10020}.
            description (string): The description of the screen scheme. The maximum length is 255 characters. Example: 'Manage employee data'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen schemes
        """
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
            "screens": screens,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screenscheme"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_screen_scheme(self, screenSchemeId: str) -> Any:
        """
        Deletes a screen scheme in Jira using the `DELETE` method, provided it is not used in an issue type screen scheme and is associated with a classic project.

        Args:
            screenSchemeId (string): screenSchemeId

        Returns:
            Any: Returned if the screen scheme is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen schemes
        """
        if screenSchemeId is None:
            raise ValueError("Missing required parameter 'screenSchemeId'.")
        url = f"{self.base_url}/rest/api/3/screenscheme/{screenSchemeId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_screen_scheme(
        self,
        screenSchemeId: str,
        description: str | None = None,
        name: str | None = None,
        screens: Any | None = None,
    ) -> Any:
        """
        Updates a specific screen scheme, identified by its ID, in Jira Cloud's classic projects, allowing modifications to its details and settings.

        Args:
            screenSchemeId (string): screenSchemeId
            description (string): The description of the screen scheme. The maximum length is 255 characters.
            name (string): The name of the screen scheme. The name must be unique. The maximum length is 255 characters. Example: 'Employee screen scheme v2'.
            screens (string): The IDs of the screens for the screen types of the screen scheme. Only screens used in classic projects are accepted. Example: {'create': '10019', 'default': '10018'}.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Screen schemes
        """
        if screenSchemeId is None:
            raise ValueError("Missing required parameter 'screenSchemeId'.")
        request_body_data = None
        request_body_data = {
            "description": description,
            "name": name,
            "screens": screens,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/screenscheme/{screenSchemeId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_for_issues_using_jql(
        self,
        jql: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        validateQuery: str | None = None,
        fields: list[str] | None = None,
        expand: str | None = None,
        properties: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        failFast: bool | None = None,
    ) -> dict[str, Any]:
        """
        Searches for Jira issues using JQL queries and returns paginated results with specified fields and expansion options.

        Args:
            jql (string): The [JQL]( that defines the search. Note: * If no JQL expression is provided, all issues are returned. * `username` and `userkey` cannot be used as search terms due to privacy reasons. Use `accountId` instead. * If a user has hidden their email address in their user profile, partial matches of the email address will not find the user. An exact match is required. Example: 'project = HSP'.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page. To manage page size, Jira may return fewer items per page where a large number of fields or properties are requested. The greatest number of items returned per page is achieved when requesting `id` or `key` only.
            validateQuery (string): Determines how to validate the JQL query and treat the validation results. Supported values are: * `strict` Returns a 400 response code if any errors are found, along with a list of all errors (and warnings). * `warn` Returns all errors as warnings. * `none` No validation is performed. * `true` *Deprecated* A legacy synonym for `strict`. * `false` *Deprecated* A legacy synonym for `warn`. Note: If the JQL is not correctly formed a 400 response code is returned, regardless of the `validateQuery` value.
            fields (array): A list of fields to return for each issue, use it to retrieve a subset of fields. This parameter accepts a comma-separated list. Expand options include: * `*all` Returns all fields. * `*navigable` Returns navigable fields. * Any issue field, prefixed with a minus to exclude. Examples: * `summary,comment` Returns only the summary and comments fields. * `-description` Returns all navigable (default) fields except description. * `*all,-comment` Returns all fields except comments. This parameter may be specified multiple times. For example, `fields=field1,field2&fields=field3`. Note: All navigable fields are returned by default. This differs from [GET issue](#api-rest-api-3-issue-issueIdOrKey-get) where the default is all fields.
            expand (string): Use [expand](#expansion) to include additional information about issues in the response. This parameter accepts a comma-separated list. Expand options include: * `renderedFields` Returns field values rendered in HTML format. * `names` Returns the display name of each field. * `schema` Returns the schema describing a field type. * `transitions` Returns all possible transitions for the issue. * `operations` Returns all possible operations for the issue. * `editmeta` Returns information about how each field can be edited. * `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent. * `versionedRepresentations` Instead of `fields`, returns `versionedRepresentations` a JSON array containing each version of a field's value, with the highest numbered item representing the most recent version.
            properties (array): A list of issue property keys for issue properties to include in the results. This parameter accepts a comma-separated list. Multiple properties can also be provided using an ampersand separated list. For example, `properties=prop1,prop2&properties=prop3`. A maximum of 5 issue property keys can be specified.
            fieldsByKeys (boolean): Reference fields by their key (rather than ID).
            failFast (boolean): Whether to fail the request quickly in case of an error while loading fields for an issue. For `failFast=true`, if one field fails, the entire operation fails. For `failFast=false`, the operation will continue even if a field fails. It will return a valid response, but without values for the failed field(s).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        url = f"{self.base_url}/rest/api/3/search"
        query_params = {
            k: v
            for k, v in [
                ("jql", jql),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("validateQuery", validateQuery),
                ("fields", fields),
                ("expand", expand),
                ("properties", properties),
                ("fieldsByKeys", fieldsByKeys),
                ("failFast", failFast),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_for_issues_using_jql_post(
        self,
        expand: list[str] | None = None,
        fields: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        jql: str | None = None,
        maxResults: int | None = None,
        properties: list[str] | None = None,
        startAt: int | None = None,
        validateQuery: str | None = None,
    ) -> dict[str, Any]:
        """
        Searches Jira issues using JQL queries and returns paginated results.

        Args:
            expand (array): Use [expand](#expansion) to include additional information about issues in the response. Note that, unlike the majority of instances where `expand` is specified, `expand` is defined as a list of values. The expand options are:

         *  `renderedFields` Returns field values rendered in HTML format.
         *  `names` Returns the display name of each field.
         *  `schema` Returns the schema describing a field type.
         *  `transitions` Returns all possible transitions for the issue.
         *  `operations` Returns all possible operations for the issue.
         *  `editmeta` Returns information about how each field can be edited.
         *  `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent.
         *  `versionedRepresentations` Instead of `fields`, returns `versionedRepresentations` a JSON array containing each version of a field's value, with the highest numbered item representing the most recent version. Example: ['names', 'schema', 'operations'].
            fields (array): A list of fields to return for each issue, use it to retrieve a subset of fields. This parameter accepts a comma-separated list. Expand options include:

         *  `*all` Returns all fields.
         *  `*navigable` Returns navigable fields.
         *  Any issue field, prefixed with a minus to exclude.

        The default is `*navigable`.

        Examples:

         *  `summary,comment` Returns the summary and comments fields only.
         *  `-description` Returns all navigable (default) fields except description.
         *  `*all,-comment` Returns all fields except comments.

        Multiple `fields` parameters can be included in a request.

        Note: All navigable fields are returned by default. This differs from [GET issue](#api-rest-api-3-issue-issueIdOrKey-get) where the default is all fields. Example: ['summary', 'status', 'assignee'].
            fieldsByKeys (boolean): Reference fields by their key (rather than ID). The default is `false`. Example: False.
            jql (string): A [JQL](https://confluence.atlassian.com/x/egORLQ) expression. Example: 'project = HSP'.
            maxResults (integer): The maximum number of items to return per page. Example: 15.
            properties (array): A list of up to 5 issue properties to include in the results. This parameter accepts a comma-separated list.
            startAt (integer): The index of the first item to return in the page of results (page offset). The base index is `0`. Example: 0.
            validateQuery (string): Determines how to validate the JQL query and treat the validation results. Supported values:

         *  `strict` Returns a 400 response code if any errors are found, along with a list of all errors (and warnings).
         *  `warn` Returns all errors as warnings.
         *  `none` No validation is performed.
         *  `true` *Deprecated* A legacy synonym for `strict`.
         *  `false` *Deprecated* A legacy synonym for `warn`.

        The default is `strict`.

        Note: If the JQL is not correctly formed a 400 response code is returned, regardless of the `validateQuery` value.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        request_body_data = None
        request_body_data = {
            "expand": expand,
            "fields": fields,
            "fieldsByKeys": fieldsByKeys,
            "jql": jql,
            "maxResults": maxResults,
            "properties": properties,
            "startAt": startAt,
            "validateQuery": validateQuery,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/search"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def count_issues(self, jql: str | None = None) -> dict[str, Any]:
        """
        Retrieves an approximate count of Jira issues matching a specified JQL query using the POST method at the "/rest/api/3/search/approximate-count" endpoint.

        Args:
            jql (string): A [JQL](https://confluence.atlassian.com/x/egORLQ) expression. For performance reasons, this parameter requires a bounded query. A bounded query is a query with a search restriction. Example: 'project = HSP'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        request_body_data = None
        request_body_data = {
            "jql": jql,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/search/approximate-count"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search_for_issues_ids(
        self,
        jql: str | None = None,
        maxResults: int | None = None,
        nextPageToken: str | None = None,
    ) -> dict[str, Any]:
        """
        Searches for Jira issues using JQL (Jira Query Language) and returns a list of matching issue IDs, along with a token for fetching additional results if needed, using the `POST` method at the path "/rest/api/3/search/id".

        Args:
            jql (string): A [JQL](https://confluence.atlassian.com/x/egORLQ) expression. Order by clauses are not allowed. Example: 'project = HSP'.
            maxResults (integer): The maximum number of items to return per page. Example: 1000.
            nextPageToken (string): The continuation token to fetch the next page. This token is provided by the response of this endpoint. Example: 'EgQIlMIC'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        request_body_data = None
        request_body_data = {
            "jql": jql,
            "maxResults": maxResults,
            "nextPageToken": nextPageToken,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/search/id"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_search_by_jql(
        self,
        jql: str | None = None,
        nextPageToken: str | None = None,
        maxResults: int | None = None,
        fields: list[str] | None = None,
        expand: str | None = None,
        properties: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        failFast: bool | None = None,
        reconcileIssues: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of Jira issues matching a JQL query with pagination support, customizable field selection, and result optimization options.

        Args:
            jql (string): A [JQL]( expression. For performance reasons, this parameter requires a bounded query. A bounded query is a query with a search restriction. * Example of an unbounded query: `order by key desc`. * Example of a bounded query: `assignee = currentUser() order by key`. Additionally, `orderBy` clause can contain a maximum of 7 fields. Example: 'project = HSP'.
            nextPageToken (string): The token for a page to fetch that is not the first page. The first page has a `nextPageToken` of `null`. Use the `nextPageToken` to fetch the next page of issues. Note: The `nextPageToken` field is **not included** in the response for the last page, indicating there is no next page. Example: '<string>'.
            maxResults (integer): The maximum number of items to return per page. To manage page size, API may return fewer items per page where a large number of fields or properties are requested. The greatest number of items returned per page is achieved when requesting `id` or `key` only. It returns max 5000 issues. Example: '114'.
            fields (array): A list of fields to return for each issue, use it to retrieve a subset of fields. This parameter accepts a comma-separated list. Expand options include: * `*all` Returns all fields. * `*navigable` Returns navigable fields. * `id` Returns only issue IDs. * Any issue field, prefixed with a minus to exclude. The default is `id`. Examples: * `summary,comment` Returns only the summary and comments fields only. * `-description` Returns all navigable (default) fields except description. * `*all,-comment` Returns all fields except comments. Multiple `fields` parameters can be included in a request. Note: By default, this resource returns IDs only. This differs from [GET issue](#api-rest-api-3-issue-issueIdOrKey-get) where the default is all fields.
            expand (string): Use [expand](#expansion) to include additional information about issues in the response. Note that, unlike the majority of instances where `expand` is specified, `expand` is defined as a comma-delimited string of values. The expand options are: * `renderedFields` Returns field values rendered in HTML format. * `names` Returns the display name of each field. * `schema` Returns the schema describing a field type. * `transitions` Returns all possible transitions for the issue. * `operations` Returns all possible operations for the issue. * `editmeta` Returns information about how each field can be edited. * `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent. * `versionedRepresentations` Instead of `fields`, returns `versionedRepresentations` a JSON array containing each version of a field's value, with the highest numbered item representing the most recent version. Examples: `"names,changelog"` Returns the display name of each field as well as a list of recent updates to an issue. Example: '<string>'.
            properties (array): A list of up to 5 issue properties to include in the results. This parameter accepts a comma-separated list.
            fieldsByKeys (boolean): Reference fields by their key (rather than ID). The default is `false`.
            failFast (boolean): Fail this request early if we can't retrieve all field data.
            reconcileIssues (array): Strong consistency issue ids to be reconciled with search results. Accepts max 50 ids

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        url = f"{self.base_url}/rest/api/3/search/jql"
        query_params = {
            k: v
            for k, v in [
                ("jql", jql),
                ("nextPageToken", nextPageToken),
                ("maxResults", maxResults),
                ("fields", fields),
                ("expand", expand),
                ("properties", properties),
                ("fieldsByKeys", fieldsByKeys),
                ("failFast", failFast),
                ("reconcileIssues", reconcileIssues),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def post_search_jql(
        self,
        expand: str | None = None,
        fields: list[str] | None = None,
        fieldsByKeys: bool | None = None,
        jql: str | None = None,
        maxResults: int | None = None,
        nextPageToken: str | None = None,
        properties: list[str] | None = None,
        reconcileIssues: list[int] | None = None,
    ) -> dict[str, Any]:
        """
        Executes a JQL query to search for issues, returning matching results and pagination tokens.

        Args:
            expand (string): Use [expand](#expansion) to include additional information about issues in the response. Note that, unlike the majority of instances where `expand` is specified, `expand` is defined as a comma-delimited string of values. The expand options are:

         *  `renderedFields` Returns field values rendered in HTML format.
         *  `names` Returns the display name of each field.
         *  `schema` Returns the schema describing a field type.
         *  `transitions` Returns all possible transitions for the issue.
         *  `operations` Returns all possible operations for the issue.
         *  `editmeta` Returns information about how each field can be edited.
         *  `changelog` Returns a list of recent updates to an issue, sorted by date, starting from the most recent.
         *  `versionedRepresentations` Instead of `fields`, returns `versionedRepresentations` a JSON array containing each version of a field's value, with the highest numbered item representing the most recent version.

        Examples: `"names,changelog"` Returns the display name of each field as well as a list of recent updates to an issue.
            fields (array): A list of fields to return for each issue. Use it to retrieve a subset of fields. This parameter accepts a comma-separated list. Expand options include:

         *  `*all` Returns all fields.
         *  `*navigable` Returns navigable fields.
         *  `id` Returns only issue IDs.
         *  Any issue field, prefixed with a dash to exclude.

        The default is `id`.

        Examples:

         *  `summary,comment` Returns the summary and comments fields only.
         *  `*all,-comment` Returns all fields except comments.

        Multiple `fields` parameters can be included in a request.

        Note: By default, this resource returns IDs only. This differs from [GET issue](#api-rest-api-3-issue-issueIdOrKey-get) where the default is all fields.
            fieldsByKeys (boolean): Reference fields by their key (rather than ID). The default is `false`.
            jql (string): A [JQL](https://confluence.atlassian.com/x/egORLQ) expression. For performance reasons, this parameter requires a bounded query. A bounded query is a query with a search restriction.

         *  Example of an unbounded query: `order by key desc`.
         *  Example of a bounded query: `assignee = currentUser() order by key`.

        Additionally, `orderBy` clause can contain a maximum of 7 fields.
            maxResults (integer): The maximum number of items to return per page. To manage page size, API may return fewer items per page where a large number of fields are requested. The greatest number of items returned per page is achieved when requesting `id` or `key` only. It returns max 5000 issues.
            nextPageToken (string): The token for a page to fetch that is not the first page. The first page has a `nextPageToken` of `null`. Use the `nextPageToken` to fetch the next page of issues.
            properties (array): A list of up to 5 issue properties to include in the results. This parameter accepts a comma-separated list.
            reconcileIssues (array): Strong consistency issue ids to be reconciled with search results. Accepts max 50 ids

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue search
        """
        request_body_data = None
        request_body_data = {
            "expand": expand,
            "fields": fields,
            "fieldsByKeys": fieldsByKeys,
            "jql": jql,
            "maxResults": maxResults,
            "nextPageToken": nextPageToken,
            "properties": properties,
            "reconcileIssues": reconcileIssues,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/search/jql"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_security_level(self, id: str) -> dict[str, Any]:
        """
        Retrieves details of a specific issue security level by its ID in Jira.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue security level
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/securitylevel/{id}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_server_info(self) -> dict[str, Any]:
        """
        Retrieves information about the Jira instance using the "GET" method at the "/rest/api/3/serverInfo" endpoint.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Server info
        """
        url = f"{self.base_url}/rest/api/3/serverInfo"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def list_columns(self) -> list[Any]:
        """
        Retrieves settings for columns using the Jira API and returns relevant data.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue navigator settings
        """
        url = f"{self.base_url}/rest/api/3/settings/columns"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_settings_columns(self, columns: list[str] | None = None) -> Any:
        """
        Updates board column configurations via a PUT request to modify their settings.

        Args:
            columns (array): columns

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Issue navigator settings
        """
        request_body_data = None
        files_data = None
        request_body_data = {}
        files_data = {}
        if columns is not None:
            request_body_data["columns"] = columns
        files_data = {k: v for k, v in files_data.items() if v is not None}
        if not files_data:
            files_data = None
        url = f"{self.base_url}/rest/api/3/settings/columns"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            files=files_data,
            params=query_params,
            content_type="multipart/form-data",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_statuses(self) -> list[Any]:
        """
        Retrieves the operational status and readiness of the Jira instance via a lightweight endpoint for monitoring.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Workflow statuses
        """
        url = f"{self.base_url}/rest/api/3/status"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_status(self, idOrName: str) -> dict[str, Any]:
        """
        Retrieves a specific status by its ID or name from Jira using the Jira REST API.

        Args:
            idOrName (string): idOrName

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Workflow statuses
        """
        if idOrName is None:
            raise ValueError("Missing required parameter 'idOrName'.")
        url = f"{self.base_url}/rest/api/3/status/{idOrName}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_status_categories(self) -> list[Any]:
        """
        Retrieves a list of all visible Jira issue status categories in JSON format.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Workflow status categories
        """
        url = f"{self.base_url}/rest/api/3/statuscategory"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_status_category(self, idOrKey: str) -> dict[str, Any]:
        """
        Retrieves a specific Jira issue status category by its ID or key using the GET method.

        Args:
            idOrKey (string): idOrKey

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Workflow status categories
        """
        if idOrKey is None:
            raise ValueError("Missing required parameter 'idOrKey'.")
        url = f"{self.base_url}/rest/api/3/statuscategory/{idOrKey}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_statuses_by_id(self, id: list[str]) -> Any:
        """
        Deletes a specific status entry identified by its ID using the provided parameters and returns a success or error code.

        Args:
            id (array): The list of status IDs. To include multiple IDs, provide an ampersand-separated list. For example, id=10000&id=10001. Min items `1`, Max items `50`

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        url = f"{self.base_url}/rest/api/3/statuses"
        query_params = {k: v for k, v in [("id", id)] if v is not None}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_statuses_by_id(self, id: list[str], expand: str | None = None) -> list[Any]:
        """
        Retrieves a list of statuses in Jira using the "/rest/api/3/statuses" endpoint, allowing you to fetch details of statuses based on query parameters like expansion and ID, though specific details about what statuses are returned are not provided.

        Args:
            id (array): The list of status IDs. To include multiple IDs, provide an ampersand-separated list. For example, id=10000&id=10001. Min items `1`, Max items `50`
            expand (string): Deprecated. See the [deprecation notice]( for details. Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `usages` Returns the project and issue types that use the status in their workflow. * `workflowUsages` Returns the workflows that use the status.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        url = f"{self.base_url}/rest/api/3/statuses"
        query_params = {
            k: v for k, v in [("expand", expand), ("id", id)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_statuses(
        self, scope: dict[str, Any], statuses: list[dict[str, Any]]
    ) -> list[Any]:
        """
        Creates commit statuses (error, failure, pending, success) with optional descriptions and target URLs via the GitHub API.

        Args:
            scope (object): The scope of the status. Example: {'project': {'id': '1'}, 'type': 'PROJECT'}.
            statuses (array): Details of the statuses being created. Example: [{'description': 'The issue is resolved', 'name': 'Finished', 'statusCategory': 'DONE'}].

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        request_body_data = None
        request_body_data = {
            "scope": scope,
            "statuses": statuses,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/statuses"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_statuses(self, statuses: list[dict[str, Any]]) -> Any:
        """
        Updates the statuses in Jira using the PUT method at the "/rest/api/3/statuses" endpoint and returns a status message.

        Args:
            statuses (array): The list of statuses that will be updated. Example: [{'description': 'The issue is resolved', 'id': '1000', 'name': 'Finished', 'statusCategory': 'DONE'}].

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        request_body_data = None
        request_body_data = {
            "statuses": statuses,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/statuses"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def search(
        self,
        expand: str | None = None,
        projectId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        searchString: str | None = None,
        statusCategory: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of Jira statuses with optional filtering by project, search string, or status category.

        Args:
            expand (string): Deprecated. See the [deprecation notice]( for details. Use [expand](#expansion) to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `usages` Returns the project and issue types that use the status in their workflow. * `workflowUsages` Returns the workflows that use the status.
            projectId (string): The project the status is part of or null for global statuses.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            searchString (string): Term to match status names against or null to search for all statuses in the search scope.
            statusCategory (string): Category of the status to filter by. The supported values are: `TODO`, `IN_PROGRESS`, and `DONE`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        url = f"{self.base_url}/rest/api/3/statuses/search"
        query_params = {
            k: v
            for k, v in [
                ("expand", expand),
                ("projectId", projectId),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("searchString", searchString),
                ("statusCategory", statusCategory),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_issue_type_usages(
        self,
        statusId: str,
        projectId: str,
        nextPageToken: str | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of issue types associated with a specific project and status, including pagination controls via `nextPageToken` and `maxResults`.

        Args:
            statusId (string): statusId
            projectId (string): projectId
            nextPageToken (string): The cursor for pagination
            maxResults (integer): The maximum number of results to return. Must be an integer between 1 and 200.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        if statusId is None:
            raise ValueError("Missing required parameter 'statusId'.")
        if projectId is None:
            raise ValueError("Missing required parameter 'projectId'.")
        url = f"{self.base_url}/rest/api/3/statuses/{statusId}/project/{projectId}/issueTypeUsages"
        query_params = {
            k: v
            for k, v in [("nextPageToken", nextPageToken), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_project_usages_for_status(
        self,
        statusId: str,
        nextPageToken: str | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves project usage information for a specific status identified by the status ID, supporting pagination through optional parameters for the next page token and maximum results.

        Args:
            statusId (string): statusId
            nextPageToken (string): The cursor for pagination
            maxResults (integer): The maximum number of results to return. Must be an integer between 1 and 200.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        if statusId is None:
            raise ValueError("Missing required parameter 'statusId'.")
        url = f"{self.base_url}/rest/api/3/statuses/{statusId}/projectUsages"
        query_params = {
            k: v
            for k, v in [("nextPageToken", nextPageToken), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_workflow_usages_for_status(
        self,
        statusId: str,
        nextPageToken: str | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the workflows associated with a specific status ID and returns their usage details.

        Args:
            statusId (string): statusId
            nextPageToken (string): The cursor for pagination
            maxResults (integer): The maximum number of results to return. Must be an integer between 1 and 200.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Status
        """
        if statusId is None:
            raise ValueError("Missing required parameter 'statusId'.")
        url = f"{self.base_url}/rest/api/3/statuses/{statusId}/workflowUsages"
        query_params = {
            k: v
            for k, v in [("nextPageToken", nextPageToken), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_task(self, taskId: str) -> dict[str, Any]:
        """
        Retrieves details for a specific task by ID using a REST API GET request.

        Args:
            taskId (string): taskId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Tasks
        """
        if taskId is None:
            raise ValueError("Missing required parameter 'taskId'.")
        url = f"{self.base_url}/rest/api/3/task/{taskId}"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def cancel_task(self, taskId: str) -> Any:
        """
        Cancels a specific task by its ID using the POST method at the "/rest/api/3/task/{taskId}/cancel" path.

        Args:
            taskId (string): taskId

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Tasks
        """
        if taskId is None:
            raise ValueError("Missing required parameter 'taskId'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/task/{taskId}/cancel"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_ui_modifications(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of UI modifications (including project, issue type, and view contexts) from Jira's REST API.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            expand (string): Use expand to include additional information in the response. This parameter accepts a comma-separated list. Expand options include: * `data` Returns UI modification data. * `contexts` Returns UI modification contexts.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            UI modifications (apps)
        """
        url = f"{self.base_url}/rest/api/3/uiModifications"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_ui_modification(
        self,
        name: str,
        contexts: list[dict[str, Any]] | None = None,
        data: str | None = None,
        description: str | None = None,
    ) -> dict[str, Any]:
        """
        Applies modifications to the user interface using the REST API at the "/rest/api/3/uiModifications" endpoint, returning status codes to indicate success or failure.

        Args:
            name (string): The name of the UI modification. The maximum length is 255 characters. Example: 'Reveal Story Points'.
            contexts (array): List of contexts of the UI modification. The maximum number of contexts is 1000. Example: [{'issueTypeId': '10000', 'projectId': '10000', 'viewType': 'GIC'}, {'issueTypeId': '10001', 'projectId': '10000', 'viewType': 'IssueView'}, {'issueTypeId': '10002', 'projectId': '10000', 'viewType': 'IssueTransition'}, {'issueTypeId': '10003', 'projectId': '10000', 'viewType': None}].
            data (string): The data of the UI modification. The maximum size of the data is 50000 characters. Example: "{field: 'Story Points', config: {hidden: false}}".
            description (string): The description of the UI modification. The maximum length is 255 characters. Example: 'Reveals Story Points field when any Sprint is selected.'.

        Returns:
            dict[str, Any]: Returned if the UI modification is created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            UI modifications (apps)
        """
        request_body_data = None
        request_body_data = {
            "contexts": contexts,
            "data": data,
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/uiModifications"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_ui_modification(self, uiModificationId: str) -> Any:
        """
        Deletes a UI modification with the specified ID from the system using the DELETE HTTP method.

        Args:
            uiModificationId (string): uiModificationId

        Returns:
            Any: Returned if the UI modification is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            UI modifications (apps)
        """
        if uiModificationId is None:
            raise ValueError("Missing required parameter 'uiModificationId'.")
        url = f"{self.base_url}/rest/api/3/uiModifications/{uiModificationId}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_ui_modification(
        self,
        uiModificationId: str,
        contexts: list[dict[str, Any]] | None = None,
        data: str | None = None,
        description: str | None = None,
        name: str | None = None,
    ) -> Any:
        """
        Updates a UI modification identified by `uiModificationId` using the PUT method.

        Args:
            uiModificationId (string): uiModificationId
            contexts (array): List of contexts of the UI modification. The maximum number of contexts is 1000. If provided, replaces all existing contexts. Example: [{'issueTypeId': '10000', 'projectId': '10000', 'viewType': 'GIC'}, {'issueTypeId': '10001', 'projectId': '10000', 'viewType': 'IssueView'}, {'issueTypeId': '10002', 'projectId': '10000', 'viewType': 'IssueTransition'}].
            data (string): The data of the UI modification. The maximum size of the data is 50000 characters. Example: "{field: 'Story Points', config: {hidden: true}}".
            description (string): The description of the UI modification. The maximum length is 255 characters.
            name (string): The name of the UI modification. The maximum length is 255 characters. Example: 'Updated Reveal Story Points'.

        Returns:
            Any: Returned if the UI modification is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            UI modifications (apps)
        """
        if uiModificationId is None:
            raise ValueError("Missing required parameter 'uiModificationId'.")
        request_body_data = None
        request_body_data = {
            "contexts": contexts,
            "data": data,
            "description": description,
            "name": name,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/uiModifications/{uiModificationId}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_avatars(self, type: str, entityId: str) -> dict[str, Any]:
        """
        Retrieves details about a universal avatar by its type and owner entity ID using the Jira API.

        Args:
            type (string): type
            entityId (string): entityId

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        if entityId is None:
            raise ValueError("Missing required parameter 'entityId'.")
        url = (
            f"{self.base_url}/rest/api/3/universal_avatar/type/{type}/owner/{entityId}"
        )
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def store_avatar(
        self,
        type: str,
        entityId: str,
        size: int,
        body_content: bytes,
        x: int | None = None,
        y: int | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new avatar for the specified entity type (e.g., project, issue) using provided parameters (x, y, size) and returns a success status.

        Args:
            type (string): type
            entityId (string): entityId
            size (integer): The length of each side of the crop region.
            body_content (bytes | None): Raw binary content for the request body.
            x (integer): The X coordinate of the top-left corner of the crop region.
            y (integer): The Y coordinate of the top-left corner of the crop region.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        if entityId is None:
            raise ValueError("Missing required parameter 'entityId'.")
        request_body_data = None
        request_body_data = body_content
        url = (
            f"{self.base_url}/rest/api/3/universal_avatar/type/{type}/owner/{entityId}"
        )
        query_params = {
            k: v for k, v in [("x", x), ("y", y), ("size", size)] if v is not None
        }
        response = self._post(
            url, data=request_body_data, params=query_params, content_type="*/*"
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_avatar(self, type: str, owningObjectId: str, id: str) -> Any:
        """
        Deletes a specified avatar associated with a resource type and owner using the Jira API.

        Args:
            type (string): type
            owningObjectId (string): owningObjectId
            id (string): id

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        if owningObjectId is None:
            raise ValueError("Missing required parameter 'owningObjectId'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/universal_avatar/type/{type}/owner/{owningObjectId}/avatar/{id}"
        query_params = {}
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_avatar_image_by_type(
        self, type: str, size: str | None = None, format: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves a Jira avatar image by type using the "GET" method, allowing specification of size and format for customization.

        Args:
            type (string): type
            size (string): The size of the avatar image. If not provided the default size is returned.
            format (string): The format to return the avatar image in. If not provided the original content format is returned.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        url = f"{self.base_url}/rest/api/3/universal_avatar/view/type/{type}"
        query_params = {
            k: v for k, v in [("size", size), ("format", format)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_avatar_image_by_id(
        self,
        type: str,
        id: str,
        size: str | None = None,
        format: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a specific avatar by type and ID using the Jira Universal Avatar API, allowing customization and display in various formats and sizes.

        Args:
            type (string): type
            id (string): id
            size (string): The size of the avatar image. If not provided the default size is returned.
            format (string): The format to return the avatar image in. If not provided the original content format is returned.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = (
            f"{self.base_url}/rest/api/3/universal_avatar/view/type/{type}/avatar/{id}"
        )
        query_params = {
            k: v for k, v in [("size", size), ("format", format)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_avatar_image_by_owner(
        self,
        type: str,
        entityId: str,
        size: str | None = None,
        format: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves an avatar image for a specified owner entity (like user, project, or issue type) by type and ID, allowing optional size and format customization.

        Args:
            type (string): type
            entityId (string): entityId
            size (string): The size of the avatar image. If not provided the default size is returned.
            format (string): The format to return the avatar image in. If not provided the original content format is returned.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Avatars
        """
        if type is None:
            raise ValueError("Missing required parameter 'type'.")
        if entityId is None:
            raise ValueError("Missing required parameter 'entityId'.")
        url = f"{self.base_url}/rest/api/3/universal_avatar/view/type/{type}/owner/{entityId}"
        query_params = {
            k: v for k, v in [("size", size), ("format", format)] if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def remove_user(
        self, accountId: str, username: str | None = None, key: str | None = None
    ) -> Any:
        """
        Deletes a user from the system using the provided query parameters such as account ID, username, or key, and returns a status code indicating success or failure.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            key (string): This parameter is no longer available. See the [deprecation notice]( for details.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user"
        query_params = {
            k: v
            for k, v in [("accountId", accountId), ("username", username), ("key", key)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user(
        self,
        accountId: str | None = None,
        username: str | None = None,
        key: str | None = None,
        expand: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a specific Jira user's details using the provided account ID, username, or user key via the Jira REST API.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Required. Example: '5b10ac8d82e05b22cc7d4ef5'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            key (string): This parameter is no longer available. See the [deprecation notice]( for details.
            expand (string): Use [expand](#expansion) to include additional information about users in the response. This parameter accepts a comma-separated list. Expand options include: * `groups` includes all groups and nested groups to which the user belongs. * `applicationRoles` includes details of all the applications to which the user has access.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("username", username),
                ("key", key),
                ("expand", expand),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_user(
        self,
        emailAddress: str,
        products: list[str],
        applicationKeys: list[str] | None = None,
        displayName: str | None = None,
        key: str | None = None,
        name: str | None = None,
        password: str | None = None,
        self_arg_body: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new user in Jira and returns the created user resource upon success.

        Args:
            emailAddress (string): The email address for the user. Example: 'mia@atlassian.com'.
            products (array): Products the new user has access to. Valid products are: jira-core, jira-servicedesk, jira-product-discovery, jira-software. To create a user without product access, set this field to be an empty array.
            applicationKeys (array): Deprecated, do not use.
            displayName (string): This property is no longer available. If the user has an Atlassian account, their display name is not changed. If the user does not have an Atlassian account, they are sent an email asking them set up an account.
            key (string): This property is no longer available. See the [migration guide](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            name (string): This property is no longer available. See the [migration guide](https://developer.atlassian.com/cloud/jira/platform/deprecation-notice-user-privacy-api-migration-guide/) for details.
            password (string): This property is no longer available. If the user has an Atlassian account, their password is not changed. If the user does not have an Atlassian account, they are sent an email asking them set up an account.
            self_arg_body (string): The URL of the user.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        request_body_data = None
        request_body_data = {
            "applicationKeys": applicationKeys,
            "displayName": displayName,
            "emailAddress": emailAddress,
            "key": key,
            "name": name,
            "password": password,
            "products": products,
            "self": self_arg_body,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/user"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_bulk_assignable_users(
        self,
        projectKeys: str,
        query: str | None = None,
        username: str | None = None,
        accountId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> list[Any]:
        """
        Retrieves a list of users who can be assigned issues in one or more specified projects, allowing filtering by various user attributes such as name or account ID.

        Args:
            projectKeys (string): A list of project keys (case sensitive). This parameter accepts a comma-separated list.
            query (string): A query string that is matched against user attributes, such as `displayName` and `emailAddress`, to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*. Required, unless `accountId` is specified. Example: 'query'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            accountId (string): A query string that is matched exactly against user `accountId`. Required, unless `query` is specified.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/assignable/multiProjectSearch"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("username", username),
                ("accountId", accountId),
                ("projectKeys", projectKeys),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_assignable_users(
        self,
        query: str | None = None,
        sessionId: str | None = None,
        username: str | None = None,
        accountId: str | None = None,
        project: str | None = None,
        issueKey: str | None = None,
        issueId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        actionDescriptorId: int | None = None,
        recommend: bool | None = None,
    ) -> list[Any]:
        """
        Searches for users who can be assigned to issues in Jira, allowing filtering by query, session ID, username, account ID, project, issue key, issue ID, and other parameters, returning a list of assignable users.

        Args:
            query (string): A query string that is matched against user attributes, such as `displayName`, and `emailAddress`, to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*. Required, unless `username` or `accountId` is specified. Example: 'query'.
            sessionId (string): The sessionId of this request. SessionId is the same until the assignee is set.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            accountId (string): A query string that is matched exactly against user `accountId`. Required, unless `query` is specified.
            project (string): The project ID or project key (case sensitive). Required, unless `issueKey` or `issueId` is specified.
            issueKey (string): The key of the issue. Required, unless `issueId` or `project` is specified.
            issueId (string): The ID of the issue. Required, unless `issueKey` or `project` is specified.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return. This operation may return less than the maximum number of items even if more are available. The operation fetches users up to the maximum and then, from the fetched users, returns only the users that can be assigned to the issue.
            actionDescriptorId (integer): The ID of the transition.
            recommend (boolean): The `recommend` parameter is used to influence the recommendation of users when searching for assignable users, potentially providing suggestions based on user activity or other relevant factors.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/assignable/search"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("sessionId", sessionId),
                ("username", username),
                ("accountId", accountId),
                ("project", project),
                ("issueKey", issueKey),
                ("issueId", issueId),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("actionDescriptorId", actionDescriptorId),
                ("recommend", recommend),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_get_users(
        self,
        accountId: list[str],
        startAt: int | None = None,
        maxResults: int | None = None,
        username: list[str] | None = None,
        key: list[str] | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a paginated list of user details for specified account IDs using the Jira REST API.

        Args:
            accountId (array): The account ID of a user. To specify multiple users, pass multiple `accountId` parameters. For example, `accountId=5b10a2844c20165700ede21g&accountId=5b10ac8d82e05b22cc7d4ef5`. Example: '5b10ac8d82e05b22cc7d4ef5'.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            username (array): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.
            key (array): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/bulk"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("username", username),
                ("key", key),
                ("accountId", accountId),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def bulk_get_users_migration(
        self,
        startAt: int | None = None,
        maxResults: int | None = None,
        username: list[str] | None = None,
        key: list[str] | None = None,
    ) -> list[Any]:
        """
        Retrieves user migration information in bulk for Jira using the GET method, allowing filtering by username, key, and pagination via startAt and maxResults parameters.

        Args:
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            username (array): Username of a user. To specify multiple users, pass multiple copies of this parameter. For example, `username=fred&username=barney`. Required if `key` isn't provided. Cannot be provided if `key` is present.
            key (array): Key of a user. To specify multiple users, pass multiple copies of this parameter. For example, `key=fred&key=barney`. Required if `username` isn't provided. Cannot be provided if `username` is present.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/bulk/migration"
        query_params = {
            k: v
            for k, v in [
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("username", username),
                ("key", key),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def reset_user_columns(
        self, accountId: str | None = None, username: str | None = None
    ) -> Any:
        """
        Deletes a user's saved column configuration in Jira based on either their account ID or username.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/columns"
        query_params = {
            k: v
            for k, v in [("accountId", accountId), ("username", username)]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_default_columns(
        self, accountId: str | None = None, username: str | None = None
    ) -> list[Any]:
        """
        Retrieves the default issue table columns for a Jira user, specified by either an accountId or the calling user if no accountId is provided, using the Jira Cloud Platform REST API.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            username (string): This parameter is no longer available See the [deprecation notice]( for details.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/columns"
        query_params = {
            k: v
            for k, v in [("accountId", accountId), ("username", username)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_user_columns(
        self, accountId: str | None = None, columns: list[str] | None = None
    ) -> Any:
        """
        Updates the columns displayed for a specific user's issue list view in Jira and returns a success status upon completion.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            columns (array): columns

        Returns:
            Any: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        request_body_data = None
        files_data = None
        request_body_data = {}
        files_data = {}
        if columns is not None:
            request_body_data["columns"] = columns
        files_data = {k: v for k, v in files_data.items() if v is not None}
        if not files_data:
            files_data = None
        url = f"{self.base_url}/rest/api/3/user/columns"
        query_params = {k: v for k, v in [("accountId", accountId)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            files=files_data,
            params=query_params,
            content_type="multipart/form-data",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_email(self, accountId: str) -> dict[str, Any]:
        """
        Retrieves a user's email address for the specified Atlassian account ID using the Jira Cloud API.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, `5b10ac8d82e05b22cc7d4ef5`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/email"
        query_params = {k: v for k, v in [("accountId", accountId)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_email_bulk(self, accountId: list[str]) -> dict[str, Any]:
        """
        Retrieves email addresses for multiple Jira users by their account IDs in a single request, bypassing profile visibility restrictions.

        Args:
            accountId (array): The account IDs of the users for which emails are required. An `accountId` is an identifier that uniquely identifies the user across all Atlassian products. For example, `5b10ac8d82e05b22cc7d4ef5`. Note, this should be treated as an opaque identifier (that is, do not assume any structure in the value).

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/email/bulk"
        query_params = {k: v for k, v in [("accountId", accountId)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_groups(
        self, accountId: str, username: str | None = None, key: str | None = None
    ) -> list[Any]:
        """
        Retrieves a list of groups associated with a specified Jira user account using their accountId, username, or key.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            key (string): This parameter is no longer available. See the [deprecation notice]( for details.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/user/groups"
        query_params = {
            k: v
            for k, v in [("accountId", accountId), ("username", username), ("key", key)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_nav_property(
        self, propertyKey: str, accountId: str | None = None
    ) -> dict[str, Any]:
        """
        Retrieves the value associated with a specified property key for a given account using the GET method.

        Args:
            propertyKey (string): propertyKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            usernavproperties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/user/nav4-opt-property/{propertyKey}"
        query_params = {k: v for k, v in [("accountId", accountId)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_user_nav_property(
        self, propertyKey: str, accountId: str | None = None
    ) -> Any:
        """
        Updates a user's navigation property (specified by propertyKey) in Jira using account-based identification and returns the operation status.

        Args:
            propertyKey (string): propertyKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.

        Returns:
            Any: Returned if the user property is updated/created.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            usernavproperties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/user/nav4-opt-property/{propertyKey}"
        query_params = {k: v for k, v in [("accountId", accountId)] if v is not None}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users_with_all_permissions(
        self,
        permissions: str,
        query: str | None = None,
        username: str | None = None,
        accountId: str | None = None,
        issueKey: str | None = None,
        projectKey: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> list[Any]:
        r"""
        Retrieves users with specified global or project permissions, filtered by query, account ID, or project/issue context, including pagination support.

        Args:
            permissions (string): A comma separated list of permissions. Permissions can be specified as any: * permission returned by [Get all permissions](#api-rest-api-3-permissions-get). * custom project permission added by Connect apps. * (deprecated) one of the following: * ASSIGNABLE\_USER * ASSIGN\_ISSUE * ATTACHMENT\_DELETE\_ALL * ATTACHMENT\_DELETE\_OWN * BROWSE * CLOSE\_ISSUE * COMMENT\_DELETE\_ALL * COMMENT\_DELETE\_OWN * COMMENT\_EDIT\_ALL * COMMENT\_EDIT\_OWN * COMMENT\_ISSUE * CREATE\_ATTACHMENT * CREATE\_ISSUE * DELETE\_ISSUE * EDIT\_ISSUE * LINK\_ISSUE * MANAGE\_WATCHER\_LIST * MODIFY\_REPORTER * MOVE\_ISSUE * PROJECT\_ADMIN * RESOLVE\_ISSUE * SCHEDULE\_ISSUE * SET\_ISSUE\_SECURITY * TRANSITION\_ISSUE * VIEW\_VERSION\_CONTROL * VIEW\_VOTERS\_AND\_WATCHERS * VIEW\_WORKFLOW\_READONLY * WORKLOG\_DELETE\_ALL * WORKLOG\_DELETE\_OWN * WORKLOG\_EDIT\_ALL * WORKLOG\_EDIT\_OWN * WORK\_ISSUE
            query (string): A query string that is matched against user attributes, such as `displayName` and `emailAddress`, to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*. Required, unless `accountId` is specified. Example: 'query'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            accountId (string): A query string that is matched exactly against user `accountId`. Required, unless `query` is specified.
            issueKey (string): The issue key for the issue.
            projectKey (string): The project key for the project (case sensitive).
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/permission/search"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("username", username),
                ("accountId", accountId),
                ("permissions", permissions),
                ("issueKey", issueKey),
                ("projectKey", projectKey),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users_for_picker(
        self,
        query: str,
        maxResults: int | None = None,
        showAvatar: bool | None = None,
        exclude: list[str] | None = None,
        excludeAccountIds: list[str] | None = None,
        avatarSize: str | None = None,
        excludeConnectUsers: bool | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves a list of users and groups for a picker field, allowing filtering by query, exclusion parameters, and pagination, to populate user or group suggestion lists in Jira applications.

        Args:
            query (string): A query string that is matched against user attributes, such as `displayName`, and `emailAddress`, to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*.
            maxResults (integer): The maximum number of items to return. The total number of matched users is returned in `total`.
            showAvatar (boolean): Include the URI to the user's avatar.
            exclude (array): This parameter is no longer available. See the [deprecation notice]( for details.
            excludeAccountIds (array): A list of account IDs to exclude from the search results. This parameter accepts a comma-separated list. Multiple account IDs can also be provided using an ampersand-separated list. For example, `excludeAccountIds=5b10a2844c20165700ede21g,5b10a0effa615349cb016cd8&excludeAccountIds=5b10ac8d82e05b22cc7d4ef5`. Cannot be provided with `exclude`.
            avatarSize (string): Specifies the size of the avatar to be returned in the response, typically in pixels (e.g., 16x16, 24x24, 32x32, or 48x48).
            excludeConnectUsers (boolean): A boolean parameter to exclude JSM customer accounts from the search results when set to true.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/picker"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("maxResults", maxResults),
                ("showAvatar", showAvatar),
                ("exclude", exclude),
                ("excludeAccountIds", excludeAccountIds),
                ("avatarSize", avatarSize),
                ("excludeConnectUsers", excludeConnectUsers),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_property_keys(
        self,
        accountId: str | None = None,
        userKey: str | None = None,
        username: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the keys of all properties for a user using the Jira Cloud REST API.

        Args:
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            userKey (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.
            username (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User properties
        """
        url = f"{self.base_url}/rest/api/3/user/properties"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("userKey", userKey),
                ("username", username),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_user_property(
        self,
        propertyKey: str,
        accountId: str | None = None,
        userKey: str | None = None,
        username: str | None = None,
    ) -> Any:
        """
        Deletes a user property identified by a specific property key using the Jira Cloud platform REST API, requiring permissions to manage user properties.

        Args:
            propertyKey (string): propertyKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            userKey (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.
            username (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.

        Returns:
            Any: Returned if the user property is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User properties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/user/properties/{propertyKey}"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("userKey", userKey),
                ("username", username),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_user_property(
        self,
        propertyKey: str,
        accountId: str | None = None,
        userKey: str | None = None,
        username: str | None = None,
    ) -> dict[str, Any]:
        """
        Retrieves the value of a specified user property using the Jira Cloud API, returning the custom data associated with a user for a given property key.

        Args:
            propertyKey (string): propertyKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            userKey (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.
            username (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User properties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        url = f"{self.base_url}/rest/api/3/user/properties/{propertyKey}"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("userKey", userKey),
                ("username", username),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def set_user_property(
        self,
        propertyKey: str,
        accountId: str | None = None,
        userKey: str | None = None,
        username: str | None = None,
    ) -> Any:
        """
        Sets or updates a custom property value for a specific Jira user, enabling per-user data storage for integrations and apps.

        Args:
            propertyKey (string): propertyKey
            accountId (string): The account ID of the user, which uniquely identifies the user across all Atlassian products. For example, *5b10ac8d82e05b22cc7d4ef5*. Example: '5b10ac8d82e05b22cc7d4ef5'.
            userKey (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.
            username (string): This parameter is no longer available and will be removed from the documentation soon. See the [deprecation notice]( for details.

        Returns:
            Any: Returned if the user property is updated.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User properties
        """
        if propertyKey is None:
            raise ValueError("Missing required parameter 'propertyKey'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/user/properties/{propertyKey}"
        query_params = {
            k: v
            for k, v in [
                ("accountId", accountId),
                ("userKey", userKey),
                ("username", username),
            ]
            if v is not None
        }
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users(
        self,
        query: str | None = None,
        username: str | None = None,
        accountId: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
        property: str | None = None,
    ) -> list[Any]:
        """
        Searches for Jira users by matching a query against display names and email addresses, supporting pagination and specific property filters.

        Args:
            query (string): A query string that is matched against user attributes ( `displayName`, and `emailAddress`) to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*. Required, unless `accountId` or `property` is specified. Example: 'query'.
            username (string): The "username" parameter is not explicitly documented in the `/rest/api/3/user/search` endpoint. Instead, the "query" parameter is used, which matches against user attributes like `displayName` and `emailAddress`. However, if "username" were to be considered, it would presumably involve searching for users based on their username or similar attributes, though this is not the standard behavior of the current API.
            accountId (string): A query string that is matched exactly against a user `accountId`. Required, unless `query` or `property` is specified.
            startAt (integer): The index of the first item to return in a page of filtered results (page offset).
            maxResults (integer): The maximum number of items to return per page.
            property (string): A query string used to search properties. Property keys are specified by path, so property keys containing dot (.) or equals (=) characters cannot be used. The query string cannot be specified using a JSON object. Example: To search for the value of `nested` from `{"something":{"nested":1,"other":2}}` use `thepropertykey.something.nested=1`. Required, unless `accountId` or `query` is specified.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/search"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("username", username),
                ("accountId", accountId),
                ("startAt", startAt),
                ("maxResults", maxResults),
                ("property", property),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users_by_query(
        self,
        query: str,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> dict[str, Any]:
        """
        Searches for users in Jira based on query parameters, returning paginated results.

        Args:
            query (string): The search query.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/search/query"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_user_keys_by_query(
        self, query: str, startAt: int | None = None, maxResult: int | None = None
    ) -> dict[str, Any]:
        """
        Searches for users based on a specified query, returning a list of matching users, with options to control the result set size and starting point.

        Args:
            query (string): The search query.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResult (integer): The maximum number of items to return per page.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/search/query/key"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("startAt", startAt),
                ("maxResult", maxResult),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def find_users_with_browse_permission(
        self,
        query: str | None = None,
        username: str | None = None,
        accountId: str | None = None,
        issueKey: str | None = None,
        projectKey: str | None = None,
        startAt: int | None = None,
        maxResults: int | None = None,
    ) -> list[Any]:
        """
        Searches for Jira issues based on specified parameters such as query, username, account ID, issue key, project key, and returns a list of matching issues with pagination options.

        Args:
            query (string): A query string that is matched against user attributes, such as `displayName` and `emailAddress`, to find relevant users. The string can match the prefix of the attribute's value. For example, *query=john* matches a user with a `displayName` of *John Smith* and a user with an `emailAddress` of *johnson@example.com*. Required, unless `accountId` is specified. Example: 'query'.
            username (string): This parameter is no longer available. See the [deprecation notice]( for details.
            accountId (string): A query string that is matched exactly against user `accountId`. Required, unless `query` is specified.
            issueKey (string): The issue key for the issue. Required, unless `projectKey` is specified.
            projectKey (string): The project key for the project (case sensitive). Required, unless `issueKey` is specified.
            startAt (integer): The index of the first item to return in a page of results (page offset).
            maxResults (integer): The maximum number of items to return per page.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            User search
        """
        url = f"{self.base_url}/rest/api/3/user/viewissue/search"
        query_params = {
            k: v
            for k, v in [
                ("query", query),
                ("username", username),
                ("accountId", accountId),
                ("issueKey", issueKey),
                ("projectKey", projectKey),
                ("startAt", startAt),
                ("maxResults", maxResults),
            ]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_users_default(
        self, startAt: int | None = None, maxResults: int | None = None
    ) -> list[Any]:
        """
        Retrieves a list of Jira users, supporting pagination via the `startAt` and `maxResults` parameters, using the GET method at the `/rest/api/3/users` endpoint.

        Args:
            startAt (integer): The index of the first item to return.
            maxResults (integer): The maximum number of items to return.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/users"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_all_users(
        self, startAt: int | None = None, maxResults: int | None = None
    ) -> list[Any]:
        """
        Searches for Jira users matching query criteria and returns paginated results.

        Args:
            startAt (integer): The index of the first item to return.
            maxResults (integer): The maximum number of items to return.

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Users
        """
        url = f"{self.base_url}/rest/api/3/users/search"
        query_params = {
            k: v
            for k, v in [("startAt", startAt), ("maxResults", maxResults)]
            if v is not None
        }
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_version(
        self,
        approvers: list[dict[str, Any]] | None = None,
        archived: bool | None = None,
        description: str | None = None,
        driver: str | None = None,
        expand: str | None = None,
        id: str | None = None,
        issuesStatusForFixVersion: Any | None = None,
        moveUnfixedIssuesTo: str | None = None,
        name: str | None = None,
        operations: list[dict[str, Any]] | None = None,
        overdue: bool | None = None,
        project: str | None = None,
        projectId: int | None = None,
        releaseDate: str | None = None,
        released: bool | None = None,
        self_arg_body: str | None = None,
        startDate: str | None = None,
        userReleaseDate: str | None = None,
        userStartDate: str | None = None,
    ) -> dict[str, Any]:
        """
        Creates a new project version in Jira and returns the details of the created version.

        Args:
            approvers (array): If the expand option `approvers` is used, returns a list containing the approvers for this version.
            archived (boolean): Indicates that the version is archived. Optional when creating or updating a version. Example: False.
            description (string): The description of the version. Optional when creating or updating a version. The maximum size is 16,384 bytes. Example: 'An excellent version'.
            driver (string): If the expand option `driver` is used, returns the Atlassian account ID of the driver.
            expand (string): Use [expand](em>#expansion) to include additional information about version in the response. This parameter accepts a comma-separated list. Expand options include:

         *  `operations` Returns the list of operations available for this version.
         *  `issuesstatus` Returns the count of issues in this version for each of the status categories *to do*, *in progress*, *done*, and *unmapped*. The *unmapped* property contains a count of issues with a status other than *to do*, *in progress*, and *done*.
         *  `driver` Returns the Atlassian account ID of the version driver.
         *  `approvers` Returns a list containing approvers for this version.

        Optional for create and update.
            id (string): The ID of the version.
            issuesStatusForFixVersion (string): If the expand option `issuesstatus` is used, returns the count of issues in this version for each of the status categories *to do*, *in progress*, *done*, and *unmapped*. The *unmapped* property contains a count of issues with a status other than *to do*, *in progress*, and *done*.
            moveUnfixedIssuesTo (string): The URL of the self link to the version to which all unfixed issues are moved when a version is released. Not applicable when creating a version. Optional when updating a version.
            name (string): The unique name of the version. Required when creating a version. Optional when updating a version. The maximum length is 255 characters. Example: 'New Version 1'.
            operations (array): If the expand option `operations` is used, returns the list of operations available for this version.
            overdue (boolean): Indicates that the version is overdue.
            project (string): Deprecated. Use `projectId`.
            projectId (integer): The ID of the project to which this version is attached. Required when creating a version. Not applicable when updating a version. Example: 10000.
            releaseDate (string): The release date of the version. Expressed in ISO 8601 format (yyyy-mm-dd). Optional when creating or updating a version. Example: '2010-07-06'.
            released (boolean): Indicates that the version is released. If the version is released a request to release again is ignored. Not applicable when creating a version. Optional when updating a version. Example: True.
            self_arg_body (string): The URL of the version.
            startDate (string): The start date of the version. Expressed in ISO 8601 format (yyyy-mm-dd). Optional when creating or updating a version.
            userReleaseDate (string): The date on which work on this version is expected to finish, expressed in the instance's *Day/Month/Year Format* date format.
            userStartDate (string): The date on which work on this version is expected to start, expressed in the instance's *Day/Month/Year Format* date format.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        request_body_data = None
        request_body_data = {
            "approvers": approvers,
            "archived": archived,
            "description": description,
            "driver": driver,
            "expand": expand,
            "id": id,
            "issuesStatusForFixVersion": issuesStatusForFixVersion,
            "moveUnfixedIssuesTo": moveUnfixedIssuesTo,
            "name": name,
            "operations": operations,
            "overdue": overdue,
            "project": project,
            "projectId": projectId,
            "releaseDate": releaseDate,
            "released": released,
            "self": self_arg_body,
            "startDate": startDate,
            "userReleaseDate": userReleaseDate,
            "userStartDate": userStartDate,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/version"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def delete_version(
        self,
        id: str,
        moveFixIssuesTo: str | None = None,
        moveAffectedIssuesTo: str | None = None,
    ) -> Any:
        """
        Deletes a Jira project version using the DELETE method, optionally allowing issues to be moved to alternative versions by specifying replacement versions for `fixVersion` and `affectedVersion` fields.

        Args:
            id (string): id
            moveFixIssuesTo (string): The ID of the version to update `fixVersion` to when the field contains the deleted version. The replacement version must be in the same project as the version being deleted and cannot be the version being deleted.
            moveAffectedIssuesTo (string): The ID of the version to update `affectedVersion` to when the field contains the deleted version. The replacement version must be in the same project as the version being deleted and cannot be the version being deleted.

        Returns:
            Any: Returned if the version is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/version/{id}"
        query_params = {
            k: v
            for k, v in [
                ("moveFixIssuesTo", moveFixIssuesTo),
                ("moveAffectedIssuesTo", moveAffectedIssuesTo),
            ]
            if v is not None
        }
        response = self._delete(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_version(self, id: str, expand: str | None = None) -> dict[str, Any]:
        """
        Retrieves detailed information about a specific Jira version using the Jira REST API, with options to expand additional fields.

        Args:
            id (string): id
            expand (string): Use [expand](#expansion) to include additional information about version in the response. This parameter accepts a comma-separated list. Expand options include: * `operations` Returns the list of operations available for this version. * `issuesstatus` Returns the count of issues in this version for each of the status categories *to do*, *in progress*, *done*, and *unmapped*. The *unmapped* property represents the number of issues with a status other than *to do*, *in progress*, and *done*. * `driver` Returns the Atlassian account ID of the version driver. * `approvers` Returns a list containing the Atlassian account IDs of approvers for this version.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/version/{id}"
        query_params = {k: v for k, v in [("expand", expand)] if v is not None}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_version(
        self,
        id: str,
        approvers: list[dict[str, Any]] | None = None,
        archived: bool | None = None,
        description: str | None = None,
        driver: str | None = None,
        expand: str | None = None,
        id_body: str | None = None,
        issuesStatusForFixVersion: Any | None = None,
        moveUnfixedIssuesTo: str | None = None,
        name: str | None = None,
        operations: list[dict[str, Any]] | None = None,
        overdue: bool | None = None,
        project: str | None = None,
        projectId: int | None = None,
        releaseDate: str | None = None,
        released: bool | None = None,
        self_arg_body: str | None = None,
        startDate: str | None = None,
        userReleaseDate: str | None = None,
        userStartDate: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates an existing version's details (e.g., name, description, release status) in Jira using the specified version ID.

        Args:
            id (string): id
            approvers (array): If the expand option `approvers` is used, returns a list containing the approvers for this version.
            archived (boolean): Indicates that the version is archived. Optional when creating or updating a version. Example: False.
            description (string): The description of the version. Optional when creating or updating a version. The maximum size is 16,384 bytes. Example: 'An excellent version'.
            driver (string): If the expand option `driver` is used, returns the Atlassian account ID of the driver.
            expand (string): Use [expand](em>#expansion) to include additional information about version in the response. This parameter accepts a comma-separated list. Expand options include:

         *  `operations` Returns the list of operations available for this version.
         *  `issuesstatus` Returns the count of issues in this version for each of the status categories *to do*, *in progress*, *done*, and *unmapped*. The *unmapped* property contains a count of issues with a status other than *to do*, *in progress*, and *done*.
         *  `driver` Returns the Atlassian account ID of the version driver.
         *  `approvers` Returns a list containing approvers for this version.

        Optional for create and update.
            id_body (string): The ID of the version. Example: '10000'.
            issuesStatusForFixVersion (string): If the expand option `issuesstatus` is used, returns the count of issues in this version for each of the status categories *to do*, *in progress*, *done*, and *unmapped*. The *unmapped* property contains a count of issues with a status other than *to do*, *in progress*, and *done*.
            moveUnfixedIssuesTo (string): The URL of the self link to the version to which all unfixed issues are moved when a version is released. Not applicable when creating a version. Optional when updating a version.
            name (string): The unique name of the version. Required when creating a version. Optional when updating a version. The maximum length is 255 characters. Example: 'New Version 1'.
            operations (array): If the expand option `operations` is used, returns the list of operations available for this version.
            overdue (boolean): Indicates that the version is overdue. Example: True.
            project (string): Deprecated. Use `projectId`.
            projectId (integer): The ID of the project to which this version is attached. Required when creating a version. Not applicable when updating a version. Example: 10000.
            releaseDate (string): The release date of the version. Expressed in ISO 8601 format (yyyy-mm-dd). Optional when creating or updating a version. Example: '2010-07-06'.
            released (boolean): Indicates that the version is released. If the version is released a request to release again is ignored. Not applicable when creating a version. Optional when updating a version. Example: True.
            self_arg_body (string): The URL of the version. Example: 'https://your-domain.atlassian.net/rest/api/~ver~/version/10000'.
            startDate (string): The start date of the version. Expressed in ISO 8601 format (yyyy-mm-dd). Optional when creating or updating a version.
            userReleaseDate (string): The date on which work on this version is expected to finish, expressed in the instance's *Day/Month/Year Format* date format. Example: '6/Jul/2010'.
            userStartDate (string): The date on which work on this version is expected to start, expressed in the instance's *Day/Month/Year Format* date format.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "approvers": approvers,
            "archived": archived,
            "description": description,
            "driver": driver,
            "expand": expand,
            "id": id_body,
            "issuesStatusForFixVersion": issuesStatusForFixVersion,
            "moveUnfixedIssuesTo": moveUnfixedIssuesTo,
            "name": name,
            "operations": operations,
            "overdue": overdue,
            "project": project,
            "projectId": projectId,
            "releaseDate": releaseDate,
            "released": released,
            "self": self_arg_body,
            "startDate": startDate,
            "userReleaseDate": userReleaseDate,
            "userStartDate": userStartDate,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/version/{id}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def merge_versions(self, id: str, moveIssuesTo: str) -> Any:
        """
        Merges a Jira version with another specified version and optionally moves associated issues to the target version.

        Args:
            id (string): id
            moveIssuesTo (string): moveIssuesTo

        Returns:
            Any: Returned if the version is deleted.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        if moveIssuesTo is None:
            raise ValueError("Missing required parameter 'moveIssuesTo'.")
        request_body_data = None
        url = f"{self.base_url}/rest/api/3/version/{id}/mergeto/{moveIssuesTo}"
        query_params = {}
        response = self._put(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def move_version(
        self, id: str, after: str | None = None, position: str | None = None
    ) -> dict[str, Any]:
        """
        Moves a project version to a new position within the ordered version list by specifying its ID and returns a status message indicating the success or failure of the operation.

        Args:
            id (string): id
            after (string): The URL (self link) of the version after which to place the moved version. Cannot be used with `position`. Example: 'https://your-domain.atlassian.net/rest/api/~ver~/version/10000'.
            position (string): An absolute position in which to place the moved version. Cannot be used with `after`.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "after": after,
            "position": position,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/version/{id}/move"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_version_related_issues(self, id: str) -> dict[str, Any]:
        """
        Retrieves counts of issues related to a specific Jira version, such as those with the version set as the fix version or affected version, using the "GET" method.

        Args:
            id (string): id

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/version/{id}/relatedIssueCounts"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def get_related_work(self, id: str) -> list[Any]:
        """
        Retrieves related work items associated with a specific version ID in Jira.

        Args:
            id (string): id

        Returns:
            list[Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        url = f"{self.base_url}/rest/api/3/version/{id}/relatedwork"
        query_params = {}
        response = self._get(url, params=query_params)
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def create_related_work(
        self,
        id: str,
        category: str,
        issueId: int | None = None,
        relatedWorkId: str | None = None,
        title: str | None = None,
        url: str | None = None,
    ) -> dict[str, Any]:
        """
        Associates external related work (e.g., design files, communication links) with a Jira project version via a POST request.

        Args:
            id (string): id
            category (string): The category of the related work Example: 'Design'.
            issueId (integer): The ID of the issue associated with the related work (if there is one). Cannot be updated via the Rest API.
            relatedWorkId (string): The id of the related work. For the native release note related work item, this will be null, and Rest API does not support updating it.
            title (string): The title of the related work Example: 'Design link'.
            url (string): The URL of the related work. Will be null for the native release note related work item, but is otherwise required. Example: 'https://www.atlassian.com'.

        Returns:
            dict[str, Any]: Returned if the request is successful.

        Raises:
            HTTPError: Raised when the API request fails (e.g., non-2XX status code).
            JSONDecodeError: Raised if the response body cannot be parsed as JSON.

        Tags:
            Project versions
        """
        if id is None:
            raise ValueError("Missing required parameter 'id'.")
        request_body_data = None
        request_body_data = {
            "category": category,
            "issueId": issueId,
            "relatedWorkId": relatedWorkId,
            "title": title,
            "url": url,
        }
        request_body_data = {
            k: v for k, v in request_body_data.items() if v is not None
        }
        url = f"{self.base_url}/rest/api/3/version/{id}/relatedwork"
        query_params = {}
        response = self._post(
            url,
            data=request_body_data,
            params=query_params,
            content_type="application/json",
        )
        response.raise_for_status()
        if (
            response.status_code == 204
            or not response.content
            or not response.text.strip()
        ):
            return None
        try:
            return response.json()
        except ValueError:
            return None

    def update_related_work(
        self,
        id: str,
        category: str,
        issueId: int | None = None,
        relatedWorkId: str | None = None,
        title: str | None = None,
        url: str | None = None,
    ) -> dict[str, Any]:
        """
        Updates a version's related work links in Jira using the PUT method at the "/rest/api/3/version/{id}/relatedwork" endpoint.

        Args:
            id (string): id
            category (string): The category of the related work Example: 'Design'.
            issueId (integer): The ID of the issue associated with the related work (if there is one). Cannot be updated via the Rest API.
            relatedWorkId (string): The id of the related work. For the native release note related work item, this will be null, and Rest API does not support updating it. Example: 'fabcdef6-7878-1234-beaf-43211234abcd'.
            title (string): The title of the related work Example: 'Design link'.
        