"""Subpackage containing the deployment functionality."""
import time
import typing

from reviser import definitions
from ..deploying import publisher
from ..deploying import uploader


def deploy_target(
    target: "definitions.Target",
    description: str = None,
    published_layers: typing.List["definitions.PublishedLayer"] = None,
    dry_run: bool = False,
) -> typing.List["definitions.PublishedLayer"]:
    """
    Deploy the bundled ZIP file to S3 for access by Lambda functions.

    The file is uploaded to the specified storage bucket with a timestamped key and
    then published to the lambda function.

    :param target:
        Configuration target to deploy.
    :param description:
        A user-specified description to apply to the target version when deployed.
    :param published_layers:
        List of published layers to potentially update in the target deployment.
    :param dry_run:
        Whether or not this is executing as a dry-run command that will echo the results
        without actually carrying out the update.
    """
    s3_keys = []
    if not target.image.configured:
        s3_keys = uploader.upload(target, dry_run=dry_run)

    # Sleep for a few seconds to make sure the S3 upload has finished
    # processing to prevent race conditions during deployment
    if not dry_run and not target.image.configured:
        print("[SYNC]: Synchronization block between S3 and lambda")
        time.sleep(2)

    if target.kind == definitions.TargetType.LAYER:
        return publisher.publish_layer(
            target=target,
            s3_keys=s3_keys,
            description=description,
            dry_run=dry_run,
        )

    publisher.publish_function(
        target=target,
        s3_keys=s3_keys,
        published_layers=published_layers or [],
        description=description,
        dry_run=dry_run,
    )
    return []


def deploy(
    context: "definitions.Context",
    selection: "definitions.Selection",
    description: str = None,
    dry_run: bool = False,
) -> typing.List["definitions.Target"]:
    """
    Execute the deploy operation on the context's configuration.

    This execution is filtered to targets by the specified selection argument value.
    """
    selected = context.get_selected_targets(selection)

    # Publish layers first so that the function publishes can update to
    # use any newly published layers.
    published_layers = []
    for target in selected.layer_targets:
        published_layers += deploy_target(
            target=target,
            description=description,
            published_layers=None,
            dry_run=dry_run,
        )

    for target in selected.function_targets:
        deploy_target(
            target=target,
            description=description,
            published_layers=published_layers,
            dry_run=dry_run,
        )

    return selected.targets
