# Beanie

**FastAPI Users** provides the necessary tools to work with MongoDB databases using the [Beanie ODM](https://github.com/roman-right/beanie).

## Setup database connection and collection

The first thing to do is to create a MongoDB connection using [mongodb/motor](https://github.com/mongodb/motor) (automatically installed with Beanie).

```py hl_lines="5-9"
--8<-- "docs/src/db_beanie.py"
```

You can choose any name for the database.

## Create the User model

As for any Beanie ODM model, we'll create a `User` model.

```py hl_lines="12-13"
--8<-- "docs/src/db_beanie.py"
```

As you can see, **FastAPI Users** provides a base class that will include base fields for our `User` table. You can of course add you own fields there to fit to your needs!

!!! info
    The base class is configured to automatically create a [unique index](https://roman-right.github.io/beanie/tutorial/defining-a-document/#indexes) on `id` and `email`.

!!! info
    If you want to add your own custom settings to your `User` document model - like changing the collection name - don't forget to let your inner `Settings` class inherit the pre-defined settings from `BeanieBaseUser` like this: `class Settings(BeanieBaseUser.Settings): # ...`! See Beanie's [documentation on `Settings`](https://beanie-odm.dev/tutorial/defining-a-document/#settings) for details.

## Create the database adapter

The database adapter of **FastAPI Users** makes the link between your database configuration and the users logic. It should be generated by a FastAPI dependency.

```py hl_lines="16-17"
--8<-- "docs/src/db_beanie.py"
```

Notice that we pass a reference to the `User` model we defined above.

## Initialize Beanie

When initializing your FastAPI app, it's important that you [**initialize Beanie**](https://roman-right.github.io/beanie/tutorial/initialization/) so it can discover your models. We can achieve this using [**Lifespan Events**](https://fastapi.tiangolo.com/advanced/events/) on the FastAPI app:

```py
from contextlib import asynccontextmanager
from beanie import init_beanie


@asynccontextmanager
async def lifespan(app: FastAPI):
    await init_beanie(
        database=db,  # (1)!
        document_models=[
            User,  # (2)!
        ],
    )
    yield

app = FastAPI(lifespan=lifespan)
```

1. This is the `db` Motor database instance we defined above.

2. This is the Beanie `User` model we defined above. Don't forget to also add your very own models!
