import concurrent.futures as concurrentfutures
import multiprocessing
import os
import sys
import warnings
from collections.abc import Callable, Generator, Iterable, Sized
from typing import Any, Generic, Literal, ParamSpec, TypeVar, cast

import loky

from cogent3.util.misc import extend_docstring_from

MPI: Any
if os.environ.get("DONT_USE_MPI", 0):
    MPI = None
else:
    try:
        from mpi4py import MPI  # type: ignore[no-redef]
        from mpi4py import futures as MPIfutures
    except ImportError:
        MPI = None
    else:
        COMM = MPI.COMM_WORLD
        if COMM.Get_attr(MPI.UNIVERSE_SIZE) == 1:
            MPI = None


USING_MPI = MPI is not None


P = ParamSpec("P")
R = TypeVar("R")
T = TypeVar("T")


def get_rank() -> int:
    """Returns the rank of the current process"""
    rank = 0
    if MPI is not None:
        rank = COMM.Get_rank()
    else:
        process_name = multiprocessing.current_process().name
        if process_name != "MainProcess":
            rank = int(process_name.split("-")[-1])
    return rank


def get_size() -> int:
    """Returns the num cpus"""
    return (
        COMM.Get_attr(MPI.UNIVERSE_SIZE) if USING_MPI else multiprocessing.cpu_count()
    )


SIZE = get_size()


def is_master_process() -> bool:
    """
    Evaluates if current process is master

    In case of MPI checks whether current process
    is being run on file generated by mpi4py.futures

    In case of Multiprocessing checks if generated
    process name included "ForkProcess" for Windows
    or "SpawnProcess" for POSIX
    """
    if MPI is None:
        ctxt = loky.backend.get_context()
        return ctxt.parent_process() is None

    process_cmd = sys.argv[0]
    process_file = process_cmd.split(os.sep)[-1]
    if process_file == "server.py":
        return False

    comm = MPI.COMM_WORLD
    return comm.Get_rank() == 0


class PicklableAndCallable(Generic[P, R]):
    def __init__(self, func: Callable[P, R]) -> None:
        self.func = func

    def __call__(self, *args: P.args, **kw: P.kwargs) -> R:
        return self.func(*args, **kw)


def get_default_chunksize(s: Sized, max_workers: int) -> int:
    chunksize, remainder = divmod(len(s), max_workers * 4)
    if remainder:
        chunksize += 1
    return chunksize


def imap(
    f: Callable[[T], R],
    s: Iterable[T],
    max_workers: int | None = None,
    use_mpi: bool = False,
    if_serial: Literal["raise", "ignore", "warn"] = "raise",
    chunksize: int | None = None,
) -> Generator[R]:
    """
    Parameters
    ----------
    f : callable
        function that operates on values in s
    s : iterable
        series of inputs to f
    max_workers : int or None
        maximum number of workers. Defaults to 1-maximum available.
    use_mpi : bool
        use MPI for parallel execution.
    if_serial : str
        action to take if conditions will result in serial execution. Valid
        values are 'raise', 'ignore', 'warn'. Defaults to 'raise'.
    chunksize : int or None
        Size of data chunks executed by worker processes. Defaults to None
        where stable chunksize is determined by get_default_chunksize()

    Returns
    -------
    imap and as_completed are generators yielding result of f(s[i]), map returns the result
    series. imap and map return results in the same order as s, as_completed returns results
    in the order completed (which can differ from the order in s).

    Notes
    -----
    To use MPI, you must have openmpi (use conda or your preferred package manager)
    and mpi4py (use pip or conda) installed. In addition, your initial script must
    have a ``if __name__ == '__main__':`` block. You then invoke your program using

    $ mpiexec -n <number CPUs> python3 -m mpi4py.futures <initial script>
    """

    if_serial = cast("Literal['raise', 'ignore', 'warn']", if_serial.lower())
    assert if_serial in ("ignore", "raise", "warn"), f"invalid choice '{if_serial}'"

    # If max_workers is not defined, get number of all processes available
    # minus 1 to leave for master process
    if use_mpi:
        if not USING_MPI:
            msg = "Cannot use MPI"
            raise RuntimeError(msg)

        if SIZE == 1:
            err_msg = (
                "Execution in serial. For parallel MPI execution, use:\n"
                " $ mpiexec -n <number CPUs> python -m mpi4py.futures <executable script>"
            )

            if if_serial == "raise":
                raise RuntimeError(err_msg)
            elif if_serial == "warn":
                warnings.warn(err_msg, UserWarning, stacklevel=2)

        max_workers = max_workers or 1

        if max_workers > SIZE:
            warnings.warn(
                "max_workers too large, reducing to UNIVERSE_SIZE-1",
                UserWarning,
                stacklevel=2,
            )

        max_workers = min(max_workers, SIZE - 1)
        if not chunksize:
            chunksize = (
                get_default_chunksize(s, max_workers) if isinstance(s, Sized) else 1
            )

        with MPIfutures.MPIPoolExecutor(max_workers=max_workers) as executor:
            yield from executor.map(f, s, chunksize=chunksize)
    else:
        if not max_workers:
            max_workers = multiprocessing.cpu_count() - 1
        assert max_workers < multiprocessing.cpu_count()

        if not chunksize:
            chunksize = (
                get_default_chunksize(s, max_workers) if isinstance(s, Sized) else 1
            )

        with loky.get_reusable_executor(max_workers=max_workers) as executor:
            yield from executor.map(f, s, chunksize=chunksize)


@extend_docstring_from(imap)
def map(
    f: Callable[[T], R],
    s: Iterable[T],
    max_workers: int | None = None,
    use_mpi: bool = False,
    if_serial: Literal["raise", "ignore", "warn"] = "raise",
    chunksize: int | None = None,
) -> list[R]:
    return list(imap(f, s, max_workers, use_mpi, if_serial, chunksize))


def _as_completed_mpi(
    f: Callable[[T], R],
    s: Iterable[T],
    max_workers: int | None,
    if_serial: Literal["raise", "ignore", "warn"],
    chunksize: int | None = None,
) -> Generator[R]:
    """MPI version of as_completed"""
    if not USING_MPI:
        msg = "Cannot use MPI"
        raise RuntimeError(msg)

    if SIZE == 1:
        err_msg = (
            "Execution in serial. For parallel MPI execution, use:\n"
            " $ mpiexec -n <number CPUs> python3 -m mpi4py.futures <executable script>"
        )

        if if_serial == "raise":
            raise RuntimeError(err_msg)
        elif if_serial == "warn":
            warnings.warn(err_msg, UserWarning, stacklevel=2)

    max_workers = max_workers or 1

    f = PicklableAndCallable(f)

    if max_workers > SIZE:
        warnings.warn(
            "max_workers too large, reducing to UNIVERSE_SIZE-1",
            UserWarning,
            stacklevel=2,
        )

    max_workers = min(max_workers, SIZE - 1)
    if not chunksize:
        chunksize = get_default_chunksize(s, max_workers) if isinstance(s, Sized) else 1

    with MPIfutures.MPIPoolExecutor(
        max_workers=max_workers,
        chunksize=chunksize,
    ) as executor:
        to_do = [executor.submit(f, e) for e in s]
        for result in concurrentfutures.as_completed(to_do):
            yield result.result()


def _as_completed_mproc(
    f: Callable[[T], R], s: Iterable[T], max_workers: int | None
) -> Generator[R]:
    """multiprocess version of as_completed"""
    if not max_workers or max_workers > multiprocessing.cpu_count():
        max_workers = multiprocessing.cpu_count() - 1

    with loky.get_reusable_executor(max_workers=max_workers) as executor:
        to_do = [executor.submit(f, e) for e in s]
        for result in concurrentfutures.as_completed(to_do):
            yield result.result()


@extend_docstring_from(imap, pre=True)
def as_completed(
    f: Callable[[T], R],
    s: Iterable[T],
    max_workers: int | None = None,
    use_mpi: bool = False,
    if_serial: Literal["raise", "ignore", "warn"] = "raise",
    chunksize: int | None = None,
) -> Generator[R]:
    if_serial = cast("Literal['raise', 'ignore', 'warn']", if_serial.lower())
    assert if_serial in ("ignore", "raise", "warn"), f"invalid choice '{if_serial}'"
    if use_mpi:
        yield from _as_completed_mpi(f, s, max_workers, if_serial, chunksize)
    else:
        yield from _as_completed_mproc(f, s, max_workers)
