"""
Character shapes that are renderable into Plotly plots. The SVG paths were derived from
the mono-spaced font Inconsolata, which is under the SIL Open Font License. The characters
are stored as SVG paths that are in array format for accessible data manipulation.
The paths are ultimately converted into a String for use in Plotly.

The SVG paths for each character were generated via the following process:

- Open the Inconsolata OTF in Fontforge.
- Export the letters as SVG files and open them with a text editor.
- Plotly only accepts SVG paths with absolute drawing commands, converted the SVG
  path found in the SVG file to all absolutes using the following online tool:
  http://lea.verou.me/2019/05/utility-convert-svg-path-to-all-relative-or-all-absolute-commands/
- A custom Python script was used to convert the resulting SVG path string into Plotly's
  SVG path formatting and then into an array format for inclusion here.
"""

import numpy

from cogent3.util.union_dict import UnionDict

_svg_paths = {
    "A": [
        ["M", 4.2679468235157464e-05, 0.0],
        ["L", 11.740364149758843, 100.0],
        ["H", 12.154707097211453],
        ["L", 25.0, 0.1358695652173956],
        ["H", 21.132589316390153],
        ["L", 17.449351208063685, 28.80434782608696],
        ["H", 6.906082122979179],
        ["L", 3.6832381083264654, 4.369800685114633e-15],
        ["H", 0.0],
        ["M", 16.62065997822494, 37.3641746750466],
        ["L", 11.924563399709868, 74.72830510982921],
        ["L", 7.642756419312122, 37.3641746750466],
        ["H", 16.62064930835788],
        ["Z"],
    ],
    "B": [
        ["M", 6.735161215100033e-05, 100.0],
        ["H", 10.832349591604315],
        [
            "C",
            13.178414420592452,
            100.0,
            15.572317262830483,
            99.93072538208159,
            17.959374732124274,
            97.21005163325559,
        ],
        [
            "C",
            21.544305115048804,
            93.12379912397438,
            23.514027503949258,
            84.55553865047374,
            23.514027503949258,
            75.56924653194388,
        ],
        [
            "C",
            23.514027503949258,
            65.98209109870407,
            21.28354416428895,
            57.262166362696185,
            17.860116824432716,
            53.60119970443926,
        ],
        [
            "C",
            22.134525660964233,
            49.82479166626545,
            25.0,
            39.33252959298769,
            25.0,
            27.687474682500095,
        ],
        [
            "C",
            25.0,
            17.380772630715338,
            22.743084214006693,
            7.588830320269986,
            18.637476886151283,
            3.0156312090065143,
        ],
        [
            "C",
            16.001053134299116,
            0.07897776040192164,
            13.374493332190399,
            0.0,
            10.779441838821471,
            0.0,
        ],
        ["H", 0.0],
        ["V", 99.99987160697354],
        ["M", 4.227293322393797, 90.44322556664656],
        ["V", 58.587310788801815],
        ["H", 10.198234163186832],
        [
            "C",
            11.896792838686766,
            58.587310788801815,
            13.72409105937963,
            58.60564820182356,
            15.480351697872914,
            60.59768879713067,
        ],
        [
            "C",
            17.88250205116273,
            63.322750538127266,
            19.190072372368693,
            68.826478108668,
            19.190072372368693,
            74.53441478779521,
        ],
        [
            "C",
            19.190072372368693,
            80.28858900557675,
            17.84949976120792,
            85.95373870863577,
            15.31968748851961,
            88.63225797754019,
        ],
        [
            "C",
            13.624055546711402,
            90.42759371567492,
            11.880352922447678,
            90.44327371403149,
            10.251099055852851,
            90.44327371403149,
        ],
        ["H", 4.227305568141459],
        ["M", 4.227293322393797, 49.169040332743364],
        ["V", 9.833910780969834],
        ["H", 11.519268683949498],
        [
            "C",
            13.168140238302252,
            9.833910780969834,
            14.785019776882478,
            9.909611309371334,
            16.432636142099657,
            11.855571326498962,
        ],
        [
            "C",
            19.068937436475185,
            14.969326905978418,
            20.52043202997759,
            21.659181353223975,
            20.52043202997759,
            28.810351944544475,
        ],
        [
            "C",
            20.52043202997759,
            36.04249038817705,
            19.0187727311691,
            43.17932910791883,
            16.18700481257883,
            46.702241164473385,
        ],
        [
            "C",
            14.282637978961803,
            49.07152582914611,
            12.389684182167741,
            49.16900823448674,
            10.462479029157125,
            49.16900823448674,
        ],
        ["H", 4.227311691015292],
        ["Z"],
    ],
    "C": [
        ["M", 13.842394554289157, 99.99999843412603],
        [
            "C",
            18.67692280314117,
            99.99999843412603,
            23.043230265612227,
            92.43823645010107,
            25.0,
            80.67586095665504,
        ],
        ["L", 21.119211574829915, 75.54059293383713],
        [
            "C",
            20.85059169665922,
            76.04552621977668,
            20.852872823198506,
            76.8945227294436,
            20.852872823198506,
            77.07787091248174,
        ],
        [
            "C",
            20.852872823198506,
            77.46719885503296,
            20.877624622965296,
            77.85713748843212,
            20.877624622965296,
            78.24554782883745,
        ],
        [
            "C",
            20.877624622965296,
            78.97358683788977,
            20.79241934907175,
            79.37805364997392,
            20.67647317943455,
            79.91480079734302,
        ],
        [
            "C",
            19.207792107737617,
            86.71625267568714,
            16.55284347480612,
            91.23909173046299,
            13.704495490940507,
            91.23909173046299,
        ],
        [
            "C",
            8.560397463301326,
            91.23909173046299,
            3.98260983124378,
            76.72707286481341,
            3.98260983124378,
            50.80681656256215,
        ],
        [
            "C",
            3.98260983124378,
            24.437968685652372,
            8.691500918704152,
            9.458818297550502,
            14.045846863754226,
            9.458818297550502,
        ],
        [
            "C",
            16.901541640211793,
            9.458818297550502,
            19.79374261737866,
            13.90352024126986,
            21.47194743933116,
            21.351505822702354,
        ],
        ["L", 24.495888438190615, 16.081212487531726],
        [
            "C",
            22.21250763663466,
            6.078425231866787,
            18.15644739964446,
            0.0,
            13.816700519612363,
            0.0,
        ],
        [
            "C",
            7.822582093118824,
            0.0,
            0.0,
            12.142256518341865,
            0.0,
            49.83879327488448,
        ],
        [
            "C",
            0.0,
            88.89717062232529,
            8.182913653471374,
            100.0,
            13.842338372934044,
            100.0,
        ],
        ["Z"],
    ],
    "D": [
        ["M", 0.0, 100.0],
        ["H", 8.96683463304388],
        [
            "C",
            12.345411911103243,
            100.0,
            15.102759517889771,
            99.36846942026692,
            17.886144372388593,
            94.64725653947468,
        ],
        [
            "C",
            22.813215486352426,
            86.28992552598834,
            25.0,
            69.46235366597153,
            25.0,
            50.23514734853414,
        ],
        [
            "C",
            25.0,
            28.382086233943404,
            22.003275713242292,
            11.368521668110159,
            16.5113653969589,
            4.008146817080622,
        ],
        [
            "C",
            13.810641887322353,
            0.3887816508138808,
            11.26908066074973,
            0.0,
            8.2240814108192,
            0.0,
        ],
        ["H", 3.4947961692850248e-15],
        ["V", 99.99996794610843],
        ["M", 4.085487016420373, 90.4564345943548],
        ["V", 8.852034748982765],
        ["H", 8.011811749776824],
        [
            "C",
            10.420500013525844,
            8.852034748982765,
            12.773117510064456,
            9.131410057100798,
            15.174790779801736,
            13.35586294525259,
        ],
        [
            "C",
            18.235639858445904,
            18.739762788380848,
            20.65831507344533,
            29.695173901977913,
            20.65831507344533,
            48.7318198340506,
        ],
        [
            "C",
            20.65831507344533,
            63.0729552982839,
            19.469461105823736,
            77.28077862749083,
            15.90950227357124,
            84.97259071732121,
        ],
        [
            "C",
            13.538637185247486,
            90.09542764058355,
            11.115236493216175,
            90.45649870213794,
            8.701580556231848,
            90.45649870213794,
        ],
        ["H", 4.085511608862132],
        ["Z"],
    ],
    "E": [
        ["M", 0.0, 99.99995191914726],
        ["H", 25.0],
        ["V", 90.04149377593362],
        ["H", 4.345426486359238],
        ["V", 57.261410788381745],
        ["H", 21.388271298161175],
        ["V", 47.02623131057186],
        ["H", 4.345426486359238],
        ["V", 9.95845814321363],
        ["H", 24.83076560906071],
        ["V", 0.0],
        ["H", 6.539195940405456e-05],
        ["V", 100.00000000000001],
        ["Z"],
    ],
    "F": [
        ["M", 0.0, 99.99993589219635],
        ["H", 25.0],
        ["V", 90.17967814677162],
        ["H", 4.927884615384611],
        ["V", 58.78288129318263],
        ["H", 21.153915790177372],
        ["V", 49.10096818810513],
        ["H", 4.92788461538461],
        ["V", 0.0],
        ["H", 2.7854532483524228e-05],
        ["V", 100.00000000000001],
        ["Z"],
    ],
    "G": [
        ["M", 14.221362338673522, 99.99970251189562],
        [
            "C",
            18.7495694028266,
            99.99970251189562,
            22.797029469362712,
            93.75682069765658,
            24.89834367553028,
            83.64709455894257,
        ],
        ["L", 21.996802963452325, 75.81005227962002],
        [
            "C",
            21.535226387938557,
            76.59582013556063,
            21.392256329188584,
            77.8352573506962,
            21.211204964254534,
            78.88783289310312,
        ],
        [
            "C",
            20.087075006488448,
            85.42566633421067,
            17.475969138569706,
            90.41244377083396,
            14.195715263195146,
            90.41244377083396,
        ],
        [
            "C",
            11.099585918882568,
            90.41244377083396,
            7.822995068777572,
            85.90806678138226,
            5.926656914328855,
            76.43618645301802,
        ],
        [
            "C",
            4.488816270674559,
            69.25372760423437,
            4.0778732038789105,
            60.50194071844944,
            4.0778732038789105,
            51.41696715574756,
        ],
        [
            "C",
            4.0778732038789105,
            41.11237065138935,
            4.50920770120095,
            27.42423839130959,
            7.577625463039423,
            18.141983243591092,
        ],
        [
            "C",
            9.576581412358726,
            12.094817287503465,
            12.18912653658306,
            9.599753241446043,
            14.566925418210129,
            9.599753241446043,
        ],
        [
            "C",
            16.757007526602646,
            9.599753241446043,
            19.11984168180638,
            11.644216752964313,
            21.18219569166883,
            15.54553838301124,
        ],
        ["V", 38.78671825214779],
        ["H", 14.513661137720312],
        ["V", 48.24514898673987],
        ["H", 25.0],
        ["V", 10.140837131525755],
        [
            "C",
            21.535155604841563,
            3.368755294113963,
            17.606764504636292,
            0.0,
            14.202274496850151,
            0.0,
        ],
        [
            "C",
            7.788618078002973,
            0.0,
            0.0,
            12.182388391074733,
            0.0,
            49.10620481899415,
        ],
        [
            "C",
            0.0,
            91.91208130506466,
            9.333282212207726,
            100.0,
            14.221267961210863,
            100.0,
        ],
        ["Z"],
    ],
    "H": [
        ["M", 0.0, 100.0],
        ["H", 4.930573099271462],
        [
            "C",
            4.975642226541251,
            99.77274284071113,
            4.999137799278301,
            99.51997400302594,
            4.999137799278301,
            99.26188422224016,
        ],
        [
            "C",
            4.999137799278301,
            98.00117169731385,
            4.601884942684537,
            97.83949355904646,
            4.601884942684537,
            96.12723971236521,
        ],
        ["V", 57.26143655783447],
        ["H", 20.105818093904897],
        ["V", 99.99998878114407],
        ["H", 24.92684815346467],
        [
            "C",
            24.974645001453847,
            99.79871288950707,
            25.000000000000004,
            99.56644731239005,
            25.000000000000004,
            99.3272949604226,
        ],
        [
            "C",
            25.000000000000004,
            98.11400614222748,
            24.543357371267696,
            98.06439155320157,
            24.543357371267696,
            96.40394958861579,
        ],
        ["V", 0.0],
        ["H", 20.051013563860916],
        ["V", 47.71780102602529],
        ["H", 4.601928109853643],
        ["V", 0.1383124667421022],
        ["H", 4.1262735174152525e-05],
        ["V", 99.99991345453996],
        ["Z"],
    ],
    "I": [
        ["M", 0.5361930294906141, 99.99995191916265],
        ["H", 24.53083109919571],
        ["V", 90.59476316341139],
        ["H", 14.94638069705094],
        ["V", 9.266988402260953],
        ["H", 25.0],
        ["V", 0.0],
        ["H", 0.0],
        ["V", 9.405349025209103],
        ["H", 9.517426273458444],
        ["V", 90.59481124424875],
        ["H", 0.5361930294906163],
        ["V", 100.0],
        ["Z"],
    ],
    "J": [
        ["M", 7.521203604686442, 100.0],
        ["H", 25.0],
        ["V", 90.74965155690462],
        ["H", 19.120768485212714],
        ["V", 34.70288310061234],
        [
            "C",
            19.120768485212714,
            27.590104184564346,
            19.095835162288466,
            20.13219242192621,
            17.63088199335979,
            13.091056748068313,
        ],
        [
            "C",
            15.858575369154337,
            4.57282500101672,
            12.5436432884466,
            0.0,
            8.92290745437923,
            0.0,
        ],
        [
            "C",
            5.5413228476512275,
            0.0,
            2.2652038953552878,
            3.935381421055138,
            0.0,
            10.896692809225318,
        ],
        ["L", 2.860142133136149, 19.87503337834221],
        [
            "C",
            3.22431226377019,
            19.015144230693448,
            2.967307242858727,
            17.928969626733192,
            3.239476020311467,
            17.027510059215622,
        ],
        [
            "C",
            3.3719341065517336,
            16.588637271447023,
            5.792445752454235,
            9.76509213368235,
            8.991404892860054,
            9.76509213368235,
        ],
        [
            "C",
            10.828621012018566,
            9.76509213368235,
            12.488723079989903,
            12.113240774573004,
            13.51200506510098,
            16.028145942249715,
        ],
        [
            "C",
            14.935840499400495,
            21.475191891971136,
            14.88345750588457,
            28.48590480291882,
            14.88345750588457,
            34.56680034817513,
        ],
        ["V", 90.74944663673992],
        ["H", 7.521176599979771],
        ["V", 99.99979507983534],
        ["Z"],
    ],
    "K": [
        ["M", 9.32594841990294e-06, 99.31796648032258],
        ["H", 4.87928142634785],
        [
            "C",
            4.944000594062908,
            98.25044269577891,
            4.685806466053076,
            97.52408743080389,
            4.550619266290327,
            97.04768834520685,
        ],
        [
            "C",
            4.3470128287746475,
            96.32936686093599,
            4.3259519228706695,
            95.90930541342132,
            4.3259519228706695,
            95.22510141343953,
        ],
        ["V", 53.75169654731439],
        ["L", 19.51710588775101, 100.0],
        [
            "C",
            20.51601405233908,
            99.31054764269084,
            21.54850728869499,
            99.1814886390137,
            22.53523343315206,
            99.1814886390137,
        ],
        ["H", 24.044265147904877],
        ["L", 10.311864378327611, 56.61667838167472],
        ["L", 25.0, 0.5457264080647144],
        [
            "C",
            22.123207202989526,
            0.5695006763425299,
            19.818876092651436,
            0.0,
            19.818876092651436,
            0.0,
        ],
        ["L", 7.193173965309337, 49.93179664803227],
        ["L", 4.32594609415291, 41.473420674430365],
        ["V", 0.6821441778138424],
        ["H", 0.0],
        ["V", 99.31790640876281],
        ["Z"],
    ],
    "L": [
        ["M", 0.0, 100.0],
        ["H", 5.439832901671773],
        [
            "C",
            5.512260202252475,
            98.86074113369752,
            5.30499800946968,
            97.93042545911167,
            5.205852379095468,
            97.47950755630063,
        ],
        [
            "C",
            5.008281984839568,
            96.58065276116096,
            4.861136034092171,
            96.05382967692098,
            4.861136034092171,
            94.60576537043463,
        ],
        ["V", 9.681905346506408],
        ["H", 24.999999999999996],
        ["V", 0.0],
        ["H", 1.475261659010098e-05],
        ["V", 99.99991986530961],
        ["Z"],
    ],
    "M": [
        ["M", 0.0, 99.86187845303867],
        ["H", 3.292204912521637],
        ["L", 12.500023842562788, 51.1049723756906],
        ["L", 21.862163760258262, 100.0],
        ["H", 25.0],
        ["V", 0.0],
        ["H", 21.03914233533134],
        ["V", 75.41436464088397],
        ["L", 13.014450977306652, 35.22099447513812],
        ["H", 11.419806732186032],
        ["L", 3.9095186663424104, 74.86187845303867],
        ["V", 0.0],
        ["H", 3.811760953178512e-15],
        ["V", 99.86187845303867],
        ["Z"],
    ],
    "N": [
        ["M", 4.3243405557191175e-05, 0.13828137378702413],
        ["V", 99.86187867409927],
        ["H", 4.2118088592745195],
        ["L", 20.365568334327847, 25.966969317219704],
        ["V", 99.99999999999999],
        ["H", 24.950530717791285],
        [
            "C",
            25.0,
            99.06831483980007,
            24.839405667477273,
            98.29870697312634,
            24.77038980996972,
            97.9584451667939,
        ],
        [
            "C",
            24.622283616985484,
            97.22885247265981,
            24.524008653514493,
            96.83107585789668,
            24.524008653514493,
            95.718238897078,
        ],
        ["V", 0.0],
        ["H", 21.111955691887438],
        ["L", 4.158440319186644, 78.5910344375037],
        ["V", 0.1381213259007094],
        ["H", 0.0],
        ["Z"],
    ],
    "O": [
        ["M", 25.0, 50.00140735829466],
        [
            "C",
            25.0,
            41.444246719291435,
            24.77633821840043,
            29.322935561754882,
            22.903391453833265,
            18.84702953575941,
        ],
        [
            "C",
            20.582567787887484,
            5.866526139334725,
            16.524443289739185,
            0.0,
            12.718800552930814,
            0.0,
        ],
        [
            "C",
            7.322898967011539,
            0.0,
            0.0,
            12.066392909416166,
            0.0,
            50.35371605139047,
        ],
        [
            "C",
            0.0,
            88.64856074158399,
            7.425707842872778,
            100.0,
            12.683438961706734,
            100.0,
        ],
        [
            "C",
            16.52083357850703,
            100.0,
            20.529129351229656,
            94.15250447227191,
            22.841041896186965,
            81.6336615084379,
        ],
        [
            "C",
            24.763829833459916,
            71.22155572513354,
            24.999977368581614,
            59.11686703278831,
            24.999977368581614,
            50.0015637314385,
        ],
        ["M", 12.517471454991995, 89.74489285312183],
        [
            "C",
            8.81317786755913,
            89.74489285312183,
            3.904745717909329,
            80.82489960844164,
            3.904745717909329,
            51.77655528728873,
        ],
        [
            "C",
            3.904745717909329,
            20.791216833256172,
            8.770336592559334,
            10.66105182831478,
            12.735808063845948,
            10.66105182831478,
        ],
        [
            "C",
            15.107908865993942,
            10.66105182831478,
            17.83931172425156,
            14.36651675694608,
            19.4705787034732,
            23.84968787920486,
        ],
        [
            "C",
            20.91543068947069,
            32.24817668914268,
            21.048701454476,
            42.18897381688078,
            21.048701454476,
            48.56993632485581,
        ],
        [
            "C",
            21.048701454476,
            56.81708720617485,
            20.86808123592892,
            66.14799779826612,
            19.676615136435775,
            74.3388230731701,
        ],
        [
            "C",
            18.033625082491522,
            85.6332643206525,
            15.018831603236839,
            89.74478339192112,
            12.517448823573615,
            89.74478339192112,
        ],
        ["Z"],
    ],
    "P": [
        ["M", 6.927046362271795e-05, 100.0],
        ["H", 11.956151292737983],
        [
            "C",
            14.486456490843953,
            100.0,
            16.9206394746931,
            99.83512080585922,
            19.381063369627693,
            96.6058645147985,
        ],
        [
            "C",
            23.08768809449801,
            91.74097015422416,
            25.0,
            81.9553785870822,
            25.0,
            71.84204761297187,
        ],
        [
            "C",
            25.0,
            61.703984920231775,
            23.09063523785966,
            52.126165468197435,
            19.46205313623913,
            47.42505254085909,
        ],
        [
            "C",
            17.078853213620537,
            44.33747160305993,
            14.72652272852074,
            44.182790029955704,
            12.282169581147956,
            44.182790029955704,
        ],
        ["V", 44.182790029955704],
        ["H", 4.565062094043597],
        ["V", 0.0],
        ["H", 0.0],
        ["V", 99.9999197543198],
        ["M", 4.56514395913697, 89.47369265744003],
        ["L", 4.510841898101379, 54.01665739829062],
        ["H", 12.445269721552883],
        [
            "C",
            13.992293283082436,
            54.01665739829062,
            15.488277107547553,
            54.11053858493164,
            17.001169222452898,
            56.06134942740696,
        ],
        [
            "C",
            19.274355054199724,
            58.992515485410145,
            20.48861590308785,
            64.95075723836098,
            20.48861590308785,
            71.32776909787005,
        ],
        [
            "C",
            20.48861590308785,
            77.83723447707501,
            19.218390816902794,
            84.19951322970407,
            16.772280718679795,
            87.32675158264543,
        ],
        [
            "C",
            15.172920173221497,
            89.37144361176176,
            13.595518364733419,
            89.4736445100319,
            11.95618876176149,
            89.4736445100319,
        ],
        ["H", 4.565156238900978],
        ["Z"],
    ],
    "Q": [
        ["M", 12.523599119470123, 91.80414885611458],
        [
            "C",
            8.68012966700862,
            91.80414885611458,
            3.8553315305418785,
            84.03368023401462,
            3.8553315305418785,
            60.54513916641597,
        ],
        [
            "C",
            3.8553315305418785,
            35.17428945843229,
            8.620344024026958,
            26.213030151677184,
            12.764563492952567,
            26.213030151677184,
        ],
        [
            "C",
            15.722739229135788,
            26.213030151677184,
            21.196619234196092,
            31.12647152969508,
            21.196619234196092,
            57.706713814396664,
        ],
        [
            "C",
            21.196619234196092,
            64.4323569090213,
            20.98180054192372,
            72.21092614802366,
            19.75460672509557,
            79.03636986990564,
        ],
        [
            "C",
            18.0557824320847,
            88.4841540251041,
            15.005503877495268,
            91.80403331641234,
            12.5235821461637,
            91.80403331641234,
        ],
        ["M", 24.999941159204408, 58.77288851129943],
        [
            "C",
            24.999941159204408,
            51.75074486520381,
            24.777056185944154,
            41.83797759715984,
            22.904354513077703,
            33.28187751297247,
        ],
        [
            "C",
            20.965963315479677,
            24.42533568969281,
            17.812198511644706,
            19.614942888340043,
            14.60010776918023,
            18.287134954557338,
        ],
        [
            "C",
            14.665912712395468,
            13.849088100332416,
            15.201358860280944,
            8.588873562618705,
            18.74453657555169,
            8.588873562618705,
        ],
        [
            "C",
            20.00996446242257,
            8.588873562618705,
            21.541211301144248,
            9.086838125359602,
            23.339154045881788,
            9.263231305015031,
        ],
        ["L", 23.241469272770765, 0.5107522402313818],
        [
            "C",
            19.170913953709945,
            0.5899852599582965,
            16.706802878582245,
            0.0,
            14.51305960142603,
            2.471011666403193,
        ],
        [
            "C",
            11.943968626087791,
            5.364395403460917,
            11.074420480398887,
            10.882122262007535,
            11.12226823119816,
            18.192242197096817,
        ],
        [
            "C",
            5.994802094641798,
            20.017730979462662,
            0.0,
            30.80532635973348,
            0.0,
            59.22565394251889,
        ],
        [
            "C",
            0.0,
            90.73320236950875,
            7.443700108380219,
            99.99999999999999,
            12.685453174834334,
            99.99999999999999,
        ],
        [
            "C",
            16.553398135154136,
            99.99999999999999,
            20.582736608457278,
            95.12729278589869,
            22.881488074668063,
            84.74079990087206,
        ],
        [
            "C",
            24.766175730751762,
            76.22476641787716,
            25.0,
            66.35702112047784,
            25.0,
            58.77299506458041,
        ],
        ["Z"],
    ],
    "R": [
        ["M", 0.0, 100.0],
        ["H", 11.257922602494077],
        [
            "C",
            13.724451199316412,
            100.0,
            16.133630345184407,
            99.85455484474001,
            18.551175504508585,
            96.63689073325384,
        ],
        [
            "C",
            22.063829375461786,
            91.96195844850583,
            23.950451152280554,
            82.61522345752303,
            23.950451152280554,
            72.57316031739384,
        ],
        [
            "C",
            23.950451152280554,
            59.33435309445502,
            20.79802581680709,
            47.90903230062058,
            16.173365846870844,
            44.59825921608713,
        ],
        ["L", 25.0, 0.0],
        ["H", 20.190264949073885],
        ["L", 11.733609473021996, 44.18274748426023],
        ["H", 4.334097176084619],
        ["V", 0.0],
        ["H", 2.4497843211567385e-05],
        ["V", 99.99982345967362],
        ["M", 4.334054304859, 89.47370279371857],
        ["V", 54.01670167782323],
        ["H", 11.73356660179638],
        [
            "C",
            13.238126140374602,
            54.01670167782323,
            14.693046924148797,
            54.11058277406162,
            16.16441188517763,
            56.061391738009036,
        ],
        [
            "C",
            18.37520137227119,
            58.99255497345397,
            19.55613215312524,
            64.95079098893238,
            19.55613215312524,
            71.32779670771552,
        ],
        [
            "C",
            19.55613215312524,
            77.8372558186488,
            18.320712044511605,
            84.19952844473913,
            15.941806108390221,
            87.32676378631497,
        ],
        [
            "C",
            14.386346176082663,
            89.37145384650077,
            12.852242238593318,
            89.47365464635685,
            11.257910353572475,
            89.47365464635685,
        ],
        ["H", 4.334023682554991],
        ["Z"],
    ],
    "S": [
        ["M", 23.74890543884093, 87.80525330714897],
        ["L", 21.021012351310745, 78.4566372443888],
        [
            "C",
            20.885492540125647,
            78.56089222121999,
            20.706937757525573,
            78.871051334628,
            20.694590165560125,
            79.74081461252045,
        ],
        [
            "C",
            20.686358520222733,
            80.31805066670486,
            20.717529898997913,
            80.64538073000308,
            20.55344135978461,
            81.25128579599536,
        ],
        [
            "C",
            19.05769812424695,
            86.77589119007125,
            16.291675137223017,
            90.94025315768891,
            12.693984004284037,
            90.94025315768891,
        ],
        [
            "C",
            8.272905210755708,
            90.94025315768891,
            5.719410548445799,
            84.1008075364036,
            5.719410548445799,
            76.60089608206187,
        ],
        [
            "C",
            5.719410548445799,
            72.60194493992336,
            6.4335094382657605,
            66.79908010480793,
            9.665470872029315,
            62.011319023852586,
        ],
        [
            "C",
            13.199161786799815,
            56.77618257436518,
            18.640314559274888,
            53.99809394480831,
            21.99961324679317,
            45.985025861612066,
        ],
        [
            "C",
            24.417062510226646,
            40.21880385742568,
            25.0,
            33.386878337260754,
            25.0,
            28.267573424446194,
        ],
        [
            "C",
            25.0,
            15.955974721910117,
            21.527985065376168,
            0.0,
            12.12812191651089,
            0.0,
        ],
        [
            "C",
            7.3625290684701925,
            0.0,
            3.0370663281749715,
            4.231995778946833,
            0.0,
            12.609655320011546,
        ],
        ["L", 2.5674338924726916, 23.85509521860981],
        [
            "C",
            2.8637742402530755,
            23.474103974456654,
            2.8500953247487346,
            22.610166812484415,
            2.848703261122873,
            22.351062434862424,
        ],
        [
            "C",
            2.845192694826928,
            21.703300705828216,
            2.829153010477045,
            21.306364963349633,
            3.0230818528453627,
            20.68436311305937,
        ],
        [
            "C",
            4.74979856603811,
            15.144890212175493,
            8.01480756548773,
            9.89304940445823,
            12.254211519295515,
            9.89304940445823,
        ],
        [
            "C",
            17.86094920641217,
            9.89304940445823,
            20.68735279825862,
            19.127749279891447,
            20.68735279825862,
            27.63907508596465,
        ],
        [
            "C",
            20.68735279825862,
            32.01379579035102,
            19.980096465174856,
            37.12336696058802,
            17.44716046291384,
            41.14486270650292,
        ],
        [
            "C",
            14.528425740905103,
            45.77853236262173,
            8.954022853147823,
            49.16361403939767,
            5.37680360870492,
            55.85402350215286,
        ],
        [
            "C",
            2.1568290451658307,
            61.876447030731256,
            1.2730111960094983,
            69.23055644101215,
            1.2730111960094983,
            74.9343568960238,
        ],
        [
            "C",
            1.2730111960094983,
            88.31304067800102,
            6.063606647196781,
            100.0,
            13.038366042076765,
            100.0,
        ],
        [
            "C",
            17.248205688247165,
            100.0,
            21.133346183787104,
            95.67663263750262,
            23.74888870432717,
            87.80506491213131,
        ],
        ["Z"],
    ],
    "T": [
        ["M", 0.0, 100.0],
        ["H", 25.0],
        ["V", 89.9171270718232],
        ["H", 14.350000000000003],
        ["V", 0.0],
        ["H", 10.250023174971034],
        ["V", 89.9171270718232],
        ["H", 2.3174971033496387e-05],
        ["V", 100.0],
        ["Z"],
    ],
    "U": [
        ["M", 0.0, 99.999929063279],
        ["H", 4.861120602279505],
        [
            "C",
            4.9033738602922465,
            99.75008047376899,
            4.925255934720305,
            99.47914160892677,
            4.925255934720305,
            99.20420033664988,
        ],
        [
            "C",
            4.925255934720305,
            98.26345803588512,
            4.681166192570425,
            97.61890649625876,
            4.517714655070472,
            97.03361549333648,
        ],
        [
            "C",
            4.289341668791038,
            96.21556534517333,
            4.273504210018197,
            95.6827690921455,
            4.273504210018197,
            94.96647215292315,
        ],
        ["V", 35.516613222415614],
        [
            "C",
            4.273504210018197,
            30.63832644826773,
            4.310982089818607,
            25.878740453887808,
            5.199567826662412,
            21.03877527278321,
        ],
        [
            "C",
            6.586487504439391,
            13.485117948850517,
            9.498095295820407,
            9.385937063049495,
            12.55699776747912,
            9.385937063049495,
        ],
        [
            "C",
            15.566956759689363,
            9.385937063049495,
            18.471309997141756,
            13.388596717174654,
            19.89672461594441,
            20.8988407678695,
        ],
        [
            "C",
            20.845276111788866,
            25.89690025445789,
            20.88674220118121,
            30.849371075031808,
            20.88674220118121,
            35.924704296337666,
        ],
        ["V", 99.99995270885267],
        ["H", 25.0],
        ["V", 36.196786030573406],
        [
            "C",
            25.0,
            29.012756314077535,
            24.914949640025572,
            22.208174999944823,
            23.45536805298469,
            15.357153779145927,
        ],
        [
            "C",
            21.261583517098863,
            5.060546855911126,
            16.92893117097117,
            0.0,
            12.471218238176364,
            0.0,
        ],
        [
            "C",
            7.920136967958936,
            0.0,
            3.6407795956583966,
            5.176473221703104,
            1.503285941536115,
            15.276017934064159,
        ],
        [
            "C",
            0.07532108705206762,
            22.023313905079103,
            2.475956967293772e-06,
            28.709194439443845,
            2.475956967293772e-06,
            35.65266985324926,
        ],
        ["V", 35.65266985324926],
        ["V", 99.99999999999999],
        ["Z"],
    ],
    "V": [
        ["M", 0.0, 99.99999999999999],
        ["H", 4.04760396544844],
        ["L", 12.857154996450909, 23.763736263736252],
        ["L", 21.19051407983162, 99.86263736263736],
        ["H", 25.0],
        ["L", 13.476154867553529, 0.0],
        ["H", 11.714261214954922],
        ["Z"],
    ],
    "W": [
        ["M", 0.0, 100.00000000000001],
        ["H", 3.440399596117574],
        ["L", 7.01835543141706, 32.87493504961731],
        ["L", 12.339457480191241, 92.70961839458593],
        ["H", 13.486257345563766],
        ["L", 18.85323096372393, 32.59967293696864],
        ["L", 21.880754543198897, 99.99984061255783],
        ["H", 25.0],
        ["L", 19.770651720720704, 0.0],
        ["H", 18.30275086964629],
        ["L", 12.614676265784233, 65.61199996174703],
        ["L", 6.834858523150209, 0.0],
        ["H", 5.321123310220018],
        ["Z"],
    ],
    "X": [
        ["M", 19.798030505695646, 100.00000000000001],
        ["H", 23.888890969690472],
        ["L", 14.661231716971864, 51.18735665895771],
        ["L", 25.0, 0.0],
        ["H", 20.404037306483502],
        ["L", 12.296675123564501, 39.57903610711774],
        ["L", 4.494976190687987, 0.0],
        ["H", 0.0],
        ["L", 9.971831928870298, 50.92691870649685],
        ["L", 0.20207834623716525, 100.00000000000001],
        ["H", 4.59601536380657],
        ["L", 12.280552162548478, 62.04208997850787],
        ["Z"],
    ],
    "Y": [
        ["M", 0.0, 100.00000000000001],
        ["H", 4.466044527864284],
        ["L", 13.058271737826903, 51.45228215767635],
        ["L", 20.825256786380685, 99.86168741355465],
        ["H", 25.0],
        ["L", 15.048550406540471, 39.41908713692947],
        ["V", 0.0],
        ["H", 10.776742379770923],
        ["V", 39.41908713692947],
        ["Z"],
    ],
    "Z": [
        ["M", 0.7322873713683093, 99.99994871378995],
        ["H", 24.058590657309427],
        ["L", 24.006272499181847, 92.11607223278041],
        ["L", 5.334769644740189, 9.958608823087257],
        ["H", 23.16945444408082],
        [
            "C",
            24.01364798855799,
            9.958608823087257,
            24.24576682060046,
            11.222608756415184,
            25.0,
            10.926799512909222,
        ],
        ["V", 0.0],
        ["H", 0.0],
        ["V", 7.883876481009519],
        ["L", 19.037610753548357, 89.90318766233288],
        ["H", 0.7322157982133979],
        ["V", 100.00000000000001],
        ["Z"],
    ],
    "-": [
        ["M", 0.0, 100.0],
        ["H", 25.0],
        ["V", 0.0],
        ["H", 0.0],
        ["V", 100.0],
        ["Z"],
    ],
}


def letter_stack(data, x, width, colours, axnum):
    """returns plotly SVG letter shapes with letters stacked on top of each other

    Parameters
    ----------
    data
        [(character, value), ...] for display. value<0 will invert the letter.
    x : float
        x-coordinate
    width : float
        width of the letter stack
    colours : dict
        colours for each letter
    axnum : int
        plotly axis number
    """
    mid_x = None
    result = []
    stack_y = 0
    xref = f"x{axnum}"
    yref = f"y{axnum}"
    for letter, value in data:
        if value == 0:
            continue

        character = get_character(
            letter=letter,
            x=x,
            y=stack_y,
            width=width,
            height=value,
            fillcolor=colours.get(letter, "gray"),
        )
        shape = character.as_shape()
        mid_x = character.middle_x
        shape.xref = xref
        shape.yref = yref
        result.append(shape)
        stack_y += abs(value)

    return mid_x, result


def get_character(
    letter,
    x=0,
    y=0,
    width=1,
    height=1,
    filled=True,
    fillcolor="rgb(0,0,256)",
    axes=(0, 0),
):
    """
    returns a Letter instance

    Parameters
    ----------
    letter : str
        the character to be represented. Must be ascii
    x, y : float
        x, y coordinates
    width, height : float
        letter width, height
    filled : bool
        letter is filled
    fillcolor : str
        color interpretable by plotly
    axes
        plotly axis numbers for x and y
    """
    letter = letter.upper()
    # convert ? ambiguity char to -
    letter = "-" if letter == "?" else letter
    return Letter(
        _svg_paths[letter],
        x,
        y,
        width=width,
        height=height,
        filled=filled,
        fillcolor=fillcolor,
        axes=axes,
    )


class Letter:
    def __init__(
        self,
        path,
        x,
        y,
        width=1,
        height=1,
        filled=True,
        fillcolor=None,
        axes=(0, 0),
    ) -> None:
        """
        Parameters
        ----------
        path
            SVG path corresponding to the desired letter
        x, y : float
            x, y coordinates
        width, height : float
            letter width, height
        filled : bool
            letter is filled
        fillcolor : str
            color interpretable by plotly
        axes
            plotly axis numbers for x and y
        """
        self.path = path
        self._update_boundaries()

        self.xref = "x" + str(axes[1] + 1 if axes[1] != 0 else "")
        self.yref = "y" + str(axes[0] + 1 if axes[0] != 0 else "")
        self.x = x
        self.y = y
        self.scale_x = width / self.width
        self.scale_y = abs(height) / self.height
        self.degrees = 0 if height >= 0 else 180
        self.filled = filled
        self.fillcolor = fillcolor

    def _update_boundaries(self) -> None:
        min_x = None
        max_x = None
        min_y = None
        max_y = None
        for seg in self.path:
            if seg[0] == "H":
                if min_x is None or (seg[1] < min_x):
                    min_x = seg[1]
                if max_x is None or (seg[1] > max_x):
                    max_x = seg[1]
            elif seg[0] == "V":
                if min_y is None or (seg[1] < min_y):
                    min_y = seg[1]
                if max_y is None or (seg[1] > max_y):
                    max_y = seg[1]
            else:
                for coord_index in range(0, len(seg[1:]), 2):
                    if min_x is None or (seg[1:][coord_index] < min_x):
                        min_x = seg[1:][coord_index]
                    if max_x is None or (seg[1:][coord_index] > max_x):
                        max_x = seg[1:][coord_index]
                    if min_y is None or (seg[1:][coord_index + 1] < min_y):
                        min_y = seg[1:][coord_index + 1]
                    if max_y is None or (seg[1:][coord_index + 1] > max_y):
                        max_y = seg[1:][coord_index + 1]
        self.left = min_x
        self.right = max_x
        self.bottom = min_y
        self.top = max_y

    def shift(self, x=0, y=0):
        self.x += x
        self.y += y
        return self

    def _shift(self, x, y) -> None:
        new_path = []
        for seg in self.path:
            edited_seg = [seg[0]]
            if seg[0] == "H":
                edited_seg.append(seg[1] + x)
            elif seg[0] == "V":
                edited_seg.append(seg[1] + y)
            else:
                for coord_index in range(0, len(seg[1:]), 2):
                    edited_seg.append(seg[1:][coord_index] + x)
                    edited_seg.append(seg[1:][coord_index + 1] + y)
            new_path.append(edited_seg)

        self.path = new_path
        self._update_boundaries()

    def scale(self, x=1, y=1):
        self.scale_x *= x
        self.scale_y *= y
        return self

    def _scale(self, x, y):
        new_path = []
        for seg in self.path:
            edited_seg = [seg[0]]
            if seg[0] == "H":
                edited_seg.append(seg[1] * x)
            elif seg[0] == "V":
                edited_seg.append(seg[1] * y)
            else:
                for coord_index in range(0, len(seg[1:]), 2):
                    edited_seg.append(seg[1:][coord_index] * x)
                    edited_seg.append(seg[1:][coord_index + 1] * y)
            new_path.append(edited_seg)

        self.path = new_path
        self._update_boundaries()
        return self

    def rotate(self, degrees=0):
        self.degrees += degrees
        return self

    def _rotate(self, degrees) -> None:
        new_path = []

        pen_location = {"x": self.path[0][1], "y": self.path[0][2]}
        for seg in self.path:
            edited_seg = [seg[0]]
            if seg[0] == "H":
                pen_location["x"] += seg[1]
                new_x, new_y = self._rotate_point(degrees, seg[1], pen_location["y"])
                edited_seg.append(new_x)
            elif seg[0] == "V":
                pen_location["y"] += seg[1]
                new_x, new_y = self._rotate_point(degrees, pen_location["x"], seg[1])
                edited_seg.append(new_y)
            else:
                for coord_index in range(0, len(seg[1:]), 2):
                    pen_location["x"] = seg[1:][coord_index]
                    pen_location["y"] = seg[1:][coord_index + 1]
                    new_x, new_y = self._rotate_point(
                        degrees,
                        seg[1:][coord_index],
                        seg[1:][coord_index + 1],
                    )
                    edited_seg.append(new_x)
                    edited_seg.append(new_y)
            new_path.append(edited_seg)

        self.path = new_path
        self._update_boundaries()

    def _rotate_point(self, degrees, x, y):
        origin_x = ((self.right - self.left) / 2) + self.left
        origin_y = self.middle_y

        xt = x - origin_x
        yt = y - origin_y

        c = numpy.cos(degrees * (numpy.pi / 180))
        s = numpy.sin(degrees * (numpy.pi / 180))
        m = numpy.array([[c, s], [-s, c]])
        d = numpy.dot(m, [xt, yt])

        new_x = float(d.T[0]) + origin_x
        new_y = float(d.T[1]) + origin_y

        return new_x, new_y

    @property
    def invert(self):
        return self.rotate(180)

    @property
    def height(self):
        return self.top - self.bottom

    @property
    def width(self):
        return self.right - self.left

    @property
    def middle_x(self):
        return numpy.mean([self.left, self.right])

    @property
    def middle_y(self):
        return self.height / 2 + self.bottom

    def as_shape(self):
        """returns component for plotly display"""
        self._scale(x=self.scale_x, y=self.scale_y)
        self._shift(x=self.x, y=self.y)
        self._rotate(degrees=self.degrees)

        path = ",".join(
            [" ".join([str(sub_seg) for sub_seg in seg]) for seg in self.path],
        )
        return UnionDict(
            type="path",
            xref=self.xref,
            yref=self.yref,
            path=path,
            line={"width": 0},
            fillcolor=self.fillcolor,
        )
