#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
@Time    : 2021/4/6 5:39
@Author  : WaveletAI-Product-Team Janus
@license : (C) Copyright 2019-2022, Visionet(Tianjin)Information Technology Co.,Ltd.
@Site    : plus.xiaobodata.com
@File    : hosted_backend.py
@Desc    :
"""
import logging
import io
import os
import platform
# import retrying
from waveletai import envs
from waveletai.model import Model
from contextlib import closing
import requests
from waveletai.envs import CHUNK_SIZE, WARN_SIZE
from multiprocessing.dummy import Pool as ThreadPool
from waveletai.backend import Backend
from waveletai.utils import with_api_exceptions_handler
from waveletai.oauth import WaveletAIAuthenticator
from waveletai.dataset import Dataset, Asset
from waveletai.model import ModelVersion
from waveletai.exceptions import LoginFailed, NotADirectory, MissingNameOrPwdException, FileUploadException
from waveletai.constants import DataType, ModelRegisterMode
from waveletai.utils.storage_utils import UploadEntry, scan_upload_entries, SilentProgressIndicator, \
    LoggingProgressIndicator, upload_to_storage
from waveletai.utils.datastream import FileStream, FileChunkStream, FileChunk

_logger = logging.getLogger(__name__)

pool = ThreadPool(9)


class HostedBackend(Backend):

    # @with_api_exceptions_handler
    # def __init__(self, api_token=None):
    #     from waveletai import __version__
    #     self.client_lib_version = __version__
    #
    #     self.credentials = Credentials(api_token)
    #
    #     ssl_verify = True
    #     if os.getenv("WAI_ALLOW_SELF_SIGNED_CERTIFICATE"):
    #         urllib3.disable_warnings()
    #         ssl_verify = False
    #
    #     self._http_client = RequestsClient(ssl_verify=ssl_verify)
    #     user_agent = 'waveletai-client/{lib_version} ({system}, python {python_version})'.format(
    #         lib_version=self.client_lib_version,
    #         system=platform.platform(),
    #         python_version=platform.python_version())
    #     self._http_client.session.headers.update({'User-Agent': user_agent})
    #     self._http_client_for_token.session.headers.update({'User-Agent': user_agent})
    #     self._http_client.session.headers.update({'X-TOKEN': })
    #
    #     self.authenticator = self._create_authenticator(self.credentials.api_token, ssl_verify)
    #     self._http_client.authenticator = self.authenticator
    @with_api_exceptions_handler
    def __init__(self, name=None, pwd=None):
        if name is None:
            name = os.getenv(envs.USER_NAME)
        if pwd is None:
            pwd = os.getenv(envs.USER_PWD)
        if all([name, pwd]):
            pass
        else:
            raise MissingNameOrPwdException()
        from waveletai import __version__
        self.client_lib_version = __version__
        self._session = requests.session()
        user_agent = 'waveletai-client/{lib_version} ({system}, python {python_version})'.format(
            lib_version=self.client_lib_version,
            system=platform.platform(),
            python_version=platform.python_version())
        headers = {'User-Agent': user_agent}
        res = self._session.post(
            url=f'{self.api_address}/account/users/login/',
            json={'username': name, 'password': pwd},
            headers=headers,
        )
        if res.json()["message"]:
            raise LoginFailed(res.json()["message"])
        print("WaveletAI Backend connected")
        self._session.headers.update({"X-TOKEN": res.json()["data"]["token"]})

    @with_api_exceptions_handler
    def _create_authenticator(self, api_token, ssl_verify):
        return WaveletAIAuthenticator(api_token, ssl_verify)

    @property
    def api_address(self):
        api_url = os.getenv(envs.API_URL)
        if api_url:
            return api_url
        return "https://ai.xiaobodata.com/api"
        # return "http://localhost:3000"

    @property
    def display_address(self):
        # return self._client_config.display_url
        pass

    @with_api_exceptions_handler
    def get_dataset(self, dataset_id):
        res = self._session.get(url=f'{self.api_address}/data/dataset/{dataset_id}')
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        dict = res.json()['data']
        return Dataset(self, dict['id'], dict['name'], dict['desc'], dict['zone'], dict['dimension'], dict['json_data'],
                       dict['create_time'], dict['type'], dict['create_user_id'], dict['update_time'],
                       dict['update_user_id'])

    @with_api_exceptions_handler
    def create_dataset(self, name, zone, path, data_type=DataType.TYPE_FILE.value, desc=None):
        res = self._session.post(
            url=f'{self.api_address}/data/dataset/',
            json={'name': name, 'zone': zone, 'type': data_type, 'desc': desc},
        )
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        print(f"Dataset id = {res.json()['data']['id']} created succ")

        dict = res.json()['data']
        self.upload_dataset(dict['id'], path)
        return Dataset(self, dict['id'], dict['name'], dict['desc'], dict['zone'], dict['dimension'], dict['json_data'],
                       dict['create_time'], dict['type'], dict['create_user_id'], dict['update_time'],
                       dict['update_user_id'])

    @with_api_exceptions_handler
    def upload_dataset(self, dataset_id, path):
        """
        :param dataset_id: 文件所属数据集
        :param path: 要上传的文件夹/文件路径
        :return:  上传文件 succ，共xxx个
        """
        entries = scan_upload_entries({UploadEntry(path)})
        s_count = f_count = 0
        for entry in entries:
            fs = FileStream(entry)
            progress_indicator = SilentProgressIndicator(fs.length, fs.filename)
            prefix = entry.prefix
            res = self._upload_raw_data(api_method=f'/data/dataset/{dataset_id}/asset', file=fs,
                                        data={"prefix": prefix.replace('\\', '/')})
            if res.json()["message"]:
                f_count = f_count + 1
                _logger.error(f"file {entry.source_path} upload failed,{res.json()['message']}")
            else:
                # print(f"file {entry.source_path} upload succ")
                progress_indicator.complete()
                s_count = s_count + 1
            fs.close()
        print(f"Dataset id = {dataset_id}  upload succ: {s_count} , fail: {f_count}")

    # @with_api_exceptions_handler
    # def _url_download(self, urls):
    #     url, destination, filename = urls
    #     destination_loc = os.path.join(destination, filename)
    #     with closing(requests.get(url, stream=True)) as response:
    #         chunk_size = 1024  # 单次请求最大值
    #         content_size = int(response.headers['content-length'])  # 内容体总大小
    #         progress = ProgressBar(filename, total=content_size,
    #                                unit="KB", chunk_size=chunk_size, run_status="正在下载", fin_status="下载完成")
    #         with open(destination_loc, 'wb') as f:
    #             for data in response.iter_content(chunk_size=chunk_size):
    #                 f.write(data)
    #                 progress.refresh(count=len(data))

    @with_api_exceptions_handler
    # @retrying(stop_max_attempt_number=3)
    def _model_upload(self, entry: UploadEntry, model_repo_id):
        fs = FileStream(entry)
        progress_indicator = SilentProgressIndicator(fs.length, fs.filename)
        payload = {"path": entry.prefix, "task_id": model_repo_id}
        res = self._session.post(url=f'{self.api_address}/deploy/model/upload', files=[('file', (
            fs.filename, fs.fobj, fs.content_type))], data=payload)
        if res.json()["message"]:
            _logger.error(f"file {entry.source_path} upload failed,{res.json()['message']}")
            raise FileUploadException(fs.filename)
        else:
            progress_indicator.complete()
        fs.close()

    @with_api_exceptions_handler
    def _model_chunk_upload_loop(self, entry: UploadEntry, model_repo_id):
        stream = FileChunkStream(entry)
        progress_indicator = LoggingProgressIndicator(stream.length, stream.filename)
        for ind, fc in enumerate(stream.generate(chunk_size=CHUNK_SIZE * 1024 * 20)):
            self._model_chunk_upload(fc, entry.filename, str(ind), entry.prefix, progress_indicator, model_repo_id)

        res = self._session.get(url=f'{self.api_address}/deploy/model/upload/success',
                                params={'filename': entry.filename, 'task_id': model_repo_id, 'path': entry.prefix})
        if res.json()["message"]:
            _logger.error(f"file {entry.source_path} upload failed,{res.json()['message']}")
            raise FileUploadException(entry.filename)
        else:
            progress_indicator.complete()
        stream.close()

    # @retrying(stop_max_attempt_number=3)
    def _model_chunk_upload(self, file: FileChunk, filename, chunk, path, progress_indicator, model_repo_id):
        payload = {"chunk": chunk, "path": path, "task_id": model_repo_id}
        res = self._session.post(url=f'{self.api_address}/deploy/model/upload', files=[('file', (file.data))],
                                 data=payload)
        if res.json()["message"]:
            _logger.error(f"file {filename} upload failed,{res.json()['message']}")
            # raise FileUploadException(filename + '_' + chunk)
        else:
            progress_indicator.progress(file.end - file.start)

    @with_api_exceptions_handler
    def _url_download(self, urls):
        url, destination, filename = urls
        destination_loc = os.path.join(destination, filename)
        with closing(requests.get(url, stream=True)) as response:
            # content_size = int(response.headers['content-length'])  # 内容体总大小
            content_size = len(response.content)
            progress_indicator = SilentProgressIndicator(content_size, filename)
            if content_size >= WARN_SIZE:
                progress_indicator = LoggingProgressIndicator(content_size, filename)
            with open(destination_loc, 'wb') as f:
                for data in response.iter_content(chunk_size=CHUNK_SIZE):
                    progress_indicator.progress(CHUNK_SIZE)
                    f.write(data)
        progress_indicator.complete()

    @with_api_exceptions_handler
    def download_dataset_artifact(self, dataset_id, path, destination):
        """
        :param dataset_id:
        :param path:
        :param destination:
        :return:
        """
        dataset = self.get_dataset(dataset_id)
        dataset_type = dataset.data_type
        name = dataset.name
        self.download_dataset_artifact_(dataset_id, dataset_type, name, path, destination)

    @with_api_exceptions_handler
    def download_dataset_artifact_(self, dataset_id, type, name, path, destination):
        """
        :param type
        :param dataset_id:
        :param path: 当数据集类型为timeseries时 path传参为 None
        :param destination:
        :return:
        """
        if not destination:
            destination = os.getcwd()

        if not os.path.exists(destination):
            os.makedirs(destination)
        elif not os.path.isdir(destination):
            raise NotADirectory(destination)

        try:
            if type == 'timeseries':
                self._timeseries_download(dataset_id, name, destination)
            else:
                urls = []
                artifacts = self._list_dataset_artifacts(dataset_id)
                if '.' not in path:
                    resultinfo, dir = self._list_dir_artifacts(dataset_id, path, [], [])
                    for i in dir:
                        if not os.path.exists(destination + i):
                            os.makedirs(destination + i)
                    for asset in artifacts:
                        for res in resultinfo:
                            if asset.name in res:
                                if '/' in res:
                                    res = res.split('/' + asset.name)
                                    res = res[0]
                                urls.append((asset.path, destination + res, asset.name))
                else:
                    for asset in artifacts:
                        if path in asset.path:
                            urls.append((asset.path, destination, asset.name))
                pool.map(self._url_download, urls)
                pool.close()
                pool.join()
        except Exception as e:
            raise e

    @with_api_exceptions_handler
    def download_dataset_artifacts(self, dataset_id, destination):
        """
        :param dataset_id:
        :param destination:
        :return:
        """
        dataset = self.get_dataset(dataset_id)
        dataset_type = dataset.data_type
        name = dataset.name
        self.download_dataset_artifacts_(dataset_id, dataset_type, name, destination)

    @with_api_exceptions_handler
    def download_dataset_artifacts_(self, dataset_id, type, name, destination):
        """
        :param type 数据集类型
        :param dataset_id:
        :param destination:
        :return:
        """
        if not destination:
            destination = os.getcwd()

        if not os.path.exists(destination):
            os.makedirs(destination)
        elif not os.path.isdir(destination):
            raise NotADirectory(destination)

        try:
            if type == 'timeseries':
                self._timeseries_download(dataset_id, name, destination)
            else:
                artifacts = self._list_dataset_artifacts(dataset_id)
                resultinfo, dir = self._list_dir_artifacts(dataset_id=dataset_id, path=None, result=[], dir=[])
                urls = []
                for d in dir:
                    if not os.path.exists(destination + d):
                        os.makedirs(destination + d)
                for asset in artifacts:
                    for res in resultinfo:
                        if res in asset.path:
                            urls.append((asset.path, destination, asset.name))
                pool.map(self._url_download, list(set(urls)))
                pool.close()
                pool.join()
        except Exception as e:
            raise e

    @with_api_exceptions_handler
    def _list_dir_artifacts(self, dataset_id, path, result, dir):
        """
        递归获取所有的文件，文件夹
        :param dataset_id:
        :path 文件路径
        :return: 返回文件，图片数据集文件，文件夹列表
        """
        res = self._session.get(url=f'{self.api_address}/data/dataset/dir/{dataset_id}?path={path}')
        if not path:
            res = self._session.get(url=f'{self.api_address}/data/dataset/dir/{dataset_id}')
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        for d in res.json()['data']:
            if d['isDir'] is True:
                dir.append(d['path'])
                self._list_dir_artifacts(dataset_id, d['path'], result, dir)
            else:
                if '/' in d['path'] and '.' in d['path']:
                    dir_path = d['path'].split('/')
                    dir_path.remove(dir_path[-1])
                    dp = ''
                    for p in dir_path:
                        dp = dp + '/' + p
                    dir.append(dp)
                result.append(d['path'])
        return list(set(result)), list(set(dir))

    @with_api_exceptions_handler
    def _list_dataset_artifacts(self, dataset_id):
        """
        :param dataset_id:
        :return: 返回文件，图片数据集文件列表
        """
        res = self._session.get(url=f'{self.api_address}/data/dataset/{dataset_id}/asset?page=-1')
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        artifacts = []
        for art in res.json()['data']['data']:
            artifacts.append(
                Asset(art['id'], art['name'], art['path'], art['content_type'], art['size'], art['type'], dataset_id))
        return artifacts

    @with_api_exceptions_handler
    def _timeseries_download(self, dataset_id, name, destination):
        """
        时序数据整体下载
        :param dataset_id:
        :param destination
        :return true or false
        """
        res = self._session.get(url=f'{self.api_address}/data/dataset/filedownload/device/{dataset_id}/all')
        try:
            destination_loc = os.path.join(destination, f'{name}.csv')
            content_size = len(res.content)
            progress_indicator = SilentProgressIndicator(content_size, f'{name}.csv')
            if content_size >= WARN_SIZE:
                progress_indicator = LoggingProgressIndicator(content_size, f'{name}.csv')
            with open(destination_loc, 'wb') as f:
                for data in res.iter_content(chunk_size=CHUNK_SIZE):
                    progress_indicator.progress(CHUNK_SIZE)
                    f.write(data)
            progress_indicator.complete()
        except Exception as e:
            return e

    @with_api_exceptions_handler
    def create_model(self, app_id, name, desc):
        pass

    @with_api_exceptions_handler
    def get_model(self, model_id):
        """
        获取模型信息
        :param model_id:
        :return: Model instance
        """
        res = self._session.get(url=f'{self.api_address}/data/model/{model_id}/')
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        dict = res.json()['data']
        return Model(self, dict['id'], dict['name'], dict['desc'], dict['app_id'],
                     dict['create_time'], dict['create_user_id'], dict['update_time'],
                     dict['update_user_id'])

    @with_api_exceptions_handler
    def register_model_version(self, model_id, desc, artifacts, mode):
        """
        注册自定义模型
        :param model_id: 模型实例
        :param desc:  模型版本说明
        :param artifacts:  上传文件路径
        :param mode:  模型模式
        :return:
        """
        if mode == ModelRegisterMode.DOCKER.value:
            _logger.warning("暂不支持此功能，功能即将上线，敬请期待。")
            pass
        if mode == ModelRegisterMode.PYFUNC.value:
            res = self._session.post(
                url=f'{self.api_address}/deploy/model/',
                json={'model_id': model_id, 'desc': desc, 'mode': mode},
            )
        if res.json()["message"]:
            raise Exception(res.json()["message"])
        model_repo_id = res.json()['data']['task_id']
        entries = scan_upload_entries({UploadEntry(artifacts, "")})
        upload_to_storage(entries, self._model_upload, self._model_chunk_upload_loop,
                          warn_limit=CHUNK_SIZE * 1024 * 50, model_repo_id=model_repo_id)

        res = self._session.get(
            url=f'{self.api_address}/deploy/model/{model_repo_id}'
        )
        dict = res.json()['data']
        return ModelVersion(self, dict['id'], dict['version'], dict['desc'], dict['model_id'], dict['model_name'],
                            dict['mode'], dict['create_user_name'], dict['create_time'], dict['create_user_id'])

    @with_api_exceptions_handler
    def download_model(self, model_id, version, destination):
        pass

    # def _upload_loop(self, fun, data, progress_indicator, **kwargs):
    #     ret = None
    #     for part in data.generate():
    #         ret = with_api_exceptions_handler(self._upload_loop_chunk)(fun, part, data, **kwargs)
    #         progress_indicator.progress(part.end - part.start)
    #
    #     data.close()
    #     return ret
    #
    #
    # def _upload_loop_chunk(self, fun, part, data, **kwargs):
    #     if data.length is not None:
    #         binary_range = "bytes=%d-%d/%d" % (part.start, part.end - 1, data.length)
    #     else:
    #         binary_range = "bytes=%d-%d" % (part.start, part.end - 1)
    #     headers = {
    #         "Content-Type": "application/octet-stream",
    #         "Content-Filename": data.filename,
    #         "X-Range": binary_range,
    #     }
    #     if data.permissions is not None:
    #         headers["X-File-Permissions"] = data.permissions
    #     response = fun(data=part.get_data(), headers=headers, **kwargs)
    #     response.raise_for_status()
    #     return response

    # import requests
    #
    # url = "localhost:3000/rest/market/upload"
    #
    # payload = {'type': 'video',
    #            'task_id': 'test1',
    #            'chunk': '1'}
    # files = [
    #     ('file', ('video.mp4', open('/C:/Users/janus/Desktop/video.mp4', 'rb'), 'application/octet-stream'))
    # ]
    # headers = {
    #     'Content-Type': 'multipart/form-data'
    # }
    #
    # response = requests.request("POST", url, headers=headers, data=payload, files=files)
    #
    # print(response.text)

    def _upload_raw_data(self, api_method, file: FileStream, headers={}, data={}):
        url = self.api_address + api_method
        with open(file.fobj.name, 'rb') as f:
            res = self._session.post(url=url,
                                     files=[('file', (
                                         file.filename.replace(".\\", ""), f, file.content_type))],
                                     data=data,
                                     headers=headers,
                                     )
        return res

    def _download_raw_data(self, api_method, headers, path_params, query_params):
        url = self.api_address + api_method.operation.path_name + "?"

        for key, val in path_params.items():
            url = url.replace("{" + key + "}", val)

        for key, val in query_params.items():
            url = url + key + "=" + val + "&"

        session = self._session

        request = self.authenticator.apply(
            requests.Request(
                method='GET',
                url=url,
                headers=headers
            )
        )

        return session.send(session.prepare_request(request), stream=True)

    @with_api_exceptions_handler
    def _upload_tar_data(self, experiment, api_method, data):
        url = self.api_address + api_method.operation.path_name
        url = url.replace("{experimentId}", experiment.internal_id)

        session = self._http_client.session

        request = self.authenticator.apply(
            requests.Request(
                method='POST',
                url=url,
                data=io.BytesIO(data),
                headers={
                    "Content-Type": "application/octet-stream"
                }
            )
        )

        response = session.send(session.prepare_request(request))
        response.raise_for_status()
        return response
