import unittest

from disco import Job
from disco.core.constants import JobStatus
from disco.task import TaskResult
from mock import patch, ANY
from discomp import Pool

from discomp.tests import func_examples


class TestPool(unittest.TestCase):

    @patch('disco.job.Job.create')
    @patch('disco.job.Job.start')
    @patch('disco.job.Job.wait_for_finish')
    @patch('disco.job.Job.get_results')
    @patch('disco.job.Job.get_status')
    @patch('disco.asset.Asset.upload', return_value='file_id')
    def test_map(self, asset_upload_mock, get_status_mock, get_results_mock, wait_for_finish_mock, start_mock,
                 create_mock):
        create_mock.return_value = Job('some_job_id')
        get_results_mock.return_value = [
            TaskResult('5e4bbcbef132fa000c164798', (('result.pickle', b'\x80\x03K\x00.'),
                                                    ('DiscoTask.stdout.0.txt',
                                                     b"Collecting analytics-python==1.2.9\n  Downloading https://files.pythonhosted.org/packages/d3/37/c49d052f88655cd96445c36979fb63f69ef859e167eaff5706ca7c8a8ee3/analytics_python-1.2.9-py2.py3-none-any.whl\nCollecting astroid==2.3.3\n  Downloading https://files.pythonhosted.org/packages/ad/ae/86734823047962e7b8c8529186a1ac4a7ca19aaf1aa0c7713c022ef593fd/astroid-2.3.3-py3-none-any.whl (205kB)\nCollecting attrs==19.3.0\n  Downloading https://files.pythonhosted.org/packages/a2/db/4313ab3be961f7a763066401fb77f7748373b6094076ae2bda2806988af6/attrs-19.3.0-py2.py3-none-any.whl\nCollecting azure-core==1.2.1\n  Downloading https://files.pythonhosted.org/packages/ce/a9/83145b00f09f043610e2e1dd6486cf32876d137455723486348dc9abc074/azure_core-1.2.1-py2.py3-none-any.whl (104kB)\nCollecting azure-storage-blob==12.1.0\n  Downloading https://files.pythonhosted.org/packages/02/16/988b90dd1506c7508bf83624daa23ad9a137a637c0094ab37fd8850b7bd1/azure_storage_blob-12.1.0-py2.py3-none-any.whl (272kB)\nCollecting certifi==2019.11.28\n  Downloading https://files.pythonhosted.org/packages/b9/63/df50cac98ea0d5b006c55a399c3bf1db9da7b5a24de7890bc9cfd5dd9e99/certifi-2019.11.28-py2.py3-none-any.whl (156kB)\nCollecting cffi==1.13.2\n  Downloading https://files.pythonhosted.org/packages/49/72/0d42f94fe94afa8030350c26e9d787219f3f008ec9bf6b86c66532b29236/cffi-1.13.2-cp36-cp36m-manylinux1_x86_64.whl (397kB)\nCollecting chardet==3.0.4\n  Downloading https://files.pythonhosted.org/packages/bc/a9/01ffebfb562e4274b6487b4bb1ddec7ca55ec7510b22e4c51f14098443b8/chardet-3.0.4-py2.py3-none-any.whl (133kB)\nCollecting Click==7.0\n  Downloading https://files.pythonhosted.org/packages/fa/37/45185cb5abbc30d7257104c434fe0b07e5a195a6847506c074527aa599ec/Click-7.0-py2.py3-none-any.whl (81kB)\nCollecting cryptography==2.8\n  Downloading https://files.pythonhosted.org/packages/ca/9a/7cece52c46546e214e10811b36b2da52ce1ea7fa203203a629b8dfadad53/cryptography-2.8-cp34-abi3-manylinux2010_x86_64.whl (2.3MB)\nCollecting decorator==4.4.1\n  Downloading https://files.pythonhosted.org/packages/8f/b7/f329cfdc75f3d28d12c65980e4469e2fa373f1953f5df6e370e84ea2e875/decorator-4.4.1-py2.py3-none-any.whl\nCollecting dill==0.3.1.1\n  Downloading https://files.pythonhosted.org/packages/c7/11/345f3173809cea7f1a193bfbf02403fff250a3360e0e118a1630985e547d/dill-0.3.1.1.tar.gz (151kB)\nCollecting idna==2.8\n  Downloading https://files.pythonhosted.org/packages/14/2c/cd551d81dbe15200be1cf41cd03869a46fe7226e7450af7a6545bfc474c9/idna-2.8-py2.py3-none-any.whl (58kB)\nCollecting importlib-metadata==1.3.0\n  Downloading https://files.pythonhosted.org/packages/e9/71/1a1e0ed0981bb6a67bce55a210f168126b7ebd2065958673797ea66489ca/importlib_metadata-1.3.0-py2.py3-none-any.whl\nCollecting isodate==0.6.0\n  Downloading https://files.pythonhosted.org/packages/9b/9f/b36f7774ff5ea8e428fdcfc4bb332c39ee5b9362ddd3d40d9516a55221b2/isodate-0.6.0-py2.py3-none-any.whl (45kB)\nCollecting isort==4.3.21\n  Downloading https://files.pythonhosted.org/packages/e5/b0/c121fd1fa3419ea9bfd55c7f9c4fedfec5143208d8c7ad3ce3db6c623c21/isort-4.3.21-py2.py3-none-any.whl (42kB)\nCollecting lazy-object-proxy==1.4.3\n  Downloading https://files.pythonhosted.org/packages/0b/dd/b1e3407e9e6913cf178e506cd0dee818e58694d9a5cd1984e3f6a8b9a10f/lazy_object_proxy-1.4.3-cp36-cp36m-manylinux1_x86_64.whl (55kB)\nCollecting mccabe==0.6.1\n  Downloading https://files.pythonhosted.org/packages/87/89/479dc97e18549e21354893e4ee4ef36db1d237534982482c3681ee6e7b57/mccabe-0.6.1-py2.py3-none-any.whl\nCollecting mock==3.0.5\n  Downloading https://files.pythonhosted.org/packages/05/d2/f94e68be6b17f46d2c353564da56e6fb89ef09faeeff3313a046cb810ca9/mock-3.0.5-py2.py3-none-any.whl\nCollecting more-itertools==8.0.2\n  Downloading https://files.pythonhosted.org/packages/68/03/0604cec1ea13c9f063dd50f900d1a36160334dd3cfb01fd0e638f61b46ba/more_itertools-8.0.2-py3-none-any.whl (40kB)\nCollecting msrest==0.6.11\n  Downloading https://files.pythonhosted.org/packages/f1/fe/a6786f141181a8f0579e4d7ed048812b1c395a6c265a46fc478be03ff4c1/msrest-0.6.11-py2.py3-none-any.whl (83kB)\nCollecting oauthlib==3.1.0\n  Downloading https://files.pythonhosted.org/packages/05/57/ce2e7a8fa7c0afb54a0581b14a65b56e62b5759dbc98e80627142b8a3704/oauthlib-3.1.0-py2.py3-none-any.whl (147kB)\nCollecting packaging==19.2\n  Downloading https://files.pythonhosted.org/packages/cf/94/9672c2d4b126e74c4496c6b3c58a8b51d6419267be9e70660ba23374c875/packaging-19.2-py2.py3-none-any.whl\nCollecting pluggy==0.13.1\n  Downloading https://files.pythonhosted.org/packages/a0/28/85c7aa31b80d150b772fbe4a229487bc6644da9ccb7e427dd8cc60cb8a62/pluggy-0.13.1-py2.py3-none-any.whl\nCollecting py==1.8.0\n  Downloading https://files.pythonhosted.org/packages/76/bc/394ad449851729244a97857ee14d7cba61ddb268dce3db538ba2f2ba1f0f/py-1.8.0-py2.py3-none-any.whl (83kB)\nCollecting pycparser==2.19\n  Downloading https://files.pythonhosted.org/packages/68/9e/49196946aee219aead1290e00d1e7fdeab8567783e83e1b9ab5585e6206a/pycparser-2.19.tar.gz (158kB)\nCollecting pylint==2.4.4\n  Downloading https://files.pythonhosted.org/packages/e9/59/43fc36c5ee316bb9aeb7cf5329cdbdca89e5749c34d5602753827c0aa2dc/pylint-2.4.4-py3-none-any.whl (302kB)\nCollecting pyparsing==2.4.5\n  Downloading https://files.pythonhosted.org/packages/c0/0c/fc2e007d9a992d997f04a80125b0f183da7fb554f1de701bbb70a8e7d479/pyparsing-2.4.5-py2.py3-none-any.whl (67kB)\nCollecting pytest==5.3.2\n  Downloading https://files.pythonhosted.org/packages/19/cf/711f1d887cb92c5155c9a1eb338f1b5d2411b50e4492a3b20e4a188a22b2/pytest-5.3.2-py3-none-any.whl (234kB)\nCollecting python-dateutil==2.8.1\n  Downloading https://files.pythonhosted.org/packages/d4/70/d60450c3dd48ef87586924207ae8907090de0b306af2bce5d134d78615cb/python_dateutil-2.8.1-py2.py3-none-any.whl (227kB)\nCollecting requests==2.22.0\n  Downloading https://files.pythonhosted.org/packages/51/bd/23c926cd341ea6b7dd0b2a00aba99ae0f828be89d72b2190f27c11d4b7fb/requests-2.22.0-py2.py3-none-any.whl (57kB)\nCollecting requests-oauthlib==1.3.0\n  Downloading https://files.pythonhosted.org/packages/a3/12/b92740d845ab62ea4edf04d2f4164d82532b5a0b03836d4d4e71c6f3d379/requests_oauthlib-1.3.0-py2.py3-none-any.whl\nCollecting requests-toolbelt==0.9.1\n  Downloading https://files.pythonhosted.org/packages/60/ef/7681134338fc097acef8d9b2f8abe0458e4d87559c689a8c306d0957ece5/requests_toolbelt-0.9.1-py2.py3-none-any.whl (54kB)\nCollecting six==1.13.0\n  Downloading https://files.pythonhosted.org/packages/65/26/32b8464df2a97e6dd1b656ed26b2c194606c16fe163c695a992b36c11cdf/six-1.13.0-py2.py3-none-any.whl\nCollecting tabulate==0.8.6\n  Downloading https://files.pythonhosted.org/packages/c4/41/523f6a05e6dc3329a5660f6a81254c6cd87e5cfb5b7482bae3391d86ec3a/tabulate-0.8.6.tar.gz (45kB)\nCollecting tqdm==4.41.0\n  Downloading https://files.pythonhosted.org/packages/8c/c3/d049cf3fb31094ee045ec1ee29fffac218c91e82c8838c49ab4c3e52627b/tqdm-4.41.0-py2.py3-none-any.whl (56kB)\nCollecting typed-ast==1.4.0\n  Downloading https://files.pythonhosted.org/packages/31/d3/9d1802c161626d0278bafb1ffb32f76b9d01e123881bbf9d91e8ccf28e18/typed_ast-1.4.0-cp36-cp36m-manylinux1_x86_64.whl (736kB)\nCollecting urllib3==1.25.7\n  Downloading https://files.pythonhosted.org/packages/b4/40/a9837291310ee1ccc242ceb6ebfd9eb21539649f193a7c8c86ba15b98539/urllib3-1.25.7-py2.py3-none-any.whl (125kB)\nCollecting wcwidth==0.1.7\n  Downloading https://files.pythonhosted.org/packages/7e/9f/526a6947247599b084ee5232e4f9190a38f398d7300d866af3ab571a5bfe/wcwidth-0.1.7-py2.py3-none-any.whl\nCollecting wrapt==1.11.2\n  Downloading https://files.pythonhosted.org/packages/23/84/323c2415280bc4fc880ac5050dddfb3c8062c2552b34c2e512eb4aa68f79/wrapt-1.11.2.tar.gz\nCollecting zipp==0.6.0\n  Downloading https://files.pythonhosted.org/packages/74/3d/1ee25a26411ba0401b43c6376d2316a71addcc72ef8690b101b4ea56d76a/zipp-0.6.0-py2.py3-none-any.whl\nCollecting discomp\n  Downloading https://files.pythonhosted.org/packages/72/bc/556cf941a790c7396020ba7f1c8c924b95bc930e22fc21db81622e8c68bf/discomp-1.7.0-py3-none-any.whl\nBuilding wheels for collected packages: dill, pycparser, tabulate, wrapt\n  Building wheel for dill (setup.py): started\n  Building wheel for dill (setup.py): finished with status 'done'\n  Created wheel for dill: filename=dill-0.3.1.1-cp36-none-any.whl size=78532 sha256=d847e624b1f83f3589883cae4c3882dcaa95fa4bf9857e65736a6269e96381e9\n  Stored in directory: /root/.cache/pip/wheels/59/b1/91/f02e76c732915c4015ab4010f3015469866c1eb9b14058d8e7\n  Building wheel for pycparser (setup.py): started\n  Building wheel for pycparser (setup.py): finished with status 'done'\n  Created wheel for pycparser: filename=pycparser-2.19-py2.py3-none-any.whl size=111029 sha256=0dbf218258a09929d3f85fe5d501a059ad05973fa86fb10e660f19d6404f88ec\n  Stored in directory: /root/.cache/pip/wheels/f2/9a/90/de94f8556265ddc9d9c8b271b0f63e57b26fb1d67a45564511\n  Building wheel for tabulate (setup.py): started\n  Building wheel for tabulate (setup.py): finished with status 'done'\n  Created wheel for tabulate: filename=tabulate-0.8.6-cp36-none-any.whl size=23271 sha256=2236db1dbe134160157ad4f9218559c6abc9d53f8fdacc08e03674d7addf7811\n  Stored in directory: /root/.cache/pip/wheels/9c/9b/f4/eb243fdb89676ec00588e8c54bb54360724c06e7fafe95278e\n  Building wheel for wrapt (setup.py): started\n  Building wheel for wrapt (setup.py): finished with status 'done'\n  Created wheel for wrapt: filename=wrapt-1.11.2-cp36-cp36m-linux_x86_64.whl size=75254 sha256=db1cbd15a6de4fa63889d658ca83cdafbc1e81f2203fc0051bf60a4c83677428\n  Stored in directory: /root/.cache/pip/wheels/d7/de/2e/efa132238792efb6459a96e85916ef8597fcb3d2ae51590dfd\nSuccessfully built dill pycparser tabulate wrapt\nInstalling collected packages: six, python-dateutil, idna, urllib3, chardet, certifi, requests, analytics-python, lazy-object-proxy, wrapt, typed-ast, astroid, attrs, azure-core, pycparser, cffi, cryptography, isodate, oauthlib, requests-oauthlib, msrest, azure-storage-blob, Click, decorator, dill, more-itertools, zipp, importlib-metadata, isort, mccabe, mock, pyparsing, packaging, pluggy, py, pylint, wcwidth, pytest, requests-toolbelt, tabulate, tqdm, discomp\nSuccessfully installed Click-7.0 analytics-python-1.2.9 astroid-2.3.3 attrs-19.3.0 azure-core-1.2.1 azure-storage-blob-12.1.0 certifi-2019.11.28 cffi-1.13.2 chardet-3.0.4 cryptography-2.8 decorator-4.4.1 dill-0.3.1.1 discomp-1.7.0 idna-2.8 importlib-metadata-1.3.0 isodate-0.6.0 isort-4.3.21 lazy-object-proxy-1.4.3 mccabe-0.6.1 mock-3.0.5 more-itertools-8.0.2 msrest-0.6.11 oauthlib-3.1.0 packaging-19.2 pluggy-0.13.1 py-1.8.0 pycparser-2.19 pylint-2.4.4 pyparsing-2.4.5 pytest-5.3.2 python-dateutil-2.8.1 requests-2.22.0 requests-oauthlib-1.3.0 requests-toolbelt-0.9.1 six-1.13.0 tabulate-0.8.6 tqdm-4.41.0 typed-ast-1.4.0 urllib3-1.25.7 wcwidth-0.1.7 wrapt-1.11.2 zipp-0.6.0\n0\n"),
                                                    ('DiscoTask.stderr.0.txt',
                                                     b"WARNING: You are using pip version 19.3.1; however, version 20.0.2 is available.\nYou should consider upgrading via the 'pip install --upgrade pip' command.\n")))
        ]
        get_status_mock.return_value = JobStatus.success
        mypool = Pool()
        results = mypool.map(func_examples.pow3, range(10))
        create_mock.assert_called_with(cluster_id=None, cluster_instance_type='s',
                                       constants_file_ids=['file_id', 'file_id'],
                                       input_file_ids=['file_id', 'file_id', 'file_id', 'file_id', 'file_id', 'file_id',
                                                       'file_id', 'file_id', 'file_id', 'file_id'],
                                       job_name=ANY, script_file_id='file_id', upload_requirements_file=False)
        assert results == [0]
        assert asset_upload_mock.call_count == 13

    @patch('disco.job.Job.create')
    @patch('disco.job.Job.start')
    @patch('disco.job.Job.wait_for_finish')
    @patch('disco.job.Job.get_results')
    @patch('disco.job.Job.get_status')
    @patch('disco.asset.Asset.upload', return_value='file_id')
    def test_map_empty_args(self, asset_upload_mock, get_status_mock, get_results_mock, wait_for_finish_mock,
                            start_mock,
                            create_mock):
        create_mock.return_value = Job('some_job_id')
        get_results_mock.return_value = [
            TaskResult('5e4bbcbef132fa000c164798', (('DiscoTask.stdout.0.txt',
                                                     b"Collecting analytics-python==1.2.9\n  Downloading https://files.pythonhosted.org/packages/d3/37/c49d052f88655cd96445c36979fb63f69ef859e167eaff5706ca7c8a8ee3/analytics_python-1.2.9-py2.py3-none-any.whl\nCollecting astroid==2.3.3\n  Downloading https://files.pythonhosted.org/packages/ad/ae/86734823047962e7b8c8529186a1ac4a7ca19aaf1aa0c7713c022ef593fd/astroid-2.3.3-py3-none-any.whl (205kB)\nCollecting attrs==19.3.0\n  Downloading https://files.pythonhosted.org/packages/a2/db/4313ab3be961f7a763066401fb77f7748373b6094076ae2bda2806988af6/attrs-19.3.0-py2.py3-none-any.whl\nCollecting azure-core==1.2.1\n  Downloading https://files.pythonhosted.org/packages/ce/a9/83145b00f09f043610e2e1dd6486cf32876d137455723486348dc9abc074/azure_core-1.2.1-py2.py3-none-any.whl (104kB)\nCollecting azure-storage-blob==12.1.0\n  Downloading https://files.pythonhosted.org/packages/02/16/988b90dd1506c7508bf83624daa23ad9a137a637c0094ab37fd8850b7bd1/azure_storage_blob-12.1.0-py2.py3-none-any.whl (272kB)\nCollecting certifi==2019.11.28\n  Downloading https://files.pythonhosted.org/packages/b9/63/df50cac98ea0d5b006c55a399c3bf1db9da7b5a24de7890bc9cfd5dd9e99/certifi-2019.11.28-py2.py3-none-any.whl (156kB)\nCollecting cffi==1.13.2\n  Downloading https://files.pythonhosted.org/packages/49/72/0d42f94fe94afa8030350c26e9d787219f3f008ec9bf6b86c66532b29236/cffi-1.13.2-cp36-cp36m-manylinux1_x86_64.whl (397kB)\nCollecting chardet==3.0.4\n  Downloading https://files.pythonhosted.org/packages/bc/a9/01ffebfb562e4274b6487b4bb1ddec7ca55ec7510b22e4c51f14098443b8/chardet-3.0.4-py2.py3-none-any.whl (133kB)\nCollecting Click==7.0\n  Downloading https://files.pythonhosted.org/packages/fa/37/45185cb5abbc30d7257104c434fe0b07e5a195a6847506c074527aa599ec/Click-7.0-py2.py3-none-any.whl (81kB)\nCollecting cryptography==2.8\n  Downloading https://files.pythonhosted.org/packages/ca/9a/7cece52c46546e214e10811b36b2da52ce1ea7fa203203a629b8dfadad53/cryptography-2.8-cp34-abi3-manylinux2010_x86_64.whl (2.3MB)\nCollecting decorator==4.4.1\n  Downloading https://files.pythonhosted.org/packages/8f/b7/f329cfdc75f3d28d12c65980e4469e2fa373f1953f5df6e370e84ea2e875/decorator-4.4.1-py2.py3-none-any.whl\nCollecting dill==0.3.1.1\n  Downloading https://files.pythonhosted.org/packages/c7/11/345f3173809cea7f1a193bfbf02403fff250a3360e0e118a1630985e547d/dill-0.3.1.1.tar.gz (151kB)\nCollecting idna==2.8\n  Downloading https://files.pythonhosted.org/packages/14/2c/cd551d81dbe15200be1cf41cd03869a46fe7226e7450af7a6545bfc474c9/idna-2.8-py2.py3-none-any.whl (58kB)\nCollecting importlib-metadata==1.3.0\n  Downloading https://files.pythonhosted.org/packages/e9/71/1a1e0ed0981bb6a67bce55a210f168126b7ebd2065958673797ea66489ca/importlib_metadata-1.3.0-py2.py3-none-any.whl\nCollecting isodate==0.6.0\n  Downloading https://files.pythonhosted.org/packages/9b/9f/b36f7774ff5ea8e428fdcfc4bb332c39ee5b9362ddd3d40d9516a55221b2/isodate-0.6.0-py2.py3-none-any.whl (45kB)\nCollecting isort==4.3.21\n  Downloading https://files.pythonhosted.org/packages/e5/b0/c121fd1fa3419ea9bfd55c7f9c4fedfec5143208d8c7ad3ce3db6c623c21/isort-4.3.21-py2.py3-none-any.whl (42kB)\nCollecting lazy-object-proxy==1.4.3\n  Downloading https://files.pythonhosted.org/packages/0b/dd/b1e3407e9e6913cf178e506cd0dee818e58694d9a5cd1984e3f6a8b9a10f/lazy_object_proxy-1.4.3-cp36-cp36m-manylinux1_x86_64.whl (55kB)\nCollecting mccabe==0.6.1\n  Downloading https://files.pythonhosted.org/packages/87/89/479dc97e18549e21354893e4ee4ef36db1d237534982482c3681ee6e7b57/mccabe-0.6.1-py2.py3-none-any.whl\nCollecting mock==3.0.5\n  Downloading https://files.pythonhosted.org/packages/05/d2/f94e68be6b17f46d2c353564da56e6fb89ef09faeeff3313a046cb810ca9/mock-3.0.5-py2.py3-none-any.whl\nCollecting more-itertools==8.0.2\n  Downloading https://files.pythonhosted.org/packages/68/03/0604cec1ea13c9f063dd50f900d1a36160334dd3cfb01fd0e638f61b46ba/more_itertools-8.0.2-py3-none-any.whl (40kB)\nCollecting msrest==0.6.11\n  Downloading https://files.pythonhosted.org/packages/f1/fe/a6786f141181a8f0579e4d7ed048812b1c395a6c265a46fc478be03ff4c1/msrest-0.6.11-py2.py3-none-any.whl (83kB)\nCollecting oauthlib==3.1.0\n  Downloading https://files.pythonhosted.org/packages/05/57/ce2e7a8fa7c0afb54a0581b14a65b56e62b5759dbc98e80627142b8a3704/oauthlib-3.1.0-py2.py3-none-any.whl (147kB)\nCollecting packaging==19.2\n  Downloading https://files.pythonhosted.org/packages/cf/94/9672c2d4b126e74c4496c6b3c58a8b51d6419267be9e70660ba23374c875/packaging-19.2-py2.py3-none-any.whl\nCollecting pluggy==0.13.1\n  Downloading https://files.pythonhosted.org/packages/a0/28/85c7aa31b80d150b772fbe4a229487bc6644da9ccb7e427dd8cc60cb8a62/pluggy-0.13.1-py2.py3-none-any.whl\nCollecting py==1.8.0\n  Downloading https://files.pythonhosted.org/packages/76/bc/394ad449851729244a97857ee14d7cba61ddb268dce3db538ba2f2ba1f0f/py-1.8.0-py2.py3-none-any.whl (83kB)\nCollecting pycparser==2.19\n  Downloading https://files.pythonhosted.org/packages/68/9e/49196946aee219aead1290e00d1e7fdeab8567783e83e1b9ab5585e6206a/pycparser-2.19.tar.gz (158kB)\nCollecting pylint==2.4.4\n  Downloading https://files.pythonhosted.org/packages/e9/59/43fc36c5ee316bb9aeb7cf5329cdbdca89e5749c34d5602753827c0aa2dc/pylint-2.4.4-py3-none-any.whl (302kB)\nCollecting pyparsing==2.4.5\n  Downloading https://files.pythonhosted.org/packages/c0/0c/fc2e007d9a992d997f04a80125b0f183da7fb554f1de701bbb70a8e7d479/pyparsing-2.4.5-py2.py3-none-any.whl (67kB)\nCollecting pytest==5.3.2\n  Downloading https://files.pythonhosted.org/packages/19/cf/711f1d887cb92c5155c9a1eb338f1b5d2411b50e4492a3b20e4a188a22b2/pytest-5.3.2-py3-none-any.whl (234kB)\nCollecting python-dateutil==2.8.1\n  Downloading https://files.pythonhosted.org/packages/d4/70/d60450c3dd48ef87586924207ae8907090de0b306af2bce5d134d78615cb/python_dateutil-2.8.1-py2.py3-none-any.whl (227kB)\nCollecting requests==2.22.0\n  Downloading https://files.pythonhosted.org/packages/51/bd/23c926cd341ea6b7dd0b2a00aba99ae0f828be89d72b2190f27c11d4b7fb/requests-2.22.0-py2.py3-none-any.whl (57kB)\nCollecting requests-oauthlib==1.3.0\n  Downloading https://files.pythonhosted.org/packages/a3/12/b92740d845ab62ea4edf04d2f4164d82532b5a0b03836d4d4e71c6f3d379/requests_oauthlib-1.3.0-py2.py3-none-any.whl\nCollecting requests-toolbelt==0.9.1\n  Downloading https://files.pythonhosted.org/packages/60/ef/7681134338fc097acef8d9b2f8abe0458e4d87559c689a8c306d0957ece5/requests_toolbelt-0.9.1-py2.py3-none-any.whl (54kB)\nCollecting six==1.13.0\n  Downloading https://files.pythonhosted.org/packages/65/26/32b8464df2a97e6dd1b656ed26b2c194606c16fe163c695a992b36c11cdf/six-1.13.0-py2.py3-none-any.whl\nCollecting tabulate==0.8.6\n  Downloading https://files.pythonhosted.org/packages/c4/41/523f6a05e6dc3329a5660f6a81254c6cd87e5cfb5b7482bae3391d86ec3a/tabulate-0.8.6.tar.gz (45kB)\nCollecting tqdm==4.41.0\n  Downloading https://files.pythonhosted.org/packages/8c/c3/d049cf3fb31094ee045ec1ee29fffac218c91e82c8838c49ab4c3e52627b/tqdm-4.41.0-py2.py3-none-any.whl (56kB)\nCollecting typed-ast==1.4.0\n  Downloading https://files.pythonhosted.org/packages/31/d3/9d1802c161626d0278bafb1ffb32f76b9d01e123881bbf9d91e8ccf28e18/typed_ast-1.4.0-cp36-cp36m-manylinux1_x86_64.whl (736kB)\nCollecting urllib3==1.25.7\n  Downloading https://files.pythonhosted.org/packages/b4/40/a9837291310ee1ccc242ceb6ebfd9eb21539649f193a7c8c86ba15b98539/urllib3-1.25.7-py2.py3-none-any.whl (125kB)\nCollecting wcwidth==0.1.7\n  Downloading https://files.pythonhosted.org/packages/7e/9f/526a6947247599b084ee5232e4f9190a38f398d7300d866af3ab571a5bfe/wcwidth-0.1.7-py2.py3-none-any.whl\nCollecting wrapt==1.11.2\n  Downloading https://files.pythonhosted.org/packages/23/84/323c2415280bc4fc880ac5050dddfb3c8062c2552b34c2e512eb4aa68f79/wrapt-1.11.2.tar.gz\nCollecting zipp==0.6.0\n  Downloading https://files.pythonhosted.org/packages/74/3d/1ee25a26411ba0401b43c6376d2316a71addcc72ef8690b101b4ea56d76a/zipp-0.6.0-py2.py3-none-any.whl\nCollecting discomp\n  Downloading https://files.pythonhosted.org/packages/72/bc/556cf941a790c7396020ba7f1c8c924b95bc930e22fc21db81622e8c68bf/discomp-1.7.0-py3-none-any.whl\nBuilding wheels for collected packages: dill, pycparser, tabulate, wrapt\n  Building wheel for dill (setup.py): started\n  Building wheel for dill (setup.py): finished with status 'done'\n  Created wheel for dill: filename=dill-0.3.1.1-cp36-none-any.whl size=78532 sha256=b63613df2dc3a3f0cf8facaa645af14ea90d6936117f1b8f8653b6aab1d15b70\n  Stored in directory: /root/.cache/pip/wheels/59/b1/91/f02e76c732915c4015ab4010f3015469866c1eb9b14058d8e7\n  Building wheel for pycparser (setup.py): started\n  Building wheel for pycparser (setup.py): finished with status 'done'\n  Created wheel for pycparser: filename=pycparser-2.19-py2.py3-none-any.whl size=111029 sha256=6b17792d51de34e6144da5c7782e0aa0a37b8b5c9e1280ae956c3582c510dc0c\n  Stored in directory: /root/.cache/pip/wheels/f2/9a/90/de94f8556265ddc9d9c8b271b0f63e57b26fb1d67a45564511\n  Building wheel for tabulate (setup.py): started\n  Building wheel for tabulate (setup.py): finished with status 'done'\n  Created wheel for tabulate: filename=tabulate-0.8.6-cp36-none-any.whl size=23271 sha256=452968bc77556d08afdbc4e01002cdf496042efa02fd49cdb0858c553ff104b0\n  Stored in directory: /root/.cache/pip/wheels/9c/9b/f4/eb243fdb89676ec00588e8c54bb54360724c06e7fafe95278e\n  Building wheel for wrapt (setup.py): started\n  Building wheel for wrapt (setup.py): finished with status 'done'\n  Created wheel for wrapt: filename=wrapt-1.11.2-cp36-cp36m-linux_x86_64.whl size=75256 sha256=ec7082fdbaff9b4dd396dc37c696ffc58890a17348996a2faa2f41acd68ec40d\n  Stored in directory: /root/.cache/pip/wheels/d7/de/2e/efa132238792efb6459a96e85916ef8597fcb3d2ae51590dfd\nSuccessfully built dill pycparser tabulate wrapt\nInstalling collected packages: six, python-dateutil, urllib3, idna, certifi, chardet, requests, analytics-python, typed-ast, lazy-object-proxy, wrapt, astroid, attrs, azure-core, oauthlib, requests-oauthlib, isodate, msrest, pycparser, cffi, cryptography, azure-storage-blob, Click, decorator, dill, more-itertools, zipp, importlib-metadata, isort, mccabe, mock, pyparsing, packaging, pluggy, py, pylint, wcwidth, pytest, requests-toolbelt, tabulate, tqdm, discomp\nSuccessfully installed Click-7.0 analytics-python-1.2.9 astroid-2.3.3 attrs-19.3.0 azure-core-1.2.1 azure-storage-blob-12.1.0 certifi-2019.11.28 cffi-1.13.2 chardet-3.0.4 cryptography-2.8 decorator-4.4.1 dill-0.3.1.1 discomp-1.7.0 idna-2.8 importlib-metadata-1.3.0 isodate-0.6.0 isort-4.3.21 lazy-object-proxy-1.4.3 mccabe-0.6.1 mock-3.0.5 more-itertools-8.0.2 msrest-0.6.11 oauthlib-3.1.0 packaging-19.2 pluggy-0.13.1 py-1.8.0 pycparser-2.19 pylint-2.4.4 pyparsing-2.4.5 pytest-5.3.2 python-dateutil-2.8.1 requests-2.22.0 requests-oauthlib-1.3.0 requests-toolbelt-0.9.1 six-1.13.0 tabulate-0.8.6 tqdm-4.41.0 typed-ast-1.4.0 urllib3-1.25.7 wcwidth-0.1.7 wrapt-1.11.2 zipp-0.6.0\n"),
                                                    ('DiscoTask.stderr.0.txt',
                                                     b'WARNING: You are using pip version 19.3.1; however, version 20.0.2 is available.\nYou should consider upgrading via the \'pip install --upgrade pip\' command.\nTraceback (most recent call last):\n  File "hello1582028783.374274-script.py", line 9, in <module>\n    result = func(arg)\nTypeError: hello() takes 0 positional arguments but 1 was given\n')))
        ]
        get_status_mock.return_value = JobStatus.success
        mypool = Pool()
        results = mypool.map(func_examples.hello, '')
        create_mock.assert_called_with(cluster_id=None, cluster_instance_type='s',
                                       constants_file_ids=['file_id', 'file_id'],
                                       input_file_ids=[],
                                       job_name=ANY, script_file_id='file_id', upload_requirements_file=False)
        assert results == []
        assert asset_upload_mock.call_count == 3

    @patch('disco.job.Job.create')
    @patch('disco.job.Job.start')
    @patch('disco.job.Job.wait_for_finish')
    @patch('disco.job.Job.get_results')
    @patch('disco.job.Job.get_status')
    @patch('disco.asset.Asset.upload', return_value='file_id')
    def test_starmap_with_files_and_context_manager(self, asset_upload_mock, get_status_mock, get_results_mock,
                                                    wait_for_finish_mock,
                                                    start_mock,
                                                    create_mock):
        create_mock.return_value = Job('some_job_id')
        get_results_mock.return_value = [
            TaskResult('5e4bbcbef132fa000c164798', (('result.pickle', b'\x80\x03K\x00.'),
                                                    ('DiscoTask.stdout.0.txt',
                                                     b"Collecting analytics-python==1.2.9\n  Downloading https://files.pythonhosted.org/packages/d3/37/c49d052f88655cd96445c36979fb63f69ef859e167eaff5706ca7c8a8ee3/analytics_python-1.2.9-py2.py3-none-any.whl\nCollecting astroid==2.3.3\n  Downloading https://files.pythonhosted.org/packages/ad/ae/86734823047962e7b8c8529186a1ac4a7ca19aaf1aa0c7713c022ef593fd/astroid-2.3.3-py3-none-any.whl (205kB)\nCollecting attrs==19.3.0\n  Downloading https://files.pythonhosted.org/packages/a2/db/4313ab3be961f7a763066401fb77f7748373b6094076ae2bda2806988af6/attrs-19.3.0-py2.py3-none-any.whl\nCollecting azure-core==1.2.1\n  Downloading https://files.pythonhosted.org/packages/ce/a9/83145b00f09f043610e2e1dd6486cf32876d137455723486348dc9abc074/azure_core-1.2.1-py2.py3-none-any.whl (104kB)\nCollecting azure-storage-blob==12.1.0\n  Downloading https://files.pythonhosted.org/packages/02/16/988b90dd1506c7508bf83624daa23ad9a137a637c0094ab37fd8850b7bd1/azure_storage_blob-12.1.0-py2.py3-none-any.whl (272kB)\nCollecting certifi==2019.11.28\n  Downloading https://files.pythonhosted.org/packages/b9/63/df50cac98ea0d5b006c55a399c3bf1db9da7b5a24de7890bc9cfd5dd9e99/certifi-2019.11.28-py2.py3-none-any.whl (156kB)\nCollecting cffi==1.13.2\n  Downloading https://files.pythonhosted.org/packages/49/72/0d42f94fe94afa8030350c26e9d787219f3f008ec9bf6b86c66532b29236/cffi-1.13.2-cp36-cp36m-manylinux1_x86_64.whl (397kB)\nCollecting chardet==3.0.4\n  Downloading https://files.pythonhosted.org/packages/bc/a9/01ffebfb562e4274b6487b4bb1ddec7ca55ec7510b22e4c51f14098443b8/chardet-3.0.4-py2.py3-none-any.whl (133kB)\nCollecting Click==7.0\n  Downloading https://files.pythonhosted.org/packages/fa/37/45185cb5abbc30d7257104c434fe0b07e5a195a6847506c074527aa599ec/Click-7.0-py2.py3-none-any.whl (81kB)\nCollecting cryptography==2.8\n  Downloading https://files.pythonhosted.org/packages/ca/9a/7cece52c46546e214e10811b36b2da52ce1ea7fa203203a629b8dfadad53/cryptography-2.8-cp34-abi3-manylinux2010_x86_64.whl (2.3MB)\nCollecting decorator==4.4.1\n  Downloading https://files.pythonhosted.org/packages/8f/b7/f329cfdc75f3d28d12c65980e4469e2fa373f1953f5df6e370e84ea2e875/decorator-4.4.1-py2.py3-none-any.whl\nCollecting dill==0.3.1.1\n  Downloading https://files.pythonhosted.org/packages/c7/11/345f3173809cea7f1a193bfbf02403fff250a3360e0e118a1630985e547d/dill-0.3.1.1.tar.gz (151kB)\nCollecting idna==2.8\n  Downloading https://files.pythonhosted.org/packages/14/2c/cd551d81dbe15200be1cf41cd03869a46fe7226e7450af7a6545bfc474c9/idna-2.8-py2.py3-none-any.whl (58kB)\nCollecting importlib-metadata==1.3.0\n  Downloading https://files.pythonhosted.org/packages/e9/71/1a1e0ed0981bb6a67bce55a210f168126b7ebd2065958673797ea66489ca/importlib_metadata-1.3.0-py2.py3-none-any.whl\nCollecting isodate==0.6.0\n  Downloading https://files.pythonhosted.org/packages/9b/9f/b36f7774ff5ea8e428fdcfc4bb332c39ee5b9362ddd3d40d9516a55221b2/isodate-0.6.0-py2.py3-none-any.whl (45kB)\nCollecting isort==4.3.21\n  Downloading https://files.pythonhosted.org/packages/e5/b0/c121fd1fa3419ea9bfd55c7f9c4fedfec5143208d8c7ad3ce3db6c623c21/isort-4.3.21-py2.py3-none-any.whl (42kB)\nCollecting lazy-object-proxy==1.4.3\n  Downloading https://files.pythonhosted.org/packages/0b/dd/b1e3407e9e6913cf178e506cd0dee818e58694d9a5cd1984e3f6a8b9a10f/lazy_object_proxy-1.4.3-cp36-cp36m-manylinux1_x86_64.whl (55kB)\nCollecting mccabe==0.6.1\n  Downloading https://files.pythonhosted.org/packages/87/89/479dc97e18549e21354893e4ee4ef36db1d237534982482c3681ee6e7b57/mccabe-0.6.1-py2.py3-none-any.whl\nCollecting mock==3.0.5\n  Downloading https://files.pythonhosted.org/packages/05/d2/f94e68be6b17f46d2c353564da56e6fb89ef09faeeff3313a046cb810ca9/mock-3.0.5-py2.py3-none-any.whl\nCollecting more-itertools==8.0.2\n  Downloading https://files.pythonhosted.org/packages/68/03/0604cec1ea13c9f063dd50f900d1a36160334dd3cfb01fd0e638f61b46ba/more_itertools-8.0.2-py3-none-any.whl (40kB)\nCollecting msrest==0.6.11\n  Downloading https://files.pythonhosted.org/packages/f1/fe/a6786f141181a8f0579e4d7ed048812b1c395a6c265a46fc478be03ff4c1/msrest-0.6.11-py2.py3-none-any.whl (83kB)\nCollecting oauthlib==3.1.0\n  Downloading https://files.pythonhosted.org/packages/05/57/ce2e7a8fa7c0afb54a0581b14a65b56e62b5759dbc98e80627142b8a3704/oauthlib-3.1.0-py2.py3-none-any.whl (147kB)\nCollecting packaging==19.2\n  Downloading https://files.pythonhosted.org/packages/cf/94/9672c2d4b126e74c4496c6b3c58a8b51d6419267be9e70660ba23374c875/packaging-19.2-py2.py3-none-any.whl\nCollecting pluggy==0.13.1\n  Downloading https://files.pythonhosted.org/packages/a0/28/85c7aa31b80d150b772fbe4a229487bc6644da9ccb7e427dd8cc60cb8a62/pluggy-0.13.1-py2.py3-none-any.whl\nCollecting py==1.8.0\n  Downloading https://files.pythonhosted.org/packages/76/bc/394ad449851729244a97857ee14d7cba61ddb268dce3db538ba2f2ba1f0f/py-1.8.0-py2.py3-none-any.whl (83kB)\nCollecting pycparser==2.19\n  Downloading https://files.pythonhosted.org/packages/68/9e/49196946aee219aead1290e00d1e7fdeab8567783e83e1b9ab5585e6206a/pycparser-2.19.tar.gz (158kB)\nCollecting pylint==2.4.4\n  Downloading https://files.pythonhosted.org/packages/e9/59/43fc36c5ee316bb9aeb7cf5329cdbdca89e5749c34d5602753827c0aa2dc/pylint-2.4.4-py3-none-any.whl (302kB)\nCollecting pyparsing==2.4.5\n  Downloading https://files.pythonhosted.org/packages/c0/0c/fc2e007d9a992d997f04a80125b0f183da7fb554f1de701bbb70a8e7d479/pyparsing-2.4.5-py2.py3-none-any.whl (67kB)\nCollecting pytest==5.3.2\n  Downloading https://files.pythonhosted.org/packages/19/cf/711f1d887cb92c5155c9a1eb338f1b5d2411b50e4492a3b20e4a188a22b2/pytest-5.3.2-py3-none-any.whl (234kB)\nCollecting python-dateutil==2.8.1\n  Downloading https://files.pythonhosted.org/packages/d4/70/d60450c3dd48ef87586924207ae8907090de0b306af2bce5d134d78615cb/python_dateutil-2.8.1-py2.py3-none-any.whl (227kB)\nCollecting requests==2.22.0\n  Downloading https://files.pythonhosted.org/packages/51/bd/23c926cd341ea6b7dd0b2a00aba99ae0f828be89d72b2190f27c11d4b7fb/requests-2.22.0-py2.py3-none-any.whl (57kB)\nCollecting requests-oauthlib==1.3.0\n  Downloading https://files.pythonhosted.org/packages/a3/12/b92740d845ab62ea4edf04d2f4164d82532b5a0b03836d4d4e71c6f3d379/requests_oauthlib-1.3.0-py2.py3-none-any.whl\nCollecting requests-toolbelt==0.9.1\n  Downloading https://files.pythonhosted.org/packages/60/ef/7681134338fc097acef8d9b2f8abe0458e4d87559c689a8c306d0957ece5/requests_toolbelt-0.9.1-py2.py3-none-any.whl (54kB)\nCollecting six==1.13.0\n  Downloading https://files.pythonhosted.org/packages/65/26/32b8464df2a97e6dd1b656ed26b2c194606c16fe163c695a992b36c11cdf/six-1.13.0-py2.py3-none-any.whl\nCollecting tabulate==0.8.6\n  Downloading https://files.pythonhosted.org/packages/c4/41/523f6a05e6dc3329a5660f6a81254c6cd87e5cfb5b7482bae3391d86ec3a/tabulate-0.8.6.tar.gz (45kB)\nCollecting tqdm==4.41.0\n  Downloading https://files.pythonhosted.org/packages/8c/c3/d049cf3fb31094ee045ec1ee29fffac218c91e82c8838c49ab4c3e52627b/tqdm-4.41.0-py2.py3-none-any.whl (56kB)\nCollecting typed-ast==1.4.0\n  Downloading https://files.pythonhosted.org/packages/31/d3/9d1802c161626d0278bafb1ffb32f76b9d01e123881bbf9d91e8ccf28e18/typed_ast-1.4.0-cp36-cp36m-manylinux1_x86_64.whl (736kB)\nCollecting urllib3==1.25.7\n  Downloading https://files.pythonhosted.org/packages/b4/40/a9837291310ee1ccc242ceb6ebfd9eb21539649f193a7c8c86ba15b98539/urllib3-1.25.7-py2.py3-none-any.whl (125kB)\nCollecting wcwidth==0.1.7\n  Downloading https://files.pythonhosted.org/packages/7e/9f/526a6947247599b084ee5232e4f9190a38f398d7300d866af3ab571a5bfe/wcwidth-0.1.7-py2.py3-none-any.whl\nCollecting wrapt==1.11.2\n  Downloading https://files.pythonhosted.org/packages/23/84/323c2415280bc4fc880ac5050dddfb3c8062c2552b34c2e512eb4aa68f79/wrapt-1.11.2.tar.gz\nCollecting zipp==0.6.0\n  Downloading https://files.pythonhosted.org/packages/74/3d/1ee25a26411ba0401b43c6376d2316a71addcc72ef8690b101b4ea56d76a/zipp-0.6.0-py2.py3-none-any.whl\nCollecting discomp\n  Downloading https://files.pythonhosted.org/packages/72/bc/556cf941a790c7396020ba7f1c8c924b95bc930e22fc21db81622e8c68bf/discomp-1.7.0-py3-none-any.whl\nBuilding wheels for collected packages: dill, pycparser, tabulate, wrapt\n  Building wheel for dill (setup.py): started\n  Building wheel for dill (setup.py): finished with status 'done'\n  Created wheel for dill: filename=dill-0.3.1.1-cp36-none-any.whl size=78532 sha256=d847e624b1f83f3589883cae4c3882dcaa95fa4bf9857e65736a6269e96381e9\n  Stored in directory: /root/.cache/pip/wheels/59/b1/91/f02e76c732915c4015ab4010f3015469866c1eb9b14058d8e7\n  Building wheel for pycparser (setup.py): started\n  Building wheel for pycparser (setup.py): finished with status 'done'\n  Created wheel for pycparser: filename=pycparser-2.19-py2.py3-none-any.whl size=111029 sha256=0dbf218258a09929d3f85fe5d501a059ad05973fa86fb10e660f19d6404f88ec\n  Stored in directory: /root/.cache/pip/wheels/f2/9a/90/de94f8556265ddc9d9c8b271b0f63e57b26fb1d67a45564511\n  Building wheel for tabulate (setup.py): started\n  Building wheel for tabulate (setup.py): finished with status 'done'\n  Created wheel for tabulate: filename=tabulate-0.8.6-cp36-none-any.whl size=23271 sha256=2236db1dbe134160157ad4f9218559c6abc9d53f8fdacc08e03674d7addf7811\n  Stored in directory: /root/.cache/pip/wheels/9c/9b/f4/eb243fdb89676ec00588e8c54bb54360724c06e7fafe95278e\n  Building wheel for wrapt (setup.py): started\n  Building wheel for wrapt (setup.py): finished with status 'done'\n  Created wheel for wrapt: filename=wrapt-1.11.2-cp36-cp36m-linux_x86_64.whl size=75254 sha256=db1cbd15a6de4fa63889d658ca83cdafbc1e81f2203fc0051bf60a4c83677428\n  Stored in directory: /root/.cache/pip/wheels/d7/de/2e/efa132238792efb6459a96e85916ef8597fcb3d2ae51590dfd\nSuccessfully built dill pycparser tabulate wrapt\nInstalling collected packages: six, python-dateutil, idna, urllib3, chardet, certifi, requests, analytics-python, lazy-object-proxy, wrapt, typed-ast, astroid, attrs, azure-core, pycparser, cffi, cryptography, isodate, oauthlib, requests-oauthlib, msrest, azure-storage-blob, Click, decorator, dill, more-itertools, zipp, importlib-metadata, isort, mccabe, mock, pyparsing, packaging, pluggy, py, pylint, wcwidth, pytest, requests-toolbelt, tabulate, tqdm, discomp\nSuccessfully installed Click-7.0 analytics-python-1.2.9 astroid-2.3.3 attrs-19.3.0 azure-core-1.2.1 azure-storage-blob-12.1.0 certifi-2019.11.28 cffi-1.13.2 chardet-3.0.4 cryptography-2.8 decorator-4.4.1 dill-0.3.1.1 discomp-1.7.0 idna-2.8 importlib-metadata-1.3.0 isodate-0.6.0 isort-4.3.21 lazy-object-proxy-1.4.3 mccabe-0.6.1 mock-3.0.5 more-itertools-8.0.2 msrest-0.6.11 oauthlib-3.1.0 packaging-19.2 pluggy-0.13.1 py-1.8.0 pycparser-2.19 pylint-2.4.4 pyparsing-2.4.5 pytest-5.3.2 python-dateutil-2.8.1 requests-2.22.0 requests-oauthlib-1.3.0 requests-toolbelt-0.9.1 six-1.13.0 tabulate-0.8.6 tqdm-4.41.0 typed-ast-1.4.0 urllib3-1.25.7 wcwidth-0.1.7 wrapt-1.11.2 zipp-0.6.0\n0\n"),
                                                    ('DiscoTask.stderr.0.txt',
                                                     b"WARNING: You are using pip version 19.3.1; however, version 20.0.2 is available.\nYou should consider upgrading via the 'pip install --upgrade pip' command.\n"))),
            TaskResult('5e4bbcbef132fa000c164799', (('result.pickle', b'\x80\x03K\x00.'),
                                                    ('DiscoTask.stdout.0.txt',
                                                     b"Collecting analytics-python==1.2.9\n  Downloading https://files.pythonhosted.org/packages/d3/37/c49d052f88655cd96445c36979fb63f69ef859e167eaff5706ca7c8a8ee3/analytics_python-1.2.9-py2.py3-none-any.whl\nCollecting astroid==2.3.3\n  Downloading https://files.pythonhosted.org/packages/ad/ae/86734823047962e7b8c8529186a1ac4a7ca19aaf1aa0c7713c022ef593fd/astroid-2.3.3-py3-none-any.whl (205kB)\nCollecting attrs==19.3.0\n  Downloading https://files.pythonhosted.org/packages/a2/db/4313ab3be961f7a763066401fb77f7748373b6094076ae2bda2806988af6/attrs-19.3.0-py2.py3-none-any.whl\nCollecting azure-core==1.2.1\n  Downloading https://files.pythonhosted.org/packages/ce/a9/83145b00f09f043610e2e1dd6486cf32876d137455723486348dc9abc074/azure_core-1.2.1-py2.py3-none-any.whl (104kB)\nCollecting azure-storage-blob==12.1.0\n  Downloading https://files.pythonhosted.org/packages/02/16/988b90dd1506c7508bf83624daa23ad9a137a637c0094ab37fd8850b7bd1/azure_storage_blob-12.1.0-py2.py3-none-any.whl (272kB)\nCollecting certifi==2019.11.28\n  Downloading https://files.pythonhosted.org/packages/b9/63/df50cac98ea0d5b006c55a399c3bf1db9da7b5a24de7890bc9cfd5dd9e99/certifi-2019.11.28-py2.py3-none-any.whl (156kB)\nCollecting cffi==1.13.2\n  Downloading https://files.pythonhosted.org/packages/49/72/0d42f94fe94afa8030350c26e9d787219f3f008ec9bf6b86c66532b29236/cffi-1.13.2-cp36-cp36m-manylinux1_x86_64.whl (397kB)\nCollecting chardet==3.0.4\n  Downloading https://files.pythonhosted.org/packages/bc/a9/01ffebfb562e4274b6487b4bb1ddec7ca55ec7510b22e4c51f14098443b8/chardet-3.0.4-py2.py3-none-any.whl (133kB)\nCollecting Click==7.0\n  Downloading https://files.pythonhosted.org/packages/fa/37/45185cb5abbc30d7257104c434fe0b07e5a195a6847506c074527aa599ec/Click-7.0-py2.py3-none-any.whl (81kB)\nCollecting cryptography==2.8\n  Downloading https://files.pythonhosted.org/packages/ca/9a/7cece52c46546e214e10811b36b2da52ce1ea7fa203203a629b8dfadad53/cryptography-2.8-cp34-abi3-manylinux2010_x86_64.whl (2.3MB)\nCollecting decorator==4.4.1\n  Downloading https://files.pythonhosted.org/packages/8f/b7/f329cfdc75f3d28d12c65980e4469e2fa373f1953f5df6e370e84ea2e875/decorator-4.4.1-py2.py3-none-any.whl\nCollecting dill==0.3.1.1\n  Downloading https://files.pythonhosted.org/packages/c7/11/345f3173809cea7f1a193bfbf02403fff250a3360e0e118a1630985e547d/dill-0.3.1.1.tar.gz (151kB)\nCollecting idna==2.8\n  Downloading https://files.pythonhosted.org/packages/14/2c/cd551d81dbe15200be1cf41cd03869a46fe7226e7450af7a6545bfc474c9/idna-2.8-py2.py3-none-any.whl (58kB)\nCollecting importlib-metadata==1.3.0\n  Downloading https://files.pythonhosted.org/packages/e9/71/1a1e0ed0981bb6a67bce55a210f168126b7ebd2065958673797ea66489ca/importlib_metadata-1.3.0-py2.py3-none-any.whl\nCollecting isodate==0.6.0\n  Downloading https://files.pythonhosted.org/packages/9b/9f/b36f7774ff5ea8e428fdcfc4bb332c39ee5b9362ddd3d40d9516a55221b2/isodate-0.6.0-py2.py3-none-any.whl (45kB)\nCollecting isort==4.3.21\n  Downloading https://files.pythonhosted.org/packages/e5/b0/c121fd1fa3419ea9bfd55c7f9c4fedfec5143208d8c7ad3ce3db6c623c21/isort-4.3.21-py2.py3-none-any.whl (42kB)\nCollecting lazy-object-proxy==1.4.3\n  Downloading https://files.pythonhosted.org/packages/0b/dd/b1e3407e9e6913cf178e506cd0dee818e58694d9a5cd1984e3f6a8b9a10f/lazy_object_proxy-1.4.3-cp36-cp36m-manylinux1_x86_64.whl (55kB)\nCollecting mccabe==0.6.1\n  Downloading https://files.pythonhosted.org/packages/87/89/479dc97e18549e21354893e4ee4ef36db1d237534982482c3681ee6e7b57/mccabe-0.6.1-py2.py3-none-any.whl\nCollecting mock==3.0.5\n  Downloading https://files.pythonhosted.org/packages/05/d2/f94e68be6b17f46d2c353564da56e6fb89ef09faeeff3313a046cb810ca9/mock-3.0.5-py2.py3-none-any.whl\nCollecting more-itertools==8.0.2\n  Downloading https://files.pythonhosted.org/packages/68/03/0604cec1ea13c9f063dd50f900d1a36160334dd3cfb01fd0e638f61b46ba/more_itertools-8.0.2-py3-none-any.whl (40kB)\nCollecting msrest==0.6.11\n  Downloading https://files.pythonhosted.org/packages/f1/fe/a6786f141181a8f0579e4d7ed048812b1c395a6c265a46fc478be03ff4c1/msrest-0.6.11-py2.py3-none-any.whl (83kB)\nCollecting oauthlib==3.1.0\n  Downloading https://files.pythonhosted.org/packages/05/57/ce2e7a8fa7c0afb54a0581b14a65b56e62b5759dbc98e80627142b8a3704/oauthlib-3.1.0-py2.py3-none-any.whl (147kB)\nCollecting packaging==19.2\n  Downloading https://files.pythonhosted.org/packages/cf/94/9672c2d4b126e74c4496c6b3c58a8b51d6419267be9e70660ba23374c875/packaging-19.2-py2.py3-none-any.whl\nCollecting pluggy==0.13.1\n  Downloading https://files.pythonhosted.org/packages/a0/28/85c7aa31b80d150b772fbe4a229487bc6644da9ccb7e427dd8cc60cb8a62/pluggy-0.13.1-py2.py3-none-any.whl\nCollecting py==1.8.0\n  Downloading https://files.pythonhosted.org/packages/76/bc/394ad449851729244a97857ee14d7cba61ddb268dce3db538ba2f2ba1f0f/py-1.8.0-py2.py3-none-any.whl (83kB)\nCollecting pycparser==2.19\n  Downloading https://files.pythonhosted.org/packages/68/9e/49196946aee219aead1290e00d1e7fdeab8567783e83e1b9ab5585e6206a/pycparser-2.19.tar.gz (158kB)\nCollecting pylint==2.4.4\n  Downloading https://files.pythonhosted.org/packages/e9/59/43fc36c5ee316bb9aeb7cf5329cdbdca89e5749c34d5602753827c0aa2dc/pylint-2.4.4-py3-none-any.whl (302kB)\nCollecting pyparsing==2.4.5\n  Downloading https://files.pythonhosted.org/packages/c0/0c/fc2e007d9a992d997f04a80125b0f183da7fb554f1de701bbb70a8e7d479/pyparsing-2.4.5-py2.py3-none-any.whl (67kB)\nCollecting pytest==5.3.2\n  Downloading https://files.pythonhosted.org/packages/19/cf/711f1d887cb92c5155c9a1eb338f1b5d2411b50e4492a3b20e4a188a22b2/pytest-5.3.2-py3-none-any.whl (234kB)\nCollecting python-dateutil==2.8.1\n  Downloading https://files.pythonhosted.org/packages/d4/70/d60450c3dd48ef87586924207ae8907090de0b306af2bce5d134d78615cb/python_dateutil-2.8.1-py2.py3-none-any.whl (227kB)\nCollecting requests==2.22.0\n  Downloading https://files.pythonhosted.org/packages/51/bd/23c926cd341ea6b7dd0b2a00aba99ae0f828be89d72b2190f27c11d4b7fb/requests-2.22.0-py2.py3-none-any.whl (57kB)\nCollecting requests-oauthlib==1.3.0\n  Downloading https://files.pythonhosted.org/packages/a3/12/b92740d845ab62ea4edf04d2f4164d82532b5a0b03836d4d4e71c6f3d379/requests_oauthlib-1.3.0-py2.py3-none-any.whl\nCollecting requests-toolbelt==0.9.1\n  Downloading https://files.pythonhosted.org/packages/60/ef/7681134338fc097acef8d9b2f8abe0458e4d87559c689a8c306d0957ece5/requests_toolbelt-0.9.1-py2.py3-none-any.whl (54kB)\nCollecting six==1.13.0\n  Downloading https://files.pythonhosted.org/packages/65/26/32b8464df2a97e6dd1b656ed26b2c194606c16fe163c695a992b36c11cdf/six-1.13.0-py2.py3-none-any.whl\nCollecting tabulate==0.8.6\n  Downloading https://files.pythonhosted.org/packages/c4/41/523f6a05e6dc3329a5660f6a81254c6cd87e5cfb5b7482bae3391d86ec3a/tabulate-0.8.6.tar.gz (45kB)\nCollecting tqdm==4.41.0\n  Downloading https://files.pythonhosted.org/packages/8c/c3/d049cf3fb31094ee045ec1ee29fffac218c91e82c8838c49ab4c3e52627b/tqdm-4.41.0-py2.py3-none-any.whl (56kB)\nCollecting typed-ast==1.4.0\n  Downloading https://files.pythonhosted.org/packages/31/d3/9d1802c161626d0278bafb1ffb32f76b9d01e123881bbf9d91e8ccf28e18/typed_ast-1.4.0-cp36-cp36m-manylinux1_x86_64.whl (736kB)\nCollecting urllib3==1.25.7\n  Downloading https://files.pythonhosted.org/packages/b4/40/a9837291310ee1ccc242ceb6ebfd9eb21539649f193a7c8c86ba15b98539/urllib3-1.25.7-py2.py3-none-any.whl (125kB)\nCollecting wcwidth==0.1.7\n  Downloading https://files.pythonhosted.org/packages/7e/9f/526a6947247599b084ee5232e4f9190a38f398d7300d866af3ab571a5bfe/wcwidth-0.1.7-py2.py3-none-any.whl\nCollecting wrapt==1.11.2\n  Downloading https://files.pythonhosted.org/packages/23/84/323c2415280bc4fc880ac5050dddfb3c8062c2552b34c2e512eb4aa68f79/wrapt-1.11.2.tar.gz\nCollecting zipp==0.6.0\n  Downloading https://files.pythonhosted.org/packages/74/3d/1ee25a26411ba0401b43c6376d2316a71addcc72ef8690b101b4ea56d76a/zipp-0.6.0-py2.py3-none-any.whl\nCollecting discomp\n  Downloading https://files.pythonhosted.org/packages/72/bc/556cf941a790c7396020ba7f1c8c924b95bc930e22fc21db81622e8c68bf/discomp-1.7.0-py3-none-any.whl\nBuilding wheels for collected packages: dill, pycparser, tabulate, wrapt\n  Building wheel for dill (setup.py): started\n  Building wheel for dill (setup.py): finished with status 'done'\n  Created wheel for dill: filename=dill-0.3.1.1-cp36-none-any.whl size=78532 sha256=d847e624b1f83f3589883cae4c3882dcaa95fa4bf9857e65736a6269e96381e9\n  Stored in directory: /root/.cache/pip/wheels/59/b1/91/f02e76c732915c4015ab4010f3015469866c1eb9b14058d8e7\n  Building wheel for pycparser (setup.py): started\n  Building wheel for pycparser (setup.py): finished with status 'done'\n  Created wheel for pycparser: filename=pycparser-2.19-py2.py3-none-any.whl size=111029 sha256=0dbf218258a09929d3f85fe5d501a059ad05973fa86fb10e660f19d6404f88ec\n  Stored in directory: /root/.cache/pip/wheels/f2/9a/90/de94f8556265ddc9d9c8b271b0f63e57b26fb1d67a45564511\n  Building wheel for tabulate (setup.py): started\n  Building wheel for tabulate (setup.py): finished with status 'done'\n  Created wheel for tabulate: filename=tabulate-0.8.6-cp36-none-any.whl size=23271 sha256=2236db1dbe134160157ad4f9218559c6abc9d53f8fdacc08e03674d7addf7811\n  Stored in directory: /root/.cache/pip/wheels/9c/9b/f4/eb243fdb89676ec00588e8c54bb54360724c06e7fafe95278e\n  Building wheel for wrapt (setup.py): started\n  Building wheel for wrapt (setup.py): finished with status 'done'\n  Created wheel for wrapt: filename=wrapt-1.11.2-cp36-cp36m-linux_x86_64.whl size=75254 sha256=db1cbd15a6de4fa63889d658ca83cdafbc1e81f2203fc0051bf60a4c83677428\n  Stored in directory: /root/.cache/pip/wheels/d7/de/2e/efa132238792efb6459a96e85916ef8597fcb3d2ae51590dfd\nSuccessfully built dill pycparser tabulate wrapt\nInstalling collected packages: six, python-dateutil, idna, urllib3, chardet, certifi, requests, analytics-python, lazy-object-proxy, wrapt, typed-ast, astroid, attrs, azure-core, pycparser, cffi, cryptography, isodate, oauthlib, requests-oauthlib, msrest, azure-storage-blob, Click, decorator, dill, more-itertools, zipp, importlib-metadata, isort, mccabe, mock, pyparsing, packaging, pluggy, py, pylint, wcwidth, pytest, requests-toolbelt, tabulate, tqdm, discomp\nSuccessfully installed Click-7.0 analytics-python-1.2.9 astroid-2.3.3 attrs-19.3.0 azure-core-1.2.1 azure-storage-blob-12.1.0 certifi-2019.11.28 cffi-1.13.2 chardet-3.0.4 cryptography-2.8 decorator-4.4.1 dill-0.3.1.1 discomp-1.7.0 idna-2.8 importlib-metadata-1.3.0 isodate-0.6.0 isort-4.3.21 lazy-object-proxy-1.4.3 mccabe-0.6.1 mock-3.0.5 more-itertools-8.0.2 msrest-0.6.11 oauthlib-3.1.0 packaging-19.2 pluggy-0.13.1 py-1.8.0 pycparser-2.19 pylint-2.4.4 pyparsing-2.4.5 pytest-5.3.2 python-dateutil-2.8.1 requests-2.22.0 requests-oauthlib-1.3.0 requests-toolbelt-0.9.1 six-1.13.0 tabulate-0.8.6 tqdm-4.41.0 typed-ast-1.4.0 urllib3-1.25.7 wcwidth-0.1.7 wrapt-1.11.2 zipp-0.6.0\n0\n"),
                                                    ('DiscoTask.stderr.0.txt',
                                                     b"WARNING: You are using pip version 19.3.1; however, version 20.0.2 is available.\nYou should consider upgrading via the 'pip install --upgrade pip' command.\n")))

        ]
        get_status_mock.return_value = JobStatus.success
        with Pool(processes=3) as mypool:
            results = mypool.starmap(func_examples.mult, [[3, 0], [5, 0]])

        create_mock.assert_called_with(cluster_id=None, cluster_instance_type='s',
                                       constants_file_ids=['file_id', 'file_id'],
                                       input_file_ids=['file_id', 'file_id'],
                                       job_name=ANY, script_file_id='file_id', upload_requirements_file=False)
        assert results == [0, 0]

    @patch('disco.job.Job.create')
    @patch('disco.job.Job.start')
    @patch('disco.job.Job.wait_for_finish')
    @patch('disco.job.Job.get_results')
    @patch('disco.job.Job.get_status')
    @patch('disco.asset.Asset.upload', return_value='file_id')
    def test_map_get_result_of_failed_job(self, asset_upload_mock, get_status_mock, get_results_mock,
                                          wait_for_finish_mock,
                                          start_mock,
                                          create_mock):
        create_mock.return_value = Job('some_job_id')
        get_results_mock.return_value = [
            TaskResult('5e4bbcbef132fa000c164798', (('result.pickle', b'\x80\x03K\x00.'),
                                                    ('DiscoTask.stdout.0.txt',
                                                     b"Collecting analytics-python==1.2.9\n  Downloading https://files.pythonhosted.org/packages/d3/37/c49d052f88655cd96445c36979fb63f69ef859e167eaff5706ca7c8a8ee3/analytics_python-1.2.9-py2.py3-none-any.whl\nCollecting astroid==2.3.3\n  Downloading https://files.pythonhosted.org/packages/ad/ae/86734823047962e7b8c8529186a1ac4a7ca19aaf1aa0c7713c022ef593fd/astroid-2.3.3-py3-none-any.whl (205kB)\nCollecting attrs==19.3.0\n  Downloading https://files.pythonhosted.org/packages/a2/db/4313ab3be961f7a763066401fb77f7748373b6094076ae2bda2806988af6/attrs-19.3.0-py2.py3-none-any.whl\nCollecting azure-core==1.2.1\n  Downloading https://files.pythonhosted.org/packages/ce/a9/83145b00f09f043610e2e1dd6486cf32876d137455723486348dc9abc074/azure_core-1.2.1-py2.py3-none-any.whl (104kB)\nCollecting azure-storage-blob==12.1.0\n  Downloading https://files.pythonhosted.org/packages/02/16/988b90dd1506c7508bf83624daa23ad9a137a637c0094ab37fd8850b7bd1/azure_storage_blob-12.1.0-py2.py3-none-any.whl (272kB)\nCollecting certifi==2019.11.28\n  Downloading https://files.pythonhosted.org/packages/b9/63/df50cac98ea0d5b006c55a399c3bf1db9da7b5a24de7890bc9cfd5dd9e99/certifi-2019.11.28-py2.py3-none-any.whl (156kB)\nCollecting cffi==1.13.2\n  Downloading https://files.pythonhosted.org/packages/49/72/0d42f94fe94afa8030350c26e9d787219f3f008ec9bf6b86c66532b29236/cffi-1.13.2-cp36-cp36m-manylinux1_x86_64.whl (397kB)\nCollecting chardet==3.0.4\n  Downloading https://files.pythonhosted.org/packages/bc/a9/01ffebfb562e4274b6487b4bb1ddec7ca55ec7510b22e4c51f14098443b8/chardet-3.0.4-py2.py3-none-any.whl (133kB)\nCollecting Click==7.0\n  Downloading https://files.pythonhosted.org/packages/fa/37/45185cb5abbc30d7257104c434fe0b07e5a195a6847506c074527aa599ec/Click-7.0-py2.py3-none-any.whl (81kB)\nCollecting cryptography==2.8\n  Downloading https://files.pythonhosted.org/packages/ca/9a/7cece52c46546e214e10811b36b2da52ce1ea7fa203203a629b8dfadad53/cryptography-2.8-cp34-abi3-manylinux2010_x86_64.whl (2.3MB)\nCollecting decorator==4.4.1\n  Downloading https://files.pythonhosted.org/packages/8f/b7/f329cfdc75f3d28d12c65980e4469e2fa373f1953f5df6e370e84ea2e875/decorator-4.4.1-py2.py3-none-any.whl\nCollecting dill==0.3.1.1\n  Downloading https://files.pythonhosted.org/packages/c7/11/345f3173809cea7f1a193bfbf02403fff250a3360e0e118a1630985e547d/dill-0.3.1.1.tar.gz (151kB)\nCollecting idna==2.8\n  Downloading https://files.pythonhosted.org/packages/14/2c/cd551d81dbe15200be1cf41cd03869a46fe7226e7450af7a6545bfc474c9/idna-2.8-py2.py3-none-any.whl (58kB)\nCollecting importlib-metadata==1.3.0\n  Downloading https://files.pythonhosted.org/packages/e9/71/1a1e0ed0981bb6a67bce55a210f168126b7ebd2065958673797ea66489ca/importlib_metadata-1.3.0-py2.py3-none-any.whl\nCollecting isodate==0.6.0\n  Downloading https://files.pythonhosted.org/packages/9b/9f/b36f7774ff5ea8e428fdcfc4bb332c39ee5b9362ddd3d40d9516a55221b2/isodate-0.6.0-py2.py3-none-any.whl (45kB)\nCollecting isort==4.3.21\n  Downloading https://files.pythonhosted.org/packages/e5/b0/c121fd1fa3419ea9bfd55c7f9c4fedfec5143208d8c7ad3ce3db6c623c21/isort-4.3.21-py2.py3-none-any.whl (42kB)\nCollecting lazy-object-proxy==1.4.3\n  Downloading https://files.pythonhosted.org/packages/0b/dd/b1e3407e9e6913cf178e506cd0dee818e58694d9a5cd1984e3f6a8b9a10f/lazy_object_proxy-1.4.3-cp36-cp36m-manylinux1_x86_64.whl (55kB)\nCollecting mccabe==0.6.1\n  Downloading https://files.pythonhosted.org/packages/87/89/479dc97e18549e21354893e4ee4ef36db1d237534982482c3681ee6e7b57/mccabe-0.6.1-py2.py3-none-any.whl\nCollecting mock==3.0.5\n  Downloading https://files.pythonhosted.org/packages/05/d2/f94e68be6b17f46d2c353564da56e6fb89ef09faeeff3313a046cb810ca9/mock-3.0.5-py2.py3-none-any.whl\nCollecting more-itertools==8.0.2\n  Downloading https://files.pythonhosted.org/packages/68/03/0604cec1ea13c9f063dd50f900d1a36160334dd3cfb01fd0e638f61b46ba/more_itertools-8.0.2-py3-none-any.whl (40kB)\nCollecting msrest==0.6.11\n  Downloading https://files.pythonhosted.org/packages/f1/fe/a6786f141181a8f0579e4d7ed048812b1c395a6c265a46fc478be03ff4c1/msrest-0.6.11-py2.py3-none-any.whl (83kB)\nCollecting oauthlib==3.1.0\n  Downloading https://files.pythonhosted.org/packages/05/57/ce2e7a8fa7c0afb54a0581b14a65b56e62b5759dbc98e80627142b8a3704/oauthlib-3.1.0-py2.py3-none-any.whl (147kB)\nCollecting packaging==19.2\n  Downloading https://files.pythonhosted.org/packages/cf/94/9672c2d4b126e74c4496c6b3c58a8b51d6419267be9e70660ba23374c875/packaging-19.2-py2.py3-none-any.whl\nCollecting pluggy==0.13.1\n  Downloading https://files.pythonhosted.org/packages/a0/28/85c7aa31b80d150b772fbe4a229487bc6644da9ccb7e427dd8cc60cb8a62/pluggy-0.13.1-py2.py3-none-any.whl\nCollecting py==1.8.0\n  Downloading https://files.pythonhosted.org/packages/76/bc/394ad449851729244a97857ee14d7cba61ddb268dce3db538ba2f2ba1f0f/py-1.8.0-py2.py3-none-any.whl (83kB)\nCollecting pycparser==2.19\n  Downloading https://files.pythonhosted.org/packages/68/9e/49196946aee219aead1290e00d1e7fdeab8567783e83e1b9ab5585e6206a/pycparser-2.19.tar.gz (158kB)\nCollecting pylint==2.4.4\n  Downloading https://files.pythonhosted.org/packages/e9/59/43fc36c5ee316bb9aeb7cf5329cdbdca89e5749c34d5602753827c0aa2dc/pylint-2.4.4-py3-none-any.whl (302kB)\nCollecting pyparsing==2.4.5\n  Downloading https://files.pythonhosted.org/packages/c0/0c/fc2e007d9a992d997f04a80125b0f183da7fb554f1de701bbb70a8e7d479/pyparsing-2.4.5-py2.py3-none-any.whl (67kB)\nCollecting pytest==5.3.2\n  Downloading https://files.pythonhosted.org/packages/19/cf/711f1d887cb92c5155c9a1eb338f1b5d2411b50e4492a3b20e4a188a22b2/pytest-5.3.2-py3-none-any.whl (234kB)\nCollecting python-dateutil==2.8.1\n  Downloading https://files.pythonhosted.org/packages/d4/70/d60450c3dd48ef87586924207ae8907090de0b306af2bce5d134d78615cb/python_dateutil-2.8.1-py2.py3-none-any.whl (227kB)\nCollecting requests==2.22.0\n  Downloading https://files.pythonhosted.org/packages/51/bd/23c926cd341ea6b7dd0b2a00aba99ae0f828be89d72b2190f27c11d4b7fb/requests-2.22.0-py2.py3-none-any.whl (57kB)\nCollecting requests-oauthlib==1.3.0\n  Downloading https://files.pythonhosted.org/packages/a3/12/b92740d845ab62ea4edf04d2f4164d82532b5a0b03836d4d4e71c6f3d379/requests_oauthlib-1.3.0-py2.py3-none-any.whl\nCollecting requests-toolbelt==0.9.1\n  Downloading https://files.pythonhosted.org/packages/60/ef/7681134338fc097acef8d9b2f8abe0458e4d87559c689a8c306d0957ece5/requests_toolbelt-0.9.1-py2.py3-none-any.whl (54kB)\nCollecting six==1.13.0\n  Downloading https://files.pythonhosted.org/packages/65/26/32b8464df2a97e6dd1b656ed26b2c194606c16fe163c695a992b36c11cdf/six-1.13.0-py2.py3-none-any.whl\nCollecting tabulate==0.8.6\n  Downloading https://files.pythonhosted.org/packages/c4/41/523f6a05e6dc3329a5660f6a81254c6cd87e5cfb5b7482bae3391d86ec3a/tabulate-0.8.6.tar.gz (45kB)\nCollecting tqdm==4.41.0\n  Downloading https://files.pythonhosted.org/packages/8c/c3/d049cf3fb31094ee045ec1ee29fffac218c91e82c8838c49ab4c3e52627b/tqdm-4.41.0-py2.py3-none-any.whl (56kB)\nCollecting typed-ast==1.4.0\n  Downloading https://files.pythonhosted.org/packages/31/d3/9d1802c161626d0278bafb1ffb32f76b9d01e123881bbf9d91e8ccf28e18/typed_ast-1.4.0-cp36-cp36m-manylinux1_x86_64.whl (736kB)\nCollecting urllib3==1.25.7\n  Downloading https://files.pythonhosted.org/packages/b4/40/a9837291310ee1ccc242ceb6ebfd9eb21539649f193a7c8c86ba15b98539/urllib3-1.25.7-py2.py3-none-any.whl (125kB)\nCollecting wcwidth==0.1.7\n  Downloading https://files.pythonhosted.org/packages/7e/9f/526a6947247599b084ee5232e4f9190a38f398d7300d866af3ab571a5bfe/wcwidth-0.1.7-py2.py3-none-any.whl\nCollecting wrapt==1.11.2\n  Downloading https://files.pythonhosted.org/packages/23/84/323c2415280bc4fc880ac5050dddfb3c8062c2552b34c2e512eb4aa68f79/wrapt-1.11.2.tar.gz\nCollecting zipp==0.6.0\n  Downloading https://files.pythonhosted.org/packages/74/3d/1ee25a26411ba0401b43c6376d2316a71addcc72ef8690b101b4ea56d76a/zipp-0.6.0-py2.py3-none-any.whl\nCollecting discomp\n  Downloading https://files.pythonhosted.org/packages/72/bc/556cf941a790c7396020ba7f1c8c924b95bc930e22fc21db81622e8c68bf/discomp-1.7.0-py3-none-any.whl\nBuilding wheels for collected packages: dill, pycparser, tabulate, wrapt\n  Building wheel for dill (setup.py): started\n  Building wheel for dill (setup.py): finished with status 'done'\n  Created wheel for dill: filename=dill-0.3.1.1-cp36-none-any.whl size=78532 sha256=d847e624b1f83f3589883cae4c3882dcaa95fa4bf9857e65736a6269e96381e9\n  Stored in directory: /root/.cache/pip/wheels/59/b1/91/f02e76c732915c4015ab4010f3015469866c1eb9b14058d8e7\n  Building wheel for pycparser (setup.py): started\n  Building wheel for pycparser (setup.py): finished with status 'done'\n  Created wheel for pycparser: filename=pycparser-2.19-py2.py3-none-any.whl size=111029 sha256=0dbf218258a09929d3f85fe5d501a059ad05973fa86fb10e660f19d6404f88ec\n  Stored in directory: /root/.cache/pip/wheels/f2/9a/90/de94f8556265ddc9d9c8b271b0f63e57b26fb1d67a45564511\n  Building wheel for tabulate (setup.py): started\n  Building wheel for tabulate (setup.py): finished with status 'done'\n  Created wheel for tabulate: filename=tabulate-0.8.6-cp36-none-any.whl size=23271 sha256=2236db1dbe134160157ad4f9218559c6abc9d53f8fdacc08e03674d7addf7811\n  Stored in directory: /root/.cache/pip/wheels/9c/9b/f4/eb243fdb89676ec00588e8c54bb54360724c06e7fafe95278e\n  Building wheel for wrapt (setup.py): started\n  Building wheel for wrapt (setup.py): finished with status 'done'\n  Created wheel for wrapt: filename=wrapt-1.11.2-cp36-cp36m-linux_x86_64.whl size=75254 sha256=db1cbd15a6de4fa63889d658ca83cdafbc1e81f2203fc0051bf60a4c83677428\n  Stored in directory: /root/.cache/pip/wheels/d7/de/2e/efa132238792efb6459a96e85916ef8597fcb3d2ae51590dfd\nSuccessfully built dill pycparser tabulate wrapt\nInstalling collected packages: six, python-dateutil, idna, urllib3, chardet, certifi, requests, analytics-python, lazy-object-proxy, wrapt, typed-ast, astroid, attrs, azure-core, pycparser, cffi, cryptography, isodate, oauthlib, requests-oauthlib, msrest, azure-storage-blob, Click, decorator, dill, more-itertools, zipp, importlib-metadata, isort, mccabe, mock, pyparsing, packaging, pluggy, py, pylint, wcwidth, pytest, requests-toolbelt, tabulate, tqdm, discomp\nSuccessfully installed Click-7.0 analytics-python-1.2.9 astroid-2.3.3 attrs-19.3.0 azure-core-1.2.1 azure-storage-blob-12.1.0 certifi-2019.11.28 cffi-1.13.2 chardet-3.0.4 cryptography-2.8 decorator-4.4.1 dill-0.3.1.1 discomp-1.7.0 idna-2.8 importlib-metadata-1.3.0 isodate-0.6.0 isort-4.3.21 lazy-object-proxy-1.4.3 mccabe-0.6.1 mock-3.0.5 more-itertools-8.0.2 msrest-0.6.11 oauthlib-3.1.0 packaging-19.2 pluggy-0.13.1 py-1.8.0 pycparser-2.19 pylint-2.4.4 pyparsing-2.4.5 pytest-5.3.2 python-dateutil-2.8.1 requests-2.22.0 requests-oauthlib-1.3.0 requests-toolbelt-0.9.1 six-1.13.0 tabulate-0.8.6 tqdm-4.41.0 typed-ast-1.4.0 urllib3-1.25.7 wcwidth-0.1.7 wrapt-1.11.2 zipp-0.6.0\n0\n"),
                                                    ('DiscoTask.stderr.0.txt',
                                                     b"WARNING: You are using pip version 19.3.1; however, version 20.0.2 is available.\nYou should consider upgrading via the 'pip install --upgrade pip' command.\n")))
        ]
        get_status_mock.return_value = JobStatus.failed
        mypool = Pool()
        results = mypool.map(func_examples.pow3, [1, 2])
        create_mock.assert_called_with(cluster_id=None, cluster_instance_type='s', constants_file_ids=['file_id', 'file_id'],
                                       input_file_ids=['file_id', 'file_id'],
                                       job_name=ANY, script_file_id='file_id', upload_requirements_file=False)
        assert results == []
        assert asset_upload_mock.call_count == 5

    def test_map_invalid_params(self):
        with self.assertRaises(ValueError):
            mypool = Pool()
            mypool.map('not a function', range(10))

        with self.assertRaises(ValueError):
            mypool = Pool()
            mypool.map(func_examples.pow3, 9)

    def test_starmap_invalid_params(self):
        with self.assertRaises(ValueError):
            mypool = Pool()
            mypool.map('not a function', [[3, 4], [5, 6]])

        with self.assertRaises(ValueError):
            mypool = Pool()
            mypool.starmap(func_examples.pow3, 9)

        with self.assertRaises(ValueError):
            mypool = Pool()
            mypool.starmap(func_examples.pow3, range(10))

    def test_pool_stubs(self):
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.join()
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.apply(func_examples.pow3)
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.apply_async(func_examples.pow3)
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.map_async(func_examples.pow3, range(10))
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.starmap_async(func_examples.pow3, range(10))
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.imap(func_examples.pow3, range(10))
        with self.assertRaises(NotImplementedError):
            mypool = Pool()
            mypool.imap_unordered(func_examples.pow3, range(10))
