# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aiocomfoconnect', 'aiocomfoconnect.protobuf']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.0,<4.0.0', 'nanopb>=0.4.6,<0.5.0', 'protobuf>=4.21.9,<5.0.0']

setup_kwargs = {
    'name': 'aiocomfoconnect',
    'version': '0.1.1',
    'description': '',
    'long_description': '# aiocomfoconnect\n\n`aiocomfoconnect` is an asyncio Python 3 library for communicating with a Zehnder ComfoAir Q350/450/600 ventilation system. It\'s the successor of\n[comfoconnect](https://github.com/michaelarnauts/comfoconnect).\n\n## Installation\n\n```shell\npip3 install aiocomfoconnect\n```\n\n## CLI Usage\n\n```shell\n$ python -m aiocomfoconnect --help\n\n$ python -m aiocomfoconnect discover\n\n$ python -m aiocomfoconnect register --host 192.168.1.213\n\n$ python -m aiocomfoconnect set-speed away --host 192.168.1.213\n$ python -m aiocomfoconnect set-speed low --host 192.168.1.213\n$ python -m aiocomfoconnect set-speed medium --host 192.168.1.213\n$ python -m aiocomfoconnect set-speed high --host 192.168.1.213\n\n$ python -m aiocomfoconnect set-speed show-sensors --host 192.168.1.213\n```\n\n## Available methods\n\n- `async connect()`: Connect to the bridge.\n- `async disconnect()`: Disconnect from the bridge.\n- `async register_sensor(sensor)`: Register a sensor.\n- `async deregister_sensor(sensor)`: Deregister a sensor.\n- `async get_mode()`: Get the ventilation mode.\n- `async set_mode(mode)`: Set the ventilation mode. (auto / manual)\n- `async get_speed()`: Get the ventilation speed.\n- `async set_speed(speed)`: Set the ventilation speed. (away / low / medium / high)\n- `async get_bypass()`: Get the bypass mode.\n- `async set_bypass(mode, timeout=3600)`: Set the bypass mode. (auto / on / off)\n- `async get_balance_mode()`: Get the balance mode.\n- `async set_balance_mode(mode, timeout=3600)`: Set the balance mode. (balance / supply only / exhaust only)\n- `async get_boost()`: Get the boost mode.\n- `async set_boost(mode, timeout=3600)`: Set the boost mode. (boolean)\n- `async get_away()`: Get the away mode.\n- `async set_away(mode, timeout=3600)`: Set the away mode. (boolean)\n- `async get_temperature_profile()`: Get the temperature profile.\n- `async set_temperature_profile(profile)`: Set the temperature profile. (warm / normal / cool)\n- `async get_sensor_ventmode_temperature_passive()`: Get the sensor based ventilation passive temperature control setting.\n- `async set_sensor_ventmode_temperature_passive(mode)`: Set the sensor based ventilation passive temperature control setting. (auto / on / off)\n- `async get_sensor_ventmode_humidity_comfort()`: Get the sensor based ventilation humidity comfort setting.\n- `async set_sensor_ventmode_humidity_comfort(mode)`: Set the sensor based ventilation humidity comfort setting. (auto / on / off)\n- `async get_sensor_ventmode_humidity_protection()`: Get the sensor based ventilation humidity protection setting.\n- `async set_sensor_ventmode_humidity_protection(mode)`: Set the sensor based ventilation humidity protection setting. (auto / on / off)\n\n### Low-level API\n\n- `async cmd_start_session()`: Start a session.\n- `async cmd_close_session()`: Close a session.\n- `async cmd_list_registered_apps()`: List registered apps.\n- `async cmd_register_app(uuid, device_name, pin)`: Register an app.\n- `async cmd_deregister_app(uuid)`: Deregister an app.\n- `async cmd_version_request()`: Request the bridge\'s version.\n- `async cmd_time_request()`: Request the bridge\'s time.\n- `async cmd_rmi_request(message, node_id)`: Send a RMI request.\n- `async cmd_rpdo_request(pdid, type, zone, timeout)`: Send a RPDO request.\n- `async cmd_keepalive()`: Send a keepalive message.\n\n## Examples\n\n### Discovery of ComfoConnect LAN C Bridges\n\n```python\nimport asyncio\n\nfrom aiocomfoconnect import discover_bridges\n\n\nasync def main():\n    """ ComfoConnect LAN C Bridge discovery example."""\n\n    # Discover all ComfoConnect LAN C Bridges on the subnet.\n    bridges = await discover_bridges()\n    print(bridges)\n\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\n### Basic Example\n\n```python\nimport asyncio\n\nfrom aiocomfoconnect import ComfoConnect\nfrom aiocomfoconnect.const import SPEED_LOW\nfrom aiocomfoconnect.sensors import SENSORS\n\n\nasync def main(local_uuid, host, uuid):\n    """ Basic example."""\n\n    def sensor_callback(sensor, value):\n        """ Print sensor updates. """\n        print(f"{sensor.name} = {value}")\n\n    # Connect to the Bridge\n    comfoconnect = ComfoConnect(host, uuid, callback=sensor_callback)\n    await comfoconnect.connect(local_uuid)\n\n    # Register all sensors\n    for key in SENSORS:\n        await comfoconnect.register_sensor(SENSORS[key])\n\n    # Set speed to LOW\n    await comfoconnect.set_speed(SPEED_LOW)\n\n    # Wait 2 minutes so we can see some sensor updates\n    await asyncio.sleep(120)\n\n    # Disconnect from the bridge\n    await comfoconnect.disconnect()\n\n\nif __name__ == "__main__":\n    asyncio.run(main(\'00000000000000000000000000001337\', \'192.168.1.20\', \'00000000000000000000000000000055\'))  # Replace with your bridge\'s IP and UUID\n```\n\n## Development Notes\n\n### Protocol Documentation\n\n- [ComfoConnect LAN C Protocol](docs/PROTOCOL.md)\n- [PDO Sensors](docs/PROTOCOL-PDO.md)\n- [RMI commands](docs/PROTOCOL-RMI.md)\n\n### Decode network traffic\n\nYou can use the `scripts/decode_pcap.py` file to decode network traffic between the Mobile App and the ComfoConnect LAN C. \nMake sure that the first TCP session in the capture is the connection between the bridge and the app. It\'s therefore recommended to start the capture before you open the app.\n\n```shell\n$ sudo tcpdump -i any -s 0 -w /tmp/capture.pcap tcp and port 56747\n$ python scripts/decode_pcap.py /tmp/capture.pcap\n```\n\n### Generate zehnder_pb2.py file\n\n```shell\npip3 install grpcio-tools\npython3 -m grpc_tools.protoc -Iprotobuf protobuf/nanopb.proto --python_out=aiocomfoconnect/protobuf\npython3 -m grpc_tools.protoc -Iprotobuf protobuf/zehnder.proto --python_out=aiocomfoconnect/protobuf\n```\n',
    'author': 'Michaël Arnauts',
    'author_email': 'michael.arnauts@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/michaelarnauts/aiocomfoconnect',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
