# -*- coding: utf-8 -*-
#
# Configuration file for the Sphinx documentation builder.
#
# This file does only contain a selection of the most common options. For a
# full list see the documentation:
# http://www.sphinx-doc.org/en/stable/config

# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
from pathlib import Path
import regex as re
import shutil
import string

from setuptools_scm import get_version
from recommonmark.parser import CommonMarkParser


CURRENT_DIR = Path(__file__).parent


def make_pypi_svg(version):
    template = CURRENT_DIR / "_static" / "pypi_template.svg"
    target = CURRENT_DIR / "_static" / "pypi.svg"
    with open(str(template), "r", encoding="utf8") as f:
        svg = string.Template(f.read()).substitute(version=version)
    with open(str(target), "w", encoding="utf8") as f:
        f.write(svg)


def make_filename(line):
    non_letters = re.compile(r"[^a-z]+")
    filename = line[3:].rstrip().lower()
    filename = non_letters.sub("_", filename)
    if filename.startswith("_"):
        filename = filename[1:]
    if filename.endswith("_"):
        filename = filename[:-1]
    return filename + ".md"


def generate_sections_from_readme():
    target_dir = CURRENT_DIR / "_build" / "generated"
    readme = CURRENT_DIR / ".." / "README.md"
    shutil.rmtree(str(target_dir), ignore_errors=True)
    target_dir.mkdir(parents=True)

    output = None
    target_dir = target_dir.relative_to(CURRENT_DIR)
    with open(str(readme), "r", encoding="utf8") as f:
        for line in f:
            if line.startswith("## "):
                if output is not None:
                    output.close()
                filename = make_filename(line)
                output_path = CURRENT_DIR / filename
                if output_path.is_symlink() or output_path.is_file():
                    output_path.unlink()
                output_path.symlink_to(target_dir / filename)
                output = open(str(output_path), "w", encoding="utf8")
                output.write(
                    "[//]: # (NOTE: THIS FILE IS AUTOGENERATED FROM README.md)\n\n"
                )

            if output is None:
                continue

            if line.startswith("##"):
                line = line[1:]

            output.write(line)


# -- Project information -----------------------------------------------------

project = "Black"
copyright = "2018, Łukasz Langa and contributors to Black"
author = "Łukasz Langa and contributors to Black"

# Autopopulate version
# The full version, including alpha/beta/rc tags.
release = get_version(root=CURRENT_DIR.parent)
# The short X.Y version.
version = release
for sp in "abcfr":
    version = version.split(sp)[0]
make_pypi_svg(release)
generate_sections_from_readme()


# -- General configuration ---------------------------------------------------

# If your documentation needs a minimal Sphinx version, state it here.
#
# needs_sphinx = '1.0'

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = ["sphinx.ext.autodoc", "sphinx.ext.intersphinx", "sphinx.ext.napoleon"]

# Add any paths that contain templates here, relative to this directory.
templates_path = ["_templates"]

source_parsers = {".md": CommonMarkParser}

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
source_suffix = [".rst", ".md"]

# The master toctree document.
master_doc = "index"

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = None

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path .
exclude_patterns = ["_build", "Thumbs.db", ".DS_Store"]

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = "sphinx"


# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
html_theme = "alabaster"

html_sidebars = {
    "**": [
        "about.html",
        "navigation.html",
        "relations.html",
        "sourcelink.html",
        "searchbox.html",
    ]
}

html_theme_options = {
    "show_related": False,
    "description": "“Any color you like.”",
    "github_button": True,
    "github_user": "psf",
    "github_repo": "black",
    "github_type": "star",
    "show_powered_by": True,
    "fixed_sidebar": True,
    "logo": "logo2.png",
    "travis_button": True,
}


# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ["_static"]

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
# html_sidebars = {}


# -- Options for HTMLHelp output ---------------------------------------------

# Output file base name for HTML help builder.
htmlhelp_basename = "blackdoc"


# -- Options for LaTeX output ------------------------------------------------

latex_elements = {
    # The paper size ('letterpaper' or 'a4paper').
    #
    # 'papersize': 'letterpaper',
    # The font size ('10pt', '11pt' or '12pt').
    #
    # 'pointsize': '10pt',
    # Additional stuff for the LaTeX preamble.
    #
    # 'preamble': '',
    # Latex figure (float) alignment
    #
    # 'figure_align': 'htbp',
}

# Grouping the document tree into LaTeX files. List of tuples
# (source start file, target name, title,
#  author, documentclass [howto, manual, or own class]).
latex_documents = [
    (
        master_doc,
        "black.tex",
        "Documentation for Black",
        "Łukasz Langa and contributors to Black",
        "manual",
    )
]


# -- Options for manual page output ------------------------------------------

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
man_pages = [(master_doc, "black", "Documentation for Black", [author], 1)]


# -- Options for Texinfo output ----------------------------------------------

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
texinfo_documents = [
    (
        master_doc,
        "Black",
        "Documentation for Black",
        author,
        "Black",
        "The uncompromising Python code formatter",
        "Miscellaneous",
    )
]


# -- Options for Epub output -------------------------------------------------

# Bibliographic Dublin Core info.
epub_title = project
epub_author = author
epub_publisher = author
epub_copyright = copyright

# The unique identifier of the text. This can be a ISBN number
# or the project homepage.
#
# epub_identifier = ''

# A unique identification for the text.
#
# epub_uid = ''

# A list of files that should not be packed into the epub file.
epub_exclude_files = ["search.html"]


# -- Extension configuration -------------------------------------------------

autodoc_member_order = "bysource"

# -- Options for intersphinx extension ---------------------------------------

# Example configuration for intersphinx: refer to the Python standard library.
intersphinx_mapping = {"https://docs.python.org/3/": None}
