# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['flake8_pantsbuild']
install_requires = \
['flake8>=3.7']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=1.3.0']}

entry_points = \
{u'flake8.extension': ['PB1 = flake8_pantsbuild:Plugin',
                       'PB2 = flake8_pantsbuild:IndentationPlugin',
                       'PB3 = flake8_pantsbuild:TrailingSlashesPlugin',
                       'PB6 = flake8_pantsbuild:SixPlugin']}

setup_kwargs = {
    'name': 'flake8-pantsbuild',
    'version': '1.0.0',
    'description': 'Various Flake8 lints used by the Pants project and its users.',
    'long_description': '# flake8-pantsbuild\n\nThis Flake8 plugin provides several custom lints used by the Pants project and its users.\n\n## Table of Contents\n\n* [Installation](#installation)\n    * [Supported Python versions](#supported-python-versions)\n* [Usage](#usage)\n* [Error Codes](#error-codes)\n* [Migrating from `lint.pythonstyle` to `flake8`](#migrating-from-lintpythonstyle-to-flake8)\n    * [Differences between the tools](#differences-between-the-tools)\n    * [Steps to migrate](#steps-to-migrate)\n    * [Alternatives to custom lints](#alternatives-to-custom-lints)\n* [Development](#development)\n\n## Installation\n\nIf using with Pants, add this to your `pants.toml`:\n\n```toml\n[GLOBAL]\nbackend_packages2.add = ["pants.backend.python.lint.flake8"]\n\n[flake8]\nextra_requirements.add = ["flake8-pantsbuild"]\n```\n\nIf using Flake8 without Pants, install with:\n\n```bash\n$ pip install flake8-pantsbuild\n```\n\n### Supported Python versions\n\nThis plugin works with Python 2.7 and 3.5+.\n\n## Usage\n\nIf using with Pants, run `./pants lint file.py` as usual.\n\nIf using without Pants, run `flake8 file.py` [as usual](http://flake8.pycqa.org/en/latest/user/invocation.html).\n\n## Error Codes\n\n| Error code | Description                                                     | Notes                |\n|:----------:|:---------------------------------------------------------------:|:--------------------:|\n| PB10       | Using class attribute that breaks inheritance                   |                      |\n| PB11       | Using a constant on the left-hand side of a logical operator    |                      |\n| PB12       | Using a constant on the right-hand side of an and operator      |                      |\n| PB13       | Using `open` without a `with` statement (context manager)       |                      |\n| PB20       | Check for 2-space indentation                                   | Disabled by default\xc2\xb9 |\n| PB30       | Using slashes instead of parentheses for line continuation      | Disabled by default\xc2\xb2 |\n| PB60       | Using `print` statements, rather than the `print` function      | Disabled by default\xc2\xb3 |\n| PB61       | Using old style `except` statements instead of the `as` keyword | Disabled by default\xc2\xb3 |\n| PB62       | Using `iteritems`, `iterkeys`, or `itervalues`                  | Disabled by default\xc2\xb3 |\n| PB63       | Using `xrange`                                                  | Disabled by default\xc2\xb3 |\n| PB64       | Using `basestring` or `unicode`                                 | Disabled by default\xc2\xb3 |\n| PB65       | Using metaclasses incompatible with Python 3                    | Disabled by default\xc2\xb3 |\n| PB66       | Using Python 2 old-style classes (not inheriting `object`)      | Disabled by default\xc2\xb3 |\n\n\xc2\xb9 To enable the `PB20` indentation lint, set `--enable-extensions=PB20`. You\'ll need to disable `E111` (check for 4-space indentation) via `--extend-ignore=E111`. You\'ll likely want to disable `E121`, `E124`, `E125`, `E127`, and `E128` as well.\n\n\xc2\xb2 To enable the `PB30` trailing slash lint, set `--enable-extensions=PB30`.\n\n\xc2\xb3 To enable the `PB6*` checks for Python 2->3 lints, set `--enable-extensions=PB6`. \n\n## Migrating from `lint.pythonstyle` to `flake8`\n\n`lint.pythonstyle` was a custom Pants task from the `pants.contrib.python.checks` plugin that behaved like Flake8, but was generally inferior to the more popular Flake8.\n\n### Differences between the tools \n\nPants\' `lint.pythonstyle` task runs `pycodestyle` and `pyflakes`, in addition to providing several custom lints. \n\nIn contrast, Flake8 runs `pycodestyle` and `pyflakes`, but it also uses `mccabe` to check for complex sections of code and it [adds its own lints](https://flake8.pycqa.org/en/latest/user/error-codes.html). Flake8 does not have any of the custom `lint.pythonstyle` lints by default, but the [below table](#alternatives-to-custom-lints) shows how to keep any of these lints you\'d like.\n\nFlake8 has hundreds of plugins that you may easily add to Pants. See [Awesome Flake8 Extensions](https://github.com/DmytroLitvinov/awesome-flake8-extensions) for a curated list of plugins.\n\n### Steps to migrate\n\nFirst, follow the [installation instructions](#installation) to hook up Flake8 with Pants. Then, disable `lint.pythonstyle` by removing `pants.contrib.python.checks` from your `pants.toml`.\n\nThen, run Flake8 by running `./pants lint file.py` (see [Usage](#usage)).\n\nThe first time you run `./pants lint` , you will likely have several errors due to its differences with `lint.pythonstyle`. We recommend starting by [disabling](https://flake8.pycqa.org/en/latest/user/violations.html) those errors in a [`.flake8` config file](https://flake8.pycqa.org/en/latest/user/configuration.html). Add this to your `pants.toml` to ensure Pants uses the config:\n\n```toml\n[flake8]\nconfig = "path/to/.flake8"\n```\n\nIf you want to keep any of the custom lints from `lint.pythonstyle`, refer to the below table. You will need to install additional plugins by adding this to your `pants.toml`:\n\n```toml\n[flake8]\nextra_requirements.add = [\n  "flake8-pantsbuild",\n  "pep8-naming",\n  "flake8-import-order",\n  # and so on...\n]\n```\n\n### Alternatives to custom lints\n\n| Old code | Old option scope            | Description                                                     | Alternative                               |\n|:--------:|:---------------------------:|:---------------------------------------------------------------:|:-----------------------------------------:|\n| T000     | pycheck-variable-names      | Class names must be `UpperCamelCase`                            | `pep8-naming` Flake8 plugin               |\n| T001     | pycheck-variable-names      | Class globals must be `UPPER_SNAKE_CASED`                       | `pep8-naming` Flake8 plugin               |\n| T002     | pycheck-variable-names      | Function names must be `lower_snake_cased`                      | `pep8-naming` Flake8 plugin               |\n| T100     | pycheck-indentation         | Enforce 2-space indentation                                     | `PB20`\xc2\xb9                                   |\n| T200     | pycheck-trailing-whitespace | Trailing whitespace                                             | Flake8\'s `W291`                           |\n| T201     | pycheck-trailing-whitespace | Using slashes instead of parentheses for line continuation      | `PB30`\xc2\xb9                                   |\n| T301     | pycheck-newlines            | Excepted 1 blank line between class methods                     | Flake8\'s `E301` and `E303`                |\n| T302     | pycheck-newlines            | Excepted 2 blank lines between top level definitions            | Flake8\'s `E302` and `E303`                |\n| T400     | pycheck-import-order        | Wildcard imports not allowed                                    | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T401     | pycheck-import-order        | `from` imports not sorted correctly                             | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T402     | pycheck-import-order        | `import` should only import one module at a time                | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T403     | pycheck-import-order        | Importing multiple module types in one statement                | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T404     | pycheck-import-order        | Unclassifiable import                                           | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T405     | pycheck-import-order        | Import block has multiple module types                          | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T406     | pycheck-import-orde         | Out of order import statements                                  | `isort` or `flake8-import-order` plugin \xc2\xb2 |\n| T601     | pycheck-except-statement    | Using old style `except` statements instead of the `as` keyword | `PB61`\xc2\xb9                                   |\n| T602     | pycheck-future-compat       | Using `iteritems`, `iterkeys`, or `itervalues`                  | `PB62`\xc2\xb9                                   |\n| T603     | pycheck-future-compat       | Using `xrange`                                                  | `PB63`\xc2\xb9                                   |\n| T604     | pycheck-future-compat       | Using `basestring` or `unicode`                                 | `PB64`\xc2\xb9                                   |\n| T605     | pycheck-future-compat       | Using metaclasses incompatible with Python 3                    | `PB65`\xc2\xb9                                   |\n| T606     | pycheck-new-style-classes   | Found Python 2 old-style classes (not inheriting `object`)      | `PB66`\xc2\xb9                                   |\n| T607     | pycheck-print-statements    | Using `print` statements, rather than the `print` function      | `PB60`\xc2\xb9                                   |\n| T800     | pycheck-class-factoring     | Using class attribute that breaks inheritance                   | `PB10`                                    |\n| T801     | pycheck-variable-names      | Shadowing a `builtin` name                                      | `flake8-builtins` plugin                  |\n| T802     | pycheck-context-manager     | Using `open` without a `with` statement (context manager)       | `PB13`                                    |\n| T803     | pycheck-except-statement    | Using a blanket `except:`                                       | Flake8\'s `E722`                           |\n| T804     | pycheck-constant-logic      | Using a constant on the left-hand side of a logical operator    | `PB11`                                    |\n| T805     | pycheck-constant-logic      | Using a constant on the right-hand side of an and operator      | `PB12`                                    |\n\n\xc2\xb9 This lint is disabled by default. See the above section [`Error Codes`](#error-codes) for instructions on how to enable this lint.\n\n\xc2\xb2 To use `isort` with Pants, set `backend_packages2.add = ["pants.backend.python.lint.isort"]` in your `pants.toml`.\n\n## Development\n\nWe use [tox](https://testrun.org/tox/en/latest/) for test automation. To run the test suite, invoke tox:\n\n```bash\n$ tox\n```\n\nYou may run certain environments with `tox -e` (run `tox -a` to see all options):\n\n```bash\n$ tox -e format-run\n$ tox -e py27\n$ tox -e py36\n```\n\nYou may also manually test by building a [PEX](https://github.com/pantsbuild/pex) with this plugin, as follows:\n\n```bash\n$ pex flake8 . --entry-point flake8 --output-file flake8.pex\n$ vim example.py  # add whatever content you want\n$ ./flake.pex example.py\n```\n',
    'author': 'Pantsbuild developers',
    'author_email': 'pantsbuild@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://pants.readme.io/docs/python-linters-and-formatters',
    'py_modules': modules,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
